#ifndef FrameCPP_VERSION_8_FrTOCAdcData_HH
#define FrameCPP_VERSION_8_FrTOCAdcData_HH

#include "framecpp/Version7/FrTOCAdcData.hh"

#include "framecpp/Common/FrTOC.hh"
#include "framecpp/Version8/STRING.hh"

namespace FrameCPP
{
  namespace Common
  {
    class TOCInfo;
  }

  namespace Version_8
  {
    //===================================================================
    /// \brief Indexed elements of the FrAdcData structure.
    //===================================================================
    class FrTOCAdcData
    {
    public:
      typedef Previous::FrTOCAdcData::nadc_type		nadc_type;
      typedef Previous::FrTOCAdcData::name_type		name_type;
      typedef Previous::FrTOCAdcData::channel_id_type	channel_id_type;
      typedef Previous::FrTOCAdcData::group_id_type	group_id_type;
      typedef Previous::FrTOCAdcData::position_type	position_type;

      typedef Previous::FrTOCAdcData::adc_info_type	adc_info_type;

      typedef std::vector< name_type >	key_container_type;

      //-----------------------------------------------------------------
      /// \brief Container of indexed FrAdcData elements.
      ///
      /// This structure is ordered by the name of the FrAdcData
      /// channel name.
      /// There is one element for each channel of indexed data.
      //-----------------------------------------------------------------
      typedef std::map< name_type, adc_info_type >	MapADC_type;

      //-----------------------------------------------------------------
      /// \brief Default constructor.
      //-----------------------------------------------------------------
      FrTOCAdcData( );

      //-----------------------------------------------------------------
      /// \brief Number of bytes needed to write this structure
      ///
      /// \param[in] Stream
      ///     The stream from which to the object is being read or
      ///     written.
      ///
      /// \return
      ///     The number of bytes need to read or write this object.
      //-----------------------------------------------------------------
      Common::FrameSpec::size_type
      Bytes( const Common::StreamBase& Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Return all channels
      ///
      /// \return
      ///     Constant container to the indexed FrAdcData elements.
      //-----------------------------------------------------------------
      const MapADC_type& GetADC( ) const;

      //-----------------------------------------------------------------
      /// \brief Return the indexed data for a named channel.
      ///
      /// \param[in] Name
      ///     The name of the channel.
      ///
      /// \return
      ///     Constant iterator to the indexed FrAdcData element.
      //-----------------------------------------------------------------
      MapADC_type::const_iterator GetADC( const std::string& Name ) const;

      //-----------------------------------------------------------------
      /// \brief Return the indexed data for an indexed channel.
      ///
      /// \param[in] index
      ///     The numeric offset of the channel.
      ///
      /// \return
      ///     Constant iterator to the indexed FrAdcData element.
      //-----------------------------------------------------------------
      MapADC_type::const_iterator GetADC( INT_4U index ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the keys
      ///
      /// \return
      ///     Constant container of keys.
      //-----------------------------------------------------------------
      const key_container_type& GetKeys( ) const;

      //-----------------------------------------------------------------
      /// \brief Gather TOC info for FrAdcData being written.
      ///
      /// \param[in] Info
      ///     Information
      ///
      /// \param[in] FrameOffset
      ///     The frame offset of the frame being written.
      ///     The frame offsets start at 0 (zero).
      ///
      /// \param[in] Position
      ///     The byte offset from the start of the file
      ///     where the structure is written.
      //-----------------------------------------------------------------
      void QueryAdc( const Common::TOCInfo& Info,
		     INT_4U FrameOffset,
		     INT_8U Position );

      //-----------------------------------------------------------------
      /// \brief asignment operator
      ///
      /// \param[in] Source
      ///     The source to be copied.
      //-----------------------------------------------------------------
      const FrTOCAdcData& operator=( const Previous::FrTOCAdcData& Source );

      //-----------------------------------------------------------------
      /// \brief equality operator
      ///
      /// \param[in] RHS
      ///     The FrTOCAdcData object to be compared.
      ///
      /// \return
      ///     The value true is returned if this object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      bool operator==( const FrTOCAdcData& RHS ) const;

      //-----------------------------------------------------------------
      /// \brief The description of structure
      ///
      /// \return
      ///     A Description object which describes this structure as
      ///     specified by the frame specification.
      //-----------------------------------------------------------------
      template< typename SE >
      static void Description( Common::Description& Desc );

      // Offset from the end of file to the 
      // start of the array of ADC positions in the TOC
      position_type		m_positions_start;
    protected:
      key_container_type	m_keys;
      MapADC_type		m_info;

      //-----------------------------------------------------------------
      /// \brief Stream constructor
      ///
      /// \param[in] Stream
      ///     The input stream from where the object is being read.
      ///
      /// \param[in] FrameCount
      ///     The frame offset.
      //-----------------------------------------------------------------
      FrTOCAdcData( Common::IStream& Stream, INT_4U FrameCount );

      //-----------------------------------------------------------------
      /// \brief Iterate over contents.
      ///
      /// \param[in] Info
      ///     Specifies the type of information to be searched.
      ///
      /// \param[in] Action
      ///     Action to be taken for each piece of information found.
      //-----------------------------------------------------------------
      void forEach( Common::FrTOC::query_info_type Info,
		    Common::FrTOC::FunctionBase& Action ) const;

      //-----------------------------------------------------------------
      /// \brief Cache where the positions of the Adc channels
      ///
      /// \param[in,out] Stream
      ///     The Stream being read
      //-----------------------------------------------------------------
      void cachePositions( istream_type& Stream );

      //-----------------------------------------------------------------
      /// \brief Advance to the specified Adc channel
      ///
      /// \param[in,out] Stream
      ///     The Stream being read
      /// \param[in] Channel
      ///     The requested channel
      //-----------------------------------------------------------------
      void seekPositions( istream_type& Stream,
			  channel_id_type Channel );

      //-----------------------------------------------------------------
      /// \brief Advance to the specified Adc channel
      ///
      /// \param[in,out] Stream
      ///     The Stream being read
      /// \param[in] Channel
      ///     The requested channel
      //-----------------------------------------------------------------
      void seekPositions( istream_type& Stream,
			  const std::string& Channel );

      //-----------------------------------------------------------------
      /// \brief Write the structure to the stream
      ///
      /// \param[in] Stream
      ///     The output stream where the object is to be written.
      //-----------------------------------------------------------------
      void write( Common::OStream& Stream ) const;

    private:
      typedef General::unordered_map< std::string, INT_4U > reverse_lookup_type;
      typedef std::vector< position_type > positions_cache_type;

      MapADC_type& getADC( );

      MapADC_type::iterator getADC( const std::string& Name );

      MapADC_type::iterator getADC( INT_4U index );

      //-----------------------------------------------------------------
      /// A cache of the positions of the FrAdcData structures within
      /// the stream.
      //-----------------------------------------------------------------
      positions_cache_type		positions_cache;
      //-----------------------------------------------------------------
      /// The number of bytes from the end of the stream where the
      ///   array containing the FrAdcData offsets.
      //-----------------------------------------------------------------
      position_type			positions_cache_offset;

      //-----------------------------------------------------------------
      /// Provide means where a channel index can be mapped to a
      /// channel name
      //-----------------------------------------------------------------
      mutable reverse_lookup_type	reverse_lookup_;

      reverse_lookup_type::mapped_type
      reverse_lookup( const reverse_lookup_type::key_type& key ) const;
    };

    inline Common::FrameSpec::size_type FrTOCAdcData::
    Bytes( const Common::StreamBase& Stream ) const
    {
      Common::FrameSpec::size_type 
	retval = sizeof( nadc_type );
      if ( m_info.size( ) )
      {
	retval
	  += ( m_info.size( )
	       * ( sizeof( channel_id_type )
		   + sizeof( group_id_type )
		   + ( sizeof( position_type )
		       * m_info.begin( )->second.m_positionADC.size( ) )
		   ) );
      }
      for ( MapADC_type::const_iterator
	      cur = m_info.begin( ),
	      last = m_info.end( );
	    cur != last;
	    ++cur )
      {
	retval += cur->first.Bytes( );
      }
      return retval;
    }

    template< typename SE >
    void FrTOCAdcData::
    Description( Common::Description& Desc )
    {
      Desc( SE( "nADC", "INT_4U",
		"Number of unique FrAdcData names in file." ) );
      Desc( SE( "name", "STRING[nADC]", "Array of FrAdcData names" ) );
      Desc( SE( "channelID", "INT_4U[nADC]",
		"Array of ADC channel IDs" ) );
      Desc( SE( "groupID", "INT_4U[nADC]",
		"Array of ADC group IDs" ) );
      Desc( SE( "positionADC", "INT_8U[nADC][nFrame]",
		"Array of lists of FrAdcData offset positions, in bytes,"
		" from beginning of file (size of nFrame*nADC)"
		) );
    }

    inline const FrTOCAdcData::MapADC_type& FrTOCAdcData::
    GetADC( ) const
    {
      return m_info;
    }

    inline const FrTOCAdcData::key_container_type& FrTOCAdcData::
    GetKeys( ) const
    {
      return m_keys;
    }

    inline FrTOCAdcData::MapADC_type& FrTOCAdcData::
    getADC( )
    {
      return m_info;
    }

    inline FrTOCAdcData::MapADC_type::iterator FrTOCAdcData::
    getADC( const std::string& Channel )
    {
      MapADC_type&
	i( getADC( ) );

      return i.find( Channel );
    }

    inline FrTOCAdcData::MapADC_type::iterator FrTOCAdcData::
    getADC( INT_4U Channel )
    {
      MapADC_type&
	i( getADC( ) );

      if ( Channel >= i.size( ) )
      {
	return i.end( );
      }
      return i.find( m_keys[ Channel ] );
    }

    inline void FrTOCAdcData::
    cachePositions( istream_type& Stream )
    {
      std::streampos	here( Stream.tellg( ) );
      
      Stream.seekg( - positions_cache_offset , Stream.end );
      Stream >> positions_cache;
      Stream.seekg( here, Stream.beg );
    }

    inline void FrTOCAdcData::
    seekPositions( istream_type& Stream,
		   channel_id_type Channel )
    {
      const INT_4U	frame_count( positions_cache.size( ) / m_keys.size( ) );
      const INT_4U	offset( Channel * frame_count );

      if ( positions_cache[ offset ] != 0 )
      {
	std::copy( &(positions_cache[ offset ]),
		   &(positions_cache[ offset + frame_count]),
		   &(getADC( Channel )->second.m_positionADC[ 0 ]) );
	positions_cache[ offset ] = 0;
      }

    }

    inline void FrTOCAdcData::
    seekPositions( istream_type& Stream,
		   const std::string& Channel )
    {
      const INT_4U	frame_count( positions_cache.size( ) / m_keys.size( ) );
      const INT_4U	offset( reverse_lookup( Channel ) * frame_count );

      if ( positions_cache[ offset ] != 0 )
      {
	std::copy( &(positions_cache[ offset ]),
		   &(positions_cache[ offset + frame_count]),
		   &(getADC( Channel )->second.m_positionADC[ 0 ]) );
	positions_cache[ offset ] = 0;
      }
    }

    inline FrTOCAdcData::reverse_lookup_type::mapped_type FrTOCAdcData::
    reverse_lookup( const reverse_lookup_type::key_type& Key ) const
    {
      if ( 0 == reverse_lookup_.size( ) )
      {
	reverse_lookup_type::mapped_type index = 0;
	for ( key_container_type::const_iterator
		cur = m_keys.begin( ),
		last = m_keys.end( );
	      cur != last;
	      ++cur, ++index )
	{
	  reverse_lookup_[ *cur ] = index;
	}
      }
      return reverse_lookup_[ Key ];
    }
  } // namespace - Version_8
} // namespace - FrameCPP

#endif /* FrameCPP_VERSION_8_FrTOCAdcData_HH */
