#include "framecpp/config.h"

#include <assert.h>

#include <stdexcept>
#include <iostream>

#include "general/AtExit.hh"
#include "general/gpstime.hh"
#include "general/mutexlock.hh"

#include "framecpp/Common/Description.hh"
#include "framecpp/Common/FrameSpec.hh"

#include "framecpp/Common/IOStream.hh"

#include "framecpp/Common/FrDetector.hh"
#include "framecpp/Common/FrStatData.hh"

#include "framecpp/Common/FrameSpec.tcc"

using General::GPSTime;

namespace FrameCPP
{
  namespace Common
  {
    //-------------------------------------------------------------------
    /// \brief Collection of frame specifications
    ///
    /// This is a lookup table on the implementation details for
    /// of defined frame specifications
    //-------------------------------------------------------------------
    class FrameSpec::frame_spec_container
      : public General::unordered_map< version_type,
				       Info* >
    {
    public:
      //-----------------------------------------------------------------
      /// \brief Default constructor
      //-----------------------------------------------------------------
      frame_spec_container( );
      //-----------------------------------------------------------------
      /// \brief Destructor
      //-----------------------------------------------------------------
      virtual ~frame_spec_container( );
    };

    //-------------------------------------------------------------------
    /// Initialize the resources needed to manage the collection of
    /// frame specification details.
    //-------------------------------------------------------------------
    FrameSpec::frame_spec_container::
    frame_spec_container( )
    {
    }

    //-------------------------------------------------------------------
    /// Release the resources associated with managing the collection of
    /// frame specification details.
    //-------------------------------------------------------------------
    FrameSpec::frame_spec_container::
    ~frame_spec_container( )
    {
      for ( const_iterator
	      cur = begin( ),
	      last = end( );
	    cur != last;
	    ++cur )
      {
	delete cur->second;
      }
      erase( begin( ), end( ) );
    }

    //-------------------------------------------------------------------
    /// Release the resources back to the system.
    /// By being virtual, all the resources associated with any derived
    /// class are given a chance for proper cleanup.
    //-------------------------------------------------------------------
    FrameSpec::ObjectInterface::
    ~ObjectInterface( )
    {
    }

    //-------------------------------------------------------------------
    /// Allocation of the object's resources.
    //-------------------------------------------------------------------
    FrameSpec::Object::
    Object( class_type Class, const Description* Description )
      : m_class( Class ),
	m_desc( Description )
    {
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    FrameSpec::Object::
    Object( const Object& Source )
      : ObjectInterface( Source ),
	m_class( Source.m_class ),
	m_desc( Source.m_desc )
    {
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    FrameSpec::Object::
    ~Object( )
    {
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    ostream_type& FrameSpec::Object::
    WriteNext( ostream_type& Stream ) const
    {
      Stream.NextPtrStruct( this ).Write( Stream );
      return Stream;
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    FrameSpec::class_type FrameSpec::Object::
    getClass( ) const
    {
      return GetClass( );
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    FrameSpec::Object* FrameSpec::ObjectInterface::
    Create( ) const
    {
      std::ostringstream	msg;
      const Description* 	d( GetDescription( ) );

      msg << "Cannot create object of class: " << getClass( );
      if ( d )
      {
	msg << " (" << d->GetName( ) << ")";
      }


      throw std::runtime_error( msg.str( ) );
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    FrameSpec::Object* FrameSpec::ObjectInterface::
    Create( std::istream& Buffer ) const
    {
      std::ostringstream	msg;
      const Description* 	d( GetDescription( ) );

      msg << "Cannot create object of class: " << getClass( )
	;
      if ( d )
      {
	msg << " (" << d->GetName( ) << ")";
      }
      msg << " from std::istream"
	;



      throw std::runtime_error( msg.str( ) );
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    FrameSpec::Object* FrameSpec::ObjectInterface::
    Clone( ) const
    {
      const Description* d( GetDescription( ) );
      std::ostringstream	msg;

      msg << "Cannot clone object of class: " << getClass( );
      if ( d )
      {
	msg << " (" << d->GetName( ) << ")";
      }

      throw std::runtime_error( msg.str( ) );
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    FrameSpec::ObjectInterface::object_type FrameSpec::ObjectInterface::
    DemoteObject(  INT_2U Target, object_type Obj, istream_type* Stream ) 
    {
      object_type	retval;
      object_type	previous;
      object_type	current( Obj );

      do
      {
	previous = retval;
	retval = current->demote( Target, current, Stream );
	current = retval;
      } while ( ( current ) && ( current != previous ) );
      return retval;
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    FrameSpec::ObjectInterface::object_type FrameSpec::ObjectInterface::
    PromoteObject( INT_2U Target, INT_2U Source,
		   object_type Obj, istream_type* Stream )
    {
      object_type retval;
      if ( ! Obj )
      {
	return retval;
      }

      //-----------------------------------------------------------------
      // Obtain information about the targeted framespec
      //-----------------------------------------------------------------
      const FrameSpec::Info* info( FrameSpec::SpecInfo( Target ) );

      if ( info )
      {
	const Object*
	  target( info->FrameObject( FrameSpec::Info::
				     frame_object_types( Obj->GetClass( ) ) ) );
	if ( target )
	{
	  //-------------------------------------------------------------
	  // Ask the highest object to upconvert. This is recursive so
	  //  earlier versions do not need to know about later versions.
	  //-------------------------------------------------------------
	  retval = target->promote( Source, Obj, Stream );
	}
      }
      return retval;
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    void FrameSpec::ObjectInterface::
    VerifyObject( Verify& Verifier, IFrameStream& Stream ) const
    {
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    FrameSpec::Info::
    Info( version_type MajorVersion, version_type MinorVersion,
	  verification_func_type VerificationFunc )
      : m_version( MajorVersion ),
	m_version_minor( MinorVersion ),
	m_verification_func( VerificationFunc )
    {
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    FrameSpec::Info::
    ~Info( )
    {
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    const FrameSpec::Object* FrameSpec::Info::
    FrameObject( frame_object_types ObjectId ) const
    {
      object_container::const_iterator retval
	= m_definitions.find( ObjectId );
      if ( retval == m_definitions.end( ) )
      {
	std::ostringstream	msg;

	msg << "Unable to locate Object: " << ObjectId
	    << " for Version: " << Version( )
	    << " of the frame specification"
	  ;
	throw std::range_error( msg.str( ) );
      }
      return retval->second.get( );
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    void FrameSpec::Info::
    FrameObject( object_type ObjectTemplate )
    {
      m_definitions[ ObjectTemplate->GetClass( ) ] = ObjectTemplate;
    }
    
    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    inline FrameSpec::frame_spec_container& FrameSpec::
    frame_specs( )
    {
      static frame_spec_container m_frame_specs;

      return m_frame_specs;
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    FrameSpec::Info* FrameSpec::
    SpecInfo( version_type Version )
    {
      frame_spec_container::const_iterator retval
	= frame_specs( ).find( Version );
      
      if ( retval == frame_specs( ).end( ) )
      {
	std::ostringstream	msg;

	msg << "No frame definition found for version " << Version 
	    << " of the frame specification"
	    << std::endl;
	assert( 0 );
	throw std::range_error( msg.str( ) );
      }
      return retval->second;
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    void FrameSpec::
    SpecInfo( version_type Version, FrameSpec::Info* I )
    {
      if ( I )
      {
	static MutexLock::lock_type	key = MutexLock::Initialize( );

	MutexLock		lock( key );

	frame_specs( )[ Version ] = I;
      }
    }

    //-------------------------------------------------------------------
    /// Returns the number of bytes required to read or write a GPS
    /// time structure.
    //-------------------------------------------------------------------
    template<> FrameSpec::size_type
    Bytes< GPSTime >( const GPSTime& Source )
    {
      return
	sizeof( INT_4U )
	+ sizeof( INT_4U )
	;
    }

    //-------------------------------------------------------------------
    ///
    //-------------------------------------------------------------------
    FrameSpec::
    FrameSpec( )
    {
    }
#define INSTANTIATE( LM_TYPE, LM_CRC_TYPE )	\
    template FrameSpec::ObjectWithChecksum< LM_TYPE, LM_CRC_TYPE > \
    ::~ObjectWithChecksum( )

    INSTANTIATE( FrameSpec::Object, INT_4U );
    INSTANTIATE( FrDetector, INT_4U );
    INSTANTIATE( FrStatData, INT_4U );

#undef INSTANTIATE

  } // namespace - Common

} // namespace - FrameCPP
