#ifndef FRAME_CPP__Common__CHECKSUM_HH
#define	FRAME_CPP__Common__CHECKSUM_HH

#include "framecpp/config.h"

#if HAVE_ENDIAN_H
#include <endian.h>
#else /* HAVE_ENDIAN_H */
#ifndef __BIG_ENDIAN
#define __BIG_ENDIAN 0xFF00
#endif /* __BIG_ENDIAN */
#ifndef __LITTLE_ENDIAN
#define __LITTLE_ENDIAN 0x00FF
#endif /* __LITTLE_ENDIAN */

#if WORDS_BIGENDIAN
#define __BYTE_ORDER __BIG_ENDIAN
#else /* WORDS_BIGENDIAN */
#define __BYTE_ORDER __LITTLE_ENDIAN
#endif /* WORDS_BIGENDIAN */

#endif /* HAVE_ENDIAN_H */

#include <stdint.h>

#include "general/types.hh"
#include "general/reverse.hh"

#include "framecpp/Common/CRC.hh"

namespace FrameCPP
{
  namespace Common
  {
    class CheckSum
    {
    public:
      enum kind_type {
	UNSET = -1,
	NONE = 0,
	CRC = 1
      };

      typedef INT_4U size_type;
      typedef INT_4U value_type;

      CheckSum( const kind_type Type );

      virtual ~CheckSum( );

      static CheckSum* Create( kind_type Type );

      static CheckSum* Create( kind_type Type, value_type Value );

      kind_type GetType( ) const;

      virtual void Reset( );

      virtual void calc( const void* Buffer, size_type NBytes ) = 0;

      //: Things to be do to complete the calculation
      value_type value( );

      virtual CheckSum* Clone( ) const = 0;

      static std::string FormatError( value_type ExpectedValue,
				      value_type CalculatedValue );

    protected:
      value_type	m_value;

      CheckSum( const CheckSum& Source );
      virtual void finish( ) = 0;

    private:
      kind_type		m_type;
      CheckSum( );
    };

    inline CheckSum::kind_type CheckSum::
    GetType( ) const
    {
      return m_type;
    }

    inline CheckSum::value_type CheckSum::
    value( )
    {
      finish( );
      return m_value;
    }

    class CheckSumCRC
      : public CheckSum
    {
    public:
      typedef CheckSum::value_type value_type;

      CheckSumCRC( );

      CheckSumCRC( value_type Value );

      virtual CheckSumCRC* Clone( ) const;

      virtual void Reset( );

      virtual void calc( const void* Buffer, size_type NBytes );

    protected:
      CheckSumCRC( const CheckSumCRC& Source );
      virtual void finish( );

    private:
      static INT_4U		m_crctab_slow[256];
      static INT_4U		m_crctab[8][ 256 ];
      static const bool		initialized;

      static bool initializer( );

      void slice_by_1( const void* Data, size_t Length );

      void slice_by_8( const void* Data, size_t Length );

      bool		m_active;
      size_type		m_number_of_bytes;
    };

    inline void CheckSumCRC::
    slice_by_1( const void* Data, size_t Length )
    {
      const uint8_t* cp( reinterpret_cast< const uint8_t* >( Data ) );
      while( Length-- )
      {
	m_value = (m_value >> 8) ^ m_crctab[0][(*cp ^ m_value) & 0xFF];
	++cp;
      }
    }

    inline void CheckSumCRC::
    slice_by_8( const void* Data, size_t Length )
    {
      static const size_t SLICE = 8;
      const uint8_t*
	current_byte( reinterpret_cast< const uint8_t*>( Data ) );
      size_t alignment( ( ( ~ reinterpret_cast< long >( Data ) + 1 ) & 0x03 ) );

      //-----------------------------------------------------------------
      // align on 32-bit boundry for effecient memory access
      //-----------------------------------------------------------------
      if ( alignment )
      {
	register size_t l( std::min( alignment, Length ) );

	slice_by_1( Data, l );
	Length -= l;
	current_byte += l;
      }
      //-----------------------------------------------------------------
      // process eight bytes at once (Slicing-by-8)
      //-----------------------------------------------------------------
      const uint32_t*
	current( reinterpret_cast< const uint32_t*>( current_byte ) );
      while ( Length >= SLICE )
      {
#if __BYTE_ORDER == __BIG_ENDIAN
	INT_4U one = *current++ ^ bswap_32(m_value);
	INT_4U two = *current++;

	m_value  =
	  m_crctab[4][ one      & 0xFF] ^
	  m_crctab[5][(one>> 8) & 0xFF] ^
	  m_crctab[6][(one>>16) & 0xFF] ^
	  m_crctab[7][(one>>24) & 0xFF] ^
	  m_crctab[0][ two      & 0xFF] ^
	  m_crctab[1][(two>> 8) & 0xFF] ^
	  m_crctab[2][(two>>16) & 0xFF] ^
	  m_crctab[3][(two>>24) & 0xFF];
#else
	INT_4U one = *current++ ^ m_value;
	INT_4U two = *current++;

	m_value  =
	  m_crctab[7][ one      & 0xFF] ^
	  m_crctab[6][(one>> 8) & 0xFF] ^
	  m_crctab[5][(one>>16) & 0xFF] ^
	  m_crctab[4][(one>>24) & 0xFF] ^
	  m_crctab[3][ two      & 0xFF] ^
	  m_crctab[2][(two>> 8) & 0xFF] ^
	  m_crctab[1][(two>>16) & 0xFF] ^
	  m_crctab[0][(two>>24) & 0xFF]
	  ;
#endif

	Length -= SLICE;
      }

      slice_by_1( current, Length );
    }
  } // namespace - Common
} // namespace - FrameCPP

#endif	/* FRAME_CPP__Common__CHECKSUM_HH */
