#include "config.h"

#include <time.h>

#include "general/AtExit.hh"

#include "diskcacheAPI/Cache/Directory.hh"
#include "diskcacheAPI/Cache/HotDirectory.hh"

#include "DirectoryManagerSingleton.hh"
#include "MountPointScanner.hh"

using diskCache::Cache::HotDirectory;

static bool initialized = false;

namespace diskCache
{
  //=====================================================================
  //=====================================================================
  MountPointScanner::ScanResults::
  ScanResults( )
    : m_count_baton( MutexLock::Initialize( ) ),
      m_directory_count( 0 ),
      m_file_count( 0 ),
      m_mount_point_count( 0 )
  {
  }

  //=====================================================================
  //=====================================================================

  MutexLock::lock_type	MountPointScanner::SyncRequest::p_request_count_baton
  = MutexLock::Initialize( );
  MountPointScanner::SyncRequest::request_count_type
  MountPointScanner::SyncRequest::p_request_count = 0;

  MountPointScanner::SyncRequest::
  SyncRequest( )
  {
    static const char* caller = "diskCache::MountPointScanner::SyncRequest::SyncRequest";
    MutexLock  l( p_request_count_baton );

    ++p_request_count;

    std::ostringstream msg;

    msg << "There are now " << p_request_count
	<< " pending request(s)"
      ;
    GenericAPI::queueLogEntry( msg.str( ),
			       GenericAPI::LogHTML::MT_DEBUG,
			       30,
			       caller,
			       "IDLE" );
  }

  MountPointScanner::SyncRequest::
  ~SyncRequest( )
  {
    static const char* caller = "diskCache::MountPointScanner::SyncRequest::~SyncRequest";
    MutexLock  l( p_request_count_baton );

    if ( p_request_count > 0 )
    {
      --p_request_count;
    }

    std::ostringstream msg;

    msg << "There are now " << p_request_count
	<< " pending request(s)"
      ;
    GenericAPI::queueLogEntry( msg.str( ),
			       GenericAPI::LogHTML::MT_DEBUG,
			       30,
			       caller,
			       "IDLE" );
  }

  MountPointScanner::SyncRequest::request_count_type MountPointScanner::SyncRequest::
  PendingRequests( )
  {
    static const char* caller = "diskCache::MountPointScanner::SyncRequest::PendingRequests";

    MutexLock  l( p_request_count_baton );

    std::ostringstream msg;

    msg << "There are now " << p_request_count
	<< " pending request(s)"
      ;
    GenericAPI::queueLogEntry( msg.str( ),
			       GenericAPI::LogHTML::MT_DEBUG,
			       50,
			       caller,
			       "IDLE" );

    return p_request_count;
  }

  //=====================================================================
  //=====================================================================
  MutexLock::lock_type		MountPointScanner::m_variable_baton = MutexLock::Initialize( );
  General::ConditionalVariable	MountPointScanner::m_scan_completed;
  MutexLock::lock_type          MountPointScanner::m_active_count_baton = MutexLock::Initialize( );
  MountPointScanner::concurrency_type MountPointScanner::m_active_count = 0;
  MountPointScanner::concurrency_type MountPointScanner::m_concurrency = 10;
  MutexLock::lock_type          MountPointScanner::m_first_scan_completed_baton = MutexLock::Initialize( );
  bool				MountPointScanner::m_first_scan_completed = false;
  MutexLock::lock_type          MountPointScanner::m_mount_point_cache_reset_baton = MutexLock::Initialize( );
  bool				MountPointScanner::m_mount_point_cache_reset = false;

  MountPointScanner::scanner_sync_type MountPointScanner::p_scanner_sync = true;
  MountPointScanner::scanner_sync_rw_type::lock_type MountPointScanner::p_scanner_sync_baton
  = General::ReadWriteLock::Initialize( );
  

  //---------------------------------------------------------------------
  /// This method allows for cooperative thread cancellation.
  /// Threads that may block the scanning of the mount points,
  /// should periodicly check if a request to terminate the current
  /// scan has been made.
  /// This allows threads to cancel themselves when it may be easier
  /// to perform any cleanup that needs to occur.
  //---------------------------------------------------------------------
  bool MountPointScanner::
  CancellationRequest( std::string& Reason )
  {
    std::ostringstream	msg;
    bool		retval = false;

    if ( retval == false )
    {
      MutexLock		l1( m_mount_point_cache_reset_baton );

      if ( m_mount_point_cache_reset == true )
      {	
	msg << "Cancelling because a request has been made to rebuild the database."
	  ;
	retval = true;
      }
    }
    if ( retval == false )
    {
      const int r = SyncRequest::PendingRequests( );

      if ( r > 0 )
      {
	msg << "Cancellation because "
	    << r
	    << " request" << ( ( r == 1 ) ? " is" : "s are" )
	    << " pending"
	  ;
	retval = true;
      }
    }
    if ( retval == false )
    {
      if ( General::AtExit::IsExiting( ) == true )
      {
	msg << "Cancelling because the system is shuting down"
	  ;
      }
    }

    if ( retval )
    {
      Reason = msg.str( );
    }
    return ( retval );
  }

  void MountPointScanner::
  OnCompletion( int TaskThreadState )
  {
#if WORKING
    static const char caller[] = "MountPointScanner::onCompletion";
#endif /* WORKING */

    if ( ( Results( ).empty( ) == false )
	 && ( General::AtExit::IsExiting( ) == false ) )
    {
      /// \todo
      ///     Report scanning results
#if WORKING
      if ( m_stream )
      {
	MutexLock      lock( m_scan_completed.Mutex( ) );

	(*m_stream) << Results( ) << std::endl;
      }
      else
      {
	GenericAPI::queueLogEntry( Results( ),
				   GenericAPI::LogEntryGroup_type::MT_NOTE,
				   10,
				   caller,
				   "SCAN_MOUNTPT" );
      }
#endif /* WORKING */
    }
    if ( m_controller )
    {
      General::ThreadPool::Relinquish( m_controller );
      m_controller = (General::TaskThread*)NULL;
    }
  }

  void MountPointScanner::
  Scan( const mount_point_container_type& MountPoints,
	ScanResults& Answer )
  {
    //-----------------------------------------------------------
    // Start recording when the start of this action
    //-----------------------------------------------------------
    Answer.Start( );

    //-----------------------------------------------------------
    // Acquire the sync lock
    //-----------------------------------------------------------
    scanner_sync_ro_type	slock( SyncRO( ) );

    if ( initialized == false )
    {
      General::AtExit::Append( on_exit, "diskCache::MountPointScanner::on_exit", 200 );
      initialized = true;
    }
    mount_point_container_type::const_iterator
      cur = MountPoints.begin( ),
      last = MountPoints.end( );

    MountPointScanner::CriticalSection( true );
    while ( ( ( cur != last )
	      || ( Active( ) > 0 ) ) )
    {
      //---------------------------------------------------------
      // Check to see if scanning should be stopped.
      //---------------------------------------------------------
      if ( ( cur != last )
	   && ( ( SyncRequest::PendingRequests( ) > 0 )
		|| ( mount_point_reset( ) ) 
		|| ( General::AtExit::IsExiting( ) == true ) ) )
      {
	cur = last;
      }
      //---------------------------------------------------------
      // Check to see if there is room to start another scanning
      // thread.
      //---------------------------------------------------------
      while ( ( Active( ) < Concurrency( ) )
	      && ( cur != last ) )
      {
	start( *cur, Answer );
	++cur; // Advance to the mount point to scan
      }
      //---------------------------------------------------------
      // Wakeup once a thread has completed.
      //---------------------------------------------------------
      // MountPointScanner::CriticalSection( true );
      if ( Active( ) )
      {
	MountPointScanner::Wait( );
      }
      // MountPointScanner::CriticalSection( false );
    }
    if ( mount_point_reset( ) )
    {
      //---------------------------------------------------------
      // Check to see if the mount point cache is to be rebuilt.
      // This needs to be above lock acquisition since the
      // MountPointHash needs to know we are not going to be
      // updating while it is trying to remove.
      //---------------------------------------------------------
      const mtime_type
	bound( HotDirectory::HotLowerBound( ) );

      HotDirectory::HotLowerBound( 0 );

      FirstScanSet( false );
      try
      {
	HotDirectory::Reset( );
#if WORKING
	DirectoryHash::Instance( ).removeDirectories( );
	MountPointHash::Reset( );
#endif /* WORKING */
	HotDirectory::HotLowerBound( bound );
      }
      catch( ... )
      {
	HotDirectory::HotLowerBound( bound );
	throw;
      }
      mount_point_reset( false );
      MountPointManagerSingleton::Reset( MountPointManagerSingleton::RESET_CACHE );
    }
    else
    {
      FirstScanSet( true );
    }
    MountPointScanner::CriticalSection( false );
    Answer.Stop( );
  }

  void MountPointScanner::
  operator()( )
  {
    try
    {
      DirectoryManagerSingleton::ScanResults      r;

      DirectoryManagerSingleton::Scan( m_mount_point, r );
      m_scan_results.FileCountInc( r.FileCount( ) );
      m_scan_results.DirectoryCountInc( r.DirectoryCount( ) );
      m_scan_results.MountPointCountInc( );
    }
    catch( std::exception& Exception )
    {
      m_results = Exception.what( );
    }
    catch( ... )
    {
      m_results = "MountPointScanner received an unknown exception";
    }
    CriticalSection( true );
    m_scan_completed.Broadcast( );
    CriticalSection( false );
  }

  void MountPointScanner::
  on_exit( )
  {
    //----------------------------------------------------------------
    // Wait till the scanner daemon has finished
    //----------------------------------------------------------------
    while ( Active( ) > 0 )
    {
      static const struct timespec pause = {
	1, 0
      };
      (void)nanosleep( &pause, (struct timespec*)NULL );
    }
    
  }

  inline void MountPointScanner::
  start( const std::string& MountPoint, ScanResults& Answer )
  {
    if ( General::AtExit::IsExiting( ) == true )
    {
      return;
    }

    General::TaskThread*   thread = General::ThreadPool::Acquire( );
    MountPointScanner*     task = new MountPointScanner( MountPoint, Answer, thread );

    try
    {
      thread->AddTask( task );
    }
    catch( ... )
    {
      throw;
    }
  }

}
