#include "general/config.h"

#include <signal.h>
#include <time.h>
#include <unistd.h>

#include <list>

#include "general/UnitTestTS.h"

#include "general/ErrorLog.hh"
#include "general/TaskTimer.hh"
#include "general/TaskThread.hh"
#include "general/ThreadPool.hh"
#include "general/Timeout.hh"
#include "general/mutexlock.hh"

General::UnitTestTS	Test;
using General::Timeout;
using General::Timer;

#define WAITER_SIGNAL SIGTERM

struct sleeper_info_type {
  pthread_t		s_thread;
  int			s_sleep;
  int			s_wakeup;
};

typedef std::list< sleeper_info_type* >	sleepers_type;

MutexLock::lock_type	sleepers_key = MutexLock::Initialize( );
sleepers_type		Sleepers;

void
wakeup( int SignalNumber )
{
  signal( SignalNumber, wakeup );
}

void*
sleeper( void* Data )
{
  //---------------------------------------------------------------------
  // Setup the timer and wait for the results
  //---------------------------------------------------------------------
  sleeper_info_type*	info = reinterpret_cast<sleeper_info_type*>( Data );
  Timer			nsleep( info->s_sleep, WAITER_SIGNAL );
  try {
    int status;
    Timeout( &nsleep, info->s_wakeup, status );
  }
  catch( ... )
  {
  }

  //---------------------------------------------------------------------
  // Check the results
  //---------------------------------------------------------------------
  bool		status;
  std::string	alarm;
  if ( info->s_sleep < info->s_wakeup )
  {
    alarm = "before";
    status = ! nsleep.Error( );
  }
  else
  {
    alarm = "after";
    status = nsleep.Error( );
  }
  {
    Test.Lock( );
    Test.Check( status )
      << " Task completes " << alarm
      << " alarm rings: (task: " << info->s_sleep
      << " timeout: " << info->s_wakeup << ")"
      << std::endl;
    Test.UnLock( );
  }
  return (void*)NULL;
}


//-----------------------------------------------------------------------
// Wrapper function to start the thread
//-----------------------------------------------------------------------
void
sleeper_test( int Sleep, int Wakeup )
{
  Test.Lock( );
  Test.Message( )
    << "Setting up sleeper_test: Task: " << Sleep
    << " Timeout: " << Wakeup
    << " Signal: " << WAITER_SIGNAL
    << std::endl;
  Test.UnLock( );
  sleeper_info_type*	retval = new sleeper_info_type;

  retval->s_sleep = Sleep;
  retval->s_wakeup = Wakeup;
  {
    MutexLock	l( sleepers_key );

    Sleepers.push_back( retval );
    pthread_create( &(retval->s_thread), NULL, sleeper, retval );
  }
}

void
task_thread_test( )
{
  using General::TaskThread;

  //---------------------------------------------------------------------
  // Test the creation and destruction of the object
  //---------------------------------------------------------------------
  {
    TaskThread	t;
    t.Mutex( );
  }
}

void
timer_test( bool Expire )
{
  Timer	nsleep( 10, WAITER_SIGNAL );
  std::ostringstream	msg;
  bool pass = false;

  try
  {
    int status;

    Timeout( &nsleep, ( Expire ) ? 5 : 15, status );
    pass = ( Expire == false );
  }
  catch( const General::TimeoutException& te )
  {
    pass = ( Expire == true );
  }
  catch( const std::exception& e )
  {
    pass = false;
    msg << "Unexpected exception: " << e.what( );
  }
  catch( ... )
  {
    pass = false;
    msg << "Unknown exception";
  }
  Test.Lock( );
  Test.Check( pass )
    << "timer_test: " << ( ( Expire ) ? "Experation" : "Normal Completion" )
    << msg.str( )
    << std::endl;
  Test.UnLock( );
}

int
main( int ArgC, char** ArgV )
{
  //---------------------------------------------------------------------
  // Initialize the test structure
  //---------------------------------------------------------------------
  Test.Init(ArgC, ArgV);

  signal( WAITER_SIGNAL, wakeup );
  //---------------------------------------------------------------------
  // Set to true for verbose output, false for no output from TimerThread
  //   class.
  //---------------------------------------------------------------------
  General::StdErrLog.IsOpen( false );

  //---------------------------------------------------------------------
  // Test TaskThread class
  //---------------------------------------------------------------------

  task_thread_test( );

  //---------------------------------------------------------------------
  // Test that no exception is thrown when the task completes in
  //   a timely manner.
  //---------------------------------------------------------------------

  timer_test( false );

  //---------------------------------------------------------------------
  // Test that an exception is thrown when the timer expires
  //---------------------------------------------------------------------

  timer_test( true );

  //---------------------------------------------------------------------
  // Setup and run the differnet threads
  //---------------------------------------------------------------------

  sleeper_test( 5, 10 );
  sleeper_test( 10, 5 );
  sleeper_test( 20, 18 );
  sleeper_test( 10, 20 );
  sleep( 30 );
  sleeper_test( 5, 10 );
  sleeper_test( 10, 5 );
  sleeper_test( 20, 18 );
  sleeper_test( 10, 20 );

  //---------------------------------------------------------------------
  // Wait for all threads to terminate
  //---------------------------------------------------------------------
  {
    MutexLock	l( sleepers_key );

    while( Sleepers.size( ) > 0 )
    {
      //-----------------------------------------------------------------
      // Get rid of any threads that did not really start
      //-----------------------------------------------------------------
      if ( Sleepers.front( )->s_thread == (pthread_t)NULL )
      {
	delete Sleepers.front( );
	Sleepers.pop_front( );
	continue;
      }
      //-------------------------------------------------------------------
      // Join threads that started for real
      //-------------------------------------------------------------------
      pthread_join( Sleepers.front( )->s_thread, NULL );
      delete Sleepers.front( );
      Sleepers.pop_front( );
    }
  }
  //---------------------------------------------------------------------
  // Exit with the appropriate exit status
  //---------------------------------------------------------------------
  Test.Exit();
  return 1;
}
