#include "general/config.h"

#include <list>

#include "general/AtExit.hh"
#include "general/mutexlock.hh"
#include "general/TaskThread.hh"
#include "general/ThreadPool.hh"

typedef	std::list< General::TaskThread* >	thread_container_type;

static bool					at_exit_initialized = false;

static void on_exit( );

//-----------------------------------------------------------------------
/// \brief Obtain lock
///
/// The use of a function ensures the proper initialization of the data
/// without having to depend on the link order initialization.
//-----------------------------------------------------------------------
inline MutexLock::lock_type&
baton( )
{
  static MutexLock::lock_type	baton =
    MutexLock::Initialize( );

  return baton;
}

//-----------------------------------------------------------------------
/// \brief Obtain the queue of available resources
///
/// The use of a function ensures the proper initialization of the data
/// without having to depend on the link order initialization.
//-----------------------------------------------------------------------
inline static thread_container_type&
available( )
{
  static thread_container_type retval;

  return retval;
}

//-----------------------------------------------------------------------
/// \brief Obtain the queue of resources currently in use.
///
/// The use of a function ensures the proper initialization of the data
/// without having to depend on the link order initialization.
//-----------------------------------------------------------------------
inline static thread_container_type&
in_use( )
{
  static thread_container_type retval;

  return retval;
}

namespace General
{
  //---------------------------------------------------------------------
  /// Secure a resource from the pool of resouces.
  /// If there is no resouce available at the time of the call,
  /// a new resouce is allocated and returned to the caller.
  //---------------------------------------------------------------------
  TaskThread* ThreadPool::
  Acquire( )
  {
    if ( General::AtExit::IsExiting( ) )
    {
      //-----------------------------------------------------------------
      // Do not give out any more threads if the application is
      // in the process of shutting down.
      //-----------------------------------------------------------------
      return (TaskThread*)NULL;
    }
    TaskThread*	retval = (TaskThread*)NULL;

    {
      MutexLock	lock( baton( ) );

      if ( at_exit_initialized == false )
      {
	General::AtExit::Append( on_exit,
				 "General::ThreadPool::on_exit",
				 150 );
	at_exit_initialized = true;
      }
      
#if old
      if ( available( ).size( ) == 0 )
      {
	available( ).push_front( new TaskThread );
      }
      in_use( ).push_front( available( ).front( ) );
      available( ).pop_front( );

      retval = in_use( ).front( );
#else
      if ( available( ).empty( ) == false )
      {
	in_use( ).push_front( available( ).front( ) );
	available( ).pop_front( );
	retval = in_use( ).front( );
      }
#endif /* old */
    }

#if old
#else
    if ( retval == (TaskThread*)NULL )
    {
      retval = new TaskThread;

      MutexLock	lock( baton( ) );
      in_use( ).push_front( retval );
    }

#endif
	
    return retval;
  }

  //---------------------------------------------------------------------
  /// Recyle the resource.
  //---------------------------------------------------------------------
  void ThreadPool::
  Relinquish( TaskThread* Resource )
  {
    if ( Resource )
    {
      bool exiting = true;
      {
	MutexLock lock( baton( ) );

	in_use( ).remove( Resource );
	if ( General::AtExit::IsExiting( ) == false )
	{
	  available( ).push_back( Resource );
	  exiting = false;
	}
      }
      if ( exiting )
      {
	// delete Resource;
      }
    }
  }
  
  //---------------------------------------------------------------------
  /// \note
  ///     This method should not be called by others.
  ///     It is provided to help facilitate cleanup at program exit.
  //---------------------------------------------------------------------
  void ThreadPool::
  Reset( )
  {
    MutexLock	lock( baton( ) );
    //-------------------------------------------------------------------
    /// Get rid of threads currently doing something interesting
    //-------------------------------------------------------------------
    for ( thread_container_type::const_iterator
	    cur = in_use( ).begin( ),
	    last = in_use( ).end( );
	  cur != last;
	  ++cur )
    {
      delete( *cur );
    }
    in_use( ).erase( in_use( ).begin( ), in_use( ).end( ) );
    //-------------------------------------------------------------------
    /// Get rid of threads waiting to do something interesting
    //-------------------------------------------------------------------
    for ( thread_container_type::const_iterator
	    cur = available( ).begin( ),
	    last = available( ).end( );
	  cur != last;
	  ++cur )
    {
      delete( *cur );
    }
    available( ).erase( available( ).begin( ), available( ).end( ) );
  }

}

static void
on_exit( )
{
  {
    MutexLock	lock( baton( ) );

    for ( thread_container_type::iterator
	    cur = in_use( ).begin( ),
	    last = in_use( ).end( );
	  cur != last;
	  ++cur )
    {
      if ( *cur )
      {
	(*cur)->Halt( );
      }
    }
  }
  while( in_use( ).empty( ) == false )
  {
    //-------------------------------------------------------------------
    // Give time for tasks to complete
    //-------------------------------------------------------------------
    {
      for ( thread_container_type::iterator
	      cur = in_use( ).begin( ),
	      last = in_use( ).end( );
	    cur != last;
	     )
      {
#if 0
	std::cerr << "DEBUG: typeid: " << typeid( *cur ).name( )
		  << " ptr: " << (void*)(*cur)
		  << " name: " << ( ( (*cur)->Name( ) )
				    ? (*cur)->Name( )
				    : "unknown" )
		  << " IsAlive: " << ( (*cur)->IsAlive( )
				       ? "true" : "false" )
		  << " state: " << (*cur)->State( )
		  << " CancellationType: " << (*cur)->CancellationType( )
	  
		  << std::endl
	  ;
#endif /* 0 */
	if ( ( (*cur)->IsAlive( ) == true )
	     && (*cur)->State( ) != General::TaskThread::TASK_THREAD_EXITING )
	{
	  (*cur)->Halt( );
	}
	if ( (*cur)->IsAlive( ) == false )
	{
	  (*cur)->Join( );
	  cur = in_use( ).erase( cur );
	  continue;
	}
	++cur;
      }
    }
    struct timespec duration;

    duration.tv_sec = 5;
    duration.tv_nsec = 0;

    nanosleep( &duration, NULL );
  }
  {
    struct timespec duration;

    duration.tv_sec = 10;
    duration.tv_nsec = 0;

    nanosleep( &duration, NULL );
  }
  General::ThreadPool::Reset( );
}
