#ifndef GENERAL__LOG_HH
#define GENERAL__LOG_HH

#include <queue>
#include <sstream>

#include "general/autoarray.hh"
#include "general/fstream.hh"
#include "general/gpstime.hh"
#include "general/mutexlock.hh"
#include "general/Thread.hh"
#include "general/unordered_map.hh"

namespace General
{
  //---------------------------------------------------------------------
  /// \brief Logging class
  ///
  /// This is a very simple logging class that is thread safe.
  /// It does support seperate debugging levels for each group.
  //---------------------------------------------------------------------
  class Log
    : public Thread
  {
  public:
    //-------------------------------------------------------------------
    /// \brief Type for group.
    ///
    /// This is the data type used when referencing parameters
    /// representing the group associated with the message.
    //-------------------------------------------------------------------
    typedef int group_type;

    //-------------------------------------------------------------------
    /// \brief Type for level.
    ///
    /// This is the data type used when referencing parameters
    /// representing the level of message.
    //-------------------------------------------------------------------
    typedef int level_type;

    //-------------------------------------------------------------------
    /// \brief Constructor
    ///
    /// \param[in] BaseName
    ///     The base name to be used for creation of the log file.
    ///
    /// \return
    ///     An instance of this object.
    //-------------------------------------------------------------------
    Log( const std::string& BaseName );

    //-------------------------------------------------------------------
    /// \brief Destructor
    //-------------------------------------------------------------------
    ~Log( );

    //-------------------------------------------------------------------
    /// \brief Send a message to the log.
    ///
    /// \param[in] Group
    ///     The group associated with the message.
    /// \param[in] Level
    ///     The level associated with the message.
    /// \param[in] Message
    ///     The text of the message.
    //-------------------------------------------------------------------
    void Message( const group_type Group,
		  const level_type Level,
		  const std::string& Message );


    //-------------------------------------------------------------------
    /// \brief Establish logging level for a given group
    ///
    /// \param[in] Group
    ///     The group for which the logging level is to be modified.
    /// \param[in] Level
    ///     The level of debugging to output.
    //-------------------------------------------------------------------
    void Verbosity( const group_type Group,
		    const level_type Level );

    //-------------------------------------------------------------------
    /// \brief Check if the message will be logged.
    ///
    /// \param[in] Group
    ///     The group for which to check.
    /// \param[in] Level
    ///     The level to check.
    //-------------------------------------------------------------------
    bool VerbosityCheck( const group_type Group,
			 const level_type Level ) const;

  protected:
    //-------------------------------------------------------------------
    /// The stream_type is used to represent the output stream class
    /// used to log the messages.
    //-------------------------------------------------------------------
    typedef General::ofstream	stream_type;

    //-------------------------------------------------------------------
    /// \brief Information about a log message
    //-------------------------------------------------------------------
    struct message_queue_entry_type {
      //-----------------------------------------------------------------
      /// Body of the message.
      //-----------------------------------------------------------------
      std::string	s_message;
      //-----------------------------------------------------------------
      /// Time when the message was generated.
      //-----------------------------------------------------------------
      GPSTime		s_timestamp;
    };

    //-------------------------------------------------------------------
    /// \brief Hook to do additional action when the stream is opened
    //-------------------------------------------------------------------
    virtual void onStreamClose(  );

    //-------------------------------------------------------------------
    /// \brief Hook to do additional action when the stream is opened
    //-------------------------------------------------------------------
    virtual void onStreamOpen( );

    //-------------------------------------------------------------------
    /// \brief Method to rotate logs
    //-------------------------------------------------------------------
    virtual void rotate( );

    //-------------------------------------------------------------------
    /// \brief Send a message immediately to the logging stream
    //-------------------------------------------------------------------
    void writeDirect( const std::string& Message );

    //-------------------------------------------------------------------
    /// \brief Active output stream.
    //-------------------------------------------------------------------
    stream_type			m_stream;

  private:
    //-------------------------------------------------------------------
    /// The message_queue_type is used to represent the storage class
    /// used for the message queue.
    //-------------------------------------------------------------------
    typedef std::queue< message_queue_entry_type >	message_queue_type;

    //-------------------------------------------------------------------
    /// \brief Container type for managing verbosity levels.
    //-------------------------------------------------------------------
    typedef General::unordered_map< group_type, level_type >
    verbose_map_type;

    //-------------------------------------------------------------------
    /// The base name for the log file.
    /// It is used as the pattern for log rotation.
    //-------------------------------------------------------------------
    const std::string	m_base_name;

    //-------------------------------------------------------------------
    /// Keeps track of the number of bytes that have been written to the
    /// stream.
    //-------------------------------------------------------------------
    int		m_bytes;

    //-------------------------------------------------------------------
    /// Holder for the maximum number of bytes that should be written
    /// to the stream before rotating.
    /// If the value is zero, then it is unlimited.
    //-------------------------------------------------------------------
    int		m_bytes_max;

    //-------------------------------------------------------------------
    /// Keeps track of the number of entries that have been written
    /// to the stream.
    //-------------------------------------------------------------------
    int		m_entries;

    //-------------------------------------------------------------------
    /// Holder for the maximum number of entries that should be
    /// written to the stream before rotating.
    /// If the value is zero, then it is unlimited.
    //-------------------------------------------------------------------
    int		m_entries_max;

    //-------------------------------------------------------------------
    /// \brief Storage for pending message.
    ///
    /// This storage holds the list of pending mesages.
    //-------------------------------------------------------------------
    message_queue_type	m_message_queue;

    //-------------------------------------------------------------------
    /// \brief Protect integrity of the message queue
    ///
    /// The holder of this resouce has the authority to modify
    /// the message queue..
    //-------------------------------------------------------------------
    mutable MutexLock::lock_type	m_message_baton;

    //-------------------------------------------------------------------
    /// \brief Storage for the stream buffer.
    //-------------------------------------------------------------------
    General::AutoArray< char >	m_stream_buffer;

    //-------------------------------------------------------------------
    /// \brief Keep track of the verbosity of each group
    //-------------------------------------------------------------------
    mutable verbose_map_type	m_verbosity;

    //-------------------------------------------------------------------
    /// \brief Output contents of the message queue to the output stream.
    //-------------------------------------------------------------------
    virtual void action( );

    //-------------------------------------------------------------------
    /// \brief Write message to the output stream
    ///
    /// \param[in] MessageInfo
    ///     The content to be written to the log.
    //-------------------------------------------------------------------
    void write_message( const message_queue_entry_type& MessageInfo );
  };

  inline void Log::
  Verbosity( const group_type Group, const level_type Level )
  {
    MutexLock	lock( m_message_baton );

    m_verbosity[ Group ] = Level;
  }

  inline bool Log::
  VerbosityCheck( const group_type Group, const level_type Level ) const
  {
    MutexLock	lock( m_message_baton );

    return ( Level <= m_verbosity[ Group ] );
  }
}

#endif /* GENERAL__LOG_HH */
