#ifndef LDASUNEXPECTED_HH
#define LDASUNEXPECTED_HH

namespace General {

    /// \brief  A singleton class to initialise unexpected()
    ///
    /// Linking against a library that includes this header file
    /// will automatically turn on special handling for bad exceptions.
    /// <p>
    /// If used inside an LDAS API, unexpected exceptions will be logged
    /// to the API's log by default. This indicates that a function threw
    /// an exception which was not in its exception specification.
    ///
    class LDASUnexpected {
    public:
	/// \brief  Destructor
	///
	/// Resets the unexpected handler back to its original value
	///
        ~LDASUnexpected();

        /// \brief  Specify if process should abort (thereby providing a core)
        ///
        /// By default, aborting is turned off. By passing true, the
        /// library will abort allowing for stack tracing with a debugger.
        /// This will allow for faster location of Unexpected exceptions.
        /// \param Abort true to dump core, false otherwise.
        static
	void makeAbort( bool Abort );

	/// \brief  Turn off logging
	///
	/// By default, logging is on. Shouldn't be turned off unless
	/// you really want to suppress error reporting
	static
	void makeQuiet();

	/// \brief  Turn on logging
	///
	/// By default, logging is on
	static
	void makeVerbose();

    private:
	/// \brief  Constructor (private)
	///
	/// No instances of this object may be created outside
	/// of the class itself
	///
	LDASUnexpected();

	/// \brief  The replacement for unexpected()
        static
        void unexpected();
	
	typedef void (*unexpected_handler) ();

	/// \brief  Storage for the original unexpected handler
        unexpected_handler m_old;

	/// \brief  The current logging status
	static
	bool m_logging;

        /// \brief  The current abort status
        static
	bool m_abort;

	/// \brief  The single instance of LDASUnexpected
	///
	/// Having a static instance of the class ensures that
	/// initialisation is done during library initialisation.
	/// It would not be thread-safe to allow initialisation
	/// after main() has been entered.
	///
	static
	const LDASUnexpected TheLDASUnexpected;
    };

} // namespace General

#endif // LDASUNEXPECTED_HH
