#ifndef GENERAL__ERROR_LOG_H
#define GENERAL__ERROR_LOG_H

#if ! defined(SWIGIMPORTED)
#include <iostream>
#endif /* ! defined(SWIGIMPORTED) */

#include "general/mutexlock.hh"
#include "general/types.hh"

namespace General
{
  //---------------------------------------------------------------------
  /// \brief  Error reporting mechanism
  class ErrorLog
  {
  public:
    enum state {
      ASSERT_FAILURE,
      //-----------------------------------------------------------------
      /// \brief  Message represents a fatal condition.
      FATAL,
      //-----------------------------------------------------------------
      /// \brief  Message is for debbugging purposes.
      DEBUG,
      //-----------------------------------------------------------------
      /// \brief  Message represents a non-fatal condition.
      WARNING,
      //-----------------------------------------------------------------
      /// \brief  Message supplies additional information.
      INFO
    };

    //-------------------------------------------------------------------
    /// \brief  Constructor
    ErrorLog( std::ostream& Stream, bool OpenState );
    //-------------------------------------------------------------------
    /// \brief  Destructor
    ~ErrorLog( );

    static bool AbortOnAssertFailure( );
    static void AbortOnAssertFailure( bool Value );

    //-------------------------------------------------------------------
    /// \brief  Check log's open state
    bool IsOpen( ) const;

    //-------------------------------------------------------------------
    /// \brief  Set log's open state
    bool IsOpen( bool State );

    void MaskBitClear( state State );

    void MaskBitSet( state State );

    //-------------------------------------------------------------------
    /// \brief  Write message to log
    void operator()( state State,
		     const char* Filename, const int Line,
		     const std::string& Message,
		     bool EOL = true );

  private:
    static bool			m_abort_on_assert_failure;

    //-------------------------------------------------------------------
    /// \brief  Stream associated with the log
    std::ostream&		m_stream;
    //-------------------------------------------------------------------
    /// \brief  Lock to ensure syncronous output
    MutexLock::lock_type	m_lock;
    //-------------------------------------------------------------------
    /// \brief  Open state of log
    bool			m_is_open;
    INT_4U			m_mask;
  };

  inline bool ErrorLog::
  AbortOnAssertFailure( )
  {
    return m_abort_on_assert_failure;
  }
  
  inline void ErrorLog::
  AbortOnAssertFailure( bool Value )
  {
    m_abort_on_assert_failure = Value;
  }

  inline bool ErrorLog::
  IsOpen( ) const
  {
    return m_is_open;
  }

  inline bool ErrorLog::
  IsOpen( bool State )
  {
    bool retval = m_is_open;
    m_is_open = State;
    return retval;
  }

  inline void ErrorLog::
  MaskBitClear( state State )
  {
    m_mask &= ~(INT_4U(1<<State));
  }

  inline void ErrorLog::
  MaskBitSet( state State )
  {
    m_mask |= (1<<State);
  }

  extern ErrorLog StdErrLog;

  extern ErrorLog JournalLog;
}

#endif /* GENERAL__ERROR_LOG_H */
