#ifndef GENERAL__BACKGROUND_TASK_COUNTER_HH
#define GENERAL__BACKGROUND_TASK_COUNTER_HH

#include "general/types.hh"
#include "general/mutexlock.hh"
#include "general/ConditionalVariable.hh"

namespace General
{
  class BackgroundTaskCounter
  {
  public:
    BackgroundTaskCounter( );

    INT_4U Active( );

    void CriticalSection( bool Value );

    void Decrement( );

    void DecrementOnly( );

    void Increment( );

    void Wait( );

  private:
    General::ConditionalVariable	m_sync;
    MutexLock::lock_type		m_baton;
    INT_4U				m_count;
  };

  inline BackgroundTaskCounter::
  BackgroundTaskCounter( )
    : m_baton( MutexLock::Initialize( ) ),
      m_count( 0 )
  {
  }

  inline INT_4U BackgroundTaskCounter::
  Active( )
  {
    MutexLock  l( m_baton );

    return m_count;
  }

  inline void BackgroundTaskCounter::
  CriticalSection( bool Value )
  {
    if ( Value )
    {
      MutexLock::Lock( m_sync.Mutex( ) );
    }
    else
    {
      MutexLock::UnLock( m_sync.Mutex( ) );
    }
  }

  inline void BackgroundTaskCounter::
  Decrement( )
  {
    CriticalSection( true );
    DecrementOnly( );
    m_sync.Broadcast( );
    CriticalSection( false );
  }

  inline void BackgroundTaskCounter::
  DecrementOnly( )
  {
    MutexLock  l( m_baton );
    
    if ( m_count )
    {
      --m_count;
    }
  }

  inline void BackgroundTaskCounter::
  Increment( )
  {
    MutexLock  l( m_baton );
	 
    ++m_count;
  }

  inline void BackgroundTaskCounter::
  Wait( )
  {
    m_sync.Wait( );
  }

}

#endif /* GENERAL__BACKGROUND_TASK_COUNTER_HH */
