#include "framecpp/config.h"

extern "C" {
#include <sys/statvfs.h>

#include <unistd.h>
}

#include <algorithm>
#include <iostream>
#include <sstream>

#include "general/autoarray.hh"
#include "general/fstream.hh"
#include "general/gpstime.hh"
#include "general/types.hh"
#include "general/unittest.h"

#include "framecpp/Common/CheckSum.hh"
#include "framecpp/Common/FrameBuffer.hh"
#include "framecpp/Common/Verify.hh"

#include "framecpp/Dimension.hh"
#include "framecpp/FrameH.hh"
#include "framecpp/FrAdcData.hh"
#include "framecpp/FrRawData.hh"
#include "framecpp/FrVect.hh"
#include "framecpp/OFrameStream.hh"

using std::filebuf;
using std::ostringstream;
using General::GPSTime;
using namespace FrameCPP;
using FrameCPP::Common::CheckSum;
using FrameCPP::Common::FrameBuffer;
using FrameCPP::Common::Verify;
using FrameCPP::Common::VerifyException;

General::UnitTest Test;

static const INT_4U FIRST_FRAME = 600000000;
static const INT_4U FRAME_INC = 1024;
static const INT_4U FRAME_COUNT = 4096;
static const INT_8U DATA_SET_SIZE = 0x20000;
static const INT_4U DATA_SET_DT = 1024;

int
main( int ArgC, char** ArgV )
{
  FrameBuffer< filebuf >*
    ofb( new FrameBuffer< filebuf >( std::ios::out ) );

  struct statvfs buf;

  try {
    Test.Init( ArgC, ArgV );

    if ( statvfs( ".", &buf) == 0 )
    {
      INT_8U avail( buf.f_frsize );

      avail *= buf.f_bavail;

      if ( ( avail >> 32 ) == 0 )
      {
	std::ostringstream msg;

	msg << "The file system does not have enough free space: "
	    << " avail: " << avail
	    << " minimum needed: " << ( INT_8U( 1 ) << 32 )
	    << " difference: " << ( ( INT_8U( 1 ) << 32 ) - avail )
	  ;
	
	throw std::runtime_error( msg.str( ) );
      }
    }

    try
    {
	std::ostringstream	frame_filename;

	//---------------------------------------------------------------
	// Create an appropriate filename
	//---------------------------------------------------------------
	frame_filename << "Z-LargeFrame"
		       << "-" << FIRST_FRAME
		       << "-" << ( DATA_SET_DT * FRAME_COUNT )
		       << ".gwf";

	//---------------------------------------------------------------
	// Create a channel of data that is at least 1 Mbyte in length
	//---------------------------------------------------------------

	General::SharedPtr< FrRawData >	raw_data( new FrRawData );
	General::SharedPtr< FrAdcData >	adc( new FrAdcData );
	Dimension				dims[ 1 ] =
	  { Dimension( DATA_SET_SIZE, DATA_SET_DT ) };
	General::AutoArray< REAL_8 >	data( new REAL_8[ DATA_SET_SIZE ] );
	std::fill( &(data[0]), &(data[ DATA_SET_SIZE ]), REAL_8( 3.1415) );

	FrVect::data_type
	  vdata( reinterpret_cast< FrVect::data_type::element_type * >( data.release( ) ));
	General::SharedPtr< FrVect >
	  vect( new FrVect( "fr_vect",
			    FrVect::RAW,			/* Compress */
			    FrVect::FR_VECT_8R,			/* Type */
			    1,					/* nDim */
			    dims,				/* dims */
			    DATA_SET_SIZE,			/* NData */
			    DATA_SET_SIZE * sizeof( REAL_8 ),	/* NBytes */
			    vdata,				/* Data */
			    "unitY"				/* unitY */
			    ) );

	adc->SetDataValid( 0 );
	adc->RefData( ).append( vect );
	raw_data->RefFirstAdc( ).append( adc );
  
	//---------------------------------------------------------------
	// Loop enough times to create a file larger than
	//    0x100000000 bytes.
	//---------------------------------------------------------------
	ofb->open( frame_filename.str( ).c_str( ),
		   std::ios::out | std::ios::binary );
  
	OFrameStream	frame_stream( ofb );
	GPSTime		frame_start( FIRST_FRAME, 0 );

	for( INT_4U frame_number = 0;
	     frame_number < FRAME_COUNT;
	     ++frame_number )
	{
	  //-------------------------------------------------------------
	  // Create frame
	  //-------------------------------------------------------------
	  General::SharedPtr< FrameH >
	    frame( new FrameH( "LargeFrame", -1,
			       frame_number, frame_start,
			       frame_start.GetLeapSeconds( ),
			       FRAME_INC ) );
	  //-------------------------------------------------------------
	  // Add in the channel
	  //-------------------------------------------------------------
	  frame->SetRawData( raw_data );

	  //-------------------------------------------------------------
	  // Write frame to frame file
	  //-------------------------------------------------------------
	  frame_stream.WriteFrame( frame );

	  //-------------------------------------------------------------
	  // Advance data for next iteration
	  //-------------------------------------------------------------
	  frame_start += FRAME_INC;
	}

	frame_stream.Close( );
	ofb->close( );
      
	//---------------------------------------------------------------
	// Check the integrity of the file
	//---------------------------------------------------------------
	Verify	v;

	v.BufferSize( 64 * 1024 );
	v.CheckDataValid( false );
	v.CheckFileChecksum( true );
	v.CheckFrameChecksum( true );
	v.MustHaveEOFChecksum( true );
	v.Strict( true );
	v.ValidateMetadata( true );

	Verify::error_type	status = v( frame_filename.str( ) );
	Test.Check( status == 0 ) << "Verify large frame file data validation: "
				  << VerifyException::StrError( status )
				  << std::endl;

	//---------------------------------------------------------------
	// Remove the file from the system.
	//---------------------------------------------------------------
#if 0
	unlink( frame_filename.str( ).c_str( ) );
#endif /* 0 */
    }
    catch( ... )
    {
      //------------------------------------------------------------------
      // Remove the large file if an error occurred while generating
      //   the file.
      //------------------------------------------------------------------
#if 0
      unlink( frame_filename.str( ).c_str( ) );
#endif /* 0 */
      // Rethrow
      throw;
    }

  }
  catch ( const std::exception& e )
  {
    //-------------------------------------------------------------------
    // Output the error
    //-------------------------------------------------------------------
    ostringstream	msg;

    msg << "Caught an exception"
	<< ": " << e.what( )
      ;

    Test.Check( false ) << msg.str( ) << std::endl;
  }
  catch ( ... )
  {
    //-------------------------------------------------------------------
    // Output the error
    //-------------------------------------------------------------------
    ostringstream	msg;

    msg << "Caught an unexpected exception";

    Test.Check( false ) << msg.str( ) << std::endl;
  }

  //---------------------------------------------------------------------
  // Completed
  //---------------------------------------------------------------------
  Test.Exit();
}


