#ifndef FrameCPP_VERSION_8_FrTOCData_HH
#define FrameCPP_VERSION_8_FrTOCData_HH

#if ! defined(SWIGIMPORTED)
#include "framecpp/Version7/FrTOCData.hh"

#include "framecpp/Version8/STRING.hh"
#endif /* ! defined(SWIGIMPORTED) */

#undef	FR_TOC_NFRAME_TYPE

#define FR_TOC_NFRAME_TYPE INT_4U

namespace FrameCPP
{
#if ! defined(SWIG)
  namespace Common
  {
    class TOCInfo;
  }
#endif /* ! defined(SWIG) */

  namespace Version_8
  {
    //===================================================================
    /// \brief Helper class for data portion of the FrTOC structure.
    //===================================================================
    class FrTOCData
    {
    public:
      typedef INT_4U			data_quality_type;
      typedef REAL_8			dt_type;
      typedef INT_4U			frame_type;
      typedef INT_4U			gtimen_type;
      typedef INT_4U			gtimes_type;
      typedef STRING			namedetector_type;
      typedef INT_4U			ndetector_type;
      typedef INT_8U			nfirstadc_type;
      typedef INT_8U			nfirstmsg_type;
      typedef INT_8U			nfirstser_type;
      typedef INT_8U			nfirsttable_type;
      typedef FR_TOC_NFRAME_TYPE	nFrame_type;
      typedef INT_4U			nsh_type;
      typedef INT_8U			positiondetector_type;
      typedef INT_8U			positionh_type;
      typedef INT_4S			runs_type;
      typedef INT_2U			shid_type;
      typedef STRING			shname_type;
      typedef INT_2S			uleaps_type;

      //-----------------------------------------------------------------
      /// \brief Array of integer QA words from each FrameH
      //-----------------------------------------------------------------
      typedef std::vector< data_quality_type >	data_quality_container_type;
      //-----------------------------------------------------------------
      /// \brief Array of frame durations in seconds.
      //-----------------------------------------------------------------
      typedef std::vector< dt_type >		dt_container_type;
      //-----------------------------------------------------------------
      /// \brief Array of frame numbers.
      //-----------------------------------------------------------------
      typedef std::vector< frame_type >		frame_container_type;
      //-----------------------------------------------------------------
      /// \brief Array of integer GPS residual nanoseconds for the frame.
      //-----------------------------------------------------------------
      typedef std::vector< gtimen_type >	gtimen_container_type;
      //-----------------------------------------------------------------
      /// \brief Array of integer GPS frame times.
      //-----------------------------------------------------------------
      typedef std::vector< gtimes_type >	gtimes_container_type;
      //-----------------------------------------------------------------
      /// \brief Array of FrDetector names.
      //-----------------------------------------------------------------
      typedef std::vector< namedetector_type >	namedetector_container_type;
      //-----------------------------------------------------------------
      /// \brief  Array of first FrADCData positions.
      //-----------------------------------------------------------------
      typedef std::vector< nfirstadc_type >	nfirstadc_container_type;
      //-----------------------------------------------------------------
      /// \brief Array of first FrMsg positions.
      //-----------------------------------------------------------------
      typedef std::vector< nfirstmsg_type >	nfirstmsg_container_type;
      //-----------------------------------------------------------------
      /// \brief Array of first FrSerData positions.
      //-----------------------------------------------------------------
      typedef std::vector< nfirstser_type >	nfirstser_container_type;
      //-----------------------------------------------------------------
      /// \brief Array of first FrTable positions.
      //-----------------------------------------------------------------
      typedef std::vector< nfirsttable_type >	nfirsttable_container_type;
      //-----------------------------------------------------------------
      /// \brief Array of FrDetector positions.
      //-----------------------------------------------------------------
      typedef std::vector< positiondetector_type >
      positiondetector_container_type;
      //-----------------------------------------------------------------
      /// \brief Array of FrSH IDs.
      //-----------------------------------------------------------------
      typedef std::vector< positionh_type >	positionh_container_type;
      //-----------------------------------------------------------------
      /// \brief Array of FrameH positions.
      //-----------------------------------------------------------------
      typedef std::vector< runs_type >		runs_container_type;
      //-----------------------------------------------------------------
      /// \brief Array of FrSH IDs.
      //-----------------------------------------------------------------
      typedef std::vector< shid_type >		shid_container_type;
      //-----------------------------------------------------------------
      /// \brief Array of FrSH names.
      //-----------------------------------------------------------------
      typedef std::vector< shname_type >	shname_container_type;

      FrTOCData( );

      Common::FrameSpec::size_type
      Bytes( const Common::StreamBase& Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Retreive array of integer QA words from each FrameH.
      //-----------------------------------------------------------------
      const data_quality_container_type&	GetDataQuality( ) const;
      //-----------------------------------------------------------------
      /// \brief Retrieve array of frame durations in seconds.
      //-----------------------------------------------------------------
      const dt_container_type&			GetDt( ) const;
      //-----------------------------------------------------------------
      /// \brief Retrieve array of frame numbers.
      //-----------------------------------------------------------------
      const frame_container_type&		GetFrame( ) const;
      //-----------------------------------------------------------------
      /// \brief Retrieve array of integer GPS residual nanoseconds for the frame.
      //-----------------------------------------------------------------
      const gtimes_container_type&		GetGTimeN( ) const;
      //-----------------------------------------------------------------
      /// \brief Retrieve array of integer GPS frame times.
      //-----------------------------------------------------------------
      const gtimen_container_type&		GetGTimeS( ) const;
      //-----------------------------------------------------------------
      /// \brief Retrieve array of first FrADCData positions.
      //-----------------------------------------------------------------
      const nfirstadc_container_type&		GetNFirstADC( ) const;
      //-----------------------------------------------------------------
      /// \brief Retrieve array of first FrMsg positions.
      //-----------------------------------------------------------------
      const nfirstmsg_container_type&		GetNFirstMsg( ) const;
      //-----------------------------------------------------------------
      /// \brief Retrieve array of first FrSerData positions.
      //-----------------------------------------------------------------
      const nfirstser_container_type&		GetNFirstSer( ) const;
      //-----------------------------------------------------------------
      /// \brief Retrieve array of first FrTable positions.
      //-----------------------------------------------------------------
      const nfirsttable_container_type&		GetNFirstTable( ) const;
      //-----------------------------------------------------------------
      /// \brief
      //-----------------------------------------------------------------
      nFrame_type				GetNFrame( ) const;
      //-----------------------------------------------------------------
      /// \brief Retrieve array of FrameH positions.
      //-----------------------------------------------------------------
      const positionh_container_type&		GetPositionH( ) const;
      //-----------------------------------------------------------------
      /// \brief
      //-----------------------------------------------------------------
      const runs_container_type&		GetRuns( ) const;
      //-----------------------------------------------------------------
      /// \brief Retrieve array of FrSH IDs.
      //-----------------------------------------------------------------
      const shid_container_type&		GetSHid( ) const;
      //-----------------------------------------------------------------
      /// \brief Retrieve array of FrSH names.
      //-----------------------------------------------------------------
      const shname_container_type&		GetSHname( ) const;
      //-----------------------------------------------------------------
      /// \brief Retrieve the number of leap seconds. 
      //-----------------------------------------------------------------
      uleaps_type				GetULeapS( ) const;
      
      //-----------------------------------------------------------------
      /// \brief Retrieve array of FrDetector names.
      //-----------------------------------------------------------------
      const namedetector_container_type&	GetNameDetector( ) const;
      //-----------------------------------------------------------------
      /// \brief Retrieve array of FrDetector positions.
      //-----------------------------------------------------------------
      const positiondetector_container_type&	GetPositionDetector( ) const;

#if ! defined(SWIG)
      //-----------------------------------------------------------------
      /// \brief asignment operator
      ///
      /// \param[in] Source
      ///     The source to be copied.
      //-----------------------------------------------------------------
      const FrTOCData& operator=( const Previous::FrTOCData& Source );
#endif /* ! defined(SWIG) */

      //-----------------------------------------------------------------
      /// \brief equality operator
      ///
      /// \param[in] RHS
      ///     The FrTOCData object to be compared.
      ///
      /// \return
      ///     The value true is returned if this object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      bool operator==( const FrTOCData& RHS ) const;

      //-----------------------------------------------------------------
      /// \brief The description of structure
      ///
      /// \return
      ///     A Description object which describes this structure as
      ///     specified by the frame specification.
      //-----------------------------------------------------------------
      template< typename SE >
      static void Description( Common::Description& Desc );

    protected:
      //-----------------------------------------------------------------
      /// \brief From the first FrameH in the file.
      //-----------------------------------------------------------------
      uleaps_type			m_ULeapS;
      //-----------------------------------------------------------------
      /// \brief Array of integer QA words from each FrameH (size of nFrame) 
      //-----------------------------------------------------------------
      data_quality_container_type	m_dataQuality;
      //-----------------------------------------------------------------
      /// \brief Array of integer GPS frame times (size of nFrame).
      //-----------------------------------------------------------------
      gtimes_container_type		m_GTimeS;
      //-----------------------------------------------------------------
      /// \brief Array of integer GPS residual nanoseconds for the frame (size of nFrame).
      //-----------------------------------------------------------------
      gtimen_container_type		m_GTimeN;
      //-----------------------------------------------------------------
      /// \brief Array of frame durations in seconds (size of nFrame).
      //-----------------------------------------------------------------
      dt_container_type			m_dt;
      //-----------------------------------------------------------------
      /// \brief Array of run numbers.
      ///
      /// Array of run numbers.
      /// The size of this array is specified by nFrame.
      //-----------------------------------------------------------------
      runs_container_type		m_runs;
      //-----------------------------------------------------------------
      /// \brief Array of frame numbers (size of nFrame).
      //-----------------------------------------------------------------
      frame_container_type		m_frame;
      //-----------------------------------------------------------------
      /// \brief Array of FrDetector positions.
      ///
      /// Array of FrDetector positions from the beginning of
      /// file.
      /// The size of array is specified by nDetector. 
      ///
      /// \note
      /// Only the first occurrence for each type of FrDetector
      /// is captured.
      //-----------------------------------------------------------------
      positiondetector_container_type	m_positionDetector;
      //-----------------------------------------------------------------
      /// \brief
      ///
      /// Array of FrameH positions, in bytes,
      /// from beginning of file.
      /// The size of this array is specified by nFrame.
      //-----------------------------------------------------------------
      positionh_container_type		m_positionH;
      //-----------------------------------------------------------------
      /// \brief Array of FrDetector names.
      ///
      /// The size of this array is specified by nDetector.
      /// They appear alphabetically.
      //-----------------------------------------------------------------
      namedetector_container_type	m_nameDetector;
      //-----------------------------------------------------------------
      /// \brief Array of first FrADCData positions.
      ///
      /// Array of first FrADCData positions, in bytes,
      /// from beginning of file.
      /// The size of this array is specified by nFrame.
      //-----------------------------------------------------------------
      nfirstadc_container_type		m_nFirstADC;
      //-----------------------------------------------------------------
      /// \brief Array of first FrSerData positions.
      ///
      /// Array of first FrSerData positions, in bytes,
      /// from beginning of file.
      /// The size of this array is specified by nFrame.
      //-----------------------------------------------------------------
      nfirstser_container_type		m_nFirstSer;
      //-----------------------------------------------------------------
      /// \brief Array of first FrTable positions.
      ///
      /// Array of first FrTable positions, in bytes,
      /// from beginning of file
      /// The size of this array is specified by nFrame.
      ///
      /// \note
      /// The pointer is to the first table associated
      /// with FrRawData for each frame.
      //-----------------------------------------------------------------
      nfirsttable_container_type	m_nFirstTable;
      //-----------------------------------------------------------------
      /// \brief Array of first FrMsg positions.
      ///
      /// Array of first FrMsg positions, in bytes,
      /// from beginning of file.
      /// The size of this array is specified by nFrame.
      //-----------------------------------------------------------------
      nfirstmsg_container_type		m_nFirstMsg;
      //-----------------------------------------------------------------
      /// \brief Array of FrSH IDs.
      ///
      /// Array of FrSH IDs.
      /// The size of this array is specified by nSH.
      //-----------------------------------------------------------------
      shid_container_type		m_SHid;
      //-----------------------------------------------------------------
      /// \brief
      ///
      /// Array of FrSH names.
      /// The size of array is specified by nSH.
      //-----------------------------------------------------------------
      shname_container_type		m_SHname;

      //-----------------------------------------------------------------
      /// \brief Stream Constructor
      ///
      /// \param[in] Stream
      ///     The stream from which the object is being read.
      ///
      /// \return
      ///    A new instance of this object.
      //-----------------------------------------------------------------
      FrTOCData( Common::IStream& Stream );

      //-----------------------------------------------------------------
      /// \brief Iterate over contents.
      ///
      /// \param[in] Info
      ///     Specifies the type of information to be searched.
      ///
      /// \param[in] Action
      ///     Action to be taken for each piece of information found.
      //-----------------------------------------------------------------
      void forEach( Common::FrTOC::query_info_type Info,
		    Common::FrTOC::FunctionBase& Action ) const;

      //-----------------------------------------------------------------
      /// \brief Write the structure to the stream
      ///
      /// \param[in] Stream
      ///     The output stream where the object is to be written.
      //-----------------------------------------------------------------
      void write( Common::OStream& Stream ) const;

    };

    inline FrTOCData::
    FrTOCData( )
      : m_ULeapS( 0 )
    {
    }

    inline Common::FrameSpec::size_type FrTOCData::
    Bytes( const Common::StreamBase& Stream ) const
    {
      Common::FrameSpec::size_type	retval = 0;

      //-----------------------------------------------------------------
      // main header block
      //-----------------------------------------------------------------
      retval +=
	sizeof( m_ULeapS )
	+ sizeof( nFrame_type )
	+ ( m_dataQuality.size( )
	    * ( sizeof( data_quality_type )
		+ sizeof( gtimes_type )
		+ sizeof( gtimen_type )
		+ sizeof( dt_type )
		+ sizeof( runs_type )
		+ sizeof( frame_type )
		+ sizeof( positionh_type )
		+ sizeof( nfirstadc_type )
		+ sizeof( nfirstmsg_type )
		+ sizeof( nfirstser_type )
		+ sizeof( nfirsttable_type )
		) )
	;
      //-----------------------------------------------------------------
      // SH elements
      //-----------------------------------------------------------------
      retval +=
	sizeof( nsh_type )
	+ ( sizeof( shid_type ) * m_SHid.size( ) )
	;
      for( shname_container_type::const_iterator
	     cur = m_SHname.begin( ),
	     last = m_SHname.end( );
	   cur != last;
	   ++cur )
      {
	retval += cur->Bytes( );
      }
      //-----------------------------------------------------------------
      // Detector information
      //-----------------------------------------------------------------
      retval +=
	sizeof( ndetector_type )
	+ ( sizeof( positiondetector_type ) * m_positionDetector.size( ) )
	;
      for ( namedetector_container_type::const_iterator
	      cur = m_nameDetector.begin( ),
	      last = m_nameDetector.end( );
	    cur != last;
	    ++cur )
      {
	retval += cur->Bytes( );
      }
      //-----------------------------------------------------------------
      // Return the results
      //-----------------------------------------------------------------
      return retval;
    }

    template< typename SE >
    void FrTOCData::
    Description( Common::Description& Desc )
    {
      Desc( SE( "ULeapS", "INT_2S",
		"From the first FrameH in the file" ) );
      //-----------------------------------------------------------------
      Desc( SE( "nFrame", "INT_4U",
		"Number of frames in the file" ) );
      Desc( SE( "dataQuality", "INT_4U[nFrame]",
		"Array of integer QA words from each FrameH"
		" (size of nFrames)" ) );
      Desc( SE( "GTimeS", "INT_4U[nFrame]",
		"Array of integer GPS frae times (size of nFrames" ) );
      Desc( SE( "GTimeN", "INT_4U[nFrame]",
		"Array of integer GPS residual nanoseconds for the frame"
		" (size of nFrame" ) );
      Desc( SE( "dt", "REAL_8[nFrame]",
		"Array of frame durations in seconds( size of nFrames)" ) );
      Desc( SE( "runs", "INT_4S[nFrame]",
		"Array of run numbers (size of nFrame)" ) );
      Desc( SE( "frame", "INT_4U[nFrame]",
		"Array of frame numbers (size of nFrame)" ) );
      Desc( SE( "positionH", "INT_8U[nFrame]",
		"Array of FrameH positions, in bytes, from the"
		" bginning of file (size of nFrame)" ) );
      Desc( SE( "nFirstADC", "INT_8U[nFrame]",
		"Array of first FrADCData positions, in bytes,"
		" from beginning of file (size of nFrame)" ) );
      Desc( SE( "nFirstSer", "INT_8U[nFrame]",
		"Array of first FrSerData positions in bytes,"
		" from beginning of file (size of nFrame)" ) );
      Desc( SE( "nFirstTable", "INT_8U[nFrame]",
		"Array of first FrTable positions, in bytes,"
		" from beginning of file (size of nFrame)" ) );
      Desc( SE( "nFirstMsg", "INT_8U[nFrame]",
		"Array of first FrMsg positions, in bytes,"
		" from beginning of file (size of nFrame)" ) );
      //-----------------------------------------------------------------
      Desc( SE( "nSH", "INT_4U",
		"Number of FrSH structures in the file" ) );
      Desc( SE( "SHid", "INT_2U[nSH]",
		"Array of FrSH IDs (size of nSH)" ) );
      Desc( SE( "SHname", "STRING[nSH]",
		"Array of FrSH names (size of nSH)" ) );
      //-----------------------------------------------------------------
      Desc( SE( "nDetector", "INT_4U",
		"Number of distinct types of FrDetector in the file" ) );
      Desc( SE( "nameDetector", "STRING[nDetector]",
		"Array of FrDetector names (size of nDetector)."
		" They appear alphabetically" ) );
      Desc( SE( "positionDetector", "INT_8U[nDetector]",
		"Array of FrDetector positions from the beginning"
		" of file (size of nDetector)." ) );
      
    }

    inline const FrTOCData::data_quality_container_type& FrTOCData::
    GetDataQuality( ) const
    {
      return m_dataQuality;
    }

    inline const FrTOCData::dt_container_type& FrTOCData::
    GetDt( ) const
    {
      return m_dt;
    }

    inline const FrTOCData::frame_container_type& FrTOCData::
    GetFrame( ) const
    {
      return m_frame;
    }

    inline const FrTOCData::gtimen_container_type& FrTOCData::
    GetGTimeN( ) const
    {
      return m_GTimeN;
    }

    inline const FrTOCData::gtimes_container_type& FrTOCData::
    GetGTimeS( ) const
    {
      return m_GTimeS;
    }

    inline FrTOCData::nFrame_type FrTOCData::
    GetNFrame( ) const
    {
      return m_dataQuality.size( );
    }

    inline const FrTOCData::nfirstadc_container_type& FrTOCData::
    GetNFirstADC( ) const
    {
      return m_nFirstADC;
    }

    inline const FrTOCData::nfirstmsg_container_type& FrTOCData::
    GetNFirstMsg( ) const
    {
      return m_nFirstMsg;
    }

    inline const FrTOCData::nfirstser_container_type& FrTOCData::
    GetNFirstSer( ) const
    {
      return m_nFirstSer;
    }

    inline const FrTOCData::nfirsttable_container_type& FrTOCData::
    GetNFirstTable( ) const
    {
      return m_nFirstTable;
    }

    inline const FrTOCData::positionh_container_type& FrTOCData::
    GetPositionH( ) const
    {
      return m_positionH;
    }

    inline const FrTOCData::runs_container_type& FrTOCData::
    GetRuns( ) const
    {
      return m_runs;
    }

    inline const FrTOCData::shid_container_type& FrTOCData::
    GetSHid( ) const
    {
      return m_SHid;
    }

    inline const FrTOCData::shname_container_type& FrTOCData::
    GetSHname( ) const
    {
      return m_SHname;
    }

    inline const FrTOCData::namedetector_container_type& FrTOCData::
    GetNameDetector( ) const
    {
      return m_nameDetector;
    }

    inline const FrTOCData::positiondetector_container_type& FrTOCData::
    GetPositionDetector( ) const
    {
      return m_positionDetector;
    }

    inline FrTOCData::uleaps_type FrTOCData::
    GetULeapS( ) const
    {
      return m_ULeapS;
    }

    inline bool FrTOCData::
    operator==( const FrTOCData& RHS ) const
    {
#define CMP__(X) (X==RHS.X)

      return ( ( &RHS == this )
	       || ( CMP__(m_ULeapS )
		    && CMP__(m_dataQuality)
		    && CMP__(m_GTimeS)
		    && CMP__(m_GTimeN)
		    && CMP__(m_dt)
		    && CMP__(m_runs)
		    && CMP__(m_frame)
		    && CMP__(m_positionH)
		    && CMP__(m_nFirstADC)
		    && CMP__(m_nFirstSer)
		    && CMP__(m_nFirstTable)
		    && CMP__(m_nFirstMsg)
		    && CMP__(m_SHid)
		    && CMP__(m_SHname)
		    && CMP__(m_nameDetector)
		    && CMP__(m_positionDetector)
		    ) )
	;
#undef CMP__
    }

  } // namespace - Version_8
} // namespace - FrameCPP

#endif /* FrameCPP_VERSION_8_FrTOCData_HH */
