#ifndef FrameCPP_VERSION_8_FrSerData_HH
#define FrameCPP_VERSION_8_FrSerData_HH

#ifndef SWIGIMPORTED
#include <vector>
#endif /* SWIGIMPORTED */

#include "framecpp/Common/SearchContainer.hh"
#include "framecpp/Common/TOCInfo.hh"

#include "framecpp/Version7/FrSerData.hh"

#include "framecpp/Version8/FrameSpec.hh"
#include "framecpp/Version8/STRING.hh"
#include "framecpp/Version8/GPSTime.hh"

#include "framecpp/Version8/FrHistory.hh"
#include "framecpp/Version8/FrTable.hh"
#include "framecpp/Version8/FrVect.hh"

namespace FrameCPP
{
  namespace Version_8
  {
    //===================================================================
    /// \brief Raw Data Structure Definition
    //===================================================================
    class FrSerData
      : public ObjectWithChecksum< Common::FrameSpec::Object >,
	public Common::TOCInfo
    {
    public:
      typedef Common::SearchContainer< FrVect, &FrVect::GetName >
      serial_type;
      typedef serial_type::iterator iterator;
      typedef serial_type::const_iterator const_iterator;

      typedef Common::SearchContainer< FrTable, &FrTable::GetNameSlow >
      table_type;
      typedef table_type::iterator table_iterator;
      typedef table_type::const_iterator const_table_iterator;

      //-----------------------------------------------------------------
      /// \brief Default constructor
      ///
      /// \return
      ///    A new instance of this object.
      //-----------------------------------------------------------------
      FrSerData( );

      //-----------------------------------------------------------------
      /// \brief Copy Constructor
      ///
      /// \param[in] Source
      ///     The object from which to copy the information.
      ///
      /// \return
      ///    A new instance of this object.
      //-----------------------------------------------------------------
      FrSerData( const FrSerData& Source );

      //-----------------------------------------------------------------
      /// \brief Constructor
      ///
      /// \param[in] name
      ///     Name of station producing serial data stream.
      /// \param[in] time
      ///     Time of data acquisition.
      /// \param[in] sampleRate
      ///     Sample rate, samples per second.
      ///
      /// \return
      ///     A new instance of the object.
      //-----------------------------------------------------------------
      FrSerData( const std::string& name,
		 const GPSTime& time,
		 REAL_8 sampleRate );

      //-----------------------------------------------------------------
      /// \brief Promotion Constructor
      ///
      /// \param[in] Source
      ///     An FrSerData structure from a previous frame specification.
      /// \param[in] Stream
      ///     The stream from which the earliest version of the
      ///     FrSerData structure was read.
      ///
      /// \return
      ///     A new instance of the object.
      //-----------------------------------------------------------------
      FrSerData( Previous::FrSerData& Source, Common::IStream* Stream );

      //-----------------------------------------------------------------
      /// \brief The name structure name of this object.
      ///
      /// \return
      ///     The name of the structure as specified by the frame
      ///     specification.
      //-----------------------------------------------------------------
      static const char* StructName( );

      //-----------------------------------------------------------------
      /// \brief The description of structure
      ///
      /// \return
      ///     A Description object which describes this structure as
      ///     specified by the frame specification.
      //-----------------------------------------------------------------
      static const Common::Description* StructDescription( );

      //-----------------------------------------------------------------
      /// \brief The name structure name of this object.
      ///
      /// \return
      ///     The name of the structure as specified by the frame
      ///     specification.
      //-----------------------------------------------------------------
      virtual const char* ObjectStructName( ) const;

      //-----------------------------------------------------------------
      /// \brief Return the name of station producing serial data stream.
      ///
      /// \return
      ///     The name associated of station producing serial data stream.
      //-----------------------------------------------------------------
      const std::string& GetName() const;
      
      //-----------------------------------------------------------------
      /// \brief Retrieve the time of data acquisition.
      ///
      /// \return
      ///     The time of data acquisition.
      //-----------------------------------------------------------------
      const GPSTime& GetTime( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the sample rate.
      ///
      /// \return
      ///     The sample rate.
      //-----------------------------------------------------------------
      REAL_8 GetSampleRate( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve string of ASCII-based data.
      ///
      /// \return
      ///     The string of ASCII-based data.
      //-----------------------------------------------------------------
      const std::string& GetData( ) const;

      //-----------------------------------------------------------------
      /// \brief Establish the string of ASCII-based data.
      //-----------------------------------------------------------------
      void SetData( const std::string& Data );

      //-----------------------------------------------------------------
      /// \brief Identifier for serial data vector.
      ///
      /// \return
      ///     A constant reference the serial data vector.
      //-----------------------------------------------------------------
      const serial_type& RefSerial( ) const;

      //-----------------------------------------------------------------
      /// \brief Identifier for serial data vector.
      ///
      /// \return
      ///     A reference the serial data vector.
      //-----------------------------------------------------------------
      serial_type& RefSerial( );

      //-----------------------------------------------------------------
      /// \brief Identifier for user-defined table structure.
      ///
      /// \return
      ///     A constant reference to the  user-defined table structure.
      //-----------------------------------------------------------------
      const table_type& RefTable( ) const;
      table_type& RefTable( );
   
      //-----------------------------------------------------------------
      /// \brief Merge with another FrSerData
      ///
      /// \param[in] RHS
      ///     The source of the information to append to this FrSerData
      ///     structure.
      ///
      /// \return
      ///     A reference to this object
      //-----------------------------------------------------------------
      FrSerData& Merge( const FrSerData& RHS );

      //-----------------------------------------------------------------
      /// \brief Promote previous version of object to this version
      ///
      /// \param[in] Source
      ///     The frame specification version of the object.
      /// \param[in] Obj
      ///     The previous version of the object.
      /// \param[in] Stream
      ///     Stream from which the origional object was created.
      ///
      /// \return
      ///     A new instance of FrameH based on Obj
      //-----------------------------------------------------------------
      static promote_ret_type
      Promote( INT_2U Source,
	       promote_arg_type Obj,
	       Common::IStream* Stream );

#if WORKING_VIRTUAL_TOCQUERY
      //-----------------------------------------------------------------
      /// \brief Answer requests for information as needed by TOC
      ///
      /// \param[in] InfoClass
      ///     Class of information being requested.
      //-----------------------------------------------------------------
      virtual void TOCQuery( int InfoClass, ... ) const;
#endif /*  WORKING_VIRTUAL_TOCQUERY */

      //-----------------------------------------------------------------
      /// \brief equality operator
      ///
      /// \param[in] RHS
      ///     The FrSerData object to be compared.
      ///
      /// \return
      ///     The value true is returned if this object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      bool operator==( const FrSerData& RHS ) const;

      //-----------------------------------------------------------------
      /// \brief equality operator for abstract data type
      ///
      /// \param[in] RHS
      ///     The object to be compared.
      ///
      /// \return
      ///     The value true is returned if this object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      virtual bool operator==( const Common::FrameSpec::Object& RHS ) const;

#if ! WORKING_VIRTUAL_TOCQUERY
      //-----------------------------------------------------------------
      /// \brief Answer requests for information as needed by TOC
      ///
      /// \param[in] InfoClass
      ///     Class of information being requested.
      /// \param[in] vl
      ///     Variable argument list of requested data.
      //-----------------------------------------------------------------
      virtual void vTOCQuery( int InfoClass, va_list vl ) const;
#endif /* ! WORKING_VIRTUAL_TOCQUERY */

    protected:
      //-----------------------------------------------------------------
      /// \brief Demotes object to previous version of the frame spec
      ///
      /// \param[in] Target
      ///     The version of the frame specification to demote too.
      /// \param[in] Obj
      ///     The version of the object to demote.
      /// \param[in] Stream
      ///     The input stream from which the original object was read.
      ///
      /// \return
      ///     An object of the previous generation.
      //-----------------------------------------------------------------
      virtual demote_ret_type
      demote( INT_2U Target,
	      demote_arg_type Obj,
	      Common::IStream* Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Promotes object to another version of the frame spec
      ///
      /// \param[in] Target
      ///     The version of the promoted frame specification.
      /// \param[in] Obj
      ///     The object to be promoted.
      /// \param[in] Stream
      ///     The input stream from which the original object was read.
      ///
      /// \return
      ///     An object promoted to the next generation.
      //-----------------------------------------------------------------
      virtual promote_ret_type
      promote( INT_2U Target,
	       promote_arg_type Obj,
	       Common::IStream* Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Number of bytes needed to write this structure
      ///
      /// \param[in] Stream
      ///     The stream from which to the object is being read or
      ///     written.
      ///
      /// \return
      ///     The number of bytes need to read or write this object.
      //-----------------------------------------------------------------
      cmn_streamsize_type pBytes( const Common::StreamBase& Stream) const;

      //-----------------------------------------------------------------
      /// \brief Virtual constructor
      ///
      /// \param[in] Stream
      ///     The input stream from where the object is being read.
      //-----------------------------------------------------------------
      virtual FrSerData* pCreate( Common::IStream& Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Write the structure to the stream
      ///
      /// \param[in] Stream
      ///     The output stream where the object is to be written.
      //-----------------------------------------------------------------
      virtual void pWrite( Common::OStream& Stream ) const;

    private:
      //-----------------------------------------------------------------
      /// \brief Object with checksum data.
      //-----------------------------------------------------------------
      typedef ObjectWithChecksum< Common::FrameSpec::Object > object_type;

      using Common::FrameSpec::Object::Create;

      struct fr_ser_data_data_type {
	bool operator==( const fr_ser_data_data_type& RHS ) const;

	//---------------------------------------------------------------
	/// Name of station producing serial data stream
	//---------------------------------------------------------------
	STRING		name;
	//---------------------------------------------------------------
	/// Time of data acquisition,
	/// GPS time in seconds since GPS standard epoch.
	//---------------------------------------------------------------
	GPSTime		time;
	//---------------------------------------------------------------
	/// Sample rate, samles per second.
	//---------------------------------------------------------------
	REAL_8		sampleRate;
	//---------------------------------------------------------------
	/// Pointer to strin for ASCII-based data.
	//---------------------------------------------------------------
	STRING		data;
	//---------------------------------------------------------------
	/// Identifier for serial data vector.
	//---------------------------------------------------------------
	serial_type	serial;
	//---------------------------------------------------------------
	/// Identifier for user-defined table structure.
	//---------------------------------------------------------------
	table_type	table;
      };

      fr_ser_data_data_type	m_data;

      //-----------------------------------------------------------------
      /// \brief Stream Constructor
      ///
      /// \param[in] Stream
      ///     The stream from which the object is being read.
      ///
      /// \return
      ///    A new instance of this object.
      //-----------------------------------------------------------------
      FrSerData( Common::IStream& Stream );
    };

    inline FrSerData::promote_ret_type FrSerData::
    Promote( INT_2U Source,
	     promote_arg_type Obj,
	     Common::IStream* Stream )
    {
      return Object::PromoteObject< Previous::FrSerData, FrSerData >
	( DATA_FORMAT_VERSION, Source, Obj, Stream );
    }

    inline const char* FrSerData::
    StructName( )
    {
      static const char* name = "FrSerData";
      return name;
    }

    inline const GPSTime& FrSerData::
    GetTime( ) const
    {
      return m_data.time;
    }

    inline REAL_8 FrSerData::
    GetSampleRate( ) const
    {
      return m_data.sampleRate;
    }

    inline const std::string& FrSerData::
    GetData( ) const
    {
      return m_data.data;
    }

    inline void FrSerData::
    SetData( const std::string& Data )
    {
         m_data.data = Data;
         return;
    }

    inline const FrSerData::serial_type& FrSerData::
    RefSerial( ) const
    {
      return m_data.serial;
    }

    inline FrSerData::serial_type& FrSerData::
    RefSerial( )
    {
      return m_data.serial;
    }
    
    inline const FrSerData::table_type& FrSerData::
    RefTable( ) const
    {
      return m_data.table;
    }

    inline FrSerData::table_type& FrSerData::
    RefTable( )
    {
      return m_data.table;
    }
   
    inline bool FrSerData::
    operator==( const FrSerData& RHS ) const
    {
      return ( m_data == RHS.m_data );
    }
  } // namespace - Version_8
} // namespace FrameCPP
#endif /* FrameCPP_VERSION_8_FrSerData_HH */
