#ifndef FrameCPP_VERSION_8_FrDetector_HH
#define FrameCPP_VERSION_8_FrDetector_HH

#ifndef SWIGIMPORTED
#include <string>
#include <vector>
#endif /* SWIGIMPORTED */

#if ! defined(SWIGIMPORTED)
#include "ldastoolsal/types.hh"

#include "framecpp/Version8/FrameSpec.hh"

#include "framecpp/Common/SearchContainer.hh"
#include "framecpp/Common/TOCInfo.hh"
#include "framecpp/Common/FrDetector.hh"

#include "framecpp/Version7/FrDetector.hh"

#include "framecpp/Version8/GPSTime.hh"
#include "framecpp/Version8/STRING.hh"

#include "framecpp/Version8/FrTable.hh"
#include "framecpp/Version8/FrVect.hh"
#endif /* ! defined(SWIGIMPORTED) */

namespace FrameCPP
{
  namespace Version_8
  {
    //===================================================================
    /// \brief Detector data
    //===================================================================
    class FrDetector
#if ! defined(SWIG)
      : public ObjectWithChecksum< Common::FrDetector >,
	public Common::TOCInfo
#endif /* ! defined(SWIG) */
    {
    public:
#if ! defined(SWIGIMPORTED)
      typedef Common::SearchContainer< FrVect, &FrVect::GetName >
      aux_type;
      typedef aux_type::iterator aux_iterator;
      typedef aux_type::const_iterator const_aux_iterator;

      typedef Common::SearchContainer< FrTable, &FrTable::GetNameSlow >
      table_type;
      typedef table_type::iterator table_iterator;
      typedef table_type::const_iterator const_table_iterator;
#endif /* ! defined(SWIGIMPORTED) */

      //-----------------------------------------------------------------
      /// \brief Default constructor
      ///
      /// \return
      ///     A new instance of an FrDetector object
      //-----------------------------------------------------------------
      FrDetector( );

      //-----------------------------------------------------------------
      /// \brief Constructor
      ///
      /// \param[in] Name
      ///     Instrument name
      /// \param[in] Prefix
      ///     Channel prefix for this detector
      /// \param[in] Longitude
      ///     Detector vertex longitude
      /// \param[in] Latitude
      ///     Detector vertex latitutde.
      /// \param[in] Elevation
      ///     Vertex elevation.
      /// \param[in] ArmXazimuth
      ///     Orientation of X arm.
      /// \param[in] ArmYazimuth
      ///     Orientation of Y arm.
      /// \param[in] ArmXaltitude
      ///     Altidue angle of X arm.
      /// \param[in] ArmYaltitude
      ///     Altidue angle of Y arm.
      /// \param[in] ArmXmidpoint
      ///     Distance between the detector vertex and the middle of
      ///     the X cavity.
      /// \param[in] ArmYmidpoint
      ///     Distance between the detector vertex and the middle of
      ///     the Y cavity.
      /// \param[in] LocalTime
      ///     Local seasonal time.
      ///
      /// \return
      ///     A new instance of an FrDetector object
      //-----------------------------------------------------------------
      FrDetector( const std::string& Name,
		  const char* Prefix,
		  const REAL_8 Longitude,
		  const REAL_8 Latitude,
		  const REAL_4 Elevation,
		  const REAL_4 ArmXazimuth,
		  const REAL_4 ArmYazimuth,
		  const REAL_4 ArmXaltitude,
		  const REAL_4 ArmYaltitude,
		  const REAL_4 ArmXmidpoint,
		  const REAL_4 ArmYmidpoint,
		  const INT_4S LocalTime );

      //-----------------------------------------------------------------
      /// \brief Copy constructor
      ///
      /// \param[in] detector
      ///     The source instance to copy
      ///
      /// \return
      ///     A new instance of an FrDetector object
      //-----------------------------------------------------------------
      FrDetector( const FrDetector& detector );

      //-----------------------------------------------------------------
      /// \brief Promotion Constructor
      ///
      /// \param[in] Source
      ///     An FrDetector structure from a previous frame specification.
      /// \param[in] Stream
      ///     The stream from which the earliest version of the
      ///     FrDetector structure was read.
      ///
      /// \return
      ///     A new instance of an FrDetector object
      //-----------------------------------------------------------------
      FrDetector( Previous::FrDetector& Source,
		  istream_type* Stream );

      //-----------------------------------------------------------------
      /// \brief The name structure name of this object.
      ///
      /// \return
      ///     The name of the structure as specified by the frame
      ///     specification.
      //-----------------------------------------------------------------
      static const char* StructName( );

      //-----------------------------------------------------------------
      /// \brief The description of structure
      ///
      /// \return
      ///     A Description object which describes this structure as
      ///     specified by the frame specification.
      //-----------------------------------------------------------------
      static const Common::Description* StructDescription( );

      //-----------------------------------------------------------------
      /// \brief The name structure name of this object.
      ///
      /// \return
      ///     The name of the structure as specified by the frame
      ///     specification.
      //-----------------------------------------------------------------
      virtual const char* ObjectStructName( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the name of the instrument.
      ///
      /// \return
      ///     The name of the instrument.
      //-----------------------------------------------------------------
      const std::string& GetName( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the prefix of the instrument.
      ///
      /// \return
      ///     The prefix for the instrument.
      //-----------------------------------------------------------------
      const CHAR* GetPrefix( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the longitude of the detector vertex.
      ///
      /// \return
      ///     The longitude of the detector vertex.
      //-----------------------------------------------------------------
      REAL_8 GetLongitude( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the latitude of the detector vertex.
      ///
      /// \return
      ///     The latitude of the detector vertex.
      //-----------------------------------------------------------------
      REAL_8 GetLatitude( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the vertex elevation of the detector.
      ///
      /// \return
      ///     The vertex elevation of the detector.
      //-----------------------------------------------------------------
      REAL_4 GetElevation( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the orientation of X arm of the detector.
      ///
      /// \return
      ///     The orientation of the X arm of the detector.
      //-----------------------------------------------------------------
      REAL_4 GetArmXazimuth( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the orientation of Y arm of the detector.
      ///
      /// \return
      ///     The orientation of the Y arm of the detector.
      //-----------------------------------------------------------------
      REAL_4 GetArmYazimuth( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the altitude angle of X arm of the detector.
      ///
      /// \return
      ///     The altitude angle of the X arm of the detector.
      //-----------------------------------------------------------------
      REAL_4 GetArmXaltitude( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the altitude angle of Y arm of the detector.
      ///
      /// \return
      ///     The altitude angle of the Y arm of the detector.
      //-----------------------------------------------------------------
      REAL_4 GetArmYaltitude( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the midpoint of the X arm of the detector.
      ///
      /// \return
      ///     The midpoint of the X arm of the detector.
      //-----------------------------------------------------------------
      REAL_4 GetArmXmidpoint( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the midpoint of the Y arm of the detector.
      ///
      /// \return
      ///     The midpoint of the Y arm of the detector.
      //-----------------------------------------------------------------
      REAL_4 GetArmYmidpoint( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve the local seasonal time.
      ///
      /// \return
      ///     The local seasonal time.
      //-----------------------------------------------------------------
      INT_4S GetLocalTime( ) const;

      //-----------------------------------------------------------------
      /// \brief Identifier for vector for user-provided information.
      ///
      /// \return
      ///     A constant reference to the user-provided information.
      //-----------------------------------------------------------------
      const aux_type& RefAux( ) const;

      //-----------------------------------------------------------------
      /// \brief Identifier for vector for user-provided information.
      ///
      /// \return
      ///     A reference to the user-provided information.
      //-----------------------------------------------------------------
      aux_type& RefAux( );

      //-----------------------------------------------------------------
      /// \brief Identifier for user-provided information in table format.
      ///
      /// \return
      ///     A constant reference to the user-provided table information.
      //-----------------------------------------------------------------
      const table_type& RefTable( ) const;

      //-----------------------------------------------------------------
      /// \brief Identifier for user-provided information in table format.
      ///
      /// \return
      ///     A reference to the user-provided table information.
      //-----------------------------------------------------------------
      table_type& RefTable( );
   
      //-----------------------------------------------------------------
      /// \brief Merge with another FrDetector
      ///
      /// \param[in] RHS
      ///     The source of the information to append to this FrDetector
      ///     structure.
      ///
      /// \return
      ///     A reference to this object
      //-----------------------------------------------------------------
      FrDetector& Merge( const FrDetector& RHS );

      //-----------------------------------------------------------------
      /// \brief Promote previous version of object to this version
      ///
      /// \param[in] Source
      ///     The frame specification version of the object.
      /// \param[in] Obj
      ///     The previous version of the object.
      /// \param[in] Stream
      ///     Stream from which the origional object was created.
      ///
      /// \return
      ///     A new instance of FrameH based on Obj
      //-----------------------------------------------------------------
      static promote_ret_type
      Promote( INT_2U Source,
	       promote_arg_type Obj,
	       istream_type* Stream );

      //-----------------------------------------------------------------
      /// \brief equality operator
      ///
      /// \param[in] RHS
      ///     The FrDetector object to be compared.
      ///
      /// \return
      ///     The value true is returned if the object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      bool operator==( const FrDetector& RHS ) const;

      //-----------------------------------------------------------------
      /// \brief equality operator for abstract data type
      ///
      /// \param[in] RHS
      ///     The object to be compared.
      ///
      /// \return
      ///     The value true is returned if this object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      virtual bool operator==( const Common::FrameSpec::Object& RHS ) const;

#if WORKING_VIRTUAL_TOCQUERY
      //-----------------------------------------------------------------
      /// \brief Answer requests for information as needed by TOC
      ///
      /// \param[in] InfoClass
      ///     Class of information being requested.
      //-----------------------------------------------------------------
      virtual void TOCQuery( int InfoClass, ... ) const;
#endif /*  WORKING_VIRTUAL_TOCQUERY */

    protected:
      //-----------------------------------------------------------------
      /// \brief Demotes object to previous version of the frame spec
      ///
      /// \param[in] Target
      ///     The version of the frame specification to demote too.
      /// \param[in] Obj
      ///     The version of the object to demote.
      /// \param[in] Stream
      ///     The input stream from which the original object was read.
      ///
      /// \return
      ///     An object of the previous generation.
      //-----------------------------------------------------------------
      virtual demote_ret_type
      demote( INT_2U Target,
	      demote_arg_type Obj,
	      istream_type* Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Promotes object to another version of the frame spec
      ///
      /// \param[in] Target
      ///     The version of the promoted frame specification.
      /// \param[in] Obj
      ///     The object to be promoted.
      /// \param[in] Stream
      ///     The input stream from which the original object was read.
      ///
      /// \return
      ///     An object promoted to the next generation.
      //-----------------------------------------------------------------
      virtual promote_ret_type
      promote( INT_2U Target,
	       promote_arg_type Obj,
	       istream_type* Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Number of bytes needed to write this structure
      ///
      /// \param[in] Stream
      ///     The stream from which to the object is being read or
      ///     written.
      ///
      /// \return
      ///     The number of bytes need to read or write this object.
      //-----------------------------------------------------------------
      cmn_streamsize_type pBytes( const Common::StreamBase& Stream) const;

      //-----------------------------------------------------------------
      /// \brief Virtual constructor
      //-----------------------------------------------------------------
      virtual FrDetector* pCreate( istream_type& Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Write the structure to the stream
      ///
      /// \param[in] Stream
      ///     The output stream where the object is to be written.
      //-----------------------------------------------------------------
      virtual void pWrite( ostream_type& Stream ) const;

#if ! WORKING_VIRTUAL_TOCQUERY
      //-----------------------------------------------------------------
      /// \brief Answer requests for information as needed by TOC
      ///
      /// \param[in] InfoClass
      ///     Class of information being requested.
      /// \param[in] vl
      ///     Variable argument list of requested data.
      //-----------------------------------------------------------------
      virtual void vTOCQuery( int InfoClass, va_list vl ) const;
#endif /* ! WORKING_VIRTUAL_TOCQUERY */

    private:
      //-----------------------------------------------------------------
      //-----------------------------------------------------------------
      typedef ObjectWithChecksum< Common::FrDetector > object_type;

      using Common::FrameSpec::Object::Create;

      struct fr_detector_data_type {
	bool operator==( const fr_detector_data_type& RHS ) const;
	//---------------------------------------------------------------
	/// Instrument name as described here
	/// (e.g., Virgo; GEO_600; TAMA_300; LHO_2k; LLO_4k; 40M; PNI;
	/// simulated pseudo data - model version etc.)
	//---------------------------------------------------------------
	STRING		name;
	//---------------------------------------------------------------
	/// Channel prefix for this detector as described here.
	//---------------------------------------------------------------
	CHAR		prefix[2];
	//---------------------------------------------------------------
	/// Detector vertex longitude, geographical coordinates:
	/// radians; Value > 0 >= E of Greenwich
	/// (-pi < Longitude <= +pi)
	//---------------------------------------------------------------
	REAL_8		longitude;
	//---------------------------------------------------------------
	/// Detector vertex latitude, geographical coordinated:
	/// radians; Value >= 0 >= N of Equator
	/// (-pi/2 < Latitude <= +pi/2).
	//---------------------------------------------------------------
	REAL_8		latitude;
	//---------------------------------------------------------------
	/// Vertex elevation, in meters, relative to WGS84 ellipsoid.
	//---------------------------------------------------------------
	REAL_4		elevation;
	//---------------------------------------------------------------
	/// Orientation of X arm, measured in radians East of North
	/// (0 <= ArmXazimuth < 2pi)
	//---------------------------------------------------------------
	REAL_4		armXazimuth;
	//---------------------------------------------------------------
	/// Orientation of Y arm, measured in radians East of North
	/// (0 <= ArmYazimuth < 2pi)
	//---------------------------------------------------------------
	REAL_4		armYazimuth;
	//---------------------------------------------------------------
	/// Altitude (pitch) angle to X arm, measured in radians
	/// above horizon (local tangent to WGS84 ellipsoid)
	/// -pi/2 < ArmXaltitude <= pi/2
	//---------------------------------------------------------------
	REAL_4		armXaltitude;
	//---------------------------------------------------------------
	/// Altitude (pitch) angle to Y arm, measured in radians
	/// above horizon (local tangent to WGS84 ellipsoid)
	/// -pi/2 < ArmXaltitude <= pi/2
	//---------------------------------------------------------------
	REAL_4		armYaltitude;
	//---------------------------------------------------------------
	/// Distance between the detector vertex and the middle of the X
	/// cavity (meters) (should be zero for bars).
	//---------------------------------------------------------------
	REAL_4		armXmidpoint;
	//---------------------------------------------------------------
	/// Distance between the detector vertex and the middle of the Y
	/// cavity (meters) (should be zero for bars).
	//---------------------------------------------------------------
	REAL_4		armYmidpoint;
	//---------------------------------------------------------------
	/// Local seasonal time - UTC in seconds.
	/// If localTime % 1800 != 0 then localTime is undefined.
	//---------------------------------------------------------------
	INT_4S		localTime;
	//---------------------------------------------------------------
	/// Indentifier for user-provided structure for additional
	/// detector data.
	//---------------------------------------------------------------
	aux_type	aux;
	//---------------------------------------------------------------
	/// Identifier fo ruser-provided table structure for additional
	/// detector data.
	//---------------------------------------------------------------
	table_type	table;	
      };

      fr_detector_data_type	m_data;

      FrDetector( istream_type& Stream );
    };

    inline const CHAR* FrDetector::
    GetPrefix( ) const
    {
      return m_data.prefix;
    }

    inline REAL_8 FrDetector::
    GetLongitude( ) const
    {
      return m_data.longitude;
    }

    inline REAL_8 FrDetector::
    GetLatitude( ) const
    {
      return m_data.latitude;
    }

    inline REAL_4 FrDetector::
    GetElevation( ) const
    {
      return m_data.elevation;
    }

    inline REAL_4 FrDetector::
    GetArmXazimuth( ) const
    {
      return m_data.armXazimuth;
    }

    inline REAL_4 FrDetector::
    GetArmYazimuth( ) const
    {
      return m_data.armYazimuth;
    }

    inline REAL_4 FrDetector::
    GetArmXaltitude( ) const
    {
      return m_data.armXaltitude;
    }

    inline REAL_4 FrDetector::
    GetArmYaltitude( ) const
    {
      return m_data.armYaltitude;
    }

    inline REAL_4 FrDetector::
    GetArmXmidpoint( ) const
    {
      return m_data.armXmidpoint;
    }

    inline REAL_4 FrDetector::
    GetArmYmidpoint( ) const
    {
      return m_data.armYmidpoint;
    }

    inline INT_4S FrDetector::
    GetLocalTime( ) const
    {
      return m_data.localTime;
    }

    inline const FrDetector::aux_type& FrDetector::
    RefAux( ) const
    {
      return m_data.aux;
    }

    inline FrDetector::aux_type& FrDetector::
    RefAux( )
    {
      return m_data.aux;
    }

    inline const FrDetector::table_type& FrDetector::
    RefTable( ) const
    {
      return m_data.table;
    }

    inline FrDetector::table_type& FrDetector::
    RefTable( )
    {
      return m_data.table;
    }
   
    inline FrDetector::promote_ret_type FrDetector::
    Promote( INT_2U Source,
	     promote_arg_type Obj,
	     istream_type* Stream )
    {
      return Object::PromoteObject< Previous::FrDetector, FrDetector >
	( DATA_FORMAT_VERSION, Source, Obj, Stream );
    }

    inline const char* FrDetector::
    StructName( )
    {
      static const char* name = "FrDetector";
      return name;
    }

    inline bool FrDetector::
    operator==( const FrDetector& RHS ) const
    {
      return ( m_data == RHS.m_data );
    }
  } // namespace - Version_8
} // namespace FrameCPP
#endif /* FrameCPP_VERSION_8_FrDetector_HH */
