/* -*- mode: c++; c-basic-offset: 3 -*- */
#include <frameutils_config.h>

#include <unistd.h>
#include <string.h>

#include <cctype>
#include <cerrno>

#include <algorithm>   
#include <iterator>
#include <memory>   
#include <sstream>
#include <stdexcept>

// General
#include "ldastoolsal/MemChecker.hh"
#include "ldastoolsal/autoarray.hh"
#include "ldastoolsal/gpstime.hh"
#include "ldastoolsal/regex.hh"
#include "ldastoolsal/regexmatch.hh"
#include "ldastoolsal/unordered_map.hh"
   
#include "framecpp/Common/CheckSum.hh"
#include "framecpp/Common/FrTOC.hh"
#include "framecpp/Common/MD5Sum.hh"
#include "framecpp/Common/Verify.hh"

#include "framecpp/FrDetector.hh"
#include "framecpp/FrHistory.hh"
#include "framecpp/FrTOC.hh"
#include "framecpp/FrVect.hh"

// GenericAPI Header Files   
#include "genericAPI/swigexception.hh"
#include "genericAPI/Logging.hh"

// Local Header Files
#include "DeviceIOConfiguration.hh"
#include "rdsframe.hh"
#include "rdsutil.hh"   
#include "RDSStream.hh"
#include "filereader.hh"

#include "FileCache.icc"

#include "util.hh"

using namespace std;
using namespace FrameCPP;

using LDASTools::AL::MemChecker;
using LDASTools::AL::AutoArray;
using LDASTools::AL::GPSTime;   

using FrameCPP::Common::MD5Sum;
using FrameCPP::Common::CheckSum;
using FrameAPI::DeviceIOConfiguration;
using FrameAPI::LogMD5Sum;
using FrameAPI::StrToCompressionScheme;

#if 0
#define	DBGMSG( a ) std::cerr << a << __FILE__ << " " << __LINE__ << std::endl
#else
#define	DBGMSG( a )
#endif

#define	AT( a ) DBGMSG( a << " " )
#define	HERE( ) DBGMSG( "" )

namespace
{
   class CollectNames
      : public FrameCPP::Common::FrTOC::FunctionString,
	public std::list< std::string >
   {
   public:
      virtual ~CollectNames( );

      virtual void operator()( const std::string& ChannelName );
   };
}

namespace FrameAPI
{
   namespace RDSFrameCC
   {
      static const std::string mHistoryName( "createRDS" );

      //----------------------------------------------------------------
      // Note: This routine is only for debugging
      //----------------------------------------------------------------
      std::ostream&
      operator<<( std::ostream& Stream, const FileCache& Source )
      {
	 for ( FileCache::entry_type::const_iterator
		  current( Source.begin( ) ),
		  end( Source.end( ) );
	       current != end;
	       current++ )
	 {
	    Stream << *current;
	 }
	 return Stream;
      }

      //----------------------------------------------------------------
      //----------------------------------------------------------------
      class ChannelEntry
      {
      public:
	 typedef ChannelCacheEntry::fr_adc_data_type fr_adc_data_type;
	 typedef ChannelCacheEntry::fr_proc_data_type fr_proc_data_type;

	 enum channel_type {
	    UNKNOWN,
	    ADC,
	    PROC
	 };


	 ChannelEntry( const std::string& Name );
	 ChannelEntry( const ChannelEntry& Source );
	 const std::string& GetName( ) const;
	 fr_adc_data_type& FrAdcData( );
	 fr_proc_data_type& FrProcData( );
	 void Reset( );

      private:
	 friend std::ostream& operator<<( std::ostream&, const ChannelEntry& );
	 channel_type		m_channel_kind;
	 std::string		m_name;
	 fr_adc_data_type	m_adc;
	 fr_proc_data_type	m_proc;
      };
      
      ChannelEntry::fr_adc_data_type& ChannelEntry::
      FrAdcData( )
      {
	 return m_adc;
      }
      
      ChannelEntry::fr_proc_data_type& ChannelEntry::
      FrProcData( )
      {
	 return m_proc;
      }

      void ChannelEntry::
      Reset( )
      {
	 m_adc.reset( );
	 m_proc.reset( );
      }
      
      inline const std::string& ChannelEntry::
      GetName( ) const
      {
	 return m_name;
      }

      std::ostream&
      operator<<( std::ostream& Stream, const ChannelEntry& Source )
      {
	 Stream << Source.m_name;
	 return Stream;
      }


   } // namespace - RDSFrameCC
} // FrameAPI

using namespace FrameAPI::RDSFrameCC;

//-----------------------------------------------------------------------
// 
//-----------------------------------------------------------------------
typedef std::vector< ChannelEntry > channel_cache_type;

//=======================================================================
// class - RDSFrame::Options
//=======================================================================
namespace FrameAPI
{
   namespace RDS
   {
      MissingChannel::
      MissingChannel( const std::string& ChannelName )
	 : std::runtime_error( create_error_msg( ChannelName ) )
      {
      }

      MissingChannel::
      MissingChannel( const std::string& ChannelName,
		      const LDASTools::AL::GPSTime& Start,
		      const LDASTools::AL::GPSTime& Stop )
	 : std::runtime_error( create_error_msg( ChannelName, Start, Stop ) )
      {
      }

      std::string MissingChannel::
      create_error_msg( const std::string& ChannelName )
      {
	 std::ostringstream	retval;

	 retval << "Missing channel: " << ChannelName
	    ;
	 return retval.str( );
      }

      std::string MissingChannel::
      create_error_msg( const std::string& ChannelName,
			const LDASTools::AL::GPSTime& Start,
			const LDASTools::AL::GPSTime& Stop )
      {
	 std::ostringstream	retval;

	 retval << "Unable to locate the channel named: " << ChannelName
		<< " in any of the input frames for the time range: "
		<< Start << " - " << Stop
	    ;
	 return retval.str( );
      }

      Options::
      Options( )
	 : history_record( true ),
	   m_compression_method( StrToCompressionScheme( "zero_suppress_otherwise_gzip" ) ),
	   m_compression_level( 1 ),
	   m_frames_per_file( 1 ),
	   m_input_verify_checksum( false ),
	   m_input_verify_checksum_per_frame( false ),
	   m_input_verify_data_valid( false ),
	   m_input_verify_filename_metadata( true ),
	   m_input_verify_time_range( true ),
	   m_output_allow_short_frames( true ),
	   m_output_checksum_per_frame( true ),
	   m_output_fill_data_valid_array( false ),
	   m_output_start( 0 ),
	   m_output_stop( 0 ),
	   m_output_type( "" ),
	   padding( 0 ),
	   m_rds_level( 0 ),
	   m_seconds_per_frame( 0 )
      {
      }

      Options::
      Options( const INT_4U Start,
	       const INT_4U Stop,
	       const char* type,
	       const char* compression_method,
	       const INT_2U level,
	       const bool VerifyChecksum,
	       const bool VerifyFrameChecksum,
	       const bool VerifyTimeRange,
	       const bool VerifyDataValid,
	       const INT_4U FramesPerFile,
	       const INT_4U SecondsPerFrame,
	       const bool AllowShortFrames,
	       const bool GenerateFrameChecksum,
	       const bool FillMissingDataValidArray,
	       const bool VerifyFilenameMetadata )
	 : history_record( true ),
	   m_compression_method( StrToCompressionScheme( compression_method ) ),
	   m_compression_level( level ),
	   m_frames_per_file( FramesPerFile ),
	   m_input_verify_checksum( VerifyChecksum ),
	   m_input_verify_checksum_per_frame( VerifyFrameChecksum ),
	   m_input_verify_data_valid( VerifyDataValid ),
	   m_input_verify_filename_metadata( VerifyFilenameMetadata ),
	   m_input_verify_time_range( VerifyTimeRange ),
	   m_output_allow_short_frames( AllowShortFrames ),
	   m_output_checksum_per_frame( GenerateFrameChecksum ),
	   m_output_fill_data_valid_array( FillMissingDataValidArray ),
	   m_output_start( Start ),
	   m_output_stop( Stop ),
	   m_output_type( type ),
	   padding( 0 ),
	   m_rds_level( 0 ),
	   m_seconds_per_frame( SecondsPerFrame )
      {
	 //-------------------------------------------------------------------
	 // Sanity check on type
	 //-------------------------------------------------------------------
	 m_output_type = validate_type( m_output_type.c_str( ) );
	 //-------------------------------------------------------------------
	 // Sanity check for output directory.
	 //-------------------------------------------------------------------
	 if ( Start > Stop )
	 {
	    std::ostringstream	msg;
	       
	    msg << "The start time (" << m_output_start
		<< ") exceeds the stop time (" << m_output_stop << ")";

	    throw SWIGEXCEPTION( msg.str( ) );
	 }
      }

      Options::
      Options( const Options& Source )
	 : history_record( Source.history_record ),
	   m_compression_method( Source.m_compression_method ),
	   m_compression_level( Source.m_compression_level ),
	   m_frames_per_file( Source.m_frames_per_file ),
	   m_input_verify_checksum( Source.m_input_verify_checksum ),
	   m_input_verify_checksum_per_frame( Source.m_input_verify_checksum_per_frame ),
	   m_input_verify_data_valid( Source.m_input_verify_data_valid ),
	   m_input_verify_filename_metadata( Source.m_input_verify_filename_metadata ),
	   m_input_verify_time_range( Source.m_input_verify_time_range ),
	   m_output_allow_short_frames( Source.m_output_allow_short_frames ),
	   m_output_checksum_per_frame( Source.m_output_checksum_per_frame ),
	   m_output_fill_data_valid_array( Source.m_output_fill_data_valid_array),
	   m_output_start( Source.m_output_start ),
	   m_output_stop( Source.m_output_stop ),
	   m_output_type( Source.m_output_type ),
	   padding( Source.Padding( ) ),
	   m_rds_level( Source.m_rds_level ),
	   m_seconds_per_frame( Source.m_seconds_per_frame )
      {
      }

      void Options::
      CompressionMethod( const std::string& Value )
      {
	 m_compression_method = StrToCompressionScheme( Value.c_str( ) );
      }

      std::string Options::
      validate_type( const char* Type )
      {
	 //-------------------------------------------------------------------
	 // Initialization
	 //-------------------------------------------------------------------
	 std::string retval( Type );
	 m_rds_level = 0;

	 //-------------------------------------------------------------------
	 // Sanity check for type specifier
	 //-------------------------------------------------------------------
	 if ( retval.length( ) > 0 )
	    {
	       static const char* good_characters =
		  "abcdefghijklmnopqrstuvwxyz"
		  "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
		  "0123456789_";

	       if ( retval.find_first_not_of( good_characters ) != std::string::npos )
		  {
		     throw SWIGEXCEPTION( "Invalid character in type field" );
		  }
	       //-----------------------------------------------------------------
	       // Search for the version level
	       //-----------------------------------------------------------------
	       std::string::size_type level_pos =
		  retval.find_last_not_of("0123456789");

	       if ( level_pos != std::string::npos )
		  {
		     std::istringstream lvl( retval.substr( ++level_pos ) );
		     lvl >> m_rds_level;		// Translate ASCII to int
		     retval.erase( level_pos );	// Remove level from string
		  }
	    }
	 else
	    {
	       retval = "RDS_R_L";
	    }
	 return retval;
      }

      FileOptions::
      FileOptions( )
	 : Options( ),
	   m_output_directory( ),
	   m_md5sum_output_directory( )
      {
      }

      FileOptions::
      FileOptions( const INT_4U Start,
		   const INT_4U Stop,
		   const char* dir,
		   const char* type,
		   const char* compression_method,
		   const INT_2U level,
		   const bool VerifyChecksum,
		   const bool VerifyFrameChecksum,
		   const bool VerifyTimeRange,
		   const bool VerifyDataValid,
		   const INT_4U FramesPerFile,
		   const INT_4U SecondsPerFrame,
		   const bool AllowShortFrames,
		   const bool GenerateFrameChecksum,
		   const bool FillMissingDataValidArray,
		   const bool VerifyFilenameMetadata,
		   const std::string& MD5SumOutputDirectory )
	 : Options( Start,
		    Stop,
		    type,
		    compression_method,
		    level,
		    VerifyChecksum,
		    VerifyFrameChecksum,
		    VerifyTimeRange,
		    VerifyDataValid,
		    FramesPerFile,
		    SecondsPerFrame,
		    AllowShortFrames,
		    GenerateFrameChecksum,
		    FillMissingDataValidArray,
		    VerifyFilenameMetadata ),
	   m_output_directory( dir ),
	   m_md5sum_output_directory( MD5SumOutputDirectory )
      {
	 //-------------------------------------------------------------------
	 // Sanity check for output directory.
	 //-------------------------------------------------------------------
	 if ( m_output_directory.empty( ) )
	 {
	       throw SWIGEXCEPTION( "Output directory must be specified." );      
	 }
	 //-------------------------------------------------------------------
	 // Make sure the output directory is writable by user
	 // :TODO: Need to create test for this new code
	 //-------------------------------------------------------------------
	 if ( access( m_output_directory.c_str( ),
		      R_OK | W_OK | X_OK ) != 0 )
	 {
	    std::ostringstream	msg;

	    switch( errno )
	    {
	    case ENOENT:
	       msg << "The specified output directory ("
		   << m_output_directory.c_str( )
		   << ") does not exist";
	       break;
	    default:
	       msg << "Insufficient permission to create files under the directory \""
		   << m_output_directory.c_str( )
		   << "\"";
	       break;
	    }
	    throw SWIGEXCEPTION( msg.str( ) );
	 }
      }

      FileOptions::
      FileOptions( const FileOptions& Source )
	 : Options( Source ),
	   m_output_directory( Source.m_output_directory ),
	   m_md5sum_output_directory( Source.m_md5sum_output_directory )
      {
      }
	   
   } // namespace - RDS
} // namespace - FrameAPI

//=======================================================================
// class RDSFrame::private_type
//=======================================================================
class RDSFrame::private_type
{
public:
   typedef LDASTools::AL::SharedPtr< FrameH > frame_h_type;
   typedef LDASTools::AL::SharedPtr< FrAdcData > fr_adc_data_type;
   typedef LDASTools::AL::SharedPtr< FrProcData > fr_proc_data_type;

   private_type( );

   void Close( );
   frame_h_type CreateFrame( const LDASTools::AL::GPSTime& Start,
			     const LDASTools::AL::GPSTime& Stop );
   frame_h_type CreateFrame( const LDASTools::AL::GPSTime& Start,
			     REAL_8 DeltaT );
   const channel_cache_type& GetChannelCache( ) const;
   channel_cache_type& GetChannelCache( );
   const FileCache& GetFileCache( ) const;
   //
   char* GetBuffer( ) const;
   unsigned int GetBufferSize( ) const;
   // Get the starting time of FileCache
   LDASTools::AL::GPSTime GetStart( ) const;
   LDASTools::AL::GPSTime GetStop( ) const;
   void InitChannelCache( const channel_container_type& Channels );
   INT_4U InitFileCache( const RDSFrame::Options& Options,
			 const std::vector< std::string >& FileList );

   FrameCPP::Common::MD5Sum& MD5Sum( );

   void PurgeFilesOlderThan( const LDASTools::AL::GPSTime& Start );

   void RetrieveChannel( const std::string& Channel,
			 frame_h_type& Frame,
			 const LDASTools::AL::GPSTime& Start,
			 REAL_8& DeltaT,
			 fr_adc_data_type& Adc,
			 fr_proc_data_type& Proc,
			 const bool VerifyDataValid,
			 const bool FillMissingDataValidArray );
   void RetrieveChannels( frame_h_type& Frame,
			  RDSFrame& OutputFrame,
			  const LDASTools::AL::GPSTime& Start,
			  const REAL_8 DeltaT,
			  const RDSFrame::Options& Options );

private:
   channel_cache_type	m_channel_cache;
   FileCache		m_file_cache;
   AutoArray< char >	m_buffer;
   unsigned int		m_buffer_size;
};

inline void RDSFrame::private_type::
Close( )
{
   m_file_cache.Close( );
}

inline const channel_cache_type& RDSFrame::private_type::
GetChannelCache( ) const
{
   return m_channel_cache;
}

inline channel_cache_type& RDSFrame::private_type::
GetChannelCache( )
{
   return m_channel_cache;
}

inline char* RDSFrame::private_type::
GetBuffer( ) const
{
   return m_buffer.get( );
}

inline unsigned int RDSFrame::private_type::
GetBufferSize( ) const
{
   return m_buffer_size;
}

inline const FileCache& RDSFrame::private_type::
GetFileCache( ) const
{
   return m_file_cache;
}

inline LDASTools::AL::GPSTime RDSFrame::private_type::
GetStart( ) const
{
   return m_file_cache.GetStart( );
}

inline LDASTools::AL::GPSTime RDSFrame::private_type::
GetStop( ) const
{
   return m_file_cache.GetStop( );
}

void RDSFrame::private_type::
InitChannelCache( const channel_container_type& Channels )
{
   for ( channel_container_type::const_iterator
	    cur = Channels.begin( ),
	    last = Channels.end( );
	 cur != last;
	 ++cur )
   {
      m_channel_cache.push_back( ChannelEntry( *cur ) );
   }
}

inline void RDSFrame::private_type::
PurgeFilesOlderThan( const LDASTools::AL::GPSTime& Start )
{
   m_file_cache.PurgeIfOlderThan( Start );
}

void RDSFrame::private_type::
RetrieveChannel( const std::string& Channel,
		 frame_h_type& Frame,
		 const LDASTools::AL::GPSTime& Start,
		 REAL_8& DeltaT,
		 fr_adc_data_type& Adc,
		 fr_proc_data_type& Proc,
		 const bool VerifyDataValid,
		 const bool FillMissingDataValidArray )
{
   //--------------------------------------------------------------------
   // Look in the file chache for the given channels
   //--------------------------------------------------------------------
   const REAL_8	orig_dt = DeltaT;
   for ( FileCache::iterator
	    current( m_file_cache.begin( ) ),
	    end( m_file_cache.end( ) );
	 current != end;
	 current++ )
   {
      //-----------------------------------------------------------------
      // Locate a file that should have the requested channel
      //-----------------------------------------------------------------
      if ( (*current).ContainsIFO( Channel[0] ) )
      {
	 if ( (*current).ReadChannel( Channel,
				      Frame,
				      Start, DeltaT,
				      Adc, Proc,
				      VerifyDataValid,
				      FillMissingDataValidArray ) )
	 {
	    //-----------------------------------------------------------
	    // Channel was found and has been read in.
	    //-----------------------------------------------------------
	    return;
	 }
      }
   }
   //--------------------------------------------------------------------
   // No channel was located, time to throw an excpetion
   //--------------------------------------------------------------------
   throw FrameAPI::RDS::MissingChannel( Channel, Start, ( Start + orig_dt ) );
}


void RDSFrame::private_type::
RetrieveChannels( frame_h_type& Frame,
		  RDSFrame&	OutputFrame,
		  const LDASTools::AL::GPSTime& Start,
		  const REAL_8 DeltaT,
		  const RDSFrame::Options& Options )
{
   static const char* caller = "RDSFrame::private_type::RetrieveChannels";

   if ( Frame )
   {
      QUEUE_LOG_MESSAGE( "Frame: " << Frame->GetName( )
			 << " FrAdcData Entries: "  << ( ( Frame->GetRawData( ) )
							 ? Frame->GetRawData( )->RefFirstAdc( ).size( )
							 : 0 )
			 << " FrProcData: " << Frame->RefProcData( ).size( )
			 ,
			 MT_DEBUG, 40,
			 caller,
			 "CreateRDS" );
   }
   //--------------------------------------------------------------------
   // Reset the dynamic parts of the channel entry
   //--------------------------------------------------------------------
   for ( channel_cache_type::iterator
	    current( GetChannelCache( ).begin( ) ),
	    end( GetChannelCache( ).end( ) );
	 current != end;
	 ++current )
   {
      (*current).Reset( );
   }

   //--------------------------------------------------------------------
   // Loop over all channels in all files creating the output data
   //--------------------------------------------------------------------

   LDASTools::AL::GPSTime			end_time( Start + DeltaT );
   LDASTools::AL::GPSTime			cur_start_time( Start );
   LDASTools::AL::GPSTime			min_start_time( end_time );

   REAL_8		cur_dt = DeltaT;

   while( cur_dt > 0 )
   {
      for ( channel_cache_type::iterator
	       current( GetChannelCache( ).begin( ) ),
	       end( GetChannelCache( ).end( ) );
	    current != end;
	    ++current )
      {
	 REAL_8	dt = cur_dt;

	 RetrieveChannel( (*current).GetName( ),
			  Frame,
			  cur_start_time, dt,
			  (*current).FrAdcData( ),
			  (*current).FrProcData( ),
			  Options.VerifyDataValid( ),
			  Options.FillMissingDataValidArray( ) );
	 //--------------------------------------------------------------
	 // Setup for next batch of reads
	 //--------------------------------------------------------------
	 if ( ( cur_start_time + dt ) < min_start_time )
	 {
	    min_start_time = ( cur_start_time + dt );
	 }
      }
      if ( cur_start_time == min_start_time )
      {
	 std::ostringstream	msg;

	 msg << "data missing at GPS time: " << cur_start_time.GetSeconds( );
	 throw std::runtime_error( msg.str( ) );
      }
      cur_start_time = min_start_time;
      cur_dt = DeltaT - ( cur_start_time - Start );
      min_start_time = cur_start_time + cur_dt;
      PurgeFilesOlderThan( cur_start_time );
   }
   QUEUE_LOG_MESSAGE( "Frame: " << Frame->GetName( )
		      << " FrAdcData Entries: "  << ( ( Frame->GetRawData( ) )
						      ? Frame->GetRawData( )->RefFirstAdc( ).size( )
						      : 0 )
		      << " FrProcData: " << Frame->RefProcData( ).size( )
		      ,
		      MT_DEBUG, 40,
		      caller,
		      "CreateRDS" );
}

inline INT_4U RDSFrame::private_type::
InitFileCache( const RDSFrame::Options& Options,
	       const std::vector< std::string >& FileList )
{
   m_file_cache.Init( Options, FileList );
   return m_file_cache.GetRDSLevel( );
}

std::ostream&
operator<<( std::ostream& Stream, const channel_cache_type& Source )
{
   Stream << "{ ";
   for ( channel_cache_type::const_iterator
	    begin( Source.begin( ) ),
	    current( Source.begin( ) ),
	    end( Source.end( ) );
	 current != end;
	 current++ )
   {
      if ( current != begin )
      {
	 Stream << ", ";
      }
      Stream << *current;
   }
   Stream << " }";
   return Stream;
}

RDSFrame::
RDSFrame( const char* frame_files, 
	  const char* channels,
	  const Options& CommandOptions )
  : // protected
    m_options( CommandOptions ),
    // private
    m_p( (private_type*)NULL )
{
   ListParser			frame_files_parser( frame_files );
   ListParser			channel_parser( channels );

   init( frame_files_parser.getTokenList( ),
	 channel_parser.getTokenList( ) );
}
   

RDSFrame::
RDSFrame( const frame_file_container_type& frame_files, 
	  const channel_container_type& channels,
	  const Options& CommandOptions )
  : // protected
    m_options( CommandOptions ),
    // private
    m_p( (private_type*)NULL )
{
   init( frame_files, channels );
}
   
   
RDSFrame::
RDSFrame( const Options& CommandOptions )
  : // protected
    m_options( CommandOptions ),
    // private
    m_p( (private_type*)NULL )
{
}

//-----------------------------------------------------------------------------
//   
//: Destructor.
//
RDSFrame::
~RDSFrame()
{
   delete m_p;
   m_p = (private_type*)NULL;
}

INT_4U RDSFrame::
GetNumberOfChannels( ) const
{
   return m_p->GetChannelCache( ).size( );
}
   
INT_4U RDSFrame::
GetNumberOfFrameGroups( ) const
{
   return m_p->GetFileCache( ).size( );
}

INT_4U RDSFrame::
GetNumberOfFrameFiles( ) const
{
   if ( m_p->GetFileCache( ).size( ) > 0 )
   {
      return m_p->GetFileCache( ).front( ).size( );
   }
   return 0;
}

//-----------------------------------------------------------------------------
//   
//: Main control loop
//
// This method loops through all files and frames within the files.
//   
void RDSFrame::
ProcessRequest( stream_type Output )
{
   static const char* caller = "RDSFrame::ProcessRequest";

   if ( ! Output )
   {
      throw std::runtime_error( "Invalid Output stream" );
   }
   m_stream = Output;
   m_stream->IFOList( m_ifo_list );

   const LDASTools::AL::GPSTime	user_start( m_options.OutputTimeStart( ), 0 );
   const LDASTools::AL::GPSTime	user_stop( m_options.OutputTimeEnd( ), 0 );

   LDASTools::AL::GPSTime	data_start;
   LDASTools::AL::GPSTime	data_stop;
   REAL_8		        data_dt;
   INT_4U 			frames_to_write( 0 );
      

   ChannelCacheEntry::fr_adc_data_type	adc;
   ChannelCacheEntry::fr_proc_data_type	proc;

   //--------------------------------------------------------------------
   // Sanity checks
   //--------------------------------------------------------------------
   if ( m_options.FramesPerFile( ) == 0 )
   {
      throw std::range_error( "The number of frames per file must be greater than 0" );
   }
   //--------------------------------------------------------------------
   // Start processing the requests
   //--------------------------------------------------------------------
   try
   {
      //-----------------------------------------------------------------
      // Initialize
      //-----------------------------------------------------------------
      data_start = m_p->GetStart( );
      data_stop = m_p->GetStop( );
      
      rangeOptimizer( user_start, user_stop,
		      data_start, data_stop );

      m_stream->UserRange( user_start.GetSeconds( ),
			   user_stop.GetSeconds( ) );

      QUEUE_LOG_MESSAGE( "data_start: " << data_start
			 << " data_stop: " << data_stop,
			 MT_DEBUG, 30,
			 caller,
			 "RDSFrame" );

      if ( data_start != GPSTime( m_options.OutputTimeStart( ), 0 ) )
      {
	 data_dt = GPSTime( m_options.OutputTimeStart( ), 0 ) -
	    data_start;
      }
      else
      {
	 data_dt = m_options.SecondsPerFrame( );
      }

      //-----------------------------------------------------------------
      // Create each frame requested by user
      //-----------------------------------------------------------------
      const GPSTime stop( ( stopRequest( ) == STOP_DATA )
			 ? data_stop
			 : user_stop );
      while( data_start < stop )
      {
	 QUEUE_LOG_MESSAGE( "data_start: " << data_start
			    << " user_start: " << user_start
			    << " user_stop: " << user_stop
			    << " data_stop: " << data_stop
			    << " stop: " << stop
			    << " data_dt: " << data_dt
			    << " frames_to_write: " << frames_to_write
			    ,
			    MT_DEBUG, 30,
			    caller,
			    "RDSFrame" );
	 if ( data_start == GPSTime( m_options.OutputTimeStart( ), 0 ) )
	 {
	    // This is the case where the dt needs to be reset.
	    data_dt = m_options.SecondsPerFrame( );
	 }
	 if ( data_start < user_stop )
	 {
	    //----------------------------------------------------------
	    // Check to see if the data_dt needs to be adjusted
	    //----------------------------------------------------------
	    if ( ( data_start + data_dt ) > user_stop )
	    {
	       data_dt = user_stop - data_start;
	    }
	 } else {
	    //-----------------------------------------------------------
	    // For the case where the data stops beyond the user's
	    //   request, squeeze everything into a final frame.
	    //-----------------------------------------------------------
	    data_dt = ( data_stop - data_start );
	 }
	 if ( frames_to_write == 0 )
	 {
	    frames_to_write = m_options.FramesPerFile( );
	    QUEUE_LOG_MESSAGE( "data_start: " << data_start
			       << " data_dt: " << data_dt
			       << " frames_to_write: " << frames_to_write
			       ,
			       MT_DEBUG, 30,
			       caller,
			       "RDSFrame" );
	    m_stream->Next( data_start, data_dt, frames_to_write );
	    QUEUE_LOG_MESSAGE( "frames_to_write: " << frames_to_write
			       ,
			       MT_DEBUG, 30,
			       caller,
			       "RDSFrame" );
	    if ( frames_to_write == 0 )
	    {
	       break;
	    }
	 }
	    
	 //--------------------------------------------------------------
	 // Make room for the new frame
	 //--------------------------------------------------------------
	 mResultFrame.reset( );
	 //--------------------------------------------------------------
	 // Add in channel information
	 //--------------------------------------------------------------
	 m_p->RetrieveChannels( mResultFrame, *this,
				data_start, data_dt,
				m_options );
	 QUEUE_LOG_MESSAGE( "Frame: " << mResultFrame->GetName( )
			    << " FrAdcData Entries: "  << ( ( mResultFrame->GetRawData( ) )
							    ? mResultFrame->GetRawData( )->RefFirstAdc( ).size( )
							    : 0 )
			    << " FrProcData: " << mResultFrame->RefProcData( ).size( )
			    ,
			    MT_DEBUG, 30,
			    caller,
			    "CreateRDS" );
	 //--------------------------------------------------------------
	 // Post process the concatinated data
	 //--------------------------------------------------------------
	 for ( channel_cache_type::iterator
		  current( m_p->GetChannelCache( ).begin( ) ),
		  end( m_p->GetChannelCache( ).end( ) );
	       current != end;
	       ++current )
	 {
	    if ( MemChecker::IsExiting( ) )
	    {
	       std::ostringstream 	msg;
	       
	       msg << "System is exiting";

	       throw std::runtime_error( msg.str( ) );
	    }
	    if ( (*current).FrAdcData( ) )
	    {
	       //--------------------------------------------------------
	       // Processing FrAdcData
	       //--------------------------------------------------------
	       QUEUE_LOG_MESSAGE( "Processing FrAdcData"
			    ,
			    MT_DEBUG, 40,
			    caller,
			    "CreateRDS" );
	       processChannel( (*current).FrAdcData( ) );
	    }
	    else if ( (*current).FrProcData( ) )
	    {
	       //--------------------------------------------------------
	       // Processing FrProcData
	       //--------------------------------------------------------
	       QUEUE_LOG_MESSAGE( "Processing FrProcData"
			    ,
			    MT_DEBUG, 40,
			    caller,
			    "CreateRDS" );
	       processChannel( (*current).FrProcData( ) );
	    }
	    else
	    {
	       std::ostringstream	msg;

	       throw SWIGEXCEPTION( msg.str() );   
	    }
	 }
	 QUEUE_LOG_MESSAGE( "Frame(after): " << mResultFrame->GetName( )
			    << " FrAdcData Entries: "  << ( ( mResultFrame->GetRawData( ) )
							    ? mResultFrame->GetRawData( )->RefFirstAdc( ).size( )
							    : 0 )
			    << " FrProcData: " << mResultFrame->RefProcData( ).size( )
			    ,
			    MT_DEBUG, 30,
			    caller,
			    "CreateRDS" );
	 if ( mResultFrame.get( ) )
	 {
	    //----------------------------------------------------------
	    // Correct some information
	    //----------------------------------------------------------
	    mResultFrame->SetGTime( data_start );
	    mResultFrame->SetDt( data_dt );
	    QUEUE_LOG_MESSAGE( "Setting result frame attributes:"
			       << " data_start: " << data_start
			       << " data_dt: " << data_dt
			       ,
			       MT_DEBUG, 30,
			       caller,
			       "RDSFrame" );
	    //----------------------------------------------------------
	    // Check to see if any data is associated with the FrRawData
	    //   structure
	    //----------------------------------------------------------
	    {
	       LDASTools::AL::SharedPtr< FrameCPP::FrRawData > rd( mResultFrame->GetRawData( ) );
	       
	       if ( ( rd ) &&
		    ( rd->RefFirstAdc( ).size( ) == 0 ) &&
		    ( rd->RefFirstSer( ).size( ) == 0 ) &&
		    ( rd->RefFirstTable( ).size( ) == 0 ) &&
		    ( rd->RefLogMsg( ).size( ) == 0 ) && 
		    ( rd->RefMore( ).size( ) == 0 ) )
	       {
		  rd.reset( );
		  mResultFrame->SetRawData( rd );
	       }
	    }
	    //----------------------------------------------------------
	    // Add our history information
	    //----------------------------------------------------------
	    createHistory( );
	    //----------------------------------------------------------
	    // Write frame to file
	    //----------------------------------------------------------
	    writeFrameToStream( );
	 } // if ( mResultFrame.get( ) )
	 //--------------------------------------------------------------
	 // Advance to the next frame and set the length of the
	 //   dt appropriately
	 //--------------------------------------------------------------
	 data_start += data_dt;
	 if ( ( data_start >= user_start )
	      && ( data_start < user_stop ) )
	 {
	    // Set data_dt to the size requested by the user
	    data_dt = m_options.SecondsPerFrame( );
	    QUEUE_LOG_MESSAGE( "Reset"
			       << " data_dt: " << data_dt
			       << std::endl << "\t[ Line: " << __LINE__ << " File: " << __FILE__ << "]"
			       ,
			       MT_DEBUG, 30,
			       caller,
			       "CreateRDS" );
	    continue;
	 }
	 else if ( data_start == user_stop )
	 {
	    // Set data_dt to the remainder of data
	    data_dt = data_stop - data_start;
	    QUEUE_LOG_MESSAGE( "Reset"
			       << " data_dt: " << data_dt
			       << std::endl << "\t[ Line: " << __LINE__ << " File: " << __FILE__ << "]"
			       ,
			       MT_DEBUG, 30,
			       caller,
			       "CreateRDS" );
	 }
	 else if ( (data_start < user_stop ) &&
		   ( ( data_start + data_dt ) > user_stop ) )
	 {
	    // Set data_dt to the remainder of user data
	    data_dt = user_stop - data_start;
	    QUEUE_LOG_MESSAGE( "Reset"
			       << " data_dt: " << data_dt
			       << std::endl << "\t[ Line: " << __LINE__ << " File: " << __FILE__ << "]"
			       ,
			       MT_DEBUG, 30,
			       caller,
			       "CreateRDS" );
	 }
      }

      //-----------------------------------------------------------------
      // Finish up
      //-----------------------------------------------------------------
      //----------------------------------------------------------------
      // Close all open input files
      //----------------------------------------------------------------
      m_p->Close( );
      //----------------------------------------------------------------
      // close out any pending requests
      //----------------------------------------------------------------
      m_stream->Close( true );
   }
   catch( const SwigException& E )
   {
      //-----------------------------------------------------------------
      // Cleanup any temporary file
      //-----------------------------------------------------------------
      m_stream->Abandon( );
      //-----------------------------------------------------------------
      // Rethrow the exception
      //-----------------------------------------------------------------
      throw;
   }
   catch( const std::exception& E )
   {
      //-----------------------------------------------------------------
      // Cleanup any temporary file
      //-----------------------------------------------------------------
      m_stream->Abandon( );
      //-----------------------------------------------------------------
      // Rethrow the exception
      //-----------------------------------------------------------------
      throw;
   }
   catch( ... )
   {
      //-----------------------------------------------------------------
      // Cleanup any temporary file
      //-----------------------------------------------------------------
      m_stream->Abandon( );
      //-----------------------------------------------------------------
      // Rethrow the exception
      //-----------------------------------------------------------------
      throw;
   }
}

const std::string& RDSFrame::
getChannelName( INT_4U offset ) const
{
   return m_p->GetChannelCache( ).operator[]( offset ).GetName( );
}

void RDSFrame::
writeFrameToStream( )
{
   static const char* caller = "RDSFrame::writeFrameToStream";

   QUEUE_LOG_MESSAGE( "Writing frame"
		      << " Start Time: " << mResultFrame->GetGTime( )
		      << " Dt: " << mResultFrame->GetDt( )
		      << " with: " << ( ( mResultFrame->GetRawData( ) )
					? mResultFrame->GetRawData( )->RefFirstAdc( ).size( )
					: 0 )
		      << " FrAdcData structures"
		      << " with: " << mResultFrame->RefProcData( ).size( )
		      << " FrProcData structures"
		      << " allow duplicate FrProcData names: " << mResultFrame->RefProcData( ).AllowDuplicates( )
		      ,
		      MT_DEBUG, 30,
		      caller,
		      "RDSFrame" );

   m_stream->Write( mResultFrame,
		    m_options.CompressionMethod( ),
		    m_options.CompressionLevel( ),
		    ( ( m_options.CreateChecksumPerFrame( ) )
		      ? CheckSum::CRC
		      : CheckSum::NONE ) );
}

//-----------------------------------------------------------------------------
//   
//: Get a pointer to the specific FrProc channel.   
//
// Caller must call 'rehash' on passed ProcData container to 
// guarantee existence of channel map for the frame.
//   
//!param: const std::string& name - Channel name.
//!param: const FrameCPP::FrameH::procData_type* proc - A pointer to the
//+       ProcData container for the frame.   
//      
//!return: FrameCPP::FrProcData* - A pointer to the channel object.
//   
//!exc: Channel not found: name. - Specified channel is not found.
//   
FrameCPP::FrProcData* RDSFrame::
getProcChannel( const std::string& name,
		FrameCPP::FrameH::procData_type* const proc )
{
   if( proc->size() == 0 )
   {
       ostringstream msg;
       msg << "Channel not found: " << name;

       throw SWIGEXCEPTION( msg.str() );   
   }

   
   pair< FrameH::const_procData_hash_iterator,
         FrameH::const_procData_hash_iterator > p( proc->hashFind( name ) );
   
   if( p.first == p.second )
   {
       ostringstream msg;
       msg << "Channel not found: " << name;

       throw SWIGEXCEPTION( msg.str() );
   }

   return p.first->second.get( );
}
   
//-----------------------------------------------------------------------------
//   
//: Create new history record.
//   
void RDSFrame::
createHistory()
{
   if ( mResultFrame.get( ) == (FrameH*)NULL )
   {
      return;
   }
   if ( m_options.HistoryRecord( ) )
   {
      GPSTime gps_time;
      gps_time.Now();

      string ldas_msg( "LDAS, Version: " );
      ldas_msg += PACKAGE_VERSION;
   
      // Insert record
      FrameCPP::FrameH::history_type::value_type
	 h( new FrHistory( getHistoryName( ),
			   gps_time.GetSeconds(),
			   ldas_msg ) );
      mResultFrame->RefHistory().append( h );

      //-----------------------------------------------------------------
      // Add record(s) relating to list of channels
      //-----------------------------------------------------------------
      string record;

      record.reserve( FrameCPP::Version::STRING::MAX_STRING_LENGTH );
      record.resize( 0 );
      record.append( "frameAPI: RDS (channels:" );

      for ( channel_cache_type::const_iterator
	       current( m_p->GetChannelCache( ).begin( ) ),
	       end( m_p->GetChannelCache( ).end( ) );
	    current != end;
	    current++ )
      {
	 if ( ( record.length( ) + (*current).GetName( ).length( ) + 2 ) >=
	      FrameCPP::Version::STRING::MAX_STRING_LENGTH )
	 {
	    //-----------------------------------------------------------
	    // Need to output the record since it is full
	    //-----------------------------------------------------------
	    record.append( ")" );

	    FrameCPP::FrameH::history_type::value_type
	       h( new FrHistory( mHistoryName,
				 gps_time.GetSeconds(),
				 record ) );
	    mResultFrame->RefHistory().append( h );
	    record.resize( 0 );
	    record.append( "frameAPI: RDS (channels (cont):" );
	 }
	 record.append( " " );
	 record.append( (*current).GetName( ) );
      }
      //-----------------------------------------------------------------
      // Write the final record of channel name information
      //-----------------------------------------------------------------
      record.append( ")" );
      {
	 FrameCPP::FrameH::history_type::value_type
	    h( new FrHistory( mHistoryName,
			      gps_time.GetSeconds(),
			      record ) );
	 mResultFrame->RefHistory().append( h );
      }
   }
   //--------------------------------------------------------------------
   return;
}
   
//-----------------------------------------------------------------------------
//   
//: Get history name for new records.
//      
const std::string& RDSFrame::getHistoryName()
{
   return mHistoryName;
}


void RDSFrame::
init( const frame_file_container_type& FrameFiles, 
      const channel_container_type& Channels )
{
   std::unique_ptr< private_type >	local_data( new private_type( ) );
   //-------------------------------------------------------------------
   // Organize the frame files into groups and arange in assending
   //   time order
   //-------------------------------------------------------------------
   {
      //----------------------------------------------------------------
      // Have local scoping for parser.
      //----------------------------------------------------------------
      INT_4U				rds_level;

      rds_level = local_data->InitFileCache( m_options, FrameFiles );
      if ( m_options.RDSLevel( ) > 0 )
      {
	 if ( rds_level >= m_options.RDSLevel( ) )
	 {
	    std::ostringstream	msg;
	    msg << "For the given input, the user specified RDS level of "
		<< m_options.RDSLevel( )
		<< " does not exceed the minimum level of "
		<< rds_level << "."
	       ;
	    throw SWIGEXCEPTION( msg.str( ) );
	 }
      }
      else
      {
	 m_options.RDSLevel( ++rds_level );
      }

      if( local_data->GetFileCache( ).size( ) <= 0 )
      {
	 throw SWIGEXCEPTION( "List of frame files is empty." );
      }
   }

   //--------------------------------------------------------------------
   // Check if either the number of frames per file or the seconds per
   //   frame are zero.
   //--------------------------------------------------------------------
   if ( m_options.FramesPerFile( ) == 0 )
   {
      m_options.FramesPerFile( local_data->GetFileCache( ).GetFramesPerFile( ) );
   }
   if ( m_options.SecondsPerFrame( ) == 0 )
   {
      m_options.SecondsPerFrame( local_data->GetFileCache( ).GetSecondsPerFrame( ) );
   }
   if ( ( m_options.AllowShortFrames( ) == false ) &&
	( ( ( m_options.OutputTimeEnd( ) - m_options.OutputTimeStart( ) ) %
	    ( m_options.SecondsPerFrame( ) * m_options.FramesPerFile( ) ) ) !=
	  0 ) )
   {
      std::ostringstream	msg;
      msg << "The choice of "
	  << m_options.FramesPerFile( ) << " frame(s) per file"
	  << " each " << m_options.SecondsPerFrame( ) << " second(s) in length"
	  << " will yield a final frame file which is shorter in length than the rest";
      throw SWIGEXCEPTION( msg.str( ) );
   }
   //--------------------------------------------------------------------
   // Generate a list of channels
   //--------------------------------------------------------------------
   {
      //----------------------------------------------------------------
      // Have local scoping for parser.
      //----------------------------------------------------------------
      local_data->InitChannelCache( Channels );

      if ( local_data->GetChannelCache( ).empty( ) )
      {
	 throw SWIGEXCEPTION( "List of channels is empty." );
      }

      //-----------------------------------------------------------------
      // Create a list of IFOs using the first character of each channel
      //   name
      //-----------------------------------------------------------------
      for ( channel_cache_type::const_iterator
	       current( local_data->GetChannelCache( ).begin( ) ),
	       end( local_data->GetChannelCache( ).end( ) );
	    current != end;
	    current++ )
      {
	 char	ifo( ((*current).GetName( ))[0] );
	 bool	added( false );

	 for ( std::string::iterator
		  current_ifo( m_ifo_list.begin( ) ),
		  end_ifo( m_ifo_list.end( ) );
	       current_ifo != end_ifo;
	       current_ifo++ )
	 {
	    if ( *current_ifo == ifo )
	    {
	       added = true;
	       break;
	    }
	    else if ( ifo < *current_ifo )
	    {
	       m_ifo_list.insert( current_ifo, ifo );
	       added = true;
	       break;
	    }
	 }
	 if ( added == false )
	 {
	    m_ifo_list += ifo;
	 }
      }
   }
   m_p = local_data.release( );
}
   
//-----------------------------------------------------------------------
//
//-----------------------------------------------------------------------
RDSFrame::private_type::
private_type( )
   : m_buffer_size( FrameAPI::StreamBufferSize )
{
   if ( m_buffer_size
	== FrameCPP::Common::FrameBufferInterface::M_BUFFER_SIZE_SYSTEM )
   {
      // Do Nothing
   }
   else if ( m_buffer_size > 0 )
   {
      m_buffer.reset( new char[ m_buffer_size ] );
   }
}

namespace FrameAPI
{
   namespace RDSFrameCC
   {
      //=================================================================
      //=================================================================
      ChannelEntry::
      ChannelEntry( const std::string& Name )
	 : m_channel_kind( UNKNOWN ),
	   m_name( Name )
      {
      }
      //=================================================================
      //=================================================================
      ChannelEntry::
      ChannelEntry( const ChannelEntry& Source )
	 : m_channel_kind( Source.m_channel_kind ),
	   m_name( Source.m_name )
      {
      }
   } // namespace - RDSFrameCC
} // namespace - FrameAPI

namespace
{
   CollectNames::
   ~CollectNames( )
   {
   }

   void CollectNames::
   operator()( const std::string& ChannelName )
   {
      push_back( ChannelName );
   }
}

