#ifndef KAISERWINDOW_HH
#define KAISERWINDOW_HH

//!ppp: {eval `cat $ENV{SRCDIR}/cc_rules.pl`}

#include "FilterConstants.hh"
#include "Window.hh"

namespace Filters {

    //
    //: A class representing a Kaiser window
    //
    class KaiserWindow : public Window {
	
    public:
	//
	//: Constructor
	//
	//!exc: std::invalid_argument - thrown if beta is less than 0
	//
	explicit
	KaiserWindow(const double beta = KaiserWindowDefaultBeta);

	//
	//: Get the beta parameter for this window
	//
	//!return: double - beta parameter
	//
	double beta() const;
	
	//
	//: Set the beta parameter for this window
	//
	//!exc: std::invalid_argument - thrown if beta is less than 0
	//
	void set_beta(const double& beta);

	//
	//: Return window name
	//
	//!return: string - window name
	//
	virtual std::string name() const;

	//
	//: Return window parameter
	//
	//!return: double - window parameter
	//
	virtual double param() const;

	//
	//: Clone a window
	//
	//!return: KaiserWindow* - copy of current window
	//
	virtual KaiserWindow* Clone() const;
	
    private:
	//
	//: Get the ith element of the window
	//
	// A Kaiser window is defined by
	//
	//     w[i] = I0( beta*sqrt(1 - [2*i/(n-1) - 1]^2) )/I0(beta)
	//
	// for i= 0,1, ... n-1, where I0(x) is the 0th order, modified Bessel
	// function of the first kind, and beta (>=0) is a shape parameter
	// related to the amplitude of the sidelobes of the Fourier transform
	// of the window. See "Discrete-time Signal Processing" by Oppenheim
	// and Schafer, p.474 for details.)
	//
	// NOTE: when beta=0, a Kaiser window reduces to a rectangular window 
	//
	//!param: i - the element at which the window is evaluated
	//
	//!return: double - the window element w[i]
	//
	virtual double element(const size_t i) const; 

	//
	//: Modified Bessel Function of first kind, order 0, evaluated by
	//+ power series expansion
	//
	double I0(const double& x) const;
	
	//: beta parameter
	double m_beta;

	//: parameter for the modified Bessel function
	double m_I0beta;
    };

    inline
    double KaiserWindow::beta() const 
    { 
        return m_beta; 
    }

} // namespace Filters

#endif // KAISERWINDOW_HH

