/* -*- mode: c++; c-basic-offset: 2; -*- */

#ifndef LDASTOOLSAL__TASK_HH
#define LDASTOOLSAL__TASK_HH

#include "ldastoolsal/Memory.hh"
#include "ldastoolsal/mutexlock.hh"
#include "ldastoolsal/Thread.hh"

namespace LDASTools
{
  namespace AL
  {
    //--------------------------------------------------------------------
    /// \brief A unit of work to be done usually within a thread.
    ///
    /// This is an abstract class that should be used as the base
    /// for a class that is to perform some task usually within
    /// a thread.
    //--------------------------------------------------------------------
    class Task {
    public:
      typedef LDASTools::AL::SharedPtr< char > name_type;
      typedef Thread::cancel_type cancel_method;

      //-----------------------------------------------------------------
      /// \brief Data class for passing signal information
      //-----------------------------------------------------------------
      typedef Thread::signal_type signal_type;

      //-----------------------------------------------------------------
      /// \brief Constructor
      ///
      /// \param[in] Name
      ///     Sudo unique name identifying the task
      /// \param[in] CancelMethod
      ///     Method to use when terminating the task
      /// \param[in] CancelSignal
      ///     Signal to use for termination of the task
      //-----------------------------------------------------------------
      Task( const std::string& Name,
	    cancel_method CancelMethod = Thread::CANCEL_ABANDON,
	    signal_type CancelSignal = SignalHandler::SIGNAL_UNKNOWN );

      //-----------------------------------------------------------------
      /// \brief Destructor
      //-----------------------------------------------------------------
      virtual ~Task( );

      //-----------------------------------------------------------------
      /// \brief Retrieve baton to gain exclusive access
      //-----------------------------------------------------------------
      MutexLock::baton_type Baton( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve method to use for terminating the task.
      ///
      /// \return
      ///     The method to be used to terminate the task.
      //-----------------------------------------------------------------
      cancel_method CancelMethod( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve signal to use for terminating the task.
      ///
      /// \return
      ///     The signal to be used to terminate the task.
      //-----------------------------------------------------------------
      signal_type CancelSignal( ) const;

      //-----------------------------------------------------------------
      /// \brief Retrieve state information concearning deletion.
      ///
      /// \return
      ///    True if the task should be deleted once terminated.
      //-----------------------------------------------------------------
      bool DeleteOnCompletion( ) const;

      //-----------------------------------------------------------------
      /// \brief Action to be done when task completes
      //-----------------------------------------------------------------
      virtual void OnCompletion( int TaskThreadState );

      
      //-----------------------------------------------------------------
      /// \brief Get the name of the current task
      //-----------------------------------------------------------------
      name_type TaskName( ) const;

      //-----------------------------------------------------------------
      /// \brief Action to perform
      //-----------------------------------------------------------------
      virtual void operator()( ) = 0;

    protected:
      void taskName( const std::string& Name );

      void delete_on_completion( bool Value );

    private:
      struct _p_type;

      LDASTools::AL::SharedPtr< char >	name;
      std::unique_ptr< _p_type >	_p;
    };

    inline Task::name_type Task::
    TaskName( ) const
    {
      return name;
    }

    inline void Task::
    taskName( const std::string& Name )
    {
      name.reset( ::strdup( Name.c_str( ) ) );
    }
  } // namespace - AL
} // namespace LDASTools

#endif /* LDASTOOLSAL__TASK_HH */
