#include "render.h"
#include "canvas.h"
#include "video.h"
#include "events.h"
#include "SDL_keyboard.h"
#include "SDL_video.h"

#include <the_Foundation/defs.h>

static SDL_Renderer *renderer_;

void SDL_ShowCursor(void); /* keyboard.c */

#define NUM_DEFAULT_COLORS  16

iDeclareType(ColorPair)    
    
struct Impl_ColorPair {
    uint8_t col[2];
};

struct SDL_Renderer_s {
    SDL_Window   *window;
    SDL_Texture  *defaultTarget;
    SDL_Texture  *target;
    SDL_Color     drawColor;
    SDL_ColorId   drawColorId;
    SDL_Color     textColor;
    SDL_ColorId   textColorId;
    SDL_Color     textFillColor;
    SDL_ColorId   textFillColorId;
    int           textAttribs;
    SDL_BlendMode blend;
    SDL_Color     palette[256];
    uint16_t      numPalette;
    iColorPair    colorPairs[256];
    uint16_t      numColorPairs;
};

#define COLOR_RESET_THRESHOLD   255

static void SDL_ResetColors(SDL_Renderer *d) {
    d->numPalette = NUM_DEFAULT_COLORS;
    d->numColorPairs = 0;
    SDL_PushEvent(&(SDL_Event){ .type = SDL_RENDER_DEVICE_RESET });
}

static SDL_ColorId SDL_GetColorId(SDL_Renderer *d, SDL_Color rgb) {
    if (rgb.a != 255) {
        return transparent_VisChar;
    }
    for (unsigned int i = NUM_DEFAULT_COLORS; i < d->numPalette; i++) {
        if (!memcmp(&rgb, &d->palette[i], 3)) {
            return i;
        }
    }
    if (d->numPalette == COLOR_RESET_THRESHOLD) {
        /* Out of colors, need to reset. The app needs to handle the reset event. */
        SDL_ResetColors(d);
        return NUM_DEFAULT_COLORS;
    }
    SDL_ColorId id = d->numPalette++;
    d->palette[id] = rgb;
    init_color(id, rgb.r * 1000 / 255, rgb.g * 1000 / 255, rgb.b * 1000 / 255);
    return d->numPalette - 1;
}

uint32_t SDL_GetColorPair(SDL_ColorId fg, SDL_ColorId bg) {
    SDL_Renderer *d = renderer_;
    for (unsigned int i = 0; i < d->numColorPairs; i++) {
        if ((d->colorPairs[i].col[0] == fg || fg == transparent_VisChar) &&
            d->colorPairs[i].col[1] == bg) {
            return i;
        }
//        if ((d->colorPairs[i].col[1] == fg || fg == transparent_VisChar) &&
//            d->colorPairs[i].col[0] == bg) {
//            return i | A_REVERSE;
//        }
    }
    if (d->numColorPairs == COLOR_RESET_THRESHOLD) {
        SDL_ResetColors(d);
        return 0;
    }
    uint32_t index = d->numColorPairs++;
    d->colorPairs[index].col[0] = fg;
    d->colorPairs[index].col[1] = bg;
    init_pair(index, fg, bg);
    return index;
}

struct SDL_Texture_s {
    iCanvas       canvas;
    SDL_BlendMode blend;
    SDL_Color     colorMod;
};

static iCanvas *activeCanvas_(SDL_Renderer *d) {
    if (d->target) {
        return &d->target->canvas;
    }
    return &d->defaultTarget->canvas;
}

void SDL_ResizeTexture(SDL_Texture *d, iInt2 size) {
    resize_Canvas(&d->canvas, size);
}

SDL_Texture *SDL_CreateTexture(SDL_Renderer *render, int format, int access, int w, int h) {
    SDL_Texture *d = calloc(sizeof(SDL_Texture), 1);
    init_Canvas(&d->canvas, init_I2(w, h));
    d->blend = SDL_BLENDMODE_NONE;
    d->colorMod = (SDL_Color){ 255, 255, 255, 255 };
    return d;
}

SDL_Texture *SDL_CreateTextureFromSurface(SDL_Renderer *render, SDL_Surface *surface) {
    SDL_Texture *d = SDL_CreateTexture(render, 0, 0, surface->w, surface->h);
    return d;
}

void SDL_DestroyTexture(SDL_Texture *d) {
    if (d) {
        deinit_Canvas(&d->canvas);
        free(d);
    }
}

void SDL_SetTextureBlendMode(SDL_Texture *d, SDL_BlendMode mode) {
    if (d) {
        d->blend = mode;
    }
}

void SDL_SetTextureColorMod(SDL_Texture *d, uint8_t r, uint8_t g, uint8_t b) {
    if (d) {
        d->colorMod.r = r;
        d->colorMod.g = g;
        d->colorMod.b = b;
    }
}

void SDL_SetTextureAlphaMod(SDL_Texture *d, uint8_t alpha) {
    if (d) {
        d->colorMod.a = alpha;
    }
}

int SDL_QueryTexture(SDL_Texture *d, uint32_t *format, int *access, int *w, int *h) {
    if (d) {
        if (w) *w = d->canvas.size.x;
        if (h) *h = d->canvas.size.y;
        return SDL_TRUE;
    }
    return SDL_FALSE;
}

/*----------------------------------------------------------------------------------------------*/

void SDL_UpdateRendererToWindowSize(void) {
    SDL_Renderer *d = renderer_;
    if (d) {
        if (d->defaultTarget->canvas.size.x != d->window->w ||
            d->defaultTarget->canvas.size.y != d->window->h) {
            SDL_ResizeTexture(d->defaultTarget, init_I2(d->window->w, d->window->h));
        }
    }
}

SDL_Renderer *SDL_CreateRenderer(SDL_Window *window, int index, uint32_t flags) {
    SDL_Renderer *d    = calloc(sizeof(SDL_Renderer), 1);
    d->window          = window;
    d->defaultTarget   = SDL_CreateTexture(d, 0, 0, window->w, window->h);
    d->target          = d->defaultTarget;
    d->blend           = SDL_BLENDMODE_NONE;
    d->drawColor       = (SDL_Color){ 255, 255, 255, 255 };
    d->drawColorId     = transparent_VisChar;
    d->textColorId     = transparent_VisChar;
    d->textFillColorId = transparent_VisChar;
    d->textAttribs     = 0;
    d->numPalette      = NUM_DEFAULT_COLORS;
    /* We can't rely on the default colors being knowable. */
    d->numColorPairs   = 0;
    renderer_          = d;
    return d;
}

int SDL_CreateWindowAndRenderer(int w, int h, int flags, SDL_Window **win_out,
                                SDL_Renderer **rend_out) {
    *win_out  = SDL_CreateWindow("", 0, 0, w, h, flags);
    *rend_out = SDL_CreateRenderer(*win_out, 0, 0);
    return SDL_TRUE;
}

void SDL_DestroyRenderer(SDL_Renderer *d) {
    if (d) {
        SDL_DestroyTexture(d->defaultTarget);
        free(d);
    }
    if (renderer_ == d) {
        renderer_ = NULL;
    }
}

int SDL_GetRendererInfo(SDL_Renderer *d, SDL_RendererInfo *info) {
    iZap(*info);
    info->name = "SEALCurses";
    return SDL_TRUE;
}

void SDL_GetRendererOutputSize(SDL_Renderer *d, int *w, int *h) {
    if (w) *w = d->window->w;
    if (h) *h = d->window->h;
}

void SDL_SetRenderDrawColor(SDL_Renderer *d, int r, int g, int b, int a) {
    d->drawColor = (SDL_Color){ r, g, b, a };
    d->drawColorId = SDL_GetColorId(d, d->drawColor);
}

void SDL_SetRenderDrawBlendMode(SDL_Renderer *d, SDL_BlendMode mode) {
    d->blend = mode;
}

void SDL_RenderClear(SDL_Renderer *d) {
    clear_Canvas(activeCanvas_(d), d->drawColorId);
}

void SDL_RenderPresent(SDL_Renderer *d) {
    curs_set(0);
    present_Canvas(&d->defaultTarget->canvas, d->window->cwin);   
    wrefresh(d->window->cwin);
    if (SDL_IsTextInputStarted()) {
        SDL_ShowCursor();
        curs_set(1);
        wrefresh(d->window->cwin);
    }
}

void SDL_RenderCopy(SDL_Renderer *d, SDL_Texture *texture, const SDL_Rect *src,
                    const SDL_Rect *dst) {
    if (!texture) {
        return;
    }
    const iCanvas *srcCanvas = &texture->canvas;
    SDL_Rect fullSrcRect = { 0, 0, srcCanvas->size.x, srcCanvas->size.y };
    if (src == NULL) {
        src = &fullSrcRect;
    }
    blit_Canvas(activeCanvas_(d),
                &texture->canvas,
                *(const iRect *) src,
                dst ? init_I2(dst->x, dst->y) : zero_I2(),
                texture->blend);
}

void SDL_RenderDrawLines(SDL_Renderer *d, const SDL_Point *points, uint32_t count) {
    /* This function is expected to be used only in a limited manner, to draw full or 
       half rectangles. */
    iCanvas *canvas = activeCanvas_(d);
    if (count == 5) {
        /* A full rectangle. */
        iRect rect = init_Rect(
            points[0].x, points[0].y, points[2].x - points[0].x, points[2].y - points[0].y);
        if (rect.size.y >= 4) {
            drawRect_Canvas(canvas, rect, d->drawColorId);
        }
        else if (rect.size.y == 1) {
            setChar_Canvas(canvas, init_I2(points[0].x, points[0].y), '[', d->drawColorId);
            setChar_Canvas(canvas, init_I2(points[1].x - 1, points[1].y), ']', d->drawColorId);
        }
    }
    else if (count == 3) {
        setChar_Canvas(canvas, init_I2(points[0].x, points[0].y), '|', d->drawColorId);
    }
    else if (count == 2 && points[1].x > points[0].x) {
        fillRectChar_Canvas(
            canvas,
            (iRect){ init_I2(points[0].x, points[0].y), init_I2(points[1].x - points[0].x, 1) },
            transparent_VisChar,
            0x2500 /* box drawings light horizontal */,
            d->drawColorId);
    }
}

void SDL_RenderFillRect(SDL_Renderer *d, const SDL_Rect *rect) {
    iCanvas *canvas = activeCanvas_(d);
    SDL_Rect fullRect;
    if (!rect) {
        fullRect = (SDL_Rect){ 0, 0, canvas->size.x, canvas->size.y };
        rect = &fullRect;
    }
    if (d->drawColor.a == 255 || d->blend == SDL_BLENDMODE_NONE) {
        if (rect->w == 1 && rect->h >= 3) {
            fillRectChar_Canvas(canvas, *(const iRect *) rect, transparent_VisChar,
                                0x2503 /* box drawings heavy vertical */, d->drawColorId);   
        }
        else if (rect->h == 0 && rect->w > 1) {
            fillRectChar_Canvas(canvas, init_Rect(rect->x, rect->y, rect->w, 1),
                                transparent_VisChar, 0x2500 /* box drawings light horizontal */,
                                d->drawColorId);
        }
        else {
            fillRect_Canvas(canvas, *(const iRect *) rect, d->drawColorId);
        }
    }
}

void SDL_RenderDrawRect(SDL_Renderer *d, const SDL_Rect *rect) {}

void SDL_RenderSetClipRect(SDL_Renderer *d, const SDL_Rect *clip) {
    if (clip) {
        setClip_Canvas(activeCanvas_(d), *(const iRect *) clip);
    }
    else {
        unsetClip_Canvas(activeCanvas_(d));
    }
}

SDL_Texture *SDL_GetRenderTarget(SDL_Renderer *d) {
    return d->target;
}

void SDL_SetRenderTarget(SDL_Renderer *d, SDL_Texture *texture) {
    d->target = texture;
}

void SDL_SetRenderTextColor(SDL_Renderer *d, int r, int g, int b) {
    d->textColor   = (SDL_Color){ r, g, b, 255 };
    d->textColorId = SDL_GetColorId(d, d->textColor);
}

void SDL_SetRenderTextFillColor(SDL_Renderer *d, int r, int g, int b, int a) {
    d->textFillColor   = (SDL_Color){ r, g, b, a };
    d->textFillColorId = SDL_GetColorId(d, d->textFillColor);
}

void SDL_SetRenderTextAttributes(SDL_Renderer *d, int attributes) {
    d->textAttribs = attributes;
}

void SDL_RenderDrawUnicode(SDL_Renderer *d, int x, int y, uint32_t ucp) {
    const iInt2 pos = init_I2(x, y);
    iCanvas *canvas = activeCanvas_(d);
    if (contains_Rect(clipBounds_Canvas(canvas), pos)) {
        iVisChar *vis = at_Canvas(canvas, pos);
        vis->ch = ucp;
        vis->attr = d->textAttribs;
        if (d->textFillColorId != transparent_VisChar) {
            vis->bg = d->textFillColorId;
        }
        vis->fg = d->textColorId;
    }
}
