/*

    This file is part of Kitlist, a program to maintain a simple list
    of items and assign items to one or more categories.

    Copyright (C) 2008-2021 Frank Dean

    Kitlist is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Kitlist is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Kitlist.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef KIT_LIST_GUI_H
#define KIT_LIST_GUI_H 1

#include "service.hpp"
#ifndef GCONF
#include "yamlconfig.hpp"
#endif

#include <iostream>
#ifdef KITLIST_DEBUG
#include <fstream>
#endif
#ifdef GCONF
#include <gconfmm/client.h>
#include <gconfmm.h>
#endif
#include <gtkmm/button.h>
#include <gtkmm/checkbutton.h>
#include <gtkmm/combobox.h>
#include <gtkmm/entry.h>
#include <gtkmm/imagemenuitem.h>
#include <gtkmm/main.h>
#include <gtkmm/statusbar.h>
#include <gtkmm/toolbutton.h>
#include <gtkmm/treemodelcolumn.h>
#include <gtkmm/liststore.h>
#include <gtkmm/pagesetup.h>
#include <gtkmm/printsettings.h>
#include <gtkmm/printoperation.h>

enum gui_state { ADD_CATEGORY, RENAME_CATEGORY };

/**
 * \brief A definition for displaying a ModelCategory in a combo box.
 */
class ModelCategoryColumns : public Gtk::TreeModel::ColumnRecord {
public:

    ModelCategoryColumns() {
        add(m_col_text);
        add(m_col_num);
    }

    Gtk::TreeModelColumn<Glib::ustring> m_col_text;
    Gtk::TreeModelColumn<int> m_col_num;
};


/// The position in the list of the tick box column.
const int CHECKED_COL_POSITION = 0;

/**
 * \brief A definition for displaying an item in a multi-column list.
 */
class ModelItemColumns : public Gtk::TreeModel::ColumnRecord {
public:

    ModelItemColumns() {
        add(m_col_text);
        add(m_col_checked);
        add(m_col_num);
    }

    Gtk::TreeModelColumn<Glib::ustring> m_col_text;
    Gtk::TreeModelColumn<bool> m_col_checked;
    Gtk::TreeModelColumn<int> m_col_num;
    
};


/**
 * \brief Encapsulates the methods for the application's GUI front
 * end.
 *
 * This is the GTK+ implementation.
 */
class KitListGui {
private:
#ifndef GCONF
    YamlConfig m_yaml_config;
#endif
protected:
#ifdef GCONF
    /// GConf client reference
    Glib::RefPtr<Gnome::Conf::Client> m_ref_gconf_client;
#endif
    /**
     * \brief The filename currently associated with the loaded model.
     *
     * Should be an empty string if not related to a file.
     */
    Glib::ustring m_filename;
    /// The page title to be used when printing the item list.
    Glib::ustring m_page_title;
#ifdef KITLIST_DEBUG
    /// File stream used for debug logging
    std::ofstream* m_slog;
#endif
    /// Holder for items pasted to the clipboard.
    Glib::ustring m_clipboard_items;
    /// Temporarily ignore events on the item list.
    bool m_ignore_list_events;
    /// The main application
    Gtk::Main m_kit;
    /// The main application window
    Gtk::Window* m_window;
    /// The 'Preferences' dialog
    Gtk::Window* m_window_preferences;
    /// the text entry field for the page title
    Gtk::Entry* m_entry_page_title;
    /// The 'Add Item' dialog
    Gtk::Window* m_window_add_item;
    /// The 'Add Category' dialog
    Gtk::Window* m_window_add_category;
    /// The text entry field of the 'Add Item' dialog
    Gtk::Entry* m_entry_add_item;
    /// The text entry field of the 'Add Category' dialog
    Gtk::Entry* m_entry_add_category;
    /// The file save menu item
    Gtk::ImageMenuItem* m_file_save_menu_item;
    /// The file save toolbar button
    Gtk::ToolButton* m_file_save_tool_button;
    /// The recent files menu item
    Gtk::MenuItem* m_recent_files_menu_item;
    /// The menu paste button
    Gtk::ImageMenuItem* m_paste_menu_item;
    /// The toolbar paste button
    Gtk::ToolButton* m_paste_tool_button;
    /// The check button field of the 'Add Item' dialog
    Gtk::CheckButton* m_checkbutton_add_item;
    /// The combo box holding a list of categories
    Gtk::ComboBox* m_category_combo;
    /// The model backing the category combo box
    Glib::RefPtr<Gtk::ListStore> m_ref_category_list_store;
    /// The definition of the category combo box columns
    ModelCategoryColumns m_category_cols;
    /// The item list view definition
    Gtk::TreeView* m_item_tree_view;
    /// The definition of the item list's columns
    ModelItemColumns m_item_cols;
    /// The business/service object
    Service& m_service;
    /// The model backing the item list
    Glib::RefPtr<Gtk::ListStore> m_ref_item_tree_model;
    /// The application status bar
    Gtk::Statusbar* m_status_bar;
    /// Printer page setup settings
    Glib::RefPtr<Gtk::PageSetup> m_ref_page_setup;
    /// Printer settings
    Glib::RefPtr<Gtk::PrintSettings> m_ref_printer_settings;
    /**
     * \brief Indicates whether a category is being created or
     * renamed.
     *
     * Only used whilst the 'Add/Rename dialog is being displayed',
     * or when it has been closed to determine the correct action.
     */
    enum gui_state m_state;
    long m_current_cat_id; ///< temporary reference to a category id, usually being renamed
    virtual void init();
    virtual gint get_max_recent_files();
    virtual ModelItemContainer* get_selected_items();
    virtual void add_items(const ModelItemContainer& items);
    virtual void set_page_title(const Glib::ustring page_title);
    virtual void close_preferences_window();
    virtual void cancel_preferences_window();
    virtual void close_add_item_window();
    virtual void cancel_add_item_window();
    virtual void close_add_category_window();
    virtual void cancel_add_category_window();
    virtual long get_selected_category();
    virtual void init_add_item_window();
    virtual void delete_selected_items();
    virtual ModelItemContainer* copy_selected_items_to_clipboard();
    virtual bool confirm_lose_changes(const Glib::ustring& message);
    virtual void update_recent_files_menu();
    virtual void update_recent_files(const Glib::ustring& filename);
    virtual bool on_delete_event(GdkEventAny* event);
    virtual void on_menu_quit();
    virtual void on_menu_file_new();
    virtual void on_menu_file_open();
    virtual void on_menu_save();
    virtual void on_menu_save_as();
    void on_printoperation_done(Gtk::PrintOperationResult result, const Glib::RefPtr<Gtk::PrintOperation>& op);
    void on_printoperation_status_changed(const Glib::RefPtr<Gtk::PrintOperation>& op);
    virtual void on_menu_print();
    virtual void on_menu_export_to_pdf();
    virtual void on_menu_recent_file(const Glib::ustring& filename);
    virtual void on_menu_preferences();
    virtual void on_menu_add();
    virtual void on_menu_delete();
    virtual void on_menu_cut();
    virtual void on_menu_copy();
    virtual void on_menu_paste();
    /// Causes all items to be displayed
    virtual void on_menu_show_all() { m_service.show_all(); refresh_item_list(); }
    /// Causes only checked items to be displayed
    virtual void on_menu_show_checked() { m_service.show_checked_only(); refresh_item_list(); }
    /// Causes only unchecked items to be displayed
    virtual void on_menu_show_unchecked() { m_service.show_unchecked_only(); refresh_item_list(); }
    virtual void on_menu_select_all();
    /// Marks all selected items as checked
    virtual void on_menu_check_selected() { set_selected(true); }
    /// Marks all selected items as unchecked
    virtual void on_menu_uncheck_selected() { set_selected(false); }
    virtual void on_menu_create_category();
    virtual void on_menu_delete_category();
    virtual void on_menu_rename_category();
    virtual void on_menu_help_about();
    virtual void on_clipboard_get(Gtk::SelectionData& selection_date, guint);
    virtual void on_clipboard_clear();
    virtual void on_clipboard_received(const Gtk::SelectionData& selection_data);
    virtual void on_category_change();
    virtual void on_cell_edit(const Glib::ustring s);
    virtual bool choose_filename(Glib::ustring& filename);
    virtual bool choose_pdf_filename(Glib::ustring& filename);
    virtual void update_paste_status();
    virtual void paste_status_received(const Glib::StringArrayHandle& targets_array);
    virtual void paste_from_xml(const Glib::ustring& document);
    virtual void refresh_item_list();
    virtual void refresh_category_list(long cat_id = -2);
    virtual void selected_row_callback(const Gtk::TreeModel::iterator& iter);
    virtual void set_selected(bool checked);
    virtual void toggle_selected();
    // virtual void on_row_deleted(Gtk::TreeModel::Path path);
    void on_row_changed(const Gtk::TreeModel::Path path, const Gtk::TreeModel::iterator iter);
    virtual void update_item_count(size_t n);
 public:
    KitListGui(int argc, char **argv, Service& service);
    ~KitListGui();
    virtual void open_file(const Glib::ustring& filename);
    virtual void raise();
    virtual void safe_open_file(const Glib::ustring& filename);
    void run();
};


#endif // KIT_LIST_GUI_H
