/*

    This file is part of Kitlist, a program to maintain a simple list
    of items and assign items to one or more categories.

    Copyright (C) 2008,2009 Frank Dean

    Kitlist is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Kitlist is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Kitlist.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "service.hpp"
#include <algorithm>
#include <cassert>
#include <iostream>

using namespace std;

class FilterItem : public ItemFunctor {
    KitModel& m_model;
public:
    FilterItem(KitModel& model) : ItemFunctor(), m_model(model) {}
    bool operator()(Item& item) {
        return m_model.filter(item.get_checked());
    }

};


/**
 * \brief Loads the data model from the persistence store.
 */
Service::Service(KitListDao& dao) : m_dao(dao), m_model(0) {
    m_model = load_model();
}


Service::~Service() {
    if (m_model)
        delete m_model;
}


/**
 * \brief Loads the data model from the persistence store.
 */
KitModel* Service::load_model() {
    return m_dao.get_model();
}


/**
 * \brief Loads a new data model from the named XML document.
 *
 * Creates a new data model based on the passed filename.  The
 * existing data model is destroyed after successfully loading the new
 * one.
 */
void Service::open_as_xml(const Glib::ustring& filename) {
#ifdef XML_DAO
    delete m_model;
    m_model = ((XmlDao&) m_dao).get_model(filename);
#endif
}


/**
 * \brief Creates a default model.
 *
 * By default, a new empty XmlDao data model is created.
 */
void Service::create_default_model() {
    delete m_model;
#ifdef XML_DAO
    ((XmlDao&) m_dao).set_filename("");
    m_model = m_dao.get_model();
#else
    m_model = new KitModel();
    m_model->reset();
#endif
}


/**
 * Saves the model's state to the persistence store.
 */
void Service::save() {
    assert(m_model);
    m_dao.save_model(m_model);
}


/**
 * \brief Saves the model's state to an XML document.
 *
 * This is primarily intended to support switching from one dao
 * implementation to the XmlDao implemenation.
 * \param filename The full pathname and filename to save the file to.
 */
void Service::save_as_xml(const Glib::ustring& filename) {
#ifdef XML_DAO
    ((XmlDao&) m_dao).save_model(m_model, filename);
#else
    // Only want a new DAO when the current DAO isn't XmlDao
    XmlDao xmldao;
    xmldao.save_model(m_model, filename);
#endif
}


/**
 * Returns the item with the specified unique ID.
 */
ModelItem* Service::find_item(long id) {
    return m_model->find_item(id);
}


/**
 * Returns the category with the specified unique ID.
 */
ModelCategory* Service::find_category(long cat_id) {
    return m_model->find_category(cat_id);
}


/**
 * \brief Copies items to the specified category.
 *
 * Only copies those items that do not already exist in the target
 * category.
 */
void Service::copy_items(const ModelItemContainer& items, long cat_id) {
    assert(m_model);
    m_model->set_dirty(true);
    m_model->copy_items(items, cat_id);
}


/**
 * \brief Creates a new ModelItem and associates it with the specified category.
 *
 * A new item is created and assigned a new unique Id.  The item is
 * added to the data model and associated with a category if the
 * cat_id is greater than or equal to zero.  Otherwise the item is
 * added to the model without being associated with a category.
 */
Item* Service::create_item(long cat_id) {
    assert(m_model);
    m_model->set_dirty(true);
    ModelItem* item = new ModelItem;
    item->set_id(get_next_item_id());
    item->set_new_flag(true);
    item->set_dirty(true);
    if (cat_id < 0)
        m_model->add_item(item);
    else
        m_model->add_item(item, cat_id);
    return item;
}


/**
 * \brief Flags an item as deleted.
 *
 * Finds the item with the specified ID and flags it as deleted.
 * \return false if the item does not exist.
 */
bool Service::delete_item(long id) {
    bool retval = false;
    assert(m_model);
    ModelItem* item = m_model->find_item(id);
    if (item) {
        m_model->set_dirty(true);
        item->set_dirty(true);
        item->set_deleted(true);
        retval = true;
    }
    return retval;
}


/**
 * \brief Flags a category as deleted.
 *
 * Finds the category with the specified ID and flags it as deleted.
 * \return false if the category does not exist.
 */
bool Service::delete_category(long cat_id) {
    bool retval = false;
    assert(m_model);
    ModelCategory* category = m_model->find_category(cat_id);
    if (category) {
        m_model->set_dirty(true);
        category->set_dirty(true);
        category->set_deleted(true);
        retval = true;
    }
    return retval;
}


/**
 * \brief Creates a new category.
 *
 * A new ModelCategory is created, assigned a new unique ID and added
 * to the model.
 */
Category* Service::create_category() {
    assert(m_model);
    m_model->set_dirty(true);
    ModelCategory* category = new ModelCategory;
    category->set_id(get_next_category_id());
    category->set_new_flag(true);
    category->set_dirty(true);
    m_model->add_category(category);
    return category;
}


/**
 * \brief Checks or unchecks all the passed items.
 *
 * \param items The items to change.
 * \checked the state to change the to.
 */
void Service::select_items(ModelItemContainer* items, bool checked) {
    m_model->set_dirty();
    for (ModelItemIter i = items->begin(); i != items->end(); ++i) {
        (*i)->set_dirty();
        (*i)->set_checked(checked);
    }
}


/**
 * \brief Toggles the checked state of all the passed items.
 *
 * \param items The items to change.
 */
void Service::toggle_selected_items(ModelItemContainer* items) {
    m_model->set_dirty();
    for (ModelItemIter i = items->begin(); i != items->end(); ++i) {
        (*i)->set_dirty();
        (*i)->set_checked(!(*i)->get_checked());
    }
}


/**
 * Flags the model as being dirty.
 */
void Service::set_model_dirty(bool flag) {
    assert(m_model);
    if (m_model)
        m_model->set_dirty(flag);
}


/**
 * \brief Updates the attributes of an item.
 *
 * Locates the item using the supplied unique ID, then assigns the
 * passed values.
 *
 * \param id The unique ID of the item to update.
 * \param description The item's descriptive text.
 * \param checked The checked/ticked state of the item.
 * \return Returns true if the item exists, false otherwise.
 */
bool Service::update_item(long id, const string description, bool checked) {
    bool retval = false;
    assert(m_model);
    ModelItem* item = m_model->find_item(id);
    if (item) {
        item->set_description(description);
        item->set_checked(checked);
        item->set_dirty(true);
        retval = true;
    }
    return retval;
}


/**
 * \brief Returns a list of items.
 *
 * If cat_id is less than zero, returns all items, otherwise only
 * those items associated with the specified category ID.
 *
 * \param cat_id the unique ID of a category or '-1' to indicate all
 * items are to be retrieved.
 */
ItemContainer* Service::get_items(long cat_id) {
    assert(m_model);
    ItemContainer* retval = 0;
    if (cat_id == -1) {
        retval = m_model->get_all_items();
    } else {
        ModelCategory* cat = m_model->find_category(cat_id);
        if (cat) {
            retval = cat->get_items();
        }
    }
    return retval;
}


/**
 * \brief Returns a list of items, applying the current filter.
 *
 * The returned list is also sorted by item name.
 *
 * If cat_id is less than zero, returns all items, otherwise only
 * those items associated with the specified category ID.
 *
 * \param cat_id the unique ID of a category or '-1' to indicate all
 * items are to be retrieved.
 */
ItemContainer* Service::get_filtered_items(long cat_id) {
    assert(m_model);
    ItemContainer* retval = 0;
    FilterItem fi = FilterItem(*m_model);
    if (cat_id == -1) {
        retval = m_model->get_all_items(fi);
    } else {
        ModelCategory* cat = m_model->find_category(cat_id);
        if (cat) {
            retval = cat->get_items(fi);
        }
    }
    if (retval)
        sort(retval->begin(), retval->end(), ItemCompareName());
    return retval;
}


/**
 * \brief Returns a list of all categories.
 *
 * Categories flagged as deleted are excluded.
 */
CategoryContainer* Service::get_categories() {
    assert(m_model);
    CategoryContainer* retval = m_model->get_categories();
    return retval;
}


/**
 * Returns an unused unique id for an Item.
 */
long Service::get_next_item_id() {
    return m_dao.get_next_item_id();
}


/**
 * Returns an unused unique id for a Category.
 */
long Service::get_next_category_id() {
    return m_dao.get_next_category_id();
}
