/*

    This file is part of Kitlist, a program to maintain a simple list
    of items and assign items to one or more categories.

    Copyright (C) 2008,2009 Frank Dean

    Kitlist is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Kitlist is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Kitlist.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "kitlistgui.hpp"
#include "printing.hpp"
#include <cassert>
// #include <gtk/gtk.h>
// #include <gtkmm.h>
#ifdef MAEMO
#include <hildon/hildon-help.h>
#endif
#include <glibmm/i18n.h>
#include <glibmm/refptr.h>
#include <glibmm/ustring.h>
#include <gtkmm/aboutdialog.h>
#include <gtkmm/cellrenderertoggle.h>
#include <gtkmm/clipboard.h>
#include <gtkmm/filechooserdialog.h>
#include <gtkmm/menuitem.h>
#include <gtkmm/messagedialog.h>
#include <gtkmm/stock.h>
#include <gtkmm/targetentry.h>
#include <gtkmm/window.h>
#include <libglademm/xml.h>
#include <libxml++/libxml++.h>
#include <string>
#include <sstream>
#include <sys/stat.h>
#include <config.h>

/// Maemo service name
#define KITLIST_SERVICE_NAME "com.nokia." + PACKAGE_NAME
/// Maemo service path
#define KITLIST_SERVICE_OBJECT "/com/nokia/" + PACKAGE_NAME
/// Maemo service interface name
#define KITLIST_SERVICE_IFACE "com.nokia." + PACKAGE_NAME


using namespace std;

namespace {

#ifdef MAEMO
    /// Resource file name
    const string GLADE_APP_FILE = "maemo.glade";
#else
    /// Resource file name
    const string GLADE_APP_FILE = "kitlist.glade";
#endif
    /// Status bar message constant for displaying item counts
    const guint SB_ITEM_COUNT = 1000;
    /// Status bar message constant for save notifications
    const guint SB_SAVE = SB_ITEM_COUNT + 1;
    /// Status bar message constant for general messages
    const guint SB_MSG = SB_SAVE + 1;

    /// Status bar message constant for printer messages
    const guint SB_PRINT = SB_MSG + 1;

    /// Key used for custom clipboard
    const char item_target_custom[] = "kitlistclipboard";
    /// Mime type for clipboard content
    const char item_target_text[] = "text/plain";

    /// Tag name for the ID element in the clipbard XML document.
    const char XML_ELEMENT_ID[] = "id";

    /// Default filename extension
    const Glib::ustring DEFAULT_FILENAME_EXTENSION = ".kit";

    /// PDF filename extension
    const Glib::ustring PDF_FILENAME_EXTENSION = ".pdf";

    /// The default filename
    const Glib::ustring DEFAULT_FILENAME = "kitlist" + DEFAULT_FILENAME_EXTENSION;

    /// The application's root key in the GConf hierarchy
#ifdef MAEMO
    const Glib::ustring GCONF_KEY = "/apps/maemo/kitlist";
#else
    const Glib::ustring GCONF_KEY = "/apps/kitlist";
#endif

    /// GConf entry for the current filename
    const Glib::ustring GCONF_KEY_CURRENT_FILENAME = GCONF_KEY + "/current_filename";

    /// GConf entry for the page title
    const Glib::ustring GCONF_KEY_PAGE_TITLE = GCONF_KEY + "/page_title";

#ifdef KITLIST_DEBUG
#ifdef GCONF
    /// GConf entry for the page title
    const Glib::ustring GCONF_KEY_DEBUG_LOG = GCONF_KEY + "/debug_log_filename";
#endif
#endif

    /// The maximum number of recent files to maintain
    const gint DEFAULT_MAX_RECENT_FILES = 4;

    /// GConf entry for recent files
    const Glib::ustring GCONF_KEY_RECENT_FILES = GCONF_KEY + "/recent_files";

    /// GConf entry for max recent files
    const Glib::ustring GCONF_KEY_MAX_RECENT_FILES = GCONF_KEY + "/max_recent_files";

} // anonymous namespace


#ifdef KITLIST_DEBUG
void my_log_handler(const gchar *log_domain,
                    GLogLevelFlags log_level,
                    const gchar *message,
                    gpointer data) {
    g_print("KITLIST: %s\n", message);
#ifdef MAEMO
    syslog(LOG_DEBUG, message);
#endif
    ofstream* fout = (ofstream*) data;
    if (fout) {
        *fout << message << endl;
    }
}

#endif


typedef Gtk::TreeModel::Children type_children;

/**
 * \brief Returns true if the passed file exists
 *
 * Note: We do not test for the file type, just it's existence
 * regardless of whether it's a directory etc.
 *
 * \param filename The file to test for existence.
 * \return true if the file exists.
 */
const bool file_exists(const Glib::ustring& filename) {
    struct stat fileinfo;
    bool retval;
    retval = !(stat(filename.c_str(), &fileinfo));
    // g_debug("file_exists: %s", retval ? "true" : "false");
    return retval;
}


/// Locate the Glade resource file from a list of potential locations.
const string load_resource_glade_file(const Glib::ustring& filename) {
    vector<string> locs;
    locs.push_back(filename);
    locs.push_back("../" + filename);
    locs.push_back("./src/" + filename);
    locs.push_back(string(PACKAGE_DATA_DIR) + "/glade/" + filename);
    for (std::vector<string>::iterator i = locs.begin(); i != locs.end(); ++i) {
        if (file_exists(*i))
            return *i;
    }
    return "";
}

/**
 * \brief Returns a reference to the Glade resource file.
 *
 * Attempts to locate the external Glade resource file from a number
 * of common locations.
 */
Glib::RefPtr<Gnome::Glade::Xml> get_glade_ref_ptr(const string& filename,
                                               const Glib::ustring& root = Glib::ustring(),
                                               const Glib::ustring& domain = Glib::ustring()) {
    Glib::RefPtr<Gnome::Glade::Xml> refXml;
    try {
#ifdef GLIBMM_EXCEPTIONS_ENABLED
        refXml = Gnome::Glade::Xml::create(load_resource_glade_file(filename), root, domain);
#else
        std::auto_ptr<Gnome::Glade::XmlError> ex;
        refXml = Gnome::Glade::Xml::create(load_resource_glade_file(filename), root, domain, ex);
        if (ex.get())
            g_error(_("Error loading Glade file: %s"), ex->what().c_str());
#endif
    } catch (const Gnome::Glade::XmlError& ex) {
        g_error(_("Error loading Glade file: %s"), ex.what().c_str());
        throw ex;
    }
    return refXml;
}


#ifdef MAEMO

void dump_osso_rpc_value(osso_rpc_t* retval) {
    switch (retval->type) {
    case (DBUS_TYPE_INVALID) :
        g_debug("invalid/void");
        break;
    case (DBUS_TYPE_STRING) :
        g_debug("string :'%s'", retval->value.s);
        break;
    case (DBUS_TYPE_UINT32) :
        g_debug("uint32 :%u", retval->value.u);
        break;
    case (DBUS_TYPE_INT32) :
        g_debug("int32 :%d", retval->value.i);
        break;
    case (DBUS_TYPE_BOOLEAN) :
        g_debug("boolean :%s", (retval->value.b == TRUE)?"TRUE":"FALSE ");
        break;
    case (DBUS_TYPE_DOUBLE) :
        g_debug("double :%.3f", retval->value.d);
        break;
    case (DBUS_TYPE_OBJECT_PATH) :
        break;
    default:
        g_debug("unknown(type=%d)", retval->type);
        break;
    }
}

/// Callback handler for dbus messages
static gint dbus_req_handler(const gchar* interface, const gchar* method,
                             GArray* arguments, gpointer data,
                             osso_rpc_t* retval) {
    KitListGui* gui = (KitListGui*) data;
    // g_debug("kitlist->dbus_req_handler(%p)", gui->m_osso_context);
//     g_debug("method: %s", method);
//     g_debug("Arguments: %d", arguments->len);
//     guint x;
//     for (x = 0; x < arguments->len; x++) {
//         osso_rpc_t e = g_array_index(arguments, osso_rpc_t, x);
//         g_debug("Argument %d:", x);
//         dump_osso_rpc_value(&e);
//     }
//     g_debug("Checking whether this is a request to open a file");
    // So we get a method named 'mime_type' and the
    // first parameter is a string much like the following:
    // string :'file:///home/user/MyDocs/.documents/kitlist.kit'
    Glib::ustring smethod = method;
    if (smethod.compare("mime_open") == 0 && arguments->len > 0) {
        if (arguments->len > 0) {
            osso_rpc_t arg = g_array_index(arguments, osso_rpc_t, 0);
            if (arg.type == DBUS_TYPE_STRING) {
                Glib::ustring filename = arg.value.s;
                if (filename.find("file://") >= 0)
                    filename.erase(0, 7);
                gui->raise();
                gui->safe_open_file(filename);
            }
        }
    }

//    osso_system_note_infoprint(gui->m_osso_context, method, retval);
//     osso_system_note_dialog (gui->m_osso_context,
//                              method,
//                              /* Icon to use */
//                              OSSO_GN_WARNING ,
//                              /* Were not interested in the RPC
//                                 return value . */
//                              NULL);

    dump_osso_rpc_value(retval);
    osso_rpc_free_val(retval);

    return OSSO_OK;
}
#endif


/**
 * Constructor to create the GUI application.
 *
 * \param argc command line argument count passed to Gtk::Main
 * \param argv command line arguments passed to Gtk::Main
 * \param service a reference to the Service object, providing all
 * business/service methods.
 */
KitListGui::KitListGui(int argc, char **argv, Service& service)
    : m_kit(argc, argv),
      m_ignore_list_events(false),
      m_window(0),
      m_window_preferences(0),
      m_entry_page_title(0),
      m_window_add_item(0),
      m_window_add_category(0),
      m_entry_add_item(0),
      m_entry_add_category(0),
      m_file_save_menu_item(0),
      m_file_save_tool_button(0),
      m_paste_menu_item(0),
      m_paste_tool_button(0),
      m_checkbutton_add_item(0),
      m_category_combo(0),
      m_item_tree_view(0),
      m_ref_item_tree_model(0),
      m_service(service),
      m_status_bar(0),
      m_state(ADD_CATEGORY),
#ifdef MAEMO
      m_full_screen(false),
#endif
      m_current_cat_id(-1) {
    m_ref_page_setup = Gtk::PageSetup::create();
    m_ref_printer_settings = Gtk::PrintSettings::create();
    init();
}


KitListGui::~KitListGui() {
#ifdef KITLIST_DEBUG
    if (m_slog) {
        m_slog->close();
        delete m_slog;
    }
#endif
}


/// Starts the GUI application running.
void KitListGui::run() {
    m_kit.run(*m_window);
#ifdef MAEMO
    if (m_osso_context != NULL)
        osso_deinitialize(m_osso_context);
#endif
    if (m_service.is_model_dirty()) {
#ifdef MAEMO
        // On maemo platform, save if still dirty
        // The methods KitListGui::on_delete_event(GdkEventAny* event)
        // and KitListGui::on_menu_quit() may leave the dirty flag set
        // if a save-on-close is required
        if (m_filename.length() > 0) {
            m_service.save_as_xml(m_filename);
        }
#else
        g_warning("WARNING: Application is closing with unsaved changes");
#endif //MAEMO
    }
}


gint KitListGui::get_max_recent_files() {
    gint retval = DEFAULT_MAX_RECENT_FILES;
#ifdef MAEMO
    Gnome::Conf::Value value;
#ifdef GLIBMM_EXCEPTIONS_ENABLED
    value = m_ref_gconf_client->get_int(GCONF_KEY_MAX_RECENT_FILES);
#else
    std::auto_ptr<Glib::Error> error;
    value = m_ref_gconf_client->get(GCONF_KEY_MAX_RECENT_FILES, error);
    if (error.get())
        g_warning("GConf error: %s", error->what().c_str());
#endif //GLIBMM_EXCEPTIONS_ENABLED
    if (value.get_type() == Gnome::Conf::VALUE_INT) {
        retval = value.get_int();
//     } else if (value.get_type() == Gnome::Conf::VALUE_INVALID) {
// #ifdef GLIBMM_EXCEPTIONS_ENABLED
//         m_ref_gconf_client->set_int(GCONF_KEY_MAX_RECENT_FILES, DEFAULT_MAX_RECENT_FILES);
// #else
//         value = m_ref_gconf_client->get(GCONF_KEY_MAX_RECENT_FILES, DEFAULT_MAX_RECENT_FILES, error);
//         if (error.get())
//             g_warning(_("GConf error: %s"), error->what().c_str());
// #endif //GLIBMM_EXCEPTIONS_ENABLED
    }
#endif
    return retval;
}

#ifdef MAEMO
/// Track window state
bool KitListGui::on_window_state_event(GdkEventWindowState* event) {
    m_full_screen = (event->new_window_state & GDK_WINDOW_STATE_FULLSCREEN);
}


/// Respond to Maemo hardware key events
bool KitListGui::on_key_pressed(GdkEventKey* key) {
    if (key->keyval == GDK_F6) {
        if (!m_full_screen) {
            m_window->fullscreen();
        } else {
            m_window->unfullscreen();
        }
        return true;
    }
    return false ;
};


/// Shows banner notifications
void KitListGui::show_banner(const Glib::ustring& msg) {
    Hildon::Banner::show_information(*m_window, msg);
}

#endif


/// Shows a confirmation message
bool KitListGui::confirm_lose_changes(const Glib::ustring& message) {
    // Prompt to abandon changes
    Gtk::MessageDialog dialog(*m_window,
                              message,
                              false,
                              Gtk::MESSAGE_WARNING,
                              Gtk::BUTTONS_NONE);
    dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
    dialog.add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_APPLY);
    dialog.add_button(_("_Discard Changes"), Gtk::RESPONSE_CLOSE);
    int result = dialog.run();
    switch (result) {
    case (Gtk::RESPONSE_CANCEL) :
        return false;
    case (Gtk::RESPONSE_APPLY) :
#ifdef XML_DAO
        if (m_filename.length() > 0) {
            m_service.save_as_xml(m_filename);
            m_service.set_model_dirty(false);
            return true;
        }
        return false;
#else
        m_service.save();
        m_service.set_model_dirty(false);
        return true;
#endif
    case (Gtk::RESPONSE_CLOSE) :
        return true;
    default :
        return false;
    }
}


/**
 * \brief Updates the recent files sub menu.
 */
void KitListGui::update_recent_files_menu() {
#ifdef GCONF
    // Get the current list from GConf
    std::deque<Glib::ustring> mru =
        m_ref_gconf_client->get_string_list(GCONF_KEY_RECENT_FILES);

    if (m_recent_files_menu_item) {
        if (m_recent_files_menu_item->has_submenu()) {
            m_recent_files_menu_item->remove_submenu();
        }
        Gtk::Menu* recent_menu = Gtk::manage(new Gtk::Menu);
        // g_debug("Creating new recent files submenu");
        m_recent_files_menu_item->set_submenu(*recent_menu);
        for (std::deque<Glib::ustring>::iterator i = mru.begin();
             i != mru.end();
             ++i) {
            Gtk::MenuItem* m = Gtk::manage(new Gtk::MenuItem(*i));
            m->signal_activate().connect(
                sigc::bind<Glib::ustring>(
                    sigc::mem_fun(*this, &KitListGui::on_menu_recent_file), *i)
                );
            recent_menu->add(*m);
            recent_menu->show_all_children();
        }
    } else {
        g_warning("Couldn't find recent files menu item");
    }
#endif // GCONF
}


/**
 * Updates the list of most recently used files.
 *
 * \param filename The filename to append to the list it it
 * does not already exist.
 */
void KitListGui::update_recent_files(const Glib::ustring& filename) {
#ifdef GCONF
    // g_debug("Adding %s to recent file list", filename.c_str());
    // Get the current list from GConf
    std::deque<Glib::ustring> mru =
        m_ref_gconf_client->get_string_list(GCONF_KEY_RECENT_FILES);
    // Remove filename from list
    for (std::deque<Glib::ustring>::iterator i = mru.begin(); i != mru.end(); ++i) {
        if (*i == filename) {
            mru.erase(i);
            break;
        }
    }
    // Add add to front
    mru.push_front(filename);
    while (mru.size() > get_max_recent_files()) {
        mru.pop_back();
    }

    // Save the updated list
    m_ref_gconf_client->set_string_list(GCONF_KEY_RECENT_FILES, mru);


    update_recent_files_menu();
#endif // GCONF
}


/// Called when the application is closed by the user
bool KitListGui::on_delete_event(GdkEventAny* event) {
    if (m_service.is_model_dirty() &&
            !confirm_lose_changes(_("Your changes will be lost if you quit now"))) {
        return true;
    }
    // user has chosen to abandon changes
    // Clear the dirty state so that it can be tested in run().
    // See the notes under KitListGui::run()
    m_service.set_model_dirty(false);
    return false;
}


/// Called when the user chooses to quit the application.
void KitListGui::on_menu_quit() {
    if (m_service.is_model_dirty() &&
            !confirm_lose_changes(_("Your changes will be lost if you quit now"))) {
        return;
    }
    // user has chosen to abandon changes
    // Clear the dirty state so that it can be tested in run().
    // See the notes under KitListGui::run()
    m_service.set_model_dirty(false);
    m_window->hide();
}


/**
 * \brief Creates a new empty model.
 *
 */
void KitListGui::on_menu_file_new() {
    if (m_status_bar)
        m_status_bar->pop(SB_SAVE);
    if (m_service.is_model_dirty() &&
        !confirm_lose_changes(_("Your changes will be lost if you continue"))) {
        return;
    }
#ifndef TRANSITION_DAO
    if (m_service.require_filename()) {
#endif
        m_filename.clear();
#ifdef GCONF
        if (m_ref_gconf_client) {
#ifdef GLIBMM_EXCEPTIONS_ENABLED
            m_ref_gconf_client->unset(GCONF_KEY_CURRENT_FILENAME);
#else
            std::auto_ptr<Glib::Error> error;
            m_ref_gconf_client->unset(GCONF_KEY_CURRENT_FILENAME, error);
            if (error.get())
                g_warning("GConf error: %s", error->what().c_str());
#endif //GLIBMM_EXCEPTIONS_ENABLED
        }
#endif // GCONF
        m_service.create_default_model();
        // Don't select a category, as it may no longer be relevant,
        // after all, we've just loaded a new model.
        refresh_category_list(-1);
        refresh_item_list();
#ifndef TRANSITION_DAO
    } else {
#ifdef MAEMO
        show_banner(_("New databases must be created manually"));
#else
        if (m_status_bar)
            m_status_bar->push(_("New databases must be created manually"), SB_SAVE);
#endif
    }
#endif
}


/**
 * \brief Allows the user to open an existing XML document.
 *
 * If there are unsaved changes, the user is first prompted to discard
 * them or cancel the operation.
 */
void KitListGui::on_menu_file_open() {
    if (m_status_bar)
        m_status_bar->pop(SB_SAVE);
    if (m_service.is_model_dirty() &&
            !confirm_lose_changes(_("Your changes will be lost if you continue"))) {
        return;
    }
#ifdef MAEMO
    Hildon::FileChooserDialog dialog(*m_window, Gtk::FILE_CHOOSER_ACTION_OPEN);
#else
    // Prompt select file to open
    Gtk::FileChooserDialog dialog(_("Open File"), Gtk::FILE_CHOOSER_ACTION_OPEN);
    dialog.set_transient_for(*m_window);

    dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
    dialog.add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_OK);
#endif
    Gtk::FileFilter filter_xml;
    filter_xml.set_name(_("Kit files"));
#ifdef WIN32
    filter_xml.add_pattern("*" + DEFAULT_FILENAME_EXTENSION);
#elif MAEMO
    // Filtering seems to be ignored on MAEMO
    // filter_xml.add_pattern("*" + DEFAULT_FILENAME_EXTENSION);
    // g_debug("Adding mime type filter of application/x-kitlist");
    filter_xml.add_pattern("*" + DEFAULT_FILENAME_EXTENSION);
    // filter_xml.add_mime_type("application/x-kitlist");
#else
    filter_xml.add_pattern("*" + DEFAULT_FILENAME_EXTENSION);
    // filter_xml.add_mime_type("application/xml");
#endif
    dialog.add_filter(filter_xml);

    Gtk::FileFilter filter_any;
    filter_any.set_name(_("Any files"));
    filter_any.add_pattern("*");
    dialog.add_filter(filter_any);

    int result = dialog.run();
    dialog.hide();
    Glib::ustring filename;
    switch(result) {
    case(Gtk::RESPONSE_OK):
        filename = dialog.get_filename();
        break;
    case(Gtk::RESPONSE_CANCEL):
        return;
    default:
        return;
    }
    open_file(filename);
}


/**
 * \brief Opens an existing XML document.
 *
 */
void KitListGui::open_file(const Glib::ustring& filename) {
    m_service.open_as_xml(filename);
    m_filename = filename;
    update_recent_files(m_filename);

#ifdef GCONF
    if (m_ref_gconf_client) {
#ifdef GLIBMM_EXCEPTIONS_ENABLED
        m_ref_gconf_client->set(GCONF_KEY_CURRENT_FILENAME, m_filename);
#else
        std::auto_ptr<Glib::Error> error;
        m_ref_gconf_client->set(GCONF_KEY_CURRENT_FILENAME, m_filename, error);
        if (error.get())
            g_warning("GConf error: %s", error->what().c_str());
#endif //GLIBMM_EXCEPTIONS_ENABLED
    }
#endif

    // Don't select a category, as it may no longer be relevant,
    // after all, we've just loaded a new model.
    refresh_category_list(-1);
    refresh_item_list();
}


/**
 * \brief Opens an existing XML document, checking for unsaved changes.
 *
 * If there are unsaved changes, the user is first prompted to discard
 * them or cancel the operation.
 */
void KitListGui::safe_open_file(const Glib::ustring& filename) {
    if (m_status_bar)
        m_status_bar->pop(SB_SAVE);
    if (m_service.is_model_dirty() &&
            !confirm_lose_changes(_("Your changes will be lost if you continue"))) {
        return;
    }
    open_file(filename);
}


/**
 * \brief Saves the current application state.
 */
void KitListGui::on_menu_save() {
    if (m_status_bar)
        m_status_bar->pop(SB_SAVE);
    if (m_service.is_model_dirty()) {
        // Some persistence models may require a filename to be
        // chosen, others (e.g. database) may be defined through
        // another mechanism.  If a filename has not been set, then
        // fire up save-as instead.
        if (m_service.require_filename() && m_filename.size() == 0) {
            on_menu_save_as();
        } else {
            Glib::ustring msg = _("Saving changes...");
            if (m_status_bar)
                m_status_bar->push(msg, SB_SAVE);
#ifndef XML_DAO
            m_service.save();
#else
            m_service.save_as_xml(m_filename);
            update_recent_files(m_filename);
#ifdef GCONF
            if (m_ref_gconf_client) {
#ifdef GLIBMM_EXCEPTIONS_ENABLED
                m_ref_gconf_client->set(GCONF_KEY_CURRENT_FILENAME,
                                        m_filename);
#else
                std::auto_ptr<Glib::Error> error;
                m_ref_gconf_client->set(GCONF_KEY_CURRENT_FILENAME, m_filename, error);
                if (error.get())
                    g_warning("GConf error: %s", error->what().c_str());
#endif //GLIBMM_EXCEPTIONS_ENABLED
            }
#endif // GCONF
#endif // !XML_DAO
            if (m_status_bar)
                m_status_bar->pop(SB_SAVE);
            msg = _("Saved");
            if (m_status_bar)
                m_status_bar->push(msg, SB_SAVE);
        }
    } else {
#ifdef MAEMO
        show_banner(_("Nothing to save"));
#else
        if (m_status_bar)
            m_status_bar->push(Glib::ustring(_("Nothing to save")), SB_SAVE);
#endif
    }
}


/**
 * \brief Saves the current application state in a new document.
 */
void KitListGui::on_menu_save_as() {
    if (m_status_bar) {
        m_status_bar->pop(SB_SAVE);
        m_status_bar->push(_("Choosing target filename"), SB_SAVE);
    }
    if (choose_filename(m_filename)) {
        if (m_status_bar) {
            m_status_bar->pop(SB_SAVE);
            m_status_bar->push(_("Saving..."), SB_SAVE);
        }

        m_service.save_as_xml(m_filename);
        update_recent_files(m_filename);

        if (m_status_bar) {
            m_status_bar->pop(SB_SAVE);
            m_status_bar->push(_("Saved"), SB_SAVE);
        }
#ifdef GCONF
        if (m_ref_gconf_client) {
#ifdef GLIBMM_EXCEPTIONS_ENABLED
            m_ref_gconf_client->set(GCONF_KEY_CURRENT_FILENAME, m_filename);
#else
            std::auto_ptr<Glib::Error> error;
            m_ref_gconf_client->set(GCONF_KEY_CURRENT_FILENAME, m_filename, error);
            if (error.get())
                g_warning("GConf error: %s", error->what().c_str());
#endif //GLIBMM_EXCEPTIONS_ENABLED
        }
#endif // GCONF
#ifdef MAEMO
        show_banner(_("Saved"));
#endif

        // Disable save if we're using a database otherwise we'll be
        // trying to save to the database, not the file with the model
        // being potentially inconsistent with the database.

// Don't think we need this as we've now disabled NEW and OPEN
// #ifndef XML_DAO
//         m_file_save_menu_item->set_sensitive(false);
//         m_file_save_tool_button->set_sensitive(false);
// #endif
    } else {
        if (m_status_bar) {
            m_status_bar->pop(SB_SAVE);
            m_status_bar->push(_("Save cancelled"), SB_SAVE);
        }
#ifdef MAEMO
        show_banner(_("Save cancelled"));
#endif
    }
}


/**
 * \brief Called when the print status changes
 */
void KitListGui::on_printoperation_status_changed(const Glib::RefPtr<Gtk::PrintOperation>& op) {
    if (m_status_bar)
        m_status_bar->pop(SB_PRINT);
    Glib::ustring status_msg;
    if (op->is_finished())
        status_msg = _("Print job completed");
    else
        status_msg = op->get_status_string();

    m_status_bar->push(status_msg, SB_PRINT);
}

/**
 * \brief Called when the printer operation is done
 */
void KitListGui::on_printoperation_done(Gtk::PrintOperationResult result, const Glib::RefPtr<Gtk::PrintOperation>& op) {
    if (result == Gtk::PRINT_OPERATION_RESULT_ERROR) {
        Gtk::MessageDialog err_dialog(*this->m_window, _("Error printing list"), false,
                                      Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK, true);
        err_dialog.run();
    } else if (result == Gtk::PRINT_OPERATION_RESULT_APPLY)
        m_ref_printer_settings = op->get_print_settings();;
    if (! op->is_finished())
        op->signal_status_changed().connect(sigc::bind(sigc::mem_fun(*this, &KitListGui::on_printoperation_status_changed), op));
}


/**
 * \brief Prints the kit list
 */
void KitListGui::on_menu_print() {
    Glib::RefPtr<KitPrintOperation> op = KitPrintOperation::create();
    ItemContainer *items = m_service.get_filtered_items(get_selected_category());
    op->set_items(items);
    op->set_track_print_status();
    op->set_default_page_setup(m_ref_page_setup);
    op->set_print_settings(m_ref_printer_settings);
    op->set_page_title(m_page_title);
    op->signal_done().connect(sigc::bind(sigc::mem_fun(*this, &KitListGui::on_printoperation_done), op));

#ifdef GLIBMM_EXCEPTIONS_ENABLED
    try {
        op->run();
    } catch (const Gtk::PrintError& ex) {
        g_error("An error occured while trying to run a print operation: %s", ex.what().c_str());
    }
#else
    std::auto_ptr<Glib::Error> error;
    op->run(Gtk::PRINT_OPERATION_ACTION_PRINT_DIALOG, error);
    if (error.get())
        g_warning("Error running print job: %s",
                  error->what().c_str());
#endif //GLIBMM_EXCEPTIONS_ENABLED
}


/**
 * Allows the user to choose a file to export the kitlist to a PDF file.
 */
void KitListGui::on_menu_export_to_pdf() {
    if (m_status_bar) {
        m_status_bar->pop(SB_SAVE);
        m_status_bar->push(_("Choosing export filename"), SB_SAVE);
    }
    Glib::ustring pdf_filename;
    if (choose_pdf_filename(pdf_filename)) {
        if (m_status_bar) {
            m_status_bar->pop(SB_SAVE);
            m_status_bar->push(_("Exporting..."), SB_SAVE);
        }
        Glib::RefPtr<KitPrintOperation> op = KitPrintOperation::create();
        ItemContainer *items = m_service.get_filtered_items(get_selected_category());
        op->set_items(items);
        op->set_track_print_status();
        op->set_default_page_setup(m_ref_page_setup);
        op->set_print_settings(m_ref_printer_settings);
        op->set_page_title(m_page_title);
        // op->signal_done().connect(sigc::bind(sigc::mem_fun(*this, &KitListGui::on_printoperation_done), op));
        op->set_export_filename(pdf_filename);

    #ifdef GLIBMM_EXCEPTIONS_ENABLED
        try {
            op->run(Gtk::PRINT_OPERATION_ACTION_EXPORT);
        } catch (const Gtk::PrintError& ex) {
            g_error("An error occured while trying to run a print operation: %s", ex.what().c_str());
        }
    #else
        std::auto_ptr<Glib::Error> error;
        op->run(Gtk::PRINT_OPERATION_ACTION_EXPORT, error);
        if (error.get())
            g_warning("Error running print job: %s",
                      error->what().c_str());
    #endif //GLIBMM_EXCEPTIONS_ENABLED

        if (m_status_bar) {
            m_status_bar->pop(SB_SAVE);
            m_status_bar->push(_("PDF exported"), SB_SAVE);
        }
#ifdef MAEMO
        show_banner(_("PDF exported"));
#endif
    } else {
        if (m_status_bar) {
            m_status_bar->pop(SB_SAVE);
            m_status_bar->push(_("Export cancelled"), SB_SAVE);
        }
#ifdef MAEMO
        show_banner(_("Export cancelled"));
#endif
    }
}

/**
 * \brief displays the most recent files menu
 */
void KitListGui::on_menu_recent_file(const Glib::ustring& filename) {
    // g_debug("Most recent files menu clicked: %s", filename.c_str());
    if (m_service.is_model_dirty() &&
            !confirm_lose_changes(_("Your changes will be lost if you continue"))) {
        return;
    }
    m_service.open_as_xml(filename);
    m_filename = filename;
    update_recent_files(m_filename);

#ifdef GCONF
    if (m_ref_gconf_client) {
#ifdef GLIBMM_EXCEPTIONS_ENABLED
        m_ref_gconf_client->set(GCONF_KEY_CURRENT_FILENAME, m_filename);
#else
        std::auto_ptr<Glib::Error> error;
        m_ref_gconf_client->set(GCONF_KEY_CURRENT_FILENAME, m_filename, error);
        if (error.get())
            g_warning("GConf error: %s", error->what().c_str());
#endif //GLIBMM_EXCEPTIONS_ENABLED
    }
#endif // GCONF

    // Don't select a category, as it may no longer be relevant,
    // after all, we've just loaded a new model.
    refresh_category_list(-1);
    refresh_item_list();
}


/**
 * \brief Returns a list of items selected in the item list.
 */
ModelItemContainer* KitListGui::get_selected_items() {
    ModelItemContainer* retval = new ModelItemContainer;
    Glib::RefPtr<Gtk::TreeSelection> ref_tree_selection =
        m_item_tree_view->get_selection();
    if (ref_tree_selection) {
        if (ref_tree_selection->get_mode() == Gtk::SELECTION_MULTIPLE) {
            Gtk::TreeSelection::ListHandle_Path rows = ref_tree_selection->get_selected_rows();
            for (Glib::Container_Helpers::ListHandleIterator<Gtk::TreePath_Traits> path = rows.begin(); path != rows.end(); ++path) {
                Gtk::TreeModel::iterator iter = m_ref_item_tree_model->get_iter(*path);
                Gtk::TreeModel::Row row = *iter;
                int id = row[m_item_cols.m_col_num];
                ModelItem* item = m_service.find_item(id);
                if (item) {
                    retval->push_back(item);
                } else {
                    g_warning("Failed to find item to delete in model");
                }
            } // for
        } else {
            Gtk::TreeModel::iterator iter = ref_tree_selection->get_selected();
            if (iter) {
                Gtk::TreeModel::Row row = *iter;
                int id = row[m_item_cols.m_col_num];
                ModelItem* item = m_service.find_item(id);
                if (item) {
                    retval->push_back(item);
                } else {
                    g_warning("Failed to find item to delete in model");
                }
            } else {
                g_warning("Couldn't find the selected item or items");
            }
        } // else {Single Selection}
    } else {
        g_warning("Couldn't identify selection");
    }
    return retval;
}


/**
 * \brief Deletes the currently selected items.
 *
 * The items are flagged as deleted.  When save is called, they will
 * no longer be persisted, i.e. they will be permanently deleted.
 */
void KitListGui::delete_selected_items() {
    Glib::RefPtr<Gtk::TreeSelection> ref_tree_selection =
        m_item_tree_view->get_selection();
    if (ref_tree_selection) {
        if (ref_tree_selection->get_mode() == Gtk::SELECTION_MULTIPLE) {
            ref_tree_selection->selected_foreach_iter( sigc::mem_fun(*this, &KitListGui::selected_row_callback));
            refresh_item_list();
        } else {
            Gtk::TreeModel::iterator iter = ref_tree_selection->get_selected();
            if (iter) {
                Gtk::TreeModel::Row row = *iter;
                int id = row[m_item_cols.m_col_num];
                m_ref_item_tree_model->erase(iter);
                if (!m_service.delete_item(id))
                    g_warning("Failed to find the item to delete in the model");
            } else {
                g_warning("Couldn't find the selected item or items");
            }
        } // else {Single Selection}
    } else {
        g_warning("Couldn't identify selection");
    }
}


/**
 * \brief Called when the user chooses to delete items.
 *
 * The user is prompted to confirm deletion, prior to the items being
 * flagged as deleted in the model.  Once the model is saved, the will
 * be permanently deleted from the persistence store.
 */
void KitListGui::on_menu_delete() {
    Gtk::MessageDialog dialog(*m_window,
                              _("Delete the selected items?"),
                              false,
                              Gtk::MESSAGE_QUESTION,
                              Gtk::BUTTONS_YES_NO);
    int result = dialog.run();
    switch (result) {
    case (Gtk::RESPONSE_YES) :
        delete_selected_items();
        break;
    case (Gtk::RESPONSE_NO) :
        break;
    default:
        g_warning("Unexpected button");
        break;
    }
}


/**
 * Copies the currently selected items to the clipboard.
 */
ModelItemContainer* KitListGui::copy_selected_items_to_clipboard() {
    xmlpp::Document document;
    xmlpp::Element* nodeRoot = document.create_root_node("kitlist");
    assert(nodeRoot);
    nodeRoot->add_child_text("\n");
    ModelItemContainer* items = get_selected_items();
    for (ModelItemIter i = items->begin(); i != items->end(); ++i) {
        xmlpp::Element* nodeChild = nodeRoot->add_child("item");
        ostringstream os;
        os << (*i)->get_id();
        nodeChild->set_attribute(XML_ELEMENT_ID, os.str());
        // Enable the following lines to create a more readable XML document
        nodeChild->set_child_text((*i)->get_description());
        nodeRoot->add_child_text("\n");
    }
    m_clipboard_items = document.write_to_string();

    Glib::RefPtr<Gtk::Clipboard> refClipboard = Gtk::Clipboard::get(/*gdk_atom_intern("CLIPBOARD", false)*/);
    std::list<Gtk::TargetEntry> listTargets;
    listTargets.push_back( Gtk::TargetEntry(item_target_custom) );
    listTargets.push_back( Gtk::TargetEntry(item_target_text) );
    refClipboard->set( listTargets,
                       sigc::mem_fun(*this, &KitListGui::on_clipboard_get),
                       sigc::mem_fun(*this, &KitListGui::on_clipboard_clear) );
    update_paste_status();
    return items;
}


/**
 * Copies the currently selected items to the clipboard as a 'cut'
 * operation.
 */
void KitListGui::on_menu_cut() {
    long cat_id = get_selected_category();
    if (cat_id != -1) {
        ModelItemContainer* items = copy_selected_items_to_clipboard();
        if (items) {
            ModelCategory* category = m_service.find_category(cat_id);
            if (category) {
                m_service.set_model_dirty();
                category->remove_items(items);
                category->set_dirty(true);
            }
            refresh_item_list();
        }
        delete items;
    } else {
        if (m_status_bar) {
            m_status_bar->pop(SB_MSG);
            m_status_bar->push(_("Items can only be cut when a category is selected"), SB_MSG);
        }
#ifdef MAEMO
        show_banner(_("Select a category before copying items"));
#endif
    }
}


/**
 * \brief Called when the use chooses the 'copy' menu option.
 */
void KitListGui::on_menu_copy() {
    ModelItemContainer* items = copy_selected_items_to_clipboard();
    delete items;
}


/**
 * \brief Called when the user chooses the 'paste' menu option.
 */
void KitListGui::on_menu_paste() {
    // Tell the clipboard to call our method when it is ready
    Glib::RefPtr<Gtk::Clipboard> refClipboard = Gtk::Clipboard::get(/*gdk_atom_intern("CLIPBOARD", false)*/);
    refClipboard->request_contents(item_target_text, sigc::mem_fun(
                                       *this,
                                       &KitListGui::on_clipboard_received) );
    paste_from_xml(m_clipboard_items);
    update_paste_status();
}


/**
 * \brief Called when the user chooses the 'select all' menu option.
 */
void KitListGui::on_menu_select_all() {
    //m_item_tree_view->select_all();
    Glib::RefPtr<Gtk::TreeSelection> selection =
        m_item_tree_view->get_selection();
    if (selection)
        selection->select_all();
    m_item_tree_view->grab_focus();
}


/// Called when the user chooses to create a new category.
void KitListGui::on_menu_create_category() {
    m_state = ADD_CATEGORY;
    m_current_cat_id = -1;
#ifdef MAEMO
    Gtk::Dialog::Dialog dialog(_("Add Category"), *m_window, true, true);
    // Gtk::Label label(_("Description"), true);
    Gtk::VBox* vbox = dialog.get_vbox();

    Glib::RefPtr<Gtk::SizeGroup> group = Gtk::SizeGroup::create(Gtk::SIZE_GROUP_VERTICAL);
    Gtk::Entry name;

    Hildon::Caption* caption = Gtk::manage(new Hildon::Caption(
                                               _("Name"),
                                               name));
    vbox->pack_start(*caption);

    dialog.set_size_request(500);

    dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
    dialog.add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
    dialog.show_all_children();
    int result = dialog.run();
    dialog.hide();
    switch (result) {
    case (Gtk::RESPONSE_CANCEL) :
        break;
    case (Gtk::RESPONSE_OK) :
//         if (m_state == ADD_CATEGORY) {
        Category* category = m_service.create_category();
//         } else {
//             category = m_service.find_category(m_current_cat_id);
//         }
        if (category != NULL) {
            category->set_name(name.get_text());
            m_service.set_model_dirty();
            ((ModelCategory*) category)->set_dirty(true);
            refresh_category_list(category->get_id());
            refresh_item_list();
        }
        break;
    default :
        break;
    }


#else
    if (m_window_add_category) {
        m_window_add_category->set_title(_("Add Category"));
        if (m_entry_add_category) {
            m_entry_add_category->set_text("");
        }
        m_entry_add_category->grab_focus();
        m_window_add_category->show();
    } else {
        g_warning("\"window_add_category\" resource is missing");
    }
#endif
}


/**
 * \brief Called when the add/rename category dialog is closed using
 * the 'OK' button.
 *
 * The same dialog is used for both creating a new category and
 * renaming an existing one.  m_state is used to indicate which
 * operation is relevant (create or rename) and m_current_cat_id is
 * used to indicate the category being renamed for the rename
 * operation.
 */
void KitListGui::close_add_category_window() {
    m_window_add_category->hide();
    if (m_entry_add_category) {
        Category* category = NULL;
        if (m_state == ADD_CATEGORY) {
            category = m_service.create_category();
        } else {
            category = m_service.find_category(m_current_cat_id);
        }
        if (category != NULL) {
            category->set_name(m_entry_add_category->get_text());
            m_service.set_model_dirty();
            ((ModelCategory*) category)->set_dirty(true);
            refresh_category_list(category->get_id());
            refresh_item_list();
        }
    }
}



/**
 * \brief Called when the add/rename category dialog is closed using
 * the 'Cancel' button.
 */
void KitListGui::cancel_add_category_window() {
    m_window_add_category->hide();
}


/**
 * \brief Called when the user chooses the delete category menu option.
 *
 * Displays a confirmation box before deleting the currently selected
 * category.
 */
void KitListGui::on_menu_delete_category() {
    long cat_id = get_selected_category();
    if (cat_id == -1) {
        if (m_status_bar) {
            m_status_bar->pop(SB_MSG);
            m_status_bar->push(_("No category selected"), SB_MSG);
        }
#ifdef MAEMO
        show_banner(_("No category selected"));
#endif

    } else {
        Gtk::MessageDialog dialog(*m_window,
                                  _("Delete the current category?"),
                                  false,
                                  Gtk::MESSAGE_QUESTION,
                                  Gtk::BUTTONS_YES_NO);
        int result = dialog.run();
        switch (result) {
        case (Gtk::RESPONSE_YES) :
            m_service.delete_category(cat_id);
            refresh_category_list();
            refresh_item_list();
            break;
        case (Gtk::RESPONSE_NO) :
            break;
        default:
            g_warning("Unexpected button");
            break;
        }
    }
}


/**
 * \brief Called when the user chooses to rename a category.
 */
void KitListGui::on_menu_rename_category() {
    m_state = RENAME_CATEGORY;
    m_current_cat_id = get_selected_category();
#ifdef MAEMO
    if (m_current_cat_id != -1) {

        ModelCategory* category = m_service.find_category(m_current_cat_id);
        if (category) {

            Gtk::Dialog::Dialog dialog(_("Rename Category"), *m_window, true, true);
            // Gtk::Label label(_("Description"), true);
            Gtk::VBox* vbox = dialog.get_vbox();

            Glib::RefPtr<Gtk::SizeGroup> group = Gtk::SizeGroup::create(Gtk::SIZE_GROUP_VERTICAL);
            Gtk::Entry name;
            name.set_text(category->get_name());

            Hildon::Caption* caption = Gtk::manage(new Hildon::Caption(
                                                       _("Name"),
                                                       name));
            vbox->pack_start(*caption);

            dialog.set_size_request(500);

            dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
            dialog.add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
            dialog.show_all_children();
            int result = dialog.run();
            dialog.hide();
            switch (result) {
            case (Gtk::RESPONSE_CANCEL) :
                break;
            case (Gtk::RESPONSE_OK) :
//         if (m_state == ADD_CATEGORY) {
//                Category* category = m_service.create_category();
//         } else {
//                 category = m_service.find_category(m_current_cat_id);
//         }
//                 if (category != NULL) {
                category->set_name(name.get_text());
                m_service.set_model_dirty();
                ((ModelCategory*) category)->set_dirty(true);
                refresh_category_list(category->get_id());
                refresh_item_list();
//                 }
                break;
            default :
                break;
            }
        } // if (category)

    } else {
        show_banner(_("No category selected"));
    }
#else
    if (m_window_add_category) {
        m_window_add_category->set_title(_("Rename Category"));
        if (m_current_cat_id != -1) {
            ModelCategory* category = m_service.find_category(m_current_cat_id);
            if (category != NULL && m_entry_add_category) {
                m_entry_add_category->set_text(category->get_name());
            }
            m_entry_add_category->grab_focus();
            m_window_add_category->show();
        } else {
            if (m_status_bar) {
                m_status_bar->pop(SB_MSG);
                m_status_bar->push(_("No category selected"), SB_MSG);
            }
        }
    } else {
        g_warning("\"window_add_category\" resource is missing");
    }
#endif
}


/**
 * Displays the help documentation
 */
void KitListGui::on_menu_help_contents() {
    // g_debug("on_menu_help_contents()");
    // Note the 2nd parameter must be in the form of xxx_foo_yyy
    // where foo is the exact name of the help file under
    // /usr/share/osso-help/en_GB/ without the '.xml' extension
#ifdef MAEMO
    hildon_help_show(m_osso_context, "contents_kitlist_help", 0 /*HILDON_HELP_SHOW_DIALOG*/);
#endif
}


/**
 * Shows the help about dialog
 */
void KitListGui::on_menu_help_about() {
    Gtk::AboutDialog dialog;
    dialog.set_name(PACKAGE_NAME);
    dialog.set_copyright("(c) 2008-2018 Frank Dean");
    dialog.set_version(PACKAGE_VERSION);
    std::vector<Glib::ustring> authors;
    authors.push_back("frank.dean@fdsd.co.uk");
    dialog.set_authors(authors);
    dialog.run();
}


/**
 * Checks or unchecks the currently selected items.
 */
void KitListGui::set_selected(bool checked) {
    ModelItemContainer* items = get_selected_items();
    m_service.select_items(items, checked);
    refresh_item_list();
    delete items;
}


/**
 * Toggles the checked state of the currently selected items.
 */
void KitListGui::toggle_selected() {
    ModelItemContainer* items = get_selected_items();
    m_service.toggle_selected_items(items);
    refresh_item_list();
    delete items;
}


/**
 * Called when the current clipboard contents are required.
 */
void KitListGui::on_clipboard_get(Gtk::SelectionData& selection_data, guint) {
    const std::string target = selection_data.get_target();

    if (target == item_target_custom) {
        selection_data.set_text(m_clipboard_items);
    } else if (target == item_target_text) {
        selection_data.set_text(m_clipboard_items);
    } else {
        g_warning("KitList::on_clipboard_get(): Unexpected clipboard target format");
    }
}


/**
 * \brief This method gets called after a second 'copy' operation.
 *
 * i.e. if we have previously called Clipboard::set() and then called
 * it a second time.  However, I think this is intended to only clean
 * up data objects that may have been allocated by a previous call to
 * on_clipboard_get() where we might have created an expensive object
 * based on a list of IDs or something.  This gives us the opportunity
 * to release the big object.
 * 
 * However, in the way we're using the clipboard, we're holding the
 * expensive object (in this case an XML document).  We should probably
 * be holding the list of selected ID's and only creating the XML
 * document when the clipboard contents are requested.
 * 
 * As things are, we don't want to clear the XML document...
 * m_clipboard_items.clear();
 */
void KitListGui::on_clipboard_clear() {
}


/**
 * \brief Associates the passed list of items with the currently
 * selected category.
 *
 * Where items already exist in the Category, then are not duplicated,
 * just silently ignored.
 */
void KitListGui::add_items(const ModelItemContainer& items) {
    long cat_id = get_selected_category();
    if (cat_id != -1) {
        m_service.copy_items(items, cat_id);
        refresh_item_list();
    }
}


/**
 * Performs a paste of items from the clipboard.
 */
void KitListGui::paste_from_xml(const Glib::ustring& document) {
    try {
        if (document.length() > 0) {
            xmlpp::DomParser parser;
            parser.set_substitute_entities();
            parser.parse_memory(document);
            const xmlpp::Node* nodeRoot = parser.get_document()->get_root_node();
            ModelItemContainer items;
            xmlpp::Node::NodeList nodes = nodeRoot->get_children();
            for (xmlpp::Node::NodeList::iterator iter = nodes.begin(); iter != nodes.end(); ++iter) {
                const xmlpp::Node* node = (*iter);
                // const xmlpp::ContentNode* nodeContent = dynamic_cast<const xmlpp::ContentNode*>(node);
                const xmlpp::Element* nodeElement = dynamic_cast<const xmlpp::Element*>(node);
                if (nodeElement && node->get_name() == "item") {
                    xmlpp::Attribute* attribute = nodeElement->get_attribute(XML_ELEMENT_ID);
                    if (attribute) {
                        Glib::ustring s = attribute->get_value();
                        long id = atol(s.c_str());
                        ModelItem* item = m_service.find_item(id);
                        if (item) {
                            items.push_back(item);
                        }
                    }
                }
            }
            add_items(items);
        }
    } catch (std::exception ex) {
        g_warning("Error pasting clipboard - perhaps it was empty");
    }
}


/**
 * Callback notification method for capturing clipboard contents.
 */
void KitListGui::on_clipboard_received(const Gtk::SelectionData& selection_data) {
    const std::string target = selection_data.get_target();
    Glib::ustring s;
    if (target == item_target_custom || target == item_target_text) {
        m_clipboard_items = selection_data.get_data_as_string();
    } else {
        g_warning("KitList::on_clipboard_get(): Unexpected clipboard target format");
    }
}


/**
 * Writes the count of currently displayed items to the status bar.
 */
void KitListGui::update_item_count(size_t n) {
    // use the user's locale for this stream
    ostringstream os;
    os.imbue(std::locale(""));
    os << n << " " << (n == 1 ? _("item") : _("items"));
    if (m_status_bar) {
        m_status_bar->pop(SB_ITEM_COUNT);
#ifdef GLIBMM_EXCEPTIONS_ENABLED
        m_status_bar->push(Glib::locale_to_utf8(os.str()), SB_ITEM_COUNT);
#else
        std::auto_ptr<Glib::Error> ex;
        m_status_bar->push(Glib::locale_to_utf8(os.str(), ex), SB_ITEM_COUNT);

        if (ex.get())
            g_warning("Error updating the status bar: %s", ex->what().c_str());
#endif // GLIBMM_EXCEPTIONS_ENABLED
    }

// #ifdef MAEMO
// #ifdef GLIBMM_EXCEPTIONS_ENABLED
//     show_banner(Glib::locale_to_utf8(os.str()));
// #else
//     std::auto_ptr<Glib::Error> ex;
//     show_banner(Glib::locale_to_utf8(os.str(), ex));
//     if (ex.get())
//         cerr << "Error showing the item count: " << ex->what() << endl;
// #endif // GLIBMM_EXCEPTIONS_ENABLED
// #endif // MAEMO
}


/**
 * \brief Callback method called when a cell has been edited in the item list.
 *
 * Flags the model as dirty.
 */
void KitListGui::on_cell_edit(const Glib::ustring s) {
    if (m_ignore_list_events)
        return;
    m_service.set_model_dirty(true);
}


/**
 * \brief Displays a file chooser dialog for a user to choose a
 * filename.
 *
 * If the file already exists, the user is asked to confirm whether to overwrite.
 *
 * \param filename A reference to a string to populate with the chosen
 * filename.
 * \return true if the user selects OK, false otherwise.
 */
bool KitListGui::choose_filename(Glib::ustring& filename) {
    bool retval = false;
    bool loop = true;
    while (loop) {
#ifdef MAEMO
        Hildon::FileChooserDialog dialog(*m_window, Gtk::FILE_CHOOSER_ACTION_SAVE);
#else
        Gtk::FileChooserDialog dialog(_("Save File"), Gtk::FILE_CHOOSER_ACTION_SAVE);
        dialog.set_transient_for(*m_window);

        dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
        dialog.add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_OK);
#endif

        Gtk::FileFilter filter_xml;
        filter_xml.set_name(_("XML files"));

#ifdef WIN32
        filter_xml.add_pattern("*" + DEFAULT_FILENAME_EXTENSION);
#elif MAEMO
        filter_xml.add_pattern("*" + DEFAULT_FILENAME_EXTENSION);
#else
        filter_xml.add_pattern("*" + DEFAULT_FILENAME_EXTENSION);
        // filter_xml.add_mime_type("application/xml");
#endif

        dialog.add_filter(filter_xml);

        Gtk::FileFilter filter_any;
        filter_any.set_name(_("Any files"));
        filter_any.add_pattern("*");
        dialog.add_filter(filter_any);

        int result = dialog.run();
        dialog.hide();

        switch(result) {
        case(Gtk::RESPONSE_OK):
            filename = dialog.get_filename();
            retval = true;
            break;
        case(Gtk::RESPONSE_CANCEL):
            break;
        default:
            break;
        }
        if (retval) {
            // Add file default extension if no extension
            Glib::ustring::size_type i = filename.find(".");
            if (i < 0 || i > filename.length()) {
                filename += DEFAULT_FILENAME_EXTENSION;

            }
            // Prompt for overwrite if the file exists
            if (file_exists(filename)) {
                Gtk::MessageDialog dialog(*m_window,
                                          _("Overwrite existing file?"),
                                          false,
                                          Gtk::MESSAGE_WARNING,
                                          Gtk::BUTTONS_NONE);
                dialog.add_button(Gtk::Stock::NO, Gtk::RESPONSE_NO);
                dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
                dialog.add_button(Gtk::Stock::YES, Gtk::RESPONSE_YES);
                dialog.set_default_response(Gtk::RESPONSE_NO);
                int result = dialog.run();
                switch (result) {
                case (Gtk::RESPONSE_CANCEL) :
                    retval = false;
                    loop = false;
                    break;
                case (Gtk::RESPONSE_NO) :
                    retval = false;
                    loop = true;
                    break;
                case (Gtk::RESPONSE_YES) :
                    loop = false;
                    break;
                default :
                    retval = false;
                    loop = false;
                    break;
                }

            } else {
                loop = false;
            }
        } else {
            loop = false;
        }
    } // while loop
    return retval;
}


/**
 * \brief Displays a file chooser dialog for a user to choose a
 * filename for export to PDF.
 *
 * If the file already exists, the user is asked to confirm whether to overwrite.
 *
 * \param filename A reference to a string to populate with the chosen
 * filename.
 * \return true if the user selects OK, false otherwise.
 */
bool KitListGui::choose_pdf_filename(Glib::ustring& filename) {
    bool retval = false;
    bool loop = true;
    while (loop) {
#ifdef MAEMO
        Hildon::FileChooserDialog dialog(*m_window, Gtk::FILE_CHOOSER_ACTION_SAVE);
#else
        Gtk::FileChooserDialog dialog(_("Export to PDF"), Gtk::FILE_CHOOSER_ACTION_SAVE);
        dialog.set_transient_for(*m_window);

        dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
        dialog.add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_OK);
#endif

        Gtk::FileFilter filter_pdf;
        filter_pdf.set_name(_("PDF files"));
        filter_pdf.add_pattern("*" + PDF_FILENAME_EXTENSION);
        dialog.add_filter(filter_pdf);

        Gtk::FileFilter filter_any;
        filter_any.set_name(_("Any files"));
        filter_any.add_pattern("*");
        dialog.add_filter(filter_any);

        int result = dialog.run();
        dialog.hide();

        switch(result) {
        case(Gtk::RESPONSE_OK):
            filename = dialog.get_filename();
            retval = true;
            break;
        case(Gtk::RESPONSE_CANCEL):
            break;
        default:
            break;
        }
        if (retval) {
            // Add file default extension if no extension
            Glib::ustring::size_type i = filename.find(".");
            if (i < 0 || i > filename.length()) {
                filename += PDF_FILENAME_EXTENSION;

            }
            // Prompt for overwrite if the file exists
            if (file_exists(filename)) {
                Gtk::MessageDialog dialog(*m_window,
                                          _("Overwrite existing file?"),
                                          false,
                                          Gtk::MESSAGE_WARNING,
                                          Gtk::BUTTONS_NONE);
                dialog.add_button(Gtk::Stock::NO, Gtk::RESPONSE_NO);
                dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
                dialog.add_button(Gtk::Stock::YES, Gtk::RESPONSE_YES);
                dialog.set_default_response(Gtk::RESPONSE_NO);
                int result = dialog.run();
                switch (result) {
                case (Gtk::RESPONSE_CANCEL) :
                    retval = false;
                    loop = false;
                    break;
                case (Gtk::RESPONSE_NO) :
                    retval = false;
                    loop = true;
                    break;
                case (Gtk::RESPONSE_YES) :
                    loop = false;
                    break;
                default :
                    retval = false;
                    loop = false;
                    break;
                }
                
            } else {
                loop = false;
            }
        } else {
            loop = false;
        }
    } // while loop
    return retval;
}


/**
 * \brief Enables or disables the paste menu option.
 */
void KitListGui::update_paste_status() {
    Glib::RefPtr<Gtk::Clipboard> refClipboard = Gtk::Clipboard::get();
    refClipboard->request_targets( sigc::mem_fun(*this, &KitListGui::paste_status_received) );
}


/**
 * Callback method for enabling or disabling the paste menu option
 * based on the clipboard contents.
 *
 * \see KitListGui::update_paste_status()
 */
void KitListGui::paste_status_received(const Glib::StringArrayHandle& targets_array) {
    std::list<std::string> targets = targets_array;
    bool can_paste = false;
    for (std::list<std::string>::iterator i = targets.begin(); i != targets.end(); ++i) {
        if ( *i == item_target_custom || *i == item_target_text) {
            can_paste = true;
            break;
        }
    }
    // Enable/disable paste button appropriately
    if (m_paste_menu_item)
        m_paste_menu_item->set_sensitive(can_paste);

    if (m_paste_tool_button)
        m_paste_tool_button->set_sensitive(can_paste);
}



/**
 * \brief Callback method called when an item has been modified in the
 * item list.
 *
 * Sets the model as dirty and copies the row details to the
 * appropriate Item in the model.  The item's state is also set to
 * dirty.
 */
void KitListGui::on_row_changed(const Gtk::TreeModel::Path path, const Gtk::TreeModel::iterator iter) {
    if (m_ignore_list_events)
        return;
    m_service.set_model_dirty(true);
    Gtk::TreeModel::Row row = *iter;
    int id = row[m_item_cols.m_col_num];
    Glib::ustring us = row[m_item_cols.m_col_text];
    string s(us);
    if (!m_service.update_item(id, s, row[m_item_cols.m_col_checked]))
        g_warning("Unable to find the item in the model: %d", id);
}


/**
 * Called to delete an Item referenced by a row iterator.
 */
void KitListGui::selected_row_callback(const Gtk::TreeModel::iterator& iter) {
    Gtk::TreeModel::Row row = *iter;
    int id = row[m_item_cols.m_col_num];
    if (!m_service.delete_item(id))
        g_warning("Failed to find item to delete in model");
}


// void KitListGui::on_row_deleted(Gtk::TreeModel::Path path) {
//     m_dirty = true;
//     Gtk::TreeModel::iterator iter = m_ref_item_tree_model->get_iter(path);
//     if (iter) {
//         Gtk::TreeModel::Row row = *iter;
//         int id = row[m_item_cols.m_col_num];
//         ModelItem* item = m_model->findItem(id);
//         if (item) {
//             item->setDirty(true);
//             item->setDeleted(true);
//         } else {
//             cerr << _("Couldn't find item with id ") << id << endl;
//         }
//     } else {
//         cerr << _("Couldn't find model row being deleted") << endl;
//     }
// }


/**
 * Refreshes the item list.
 */
void KitListGui::refresh_item_list() {
    m_ignore_list_events = true;
    int cat_id = get_selected_category();
    // Fetch new list of items
    m_ref_item_tree_model->clear();
    ItemContainer *items = m_service.get_items(cat_id);
    if (items)
        sort(items->begin(), items->end(), ItemCompareName());

    // Rebuild the model for the list view
    size_t count = 0;
    if (items && !items->empty()) {
        for (ItemIter i = items->begin(); i != items->end(); ++i) {
            if (m_service.filter((*i)->get_checked())) {
                // Add a row to the model
                count++;
                Gtk::TreeModel::iterator item_iter = m_ref_item_tree_model->append();
                Gtk::TreeModel::Row row = *item_iter;
                row[m_item_cols.m_col_text] = (*i)->get_description();
                row[m_item_cols.m_col_num] = (*i)->get_id();
                row[m_item_cols.m_col_checked] = (*i)->get_checked();
            }
        }
    }
    if (items)
        delete items;
    update_item_count(count);
    m_ignore_list_events = false;
}


/**
 * \brief Refreshes the category combo box list.
 *
 * \param cat_id the id of the category to select in the combo box.
 * If set to -2 the currently selected category is used, otherwise the
 * specified category ID is used.  If the category ID does not exist,
 * or if -1 is specified, then no category is selected.
 */
void KitListGui::refresh_category_list(long cat_id) {
    m_ignore_list_events = true;

    if (cat_id == -2)
        cat_id = get_selected_category();
    CategoryContainer* c = m_service.get_categories();
    sort(c->begin(), c->end(), CategoryCompareName());
    m_ref_category_list_store->clear();
    Gtk::TreeModel::iterator iter = m_ref_category_list_store->append();
    Gtk::TreeModel::iterator active; // the row containing our category id
    Gtk::TreeModel::Row row = *iter;
    ostringstream os;
    // use the user's locale for this stream
    os.imbue(std::locale(""));
#ifdef MAEMO
    os << "[" << _("All") << "]";
#else
    os << "-- " << _("Show all items") << " --";
#endif
#ifdef GLIBMM_EXCEPTIONS_ENABLED
    row[m_category_cols.m_col_text] = Glib::locale_to_utf8(os.str());
#else
    std::auto_ptr<Glib::Error> ex;
    row[m_category_cols.m_col_text] = Glib::locale_to_utf8(os.str(), ex);
    if (ex.get()) {
        g_warning("Failed to add item to category combo model: %s", ex->what().c_str());
    }
#endif
    row[m_category_cols.m_col_num] = -1;
    if (!c->empty()) {
        for (CategoryIter i = c->begin(); i != c->end(); ++i) {
            ModelCategory* category = (ModelCategory*) *i;
            if ( !category->is_deleted() ) {
                // Add a row to the model
                iter = m_ref_category_list_store->append();
                row = *iter;
                row[m_category_cols.m_col_text] = category->get_name();
                row[m_category_cols.m_col_num] = category->get_id();
                if (category->get_id() == cat_id)
                    active = iter;
            }
        }
    }
    delete c;
    if (!active || cat_id == -1)
        m_category_combo->set_active(0);
    else
        m_category_combo->set_active(active);
    m_ignore_list_events = false;
}


/**
 * Called when the user chooses the preferences option from the menu.
 */
void KitListGui::on_menu_preferences() {
#ifdef MAEMO
    Gtk::Dialog::Dialog dialog(_("Preferences"), *m_window, true, true);
    Gtk::VBox* vbox = dialog.get_vbox();

    Glib::RefPtr<Gtk::SizeGroup> group = Gtk::SizeGroup::create(Gtk::SIZE_GROUP_VERTICAL);
    Gtk::Entry page_title;
    page_title.set_text(m_page_title);

    Hildon::Caption* caption = Gtk::manage(new Hildon::Caption(
                                               _("Page Title"),
                                               page_title));
    vbox->pack_start(*caption);

    dialog.set_size_request(500);

    dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
    dialog.add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
    dialog.show_all_children();
    int result = dialog.run();
    dialog.hide();
    switch (result) {
    case (Gtk::RESPONSE_CANCEL) :
        break;
    case (Gtk::RESPONSE_OK) :
        set_page_title(page_title.get_text());
        break;
    default :
        break;
    }
#else
    if (m_window_preferences) {
        if (m_entry_page_title) {
            m_entry_page_title->set_text(m_page_title);
            m_entry_page_title->grab_focus();
            m_window_preferences->show();
        } else {
            g_warning("\"entry_page_title\" resource is missing");
        }
    } else {
        g_warning("\"window_preferences\" resource is missing");
    }
#endif
}


void KitListGui::set_page_title(const Glib::ustring page_title) {
    m_page_title = page_title;
#ifdef GCONF
#ifdef GLIBMM_EXCEPTIONS_ENABLED
    m_ref_gconf_client->set(GCONF_KEY_PAGE_TITLE, m_page_title);
#else
    std::auto_ptr<Glib::Error> error;
    m_ref_gconf_client->set(GCONF_KEY_PAGE_TITLE, m_page_title, error);
    if (error.get())
        g_warning("GConf error setting page title: %s",
                  error->what().c_str());
#endif //GLIBMM_EXCEPTIONS_ENABLED
#endif //GCONF
}

/**
 * Called when the user closes the 'Preferences' dialog using the 'OK'
 * button.
 */
void KitListGui::close_preferences_window() {
    set_page_title(m_entry_page_title->get_text());
    m_window_preferences->hide();
}


void KitListGui::cancel_preferences_window() {
    m_window_preferences->hide();
}


/**
 * Called when the users chooses to create a new Item.
 */
void KitListGui::on_menu_add() {
#ifdef MAEMO
    Gtk::Dialog::Dialog dialog(_("Add Item"), *m_window, true, true);
    // Gtk::Label label(_("Description"), true);
    Gtk::VBox* vbox = dialog.get_vbox();

    Glib::RefPtr<Gtk::SizeGroup> group = Gtk::SizeGroup::create(Gtk::SIZE_GROUP_VERTICAL);
    Gtk::Entry description;

    Hildon::Caption* caption = Gtk::manage(new Hildon::Caption(
                                               _("Description"),
                                               description));
    vbox->pack_start(*caption);
    Gtk::CheckButton checkbutton(_("Chec_ked"), true);
    checkbutton.set_active(false);
    vbox->pack_start(checkbutton);

    dialog.set_size_request(500);

    dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
    dialog.add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
    dialog.show_all_children();
    int result = dialog.run();
    dialog.hide();
    switch (result) {
    case (Gtk::RESPONSE_CANCEL) :
        break;
    case (Gtk::RESPONSE_OK) :
        // Find the current category
        long cat_id = get_selected_category();
        Item* item = m_service.create_item(cat_id);
        item->set_description(description.get_text());
        item->set_checked(checkbutton.get_active());
        refresh_item_list();
        break;
    default :
        break;
    }

#else
    if (m_window_add_item) {
        init_add_item_window();
        m_entry_add_item->grab_focus();
        m_window_add_item->show();
    } else {
        g_warning("\"window_add_item\" resource is missing");
    }
#endif
}


/**
 * Initialises the 'Add Item' dialog prior to it being displayed.
 */
void KitListGui::init_add_item_window() {
    if (m_entry_add_item) {
        m_entry_add_item->set_text("");
    }
}


/**
 * \brief Returns the ID of the currently selected Category.
 *
 * Returns -1 if no Category is currently selected, or the 'all items'
 * option is selected.
 */
long KitListGui::get_selected_category() {
    long retval = -1;
    Gtk::TreeModel::iterator iter = m_category_combo->get_active();
    if (iter) {
        Gtk::TreeModel::Row row = *iter;
        if (row) {
            retval = row[m_category_cols.m_col_num];
        }
    }
    return retval;
}


/**
 * Called when the user closes the 'Add Item' dialog using the 'OK'
 * button.
 */
void KitListGui::close_add_item_window() {
    m_window_add_item->hide();

    // Find the current category
    long cat_id = get_selected_category();

    Item* item = m_service.create_item(cat_id);
    if (m_entry_add_item)
        item->set_description(m_entry_add_item->get_text());
    if (m_checkbutton_add_item)
        item->set_checked(m_checkbutton_add_item->get_active());

    refresh_item_list();
}


/**
 * Called when the user closes the 'Add Item' dialog using the
 * 'Cancel' button.
 */
void KitListGui::cancel_add_item_window() {
    m_window_add_item->hide();
}


/**
 * Called after the user has changed the currently selected Category.
 */
void KitListGui::on_category_change() {
    if (m_ignore_list_events)
        return;
    refresh_item_list();
}


/// Make this application topmost
void KitListGui::raise() {
    m_window->present();
}


/**
 * Initialises all the GUI components prior to the GUI application
 * being run.
 */
void KitListGui::init() {

    // Fetch startup preferences
#ifdef XML_DAO
    Glib::ustring filename;
#endif //XML_DAO

#ifdef KITLIST_DEBUG
    Glib::ustring log_filename;
#endif //KITLIST_DEBUB

#ifdef GCONF
#ifdef GLIBMM_EXCEPTIONS_ENABLED
    try {
#endif //GLIBMM_EXCEPTIONS_ENABLED
        Gnome::Conf::init();
        m_ref_gconf_client = Gnome::Conf::Client::get_default_client();

#ifdef XML_DAO
        Gnome::Conf::Value value = Gnome::Conf::VALUE_INVALID;
#ifdef GLIBMM_EXCEPTIONS_ENABLED
        m_ref_gconf_client->add_dir(GCONF_KEY);
        value = m_ref_gconf_client->get(GCONF_KEY_CURRENT_FILENAME);
#else
        std::auto_ptr<Glib::Error> error;
        m_ref_gconf_client->add_dir(GCONF_KEY,
                                    Gnome::Conf::CLIENT_PRELOAD_NONE,
                                    error);
        if (error.get())
            g_warning("GConf error: %s", error->what().c_str());
        value = m_ref_gconf_client->get(GCONF_KEY_CURRENT_FILENAME, error);
        if (error.get())
            g_warning("GConf error getting current filename: %s",
                      error->what().c_str());
#endif //GLIBMM_EXCEPTIONS_ENABLED
        if (value.get_type() == Gnome::Conf::VALUE_STRING) {
            filename = value.to_string();
        }
#endif //XML_DAO

#ifdef GLIBMM_EXCEPTIONS_ENABLED
        value = m_ref_gconf_client->get(GCONF_KEY_PAGE_TITLE);
#else
        value = m_ref_gconf_client->get(GCONF_KEY_PAGE_TITLE, error);
        if (error.get())
            g_warning("GConf error getting page title: %s",
                      error->what().c_str());
#endif // GLIBMM_EXCEPTIONS_ENABLED
        if (value.get_type() == Gnome::Conf::VALUE_STRING) {
            m_page_title = value.to_string();
        } else if (value.get_type() == Gnome::Conf::VALUE_INVALID) {
            // The default page title for printing
            m_page_title = _("Kitlist");
        }

        // Get default logfile name from GConf
#ifdef KITLIST_DEBUG
#ifdef GLIBMM_EXCEPTIONS_ENABLED
        value = m_ref_gconf_client->get(GCONF_KEY_DEBUG_LOG);
#else
        value = m_ref_gconf_client->get(GCONF_KEY_DEBUG_LOG, error);
        if (error.get())
            g_warning("GConf error getting log filename: %s",
                      error->what().c_str());
#endif // GLIBMM_EXCEPTIONS_ENABLED
        if (value.get_type() == Gnome::Conf::VALUE_STRING) {
            log_filename = value.to_string();
        } else if (value.get_type() == Gnome::Conf::VALUE_INVALID) {
            log_filename = "/tmp/kitlist.log";
#ifdef GLIBMM_EXCEPTIONS_ENABLED
            m_ref_gconf_client->set(GCONF_KEY_DEBUG_LOG, log_filename);
#else
    std::auto_ptr<Glib::Error> error;
    m_ref_gconf_client->set(GCONF_KEY_DEBUG_LOG, log_filename, error);
    if (error.get())
        g_warning("GConf error setting log filename: %s",
                  error->what().c_str());
#endif //GLIBMM_EXCEPTIONS_ENABLED
        }
#endif //KITLIST_DEBUG

#ifdef GLIBMM_EXCEPTIONS_ENABLED
    } catch (const Glib::Error& ex) {
        g_warning("GConf error: %s", ex.what().c_str());
    }
#endif //GLIBMM_EXCEPTIONS_ENABLED
#else // GCONF
    // The default page title for printing
    m_page_title = _("Kitlist");
#endif // GCONF

#ifdef KITLIST_DEBUG
    m_slog = new ofstream(log_filename.c_str(), ios_base::trunc);
    g_log_set_handler(NULL, (GLogLevelFlags) (G_LOG_LEVEL_MASK | 
                      G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION),
                      my_log_handler, m_slog);
#endif //KITLIST_DEBUG

#ifdef MAEMO
    Hildon::init();
    Hildon::fm_init();
    m_osso_context = osso_initialize(KITLIST_SERVICE_NAME, PACKAGE_VERSION, TRUE /* deprecated parameter */, NULL /* Use default Glib main loop context */);
    if(!m_osso_context) {
        g_error("osso_initialize() failed.");
    }
    // Add handler for session bus D-BUS messages
//     g_debug("Adding handler for session bus D-BUS messages");
//     g_debug("Service name     : \"%s\"", KITLIST_SERVICE_NAME);
//     g_debug("Service object   : \"%s\"", KITLIST_SERVICE_OBJECT);
//     g_debug("Service interface: \"%s\"", KITLIST_SERVICE_IFACE);
    osso_return_t result = osso_rpc_set_cb_f(m_osso_context,
                                             KITLIST_SERVICE_NAME,
                                             KITLIST_SERVICE_OBJECT,
                                             KITLIST_SERVICE_IFACE,
                                             dbus_req_handler,
                                             this);
    Glib::set_application_name(_("Kit List"));
    Gtk::Widget* container;
    Glib::RefPtr<Gnome::Glade::Xml> refXml = get_glade_ref_ptr(GLADE_APP_FILE);
    refXml->get_widget("main", container);
    m_window = new Hildon::Window;
    Hildon::Program::get_instance()->add_window(*m_window);
    m_window->add(*container);
    // Handle hardware key events
    m_window->signal_key_press_event().connect( sigc::mem_fun(*this, &KitListGui::on_key_pressed) );
    m_window->signal_window_state_event().connect( sigc::mem_fun(*this, &KitListGui::on_window_state_event) );
    // Create Maemo Menu
    m_refActionGroup = Gtk::ActionGroup::create();

    // File|New sub-menu:
    m_refActionGroup->add(
        Gtk::Action::create("FileNewStandard", Gtk::Stock::NEW, _("New")),
        sigc::mem_fun(*this, &KitListGui::on_menu_file_new)
        );

    // File|Open sub-menu:
    m_refActionGroup->add(
        Gtk::Action::create("FileOpenStandard", Gtk::Stock::OPEN, _("Open")),
        sigc::mem_fun(*this, &KitListGui::on_menu_file_open)
        );
    // File|Save sub-menu:
    m_refActionGroup->add(
        Gtk::Action::create("FileSaveStandard", Gtk::Stock::SAVE, _("Save")),
        sigc::mem_fun(*this, &KitListGui::on_menu_save)
        );
    // File|Save_As sub-menu:
    m_refActionGroup->add(
        Gtk::Action::create("FileSaveAsStandard", Gtk::Stock::SAVE_AS, _("Save As")),
        sigc::mem_fun(*this, &KitListGui::on_menu_save_as)
        );
    // PDF export in MAEMO writes infinitely to file
//     // File|Export to PDF sub-menu:
//     m_refActionGroup->add(
//         Gtk::Action::create("FileExportPDF", Gtk::Stock::PRINT, _("Export to PDF")),
//         sigc::mem_fun(*this, &KitListGui::on_menu_export_to_pdf)
//         );

    // File|Recent sub-menu:
    m_refActionGroup->add(Gtk::Action::create("FileRecent", _("Recent")));

    // Edit|Select All sub-menu
    m_refActionGroup->add(
        Gtk::Action::create("EditSelectAll", Gtk::Stock::JUSTIFY_FILL, _("Select All")),
        sigc::mem_fun(*this, &KitListGui::on_menu_select_all)
        );

    // Edit|Add sub-menu
    m_refActionGroup->add(
        Gtk::Action::create("EditAdd", Gtk::Stock::ADD, _("Add")),
        sigc::mem_fun(*this, &KitListGui::on_menu_add)
        );

    // Edit|Delete sub-menu
    m_refActionGroup->add(
        Gtk::Action::create("EditDelete", Gtk::Stock::DELETE, _("Delete")),
        sigc::mem_fun(*this, &KitListGui::on_menu_delete)
        );

    // Edit|Check Selected sub-menu
    m_refActionGroup->add(
        Gtk::Action::create("EditCheckSelected", Gtk::Stock::APPLY, _("Check Selected")),
        sigc::mem_fun(*this, &KitListGui::on_menu_check_selected)
        );

    // Edit|Uncheck Selected sub-menu
    m_refActionGroup->add(
        Gtk::Action::create("EditUncheckSelected", Gtk::Stock::CLEAR, _("Uncheck Selected")),
        sigc::mem_fun(*this, &KitListGui::on_menu_uncheck_selected)
        );

    // Edit|Toggle Selected sub-menu
    m_refActionGroup->add(
        Gtk::Action::create("EditToggleSelected", Gtk::Stock::CONVERT, _("Toggle Selected")),
        sigc::mem_fun(*this, &KitListGui::toggle_selected)
        );

    // Edit|Cut sub-menu
    m_refActionGroup->add(
        Gtk::Action::create("EditCut", Gtk::Stock::CUT, _("Cut")),
        sigc::mem_fun(*this, &KitListGui::on_menu_cut)
        );

    // Edit|Copy sub-menu
    m_refActionGroup->add(
        Gtk::Action::create("EditCopy", Gtk::Stock::COPY, _("Copy")),
        sigc::mem_fun(*this, &KitListGui::on_menu_copy)
        );

    // Edit|Paste sub-menu
    m_refActionGroup->add(
        Gtk::Action::create("EditPaste", Gtk::Stock::PASTE, _("Paste")),
        sigc::mem_fun(*this, &KitListGui::on_menu_paste)
        );
    // Edit|Preferences sub-menu
    m_refActionGroup->add(
        Gtk::Action::create("EditPreferences", Gtk::Stock::PREFERENCES, _("Preferences")),
        sigc::mem_fun(*this, &KitListGui::on_menu_preferences)
        );

    // View|Show All sub-menu
    m_refActionGroup->add(
        Gtk::Action::create("ViewShowAll", Gtk::Stock::FIND, _("Show All")),
        sigc::mem_fun(*this, &KitListGui::on_menu_show_all)
        );

    // View|Show Checked sub-menu
    m_refActionGroup->add(
        Gtk::Action::create("ViewShowChecked", Gtk::Stock::YES, _("Show Checked")),
        sigc::mem_fun(*this, &KitListGui::on_menu_show_checked)
        );

    // View|Show Unchecked sub-menu
    m_refActionGroup->add(
        Gtk::Action::create("ViewShowUnchecked", Gtk::Stock::NO, _("Show Unchecked")),
        sigc::mem_fun(*this, &KitListGui::on_menu_show_unchecked)
        );

    // View|Refresh sub-menu
    m_refActionGroup->add(
        Gtk::Action::create("ViewRefresh", Gtk::Stock::REFRESH, _("Refresh")),
        sigc::mem_fun(*this, &KitListGui::refresh_item_list)
        );

    // Category|New sub-menu
    m_refActionGroup->add(
        Gtk::Action::create("CategoryAdd", Gtk::Stock::ADD, _("New")),
        sigc::mem_fun(*this, &KitListGui::on_menu_create_category)
        );

    // Category|Rename sub-menu
    m_refActionGroup->add(
        Gtk::Action::create("CategoryRename", Gtk::Stock::EDIT, _("Rename")),
        sigc::mem_fun(*this, &KitListGui::on_menu_rename_category)
        );

    // Category|Delete sub-menu
    m_refActionGroup->add(
        Gtk::Action::create("CategoryDelete", Gtk::Stock::DELETE, _("Delete")),
        sigc::mem_fun(*this, &KitListGui::on_menu_delete_category)
        );

    // Category|Delete sub-menu
    m_refActionGroup->add(
        Gtk::Action::create("HelpContents", Gtk::Stock::HELP, _("Help")),
        sigc::mem_fun(*this, &KitListGui::on_menu_help_contents)
        );

    // Category|Delete sub-menu
    m_refActionGroup->add(
        Gtk::Action::create("HelpAbout", Gtk::Stock::ABOUT, _("About")),
        sigc::mem_fun(*this, &KitListGui::on_menu_help_about)
        );

    // Main menu:
    m_refActionGroup->add( Gtk::Action::create("FileMenu", _("File")) );
    m_refActionGroup->add( Gtk::Action::create("FileNew", _("File")) ); //Sub-menu
    m_refActionGroup->add( Gtk::Action::create("EditMenu", _("Edit")) );
    m_refActionGroup->add( Gtk::Action::create("EditSubMenu", _("Edit")) ); //Sub-menu
    m_refActionGroup->add( Gtk::Action::create("ViewMenu", _("View")) );
    m_refActionGroup->add( Gtk::Action::create("ViewSubMenu", _("View")) ); //Sub-menu
    m_refActionGroup->add( Gtk::Action::create("CategoryMenu", _("Category")) );
    m_refActionGroup->add( Gtk::Action::create("CategorySubMenu", _("Category")) ); //Sub-menu
    m_refActionGroup->add( Gtk::Action::create("HelpMenu", _("Help")) );
    m_refActionGroup->add( Gtk::Action::create("HelpSubMenu", _("Help")) ); //Sub-menu
    m_refActionGroup->add(
        Gtk::Action::create("FileQuit", _("Close")),
        sigc::mem_fun(*this, &KitListGui::on_menu_quit)
        );
    m_refUIManager = Gtk::UIManager::create();
    m_refUIManager->insert_action_group(m_refActionGroup);

    m_window->add_accel_group(m_refUIManager->get_accel_group());

    Glib::ustring ui_info =
        "<ui>"
        "  <popup action='FileMenu'>"
        "    <menu action='FileNew'>"
        "      <menuitem action='FileNewStandard'/>"
        "      <menuitem action='FileOpenStandard'/>"
        "      <menuitem action='FileSaveStandard'/>"
        "      <menuitem action='FileSaveAsStandard'/>"
// PDF export in MAEMO writes infinitely to file
//        "      <menuitem action='FileExportPDF'/>"
        "      <menuitem action='FileRecent'/>"
        "    </menu>"
        "    <menu action='EditSubMenu'>"
        "      <menuitem action='EditSelectAll'/>"
        "      <menuitem action='EditAdd'/>"
        "      <menuitem action='EditDelete'/>"
        "      <separator/>"
        "      <menuitem action='EditCheckSelected'/>"
        "      <menuitem action='EditUncheckSelected'/>"
        "      <menuitem action='EditToggleSelected'/>"
        "      <separator/>"
        "      <menuitem action='EditCut'/>"
        "      <menuitem action='EditCopy'/>"
        "      <menuitem action='EditPaste'/>"
        "      <separator/>"
        "      <menuitem action='EditPreferences'/>"
        "    </menu>"
        "    <menu action='ViewSubMenu'>"
        "      <menuitem action='ViewShowAll'/>"
        "      <menuitem action='ViewShowChecked'/>"
        "      <menuitem action='ViewShowUnchecked'/>"
        "      <separator/>"
        "      <menuitem action='ViewRefresh'/>"
        "    </menu>"
        "    <menu action='CategorySubMenu'>"
        "      <menuitem action='CategoryAdd'/>"
        "      <menuitem action='CategoryRename'/>"
        "      <menuitem action='CategoryDelete'/>"
        "    </menu>"
        "    <menu action='HelpSubMenu'>"
        "      <menuitem action='HelpContents'/>"
        "      <menuitem action='HelpAbout'/>"
        "    </menu>"
        "    <menuitem action='FileQuit'/>"
        "  </popup>"
        "  <toolbar name='ToolBar'>"
//         "    <toolitem action='FileNewStandard'/>"
//         "    <toolitem action='FileOpenStandard'/>"
//         "    <toolitem action='FileSaveStandard'/>"
//         "    <separator/>"
        "    <toolitem action='EditAdd'/>"
//         "    <toolitem action='EditDelete'/>"
//         "    <separator/>"
        "    <toolitem action='EditSelectAll'/>"
        "    <toolitem action='EditCut'/>"
        "    <toolitem action='EditCopy'/>"
        "    <toolitem action='EditPaste'/>"
        "    <toolitem action='EditCheckSelected'/>"
        "    <toolitem action='EditUncheckSelected'/>"
        "    <toolitem action='EditToggleSelected'/>"
        "    <toolitem action='ViewShowAll'/>"
        "    <toolitem action='ViewShowChecked'/>"
        "    <toolitem action='ViewShowUnchecked'/>"
        "    <toolitem action='ViewRefresh'/>"
//         "    <separator/>"
//         "    <toolitem action='FileQuit'/>"
        "  </toolbar>"
        "</ui>";

    std::auto_ptr<Glib::Error> ex;
    m_refUIManager->add_ui_from_string(ui_info, ex);
    if (ex.get()) {
        g_warning("building menus failed: %s", ex->what().c_str());
    }
    Gtk::Menu* pMenu = dynamic_cast<Gtk::Menu*>(m_refUIManager->get_widget("/FileMenu"));
    if (pMenu) {
        m_window->set_main_menu(*pMenu);
    }

    m_recent_files_menu_item = dynamic_cast<Gtk::MenuItem*>(m_refUIManager->get_widget("/FileMenu/FileNew/FileRecent"));

    m_paste_menu_item = dynamic_cast<Gtk::ImageMenuItem*>(m_refUIManager->get_widget("/FileMenu/EditSubMenu/EditPaste"));
    if (m_paste_menu_item)
        m_paste_menu_item->set_sensitive(false);

    m_paste_tool_button = dynamic_cast<Gtk::ToolButton*>(m_refUIManager->get_widget("/ToolBar/EditPaste"));
    if (m_paste_tool_button)
        m_paste_tool_button->set_sensitive(false);

    Gtk::Toolbar* pToolbar = dynamic_cast<Gtk::Toolbar*>(m_refUIManager->get_widget("/ToolBar"));
    if(pToolbar)
        m_window->add_toolbar(*pToolbar);

//     m_window_add_item = new Hildon::Window;
//     Gtk::Widget* item_window_container;
//     refXml->get_widget("window_add_item", item_window_container);
//     m_window_add_item->add(*item_window_container);
    refXml->get_widget("window_add_item", m_window_add_item);
    if (m_window_add_item == NULL) {
        g_warning("Cannot find window_add_item");
    }


#else
    Glib::RefPtr<Gnome::Glade::Xml> refXml = get_glade_ref_ptr(GLADE_APP_FILE);
    refXml->get_widget("main", m_window);
    refXml->get_widget("statusbar", m_status_bar);
#endif // MAEMO
    refXml->get_widget("window_preferences", m_window_preferences);
    refXml->get_widget("entry_page_title", m_entry_page_title);
    refXml->get_widget("window_add_item", m_window_add_item);
    refXml->get_widget("window_add_category", m_window_add_category);
    m_window->signal_delete_event().connect( sigc::mem_fun(*this, &KitListGui::on_delete_event) );

#ifndef MAEMO
    // Handle quit
    Gtk::MenuItem* p_menu_item = 0;
    refXml->get_widget("menu_file_quit", p_menu_item);
    if (p_menu_item)
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_quit) );
    Gtk::ToolButton* p_tool_button = 0;
    refXml->get_widget("tool_button_quit", p_tool_button);
    if (p_tool_button)
        p_tool_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::on_menu_quit) );

    // Handle preferences
    p_menu_item = 0;
    refXml->get_widget("menu_preferences", p_menu_item);
    if (p_menu_item)
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_preferences) );

    // Handle delete item
    p_menu_item = 0;
    refXml->get_widget("menu_delete_item", p_menu_item);
    if (p_menu_item)
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_delete) );
    p_tool_button = 0;
    refXml->get_widget("tool_button_delete", p_tool_button);
    if (p_tool_button)
        p_tool_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::on_menu_delete) );

    // Handle add item
    p_menu_item = 0;
    refXml->get_widget("menu_add_item", p_menu_item);
    if (p_menu_item)
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_add) );
    p_tool_button = 0;
    refXml->get_widget("tool_button_add", p_tool_button);
    if (p_tool_button)
        p_tool_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::on_menu_add) );
#endif
    Gtk::Button* p_button = 0;
    refXml->get_widget("preferences_ok_button", p_button);
    if (p_button)
        p_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::close_preferences_window) );
    refXml->get_widget("preferences_cancel_button", p_button);
    if (p_button)
        p_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::cancel_preferences_window) );
    refXml->get_widget("add_item_ok_button", p_button);
    if (p_button)
        p_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::close_add_item_window) );
    p_button = 0;
    refXml->get_widget("add_item_cancel_button", p_button);
    if (p_button)
        p_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::cancel_add_item_window) );
    refXml->get_widget("entry_item_desc", m_entry_add_item);
    assert(m_entry_add_item);
    refXml->get_widget("item_checkbutton", m_checkbutton_add_item);
    assert(m_checkbutton_add_item);

#ifndef MAEMO
    // Handle file new
    p_menu_item = 0;
    refXml->get_widget("menu_new", p_menu_item);
    if (p_menu_item) {
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_file_new) );
        p_menu_item->set_sensitive(m_service.require_filename());
    }
    p_tool_button = 0;
    refXml->get_widget("tool_button_new", p_tool_button);
    if (p_tool_button) {
        p_tool_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::on_menu_file_new) );
        p_tool_button->set_sensitive(m_service.require_filename());
    }

    // Handle file open
    p_menu_item = 0;
    refXml->get_widget("menu_open", p_menu_item);
    if (p_menu_item) {
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_file_open) );
        p_menu_item->set_sensitive(m_service.require_filename());
    }
    p_tool_button = 0;
    refXml->get_widget("tool_button_open", p_tool_button);
    if (p_tool_button) {
        p_tool_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::on_menu_file_open) );
        p_tool_button->set_sensitive(m_service.require_filename());
    }

    // Handle save
    refXml->get_widget("menu_save", m_file_save_menu_item);
    if (m_file_save_menu_item)
        m_file_save_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_save) );
    refXml->get_widget("tool_button_save", m_file_save_tool_button);
    if (m_file_save_tool_button) {
        m_file_save_tool_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::on_menu_save) );
    }

    // Handle save_as
    p_menu_item = 0;
    refXml->get_widget("menu_save_as", p_menu_item);
    if (p_menu_item) {
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_save_as) );
#ifdef TRANSITION_DAO
        p_menu_item->set_sensitive(true);
#else
        p_menu_item->set_sensitive(m_service.require_filename());
#endif
    }

    // Handle print
    p_menu_item = 0;
    refXml->get_widget("menu_print", p_menu_item);
    if (p_menu_item)
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_print) );
    p_tool_button = 0;
    refXml->get_widget("tool_button_print", p_tool_button);
    if (p_tool_button)
        p_tool_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::on_menu_print) );
    // Export to PDF
    p_menu_item = 0;
    refXml->get_widget("menu_export_pdf", p_menu_item);
    if (p_menu_item)
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_export_to_pdf) );

    // Recent files
    refXml->get_widget("menu_recent", m_recent_files_menu_item);

    // Handle cut
    p_menu_item = 0;
    refXml->get_widget("menu_cut", p_menu_item);
    if (p_menu_item)
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_cut) );
    p_tool_button = 0;
    refXml->get_widget("tool_button_cut", p_tool_button);
    if (p_tool_button)
        p_tool_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::on_menu_cut) );

    // Handle copy
    p_menu_item = 0;
    refXml->get_widget("menu_copy", p_menu_item);
    if (p_menu_item)
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_copy) );
    p_tool_button = 0;
    refXml->get_widget("tool_button_copy", p_tool_button);
    if (p_tool_button)
        p_tool_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::on_menu_copy) );

    // Handle paste
    refXml->get_widget("menu_paste", m_paste_menu_item);
    if (m_paste_menu_item)
        m_paste_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_paste) );
    refXml->get_widget("tool_button_paste", m_paste_tool_button);
    if (m_paste_tool_button)
        m_paste_tool_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::on_menu_paste) );

    // Handle menu show_all
    p_menu_item = 0;
    refXml->get_widget("show_all", p_menu_item);
    if (p_menu_item)
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_show_all) );
    p_tool_button = 0;
    refXml->get_widget("tool_button_show_all", p_tool_button);
    if (p_tool_button)
        p_tool_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::on_menu_show_all) );

    // Handle menu show_checked
    p_menu_item = 0;
    refXml->get_widget("show_checked", p_menu_item);
    if (p_menu_item)
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_show_checked) );
    p_tool_button = 0;
    refXml->get_widget("tool_button_show_checked", p_tool_button);
    if (p_tool_button)
        p_tool_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::on_menu_show_checked) );

    // Handle menu show_unchecked
    p_menu_item = 0;
    refXml->get_widget("show_unchecked", p_menu_item);
    if (p_menu_item)
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_show_unchecked) );
    p_tool_button = 0;
    refXml->get_widget("tool_button_show_unchecked", p_tool_button);
    if (p_tool_button)
        p_tool_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::on_menu_show_unchecked) );

    // Handle menu refresh_view
    p_menu_item = 0;
    refXml->get_widget("refresh_view", p_menu_item);
    if (p_menu_item)
        p_menu_item->signal_activate().connect ( sigc::mem_fun(*this, &KitListGui::refresh_item_list) );
    p_tool_button = 0;
    refXml->get_widget("tool_button_refresh", p_tool_button);
    if (p_tool_button)
        p_tool_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::refresh_item_list) );

    // Handle menu select_all
    p_menu_item = 0;
    refXml->get_widget("select_all", p_menu_item);
    if (p_menu_item)
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_select_all) );
    p_tool_button = 0;
    refXml->get_widget("tool_button_select_all", p_tool_button);
    if (p_tool_button)
        p_tool_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::on_menu_select_all) );

    // Handle menu check_selected
    p_menu_item = 0;
    refXml->get_widget("check_selected", p_menu_item);
    if (p_menu_item)
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_check_selected) );
    p_tool_button = 0;
    refXml->get_widget("tool_button_check_selected", p_tool_button);
    if (p_tool_button)
        p_tool_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::on_menu_check_selected) );

    // Handle menu uncheck_selected
    p_menu_item = 0;
    refXml->get_widget("uncheck_selected", p_menu_item);
    if (p_menu_item)
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_uncheck_selected) );
    p_tool_button = 0;
    refXml->get_widget("tool_button_uncheck_selected", p_tool_button);
    if (p_tool_button)
        p_tool_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::on_menu_uncheck_selected) );

    // Handle menu toggle_selected
    p_menu_item = 0;
    refXml->get_widget("toggle_selected", p_menu_item);
    if (p_menu_item)
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::toggle_selected) );
    p_tool_button = 0;
    refXml->get_widget("tool_button_toggle_selected", p_tool_button);
    if (p_tool_button)
        p_tool_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::toggle_selected) );

    // Handle create_category
    p_menu_item = 0;
    refXml->get_widget("create_category", p_menu_item);
    if (p_menu_item)
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_create_category) );
#endif // ifndef MAEMO

#ifdef XML_DAO
    update_recent_files_menu();
#else
    m_recent_files_menu_item->set_sensitive(false);
#endif

    p_button = 0;
    refXml->get_widget("add_category_ok_button", p_button);
    if (p_button)
        p_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::close_add_category_window) );
    p_button = 0;
    refXml->get_widget("add_category_cancel_button", p_button);
    if (p_button)
        p_button->signal_clicked().connect( sigc::mem_fun(*this, &KitListGui::cancel_add_category_window) );
    refXml->get_widget("entry_category_name", m_entry_add_category);
    assert(m_entry_add_category);

#ifndef MAEMO
    // Handle delete_category
    p_menu_item = 0;
    refXml->get_widget("delete_category", p_menu_item);
    if (p_menu_item)
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_delete_category) );

    // Handle rename_category
    p_menu_item = 0;
    refXml->get_widget("rename_category", p_menu_item);
    if (p_menu_item)
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_rename_category) );

    // Handle menu help_about
    p_menu_item = 0;
    refXml->get_widget("help_about", p_menu_item);
    if (p_menu_item)
        p_menu_item->signal_activate().connect( sigc::mem_fun(*this, &KitListGui::on_menu_help_about) );
#endif
    // Populate categories
    m_ref_category_list_store = Gtk::ListStore::create(m_category_cols);
#ifdef MAEMO
//     Gtk::SeparatorToolItem* sep = new Gtk::SeparatorToolItem;
//     pToolbar->append(*sep);
    Gtk::ToolItem* ti = new Gtk::ToolItem;
    m_category_combo = new Gtk::ComboBox(m_ref_category_list_store);
    m_category_combo->pack_start(m_category_cols.m_col_text);
    m_category_combo->set_column_span_column(1);
    ti->add(*m_category_combo);
    ti->set_expand();
    m_category_combo->set_size_request(40, 10);
    pToolbar->append(*ti);
    pToolbar->show_all();
#else
    // Link the model to the category combo
    refXml->get_widget("categoryCombo", m_category_combo);
    m_category_combo->set_model(m_ref_category_list_store);
#endif
    refresh_category_list();
    m_category_combo->signal_changed().connect( sigc::mem_fun(*this, &KitListGui::on_category_change) );

    // Populate items
    m_ref_item_tree_model = Gtk::ListStore::create(m_item_cols);
    refXml->get_widget("item_treeview", m_item_tree_view);
    // Link the model to the item list
    m_item_tree_view->set_model(m_ref_item_tree_model);
    ItemContainer* items = m_service.get_items();
    if (!items->empty()) {
        sort(items->begin(), items->end(), ItemCompareName());
        for (ItemIter i = items->begin(); i != items->end(); ++i) {
            // Add a row to the model
            Gtk::TreeModel::iterator itemIter = m_ref_item_tree_model->append();
            Gtk::TreeModel::Row row = *itemIter;
            row[m_item_cols.m_col_text] = (*i)->get_description();
            row[m_item_cols.m_col_num] = (*i)->get_id();
            row[m_item_cols.m_col_checked] = (*i)->get_checked();
        }
    }
    m_ref_item_tree_model->signal_row_changed().connect( sigc::mem_fun(*this, &KitListGui::on_row_changed) );
//    m_ref_item_tree_model->signal_row_deleted().connect( sigc::mem_fun(*this, &KitListGui::on_row_deleted) );
    update_item_count(items->size());
    delete items;
    // Column heading
    m_item_tree_view->append_column_editable(_("Checked"), m_item_cols.m_col_checked);
    // Column heading
    m_item_tree_view->append_column_editable(_("Item"), m_item_cols.m_col_text);
#ifdef MAEMO
    m_item_tree_view->set_headers_visible(false);
#endif
//     Gtk::CellRendererToggle* cr = (Gtk::CellRendererToggle*) m_item_tree_view->get_column_cell_renderer(CHECKED_COL_POSITION);
//     assert(cr);
//     if (cr)
//         cr->signal_toggled().connect( sigc::mem_fun(*this, &KitListGui::on_cell_edit) );
//     else
//         cerr << _("Error: Can't find toggle cell renderer for column ")
//              << CHECKED_COL_POSITION << endl;
    Glib::RefPtr<Gtk::TreeSelection> ref_tree_selection =
        m_item_tree_view->get_selection();
    ref_tree_selection->set_mode(Gtk::SELECTION_MULTIPLE);
    m_ignore_list_events = false;
    m_service.set_model_dirty(false);
#ifdef XML_DAO
    if (filename.length() == 0 || !file_exists(filename)) {
        // Use default filename
        filename = Glib::build_filename(
            Glib::get_home_dir(),
            DEFAULT_FILENAME);
        // g_debug("Set filename to: %s", filename.c_str());
        m_filename = filename;
    }
    if (filename.length() > 0 && file_exists(filename)) {
        // g_debug("Loading %s", filename.c_str());
        m_filename = filename;
        m_service.open_as_xml(m_filename);
        refresh_category_list(-1);
        refresh_item_list();
    }
#endif
}
