/*

    This file is part of Kitlist, a program to maintain a simple list
    of items and assign items to one or more categories.

    Copyright (C) 2008,2009 Frank Dean

    Kitlist is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Kitlist is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Kitlist.  If not, see <http://www.gnu.org/licenses/>.

*/

#include <config.h>
#include <locale.h>
#include <iostream>
#include <getopt.h>
#ifdef MAEMO
#include <locale.h>
#endif
#ifndef WIN32
#include <libintl.h>
#endif
#ifdef MAEMO
#include <syslog.h>
#endif
#include "kitlistgui.hpp"
#include "kitlistpgsqldao.hpp"
#include "xmldao.hpp"

/** \mainpage Kitlist Documentation
 *
 * \section intro_sec Introduction
 *
 * The kitlist program has been developed to be run on multiple
 * platforms.  It is known to run on the following platforms:
 *
 *    - Debian 5.0 (Lenny)
 *    - Maemo (Nokia Tablet)
 *    - Windows XP
 *
 * It can be compiled to use either a PostgreSQL database or XML
 * documents as the data store (Debian Linux only).  When compiled to
 * use PostgreSQL, the program can be used from the command line
 * without a GUI.
 *
 * Where the program is executed without any arguments, the GUI is
 * shown.  If there are any arguments, the command line version is
 * executed unless the '-g' option is specified to force running the
 * GUI.
 *
 * \section extra_doc_sec Additional Documentation
 *
 * See the README in the root of the source code for information on
 * building the application.
 *
 */

using namespace std;

/// Flag set by `--verbose'
static int verbose_flag = 0;

/// Flag set by `--html'
static int html_flag = 0;

/**
 * \brief Main application class.
 *
 * Primarily performs application startup and command line parsing.
 * Allows the application to be run either from the command line or as
 * an interactive GUI application.
 */
class KitList {
protected:
    KitListDao* m_dao;  ///< Reference to an implementation DAO class
public :
    KitList(const string dbname, const string user, const string port, bool verbose = false);
    ~KitList();
    KitListDao* get_dao() { return m_dao; }
    void add_item(const string name);
    void add_item(const string name, long cat_id);
    void append_items_to_category(long from_cat_id, long to_cat_id, item_choice choice);
    void associate_item_with_category(long id, long cat_id);
    void list_items_start(bool empty_list);
    void list_item(Item& item);
    void list_items_end(bool empty_list, size_t count);
    void list_items(Category& c);
    bool on_list_item(Item& item);
    void list_items(ItemContainer& items);
    void list_items(long cat_id, item_choice choice);
    void execute(ItemContainer& items, ItemFunctor& functor);
    void tick_items(Category& c);
    void tick_items(ItemContainer& items);
    void tick_items(long cat_id, item_choice choice);
    void list_categories();
    void new_category(const string name);
    void delete_category(long id);
    void delete_item(long id);
    void remove_item_from_category(long id, long cat_id);
    void set_item_flag(long id);
    void unset_item_flag(long id);
    void set_category_flag(long id);
    void unset_category_flag(long id);
    void set_all_flags();
    void unset_all_flags();
};


class TickItem : public ItemFunctor {
    ItemContainer& m_changed;
public:
    TickItem(ItemContainer& changed) : ItemFunctor(), m_changed(changed) {}
    bool operator()(Item& item) {
        string s;
        cout << "Ticking item " << item.get_id() << " " << item.get_description() << endl;
        cout.width(6);
        cout << right << item.get_id()
             << '\t';
        cout.width(40);
        cout << left << item.get_description()
             << '\t' << " (" << (item.get_checked() ? "Yn" : "yN") << "q): ";
        getline(cin, s);
        bool checked = false;
        if (!s.empty()) {
            if (s == "y" || s == "Y") {
                checked = true;
            } else if (s == "n" || s == "N") {
                checked = false;
            } else if (s == "q" || s == "Q") {
                return true;
            }
            if (checked != item.get_checked()) {
                item.set_checked(checked);
                m_changed.push_back(&item);
            }
        }
        return false;
    }

};


/**
 * \brief Constructor specifying Postgresql database connection parameters.
 *
 * Note that the PostgreSQL libraries will use default values for
 * these parameters if they are not specified, including examining
 * environment variables.  See the PostgreSQL documentation for full
 * details.
 *
 * \param dbname The name of the database.
 * \param user The database user name to connect with.
 * \param port The database port to connect to.
 * \param verbose Optional parameter indicating whether to include
 *        verbose output to STDOUT.  Defaults to false.
 */
KitList::KitList(const string dbname, const string user, string port, bool verbose) {
#ifdef XML_DAO
    m_dao = new XmlDao(verbose);
#else
    m_dao = new KitListPgsqlDao(dbname, user, port, verbose);
#endif
}


KitList::~KitList() {
    delete m_dao;
}


/**
 * Creates a new item with the specified name and optionally
 * associates it with a category.
 * \param name The name of the item to create.
 * \param cat_id The ID of the existing category to associate the item
 * with.  If the ID is less than zero, the item will not be associated
 * with a category.  Otherwise, the category must already exist.
 */
void KitList::add_item(const string name, long cat_id) {
    if (cat_id >= 0) {
        long id = m_dao->add_item(name, cat_id);
        cout << "Added item with id " << id << " to category " << cat_id << endl << endl;
    } else {
        long id = m_dao->add_item(name);
        cout << "Added item with id " << id << endl << endl;
    }
}


/**
 * \brief Copies items from one category to another.
 *
 * Optionally, only checked or unchecked items are copied.
 * \param from_cat_id The ID of the source category.
 * \param to_cat_id The ID of the target category.
 * \param choice One of ALL_ITEMS, CHECKED_ITEMS or UNCHECKED_ITEMS.
 */
void KitList::append_items_to_category(long from_cat_id, long to_cat_id, item_choice choice) {
    m_dao->append_items_to_category(to_cat_id, from_cat_id, choice);
}


/**
 * \brief Associates an existing item with an existing category.
 *
 * \param id The Item ID
 * \param cat_id The Category ID
 */
void KitList::associate_item_with_category(long id, long cat_id) {
    m_dao->associate_item_with_category(id, cat_id);
    cout << "Associated item " << id << " with category " << cat_id << endl << endl;
}


/**
 * A callback function to output details of the passed Item to STDOUT.
 */
bool KitList::on_list_item(Item& item) {
    list_item(item);
    return false;
}


/**
 * \brief Called before writing a list of items to STDOUT.
 *
 * Fundamentally outputs headings, in either HTML or text format,
 * depending on the value of html_flag.  The html_flag is set by the
 * command line option, '--html'.
 * \param empty_list Set to true if the list is empty.  If so, table
 * headings will not be output.
 */
void KitList::list_items_start(bool empty_list) {
    if (html_flag)
        cout << "<html>" << endl
             << "<head>" << endl
             << "<title>Items</title>" << endl
             << "</head>" << endl
             << "<body>" << endl;

    if (!empty_list) {
        if (html_flag) {
            cout << "<table cellpadding=\"3\" cellspacing=\"0\" border=\"1\">" << endl
                 << "<tr><th>ID</th><th>Description</th></tr>" << endl;
        } else {
            cout.width(6);
            cout << right << "ID" << "\tDescription" << endl;
            cout << right << "======" << "\t===========" << endl;
        }
    }
}


/**
 * \brief Called after writing a list of items to STDOUT.
 *
 * Fundamentally outputs a footer, in either HTML or text format,
 * depending on the value of html_flag.  The html_flag is set by the
 * command line option, '--html'.
 *
 * \param empty_list Set to true if the list is empty.  If so, a table
 * footer will not be output.
 */
void KitList::list_items_end(bool empty_list, size_t count) {
    if (!empty_list) {
        if (html_flag)
            cout << "</table>" << endl;
    }

    if (html_flag) {
        cout << "<p>" << count << " items</p>" << endl
             << "</body>" << endl << "</html>" << endl;
    } else {
        cout << endl;
        cout.width(6);
        cout << count << " items" << endl << endl;
    }
}


/**
 * \brief Outputs details of the passed item to STDOUT.
 *
 * Details are wrapped in HTML format if the html_flag has been set.
 * The html_flag is set by the command line option, '--html'.
 */
void KitList::list_item(Item& item) {
    if (html_flag) {
        cout << "<tr><td>" << item.get_id() << "</td>"
             << "<td>" << item.get_description() << "</td></tr>" << endl;
    } else {
        cout.width(6);
        cout << item.get_id();
        cout << '\t' << item.get_description()
             << endl;
    }
}


/**
 * Lists details of all items in the passed Category to STDOUT.
 */
void KitList::list_items(Category& c) {
    list_items_start(!c.has_items());
    c.foreach_item( sigc::mem_fun(*this, &KitList::on_list_item) );
    list_items_end(!c.has_items(), c.item_count());
}


/**
 * Lists details of all items in the passed ItemContainer to STDOUT.
 */
void KitList::list_items(ItemContainer& items) {
    list_items_start(items.empty());
    for (ItemIter i = items.begin(); i != items.end(); ++i ) {
        list_item(**i);
    }
    list_items_end(items.empty(), items.size());
}


/**
 * Lists details of items to STDOUT.
 *
 * \param cat_id The ID Of the Category to list.  If the value is less
 * than zero, all items are listed.
 *
 * \param choice Optional.  One of ALL_ITEMS (default), CHECKED_ITEMS
 * or UNCHECKED_ITEMS.
 */
void KitList::list_items(long cat_id, item_choice choice = ALL_ITEMS) {
    if (cat_id >=0 ) {
        Category* cat = m_dao->get_category(cat_id, choice);
        list_items(*cat);
        delete cat;
    } else {
        ItemContainer* items = m_dao->get_all_items(choice);
        list_items(*items);
        for (ItemIter i = items->begin(); i != items->end(); ++i) {
            delete (*i);
        }
        delete items;
    }
}


/**
 * Executes the passed ItemFunctor for each item in the ItemContainer.
 * \see ItemFunctor.
 */
void KitList::execute(ItemContainer& items, ItemFunctor& functor) {
    for (ItemIter i = items.begin(); i != items.end(); ++i) {
        if (functor(**i))
            break;
    }
}


/**
 * Checks/ticks all items in the passed ItemContainer.
 */
void KitList::tick_items(ItemContainer& items) {
    ItemContainer changed;
    TickItem t(changed);
    execute(items, t);
    m_dao->update_item_checked_state(changed);
}


/**
 * Checks/ticks all items in the passed Category.
 */
void KitList::tick_items(Category& c) {
    ItemContainer changed;
    TickItem t(changed);
    c.execute(t);
    m_dao->update_item_checked_state(changed);
}


/**
 * \brief Checks/ticks all items belonging to a category.
 *
 * Acts on all items if the cat_id is less than zero.  Optionally
 * operates on checked or unchecked items, depending on the value of choice.
 *
 * \param choice One of ALL_ITEMS, CHECKED_ITEMS or UNCHECKED_ITEMS.
 */
void KitList::tick_items(long cat_id, item_choice choice = ALL_ITEMS) {
    if (cat_id >=0 ) {
        Category* cat = m_dao->get_category(cat_id, choice);
        tick_items(*cat);
        delete cat;
    } else {
        ItemContainer* items = m_dao->get_all_items(choice);
        tick_items(*items);
        for (ItemIter i = items->begin(); i != items->end(); ++i) {
            delete (*i);
        }
        delete items;
    }
}


/**
 * Deletes the item with the passed id.
 */
void KitList::delete_item(long id) {
    m_dao->delete_item(id);
    cout << "Deleted item with id " << id << endl << endl;
}


/**
 * Un-associates the specified item from the specified category.
 *
 * \param id The Item id.
 * \param cat_id The Category id.
 */
void KitList::remove_item_from_category(long id, long cat_id) {
    m_dao->remove_item_from_category(id, cat_id);
    cout << "Removed item " << id << " from category " << cat_id << endl << endl;
}


/**
 * Deletes the Category with the specified id.
 * \param id The Category id.
 */
void KitList::delete_category(long id) {
    m_dao->delete_category(id);
    cout << "Deleted category with id " << id << endl << endl;
}


/**
 * \brief Lists details of all categories to STDOUT.
 *
 * Renders with HTML if the html_flag has been set using the '--html'
 * command line option.
 */
void KitList::list_categories() {
    CategoryContainer c = m_dao->get_categories();

    if (html_flag)
        cout << "<html>" << endl
             << "<head>" << endl
             << "<title>Categories</title>" << endl
             << "</head>" << endl
             << "<body>" << endl;

    if (!c.empty()) {
        if (html_flag) {
            cout << "<table cellpadding=\"3\" cellspacing=\"0\" border=\"1\">" << endl
                 << "<tr><th>ID</th><th>Description</th></tr>" << endl;
        } else {
            cout.width(6);
            cout << right << "ID" << "\tDescription" << endl;
            cout << right << "======" << "\t===========" << endl;
        }
        for (CategoryIter i = c.begin(); i != c.end(); ++i) {
            if (html_flag) {
                cout << "<tr><td>" << (*i)->get_id() << "</td>"
                     << "<td>" << (*i)->get_name() << "</td></tr>" << endl;
            } else {
                cout.width(6);
                cout << (*i)->get_id()
                     << "\t" << (*i)->get_name() << endl;
            }
            delete (*i);
        }

        if (html_flag)
            cout << "</table>" << endl;

    }
    if (html_flag) {
        cout << "<p>" << c.size() << " categories</p>" << endl
             << "</body>" << endl << "</html>" << endl;
    } else {
        cout << endl;
        cout.width(6);
        cout << c.size() << " categories" << endl << endl;
    }
}


/**
 * Creates a new category with the passed name.
 */
void KitList::new_category(const string name) {
    long id = m_dao->new_category(name);
    cout << "Added category id " << id << " named \"" << name << '"' << endl << endl;
}


/**
 * Checks/ticks the specified item.
 */
void KitList::set_item_flag(long id) {
    m_dao->set_item_flag(id);
    cout << "Checked item with id " << id << endl << endl;
}


/**
 * Unchecks/unticks the specified item.
 */
void KitList::unset_item_flag(long id) {
    m_dao->unset_item_flag(id);
    cout << "Unchecked items with id " << id << endl << endl;
}


/**
 * Checks/ticks all items in the specified Category.
 */
void KitList::set_category_flag(long id) {
    m_dao->set_category_flag(id);
    cout << "Checked items with category id " << id << endl << endl;
}


/**
 * Unchecks/unticks all items in the specified Category.
 */
void KitList::unset_category_flag(long id) {
    m_dao->unset_category_flag(id);
    cout << "Unchecked items with category id " << id << endl << endl;
}


/**
 * Checks/ticks all items.
 */
void KitList::set_all_flags() {
    m_dao->set_all_flags();
    cout << "Checked all items" << endl << endl;
}


/**
 * Unchecks/unticks all items.
 */
void KitList::unset_all_flags() {
    m_dao->unset_all_flags();
    cout << "Unchecked all items" << endl << endl;
}


/**
 * Parses the command line and executes the main application.
 */
int main(int argc, char **argv) {
    setlocale(LC_CTYPE, "");

#ifndef WIN32
    // i18n initialisation
    bindtextdomain(GETTEXT_PACKAGE, PROGRAMNAME_LOCALEDIR);
    bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
    textdomain(GETTEXT_PACKAGE);
#endif //WIN32
    int c;
    string port;
    string item_desc;
    string cat_desc;
    int option_index = 0;
    int invalid_opt = false;
    long cat_n = -1;
    long dest_cat_n = -1;
    long item_n = -1;;
    int force_flag = 0;
    item_choice checked_items_flag = ALL_ITEMS;
    enum actions {NOP,
                  SHOW_GUI,
                  ADD_ITEM,
                  COPY_ITEMS,
                  ASSOCIATE_ITEM,
                  LIST,
                  LIST_CHECKED,
                  LIST_UNCHECKED,
                  TICK_CHECKED,
                  TICK_UNCHECKED,
                  NEW_CATEGORY,
                  LIST_CATEGORIES,
                  DELETE,
                  SET,
                  CLEAR,
                  SET_ALL,
                  CLEAR_ALL};
    enum actions action = NOP;
    
    while (1) {
        static struct option long_options[] = {
#ifndef XML_DAO
            {"port", required_argument, 0, 'p'},
            {"html", no_argument, &html_flag, 1},
            {"no-html", no_argument, &html_flag, 0},
            {"force", no_argument, &force_flag, 1},
            {"add", required_argument, 0, 'a'},
            {"add-to-category", no_argument, 0, 'A'},
            {"category", required_argument, 0, 'c'},
            {"copy-to-category", required_argument, 0, 'C'},
            {"only-unchecked", no_argument, 0, 'k'},
            {"only-checked", no_argument, 0, 'K'},
            {"item", required_argument, 0, 'i'},
            {"list", no_argument, 0, 'l'},
            {"list-checked", no_argument, 0, 'X'},
            {"list-unchecked", no_argument, 0, 'U'},
            {"list-categories", no_argument, 0, 'q'},
            {"tick-unchecked", no_argument, 0, 't'},
            {"tick-checked", no_argument, 0, 'T'},
            {"new-category", required_argument, 0, 'n'},
            {"delete", no_argument, 0, 'd'},
            {"set", no_argument, 0, 'x'},
            {"clear", no_argument, 0, 'u'},
            {"set-all", no_argument, 0, 's'},
            {"clear-all", no_argument, 0, 'z'},
            {"interactive", no_argument, 0, 'g'},
#endif
            {"verbose", no_argument, &verbose_flag, 1},
            {"brief", no_argument, &verbose_flag, 0},
            {"help", no_argument, 0, 'h'},
            {"version", no_argument, 0, 'v'},
            {0, 0, 0, 0}
        };
#ifndef XML_DAO
        c = getopt_long(argc, argv, "hvVHp:fa:Ac:C:i:lXUqtTn:dxuszkKg",
                        long_options, &option_index);
#else
        c = getopt_long(argc, argv, "hvV",
                        long_options, &option_index);
#endif
        if (c == -1)
            break;
           
        switch (c) {
        case 0:
            break;

        case 'p':
            port = optarg;
            break;

        case 'h':
            cout << "Usage:" << endl
#ifndef XML_DAO
                 << "  " << argv[0] << " [OPTIONS] [DBNAME [USERNAME]]" << endl << endl
#else
                 << "  " << argv[0] << " [FILENAME]" << endl << endl
#endif
                 << "Options:" << endl
                 << "  -h, --help\t\t\t\tshow this help, then exit" << endl
                 << "  -v, --version\t\t\t\tshow version information, then exit" << endl
                 << "  -V, --verbose\t\t\t\tverbose output" << endl
#ifndef XML_DAO
                 << "  -g, --interactive\t\t\tDisplay interactive GUI" << endl
                 << "  -p, --port=PORT\t\tspecify the port to connect to, default 5432" << endl
                 << "  -a, --add=NAME\t\t\tcreate item named NAME," << endl
                 << "                \t\t\tif -c specified, associates item with that category" << endl
                 << "  -A, --add-to-category\t\tadd existing item to category" << endl
                 << "        \t\t\t\t\tboth -i and -c must be specified" << endl
                 << "  -l, --list\t\t\t\tlist all items, limit to category, if -c specified" << endl
                 << "  -X, --list-checked\t\tsame as -l, except only checked items" << endl
                 << "  -U, --list-unchecked\t\tsame as -l, except only unchecked items" << endl
                 << "  -T, --tick-checked\t\tinteratively choose to tick/untick checked items" << endl
                 << "  -t, --tick-unchecked\t\tinteratively choose to tick/untick unchecked items" << endl
                 << "  -q, --list-categories\t\tlist categories" << endl
                 << "  -n, --new-category=CATEGORY\tcreate category named CATEGORY" << endl
                 << "  -c, --category=NUMBER\t\tspecify category number for action" << endl
                 << "  -C, --copy-to-category=NUMBER\t\tcopy all or specific category to" << endl
                 << "        \t\t\t\t\tcategory NUMBER.  Specify source category with -c" << endl
                 << "        \t\t\t\t\tor omit to copy all items" << endl
                 << "  -k, --only-unchecked\t\tWhen using -C, operate on unchecked items only" << endl
                 << "  -K, --only-checked\t\tWhen using -C, operate on checked items only" << endl
                 << "  -i, --item=NUMBER\t\t\tspecify item number for action" << endl
                 << "  -d, --delete\t\t\t\tdelete category or item specified by -c or -i, if" << endl
                 << "        \t\t\t\t\tboth -i and -c are specifed, simply removes the " << endl
                 << "        \t\t\t\t\tassociation between them" << endl
                 << "  -x, --set\t\t\t\t\tset flag for item specified by -i" << endl
                 << "  -u, --clear\t\t\t\tclear flag for item specified by -i" << endl
                 << "  -s, --set-all\t\t\t\tsets all flags" << endl
                 << "  -z, --clear-all\t\t\tclears all flags" << endl
                 << "  -H, --html\t\t\t\thtml formatted output" << endl
#endif
                ;
            return 0;

        case 'v':
            cout << "kitlist " << VERSION << endl;
            return 0;

        case 'V':
            verbose_flag = 1;
            break;

        case 'H':
            html_flag = 1;
            break;

        case '?':
            invalid_opt = true;
            break;

        case 'i':
            item_n = strtol(optarg, NULL, 0);
            break;
        case 'c':
            cat_n = strtol(optarg, NULL, 0);
            break;
        case 'C':
            action = COPY_ITEMS;
            dest_cat_n = strtol(optarg, NULL, 0);
            break;
        case 'a':
            action = ADD_ITEM;
            item_desc = optarg;
            break;
        case 'A':
            action = ASSOCIATE_ITEM;
            break;
        case 'l':
            action = LIST;
            break;
        case 'X':
            action = LIST_CHECKED;
            break;
        case 'U':
            action = LIST_UNCHECKED;
            break;
        case 'T':
            action = TICK_CHECKED;
            break;
        case 't':
            action = TICK_UNCHECKED;
            break;
        case 'q':
            action = LIST_CATEGORIES;
            break;

        case 'n':
            action = NEW_CATEGORY;
            cat_desc = optarg;
            break;

        case 'd':
            action = DELETE;
            break;

        case 'x':
            action = SET;
            break;

        case 'u':
            action = CLEAR;
            break;

        case 's':
            action = SET_ALL;
            break;

        case 'z':
            action = CLEAR_ALL;
            break;

        case 'f':
            force_flag = 1;
            break;

        case 'k':
            checked_items_flag = UNCHECKED_ITEMS;
            break;

        case 'K':
            checked_items_flag = CHECKED_ITEMS;
            break;

        case 'g':
            action = SHOW_GUI;
            break;

        default:
            abort();
        }
    }
    if (invalid_opt)
        return 1;
    int index;
    index = optind;
    string dbname = (index < argc) ? argv[index++] : "";
    string user = (index < argc) ? argv[index++] : "";

    KitList kit(dbname, user, port, verbose_flag);

#ifndef XML_DAO
    // No options, run GUI
    action = (argc == 1) ? SHOW_GUI : action;
#else
    // Always show the GUI
    action = SHOW_GUI;
#endif

    switch (action) {
    case ADD_ITEM:
        kit.add_item(item_desc, cat_n);
        break;
    case COPY_ITEMS:
        if (cat_n >= 0 || force_flag) {
            kit.append_items_to_category(cat_n, dest_cat_n, checked_items_flag);
        } else {
            cerr << "Specify -f (--force) to copy all items to a category" << endl << endl;
        }
        break;
    case ASSOCIATE_ITEM:
        if (item_n >= 0 && cat_n >=0) {
            kit.associate_item_with_category(item_n, cat_n);
        } else {
            cerr << "You must specify the item and category using -i and -c" << endl << endl;
        }
        break;
    case LIST:
        kit.list_items(cat_n);
        break;
    case LIST_CHECKED:
        kit.list_items(cat_n, CHECKED_ITEMS);
        break;
    case LIST_UNCHECKED:
        kit.list_items(cat_n, UNCHECKED_ITEMS);
        break;
    case TICK_CHECKED:
        kit.tick_items(cat_n, CHECKED_ITEMS);
        break;
    case TICK_UNCHECKED:
        kit.tick_items(cat_n, UNCHECKED_ITEMS);
        break;
    case LIST_CATEGORIES:
        kit.list_categories();
        break;
    case NEW_CATEGORY:
        kit.new_category(cat_desc);
        break;
    case DELETE:
        if (cat_n >= 0 && item_n >=0) {
            kit.remove_item_from_category(item_n, cat_n);
        } else if (item_n >=0) {
            kit.delete_item(item_n);
        } else {
            if (force_flag) {
                kit.delete_category(cat_n);
            } else {
                cerr << "Specify -f (--force) to delete a category" << endl << endl;
                return 1;
            }
        }
        break;

    case SET:
        if (item_n >=0) {
            kit.set_item_flag(item_n);
        } else {
            cerr << "Specify an item set using -i or --item" << endl << endl;
            return 1;
        }
        break;
    case CLEAR:
        if (item_n >=0) {
            kit.unset_item_flag(item_n);
        } else {
            cerr << "Specify an item to clear using -i or --item" << endl << endl;
            return 1;
        }
        break;
    case SET_ALL:
        if (cat_n >=1) {
            kit.set_category_flag(cat_n);
        } else {
            if (force_flag) {
                kit.set_all_flags();
            } else {
                cerr << "Specify -f (--force) to set all flags" << endl << endl;
            }
        }
        break;
    case CLEAR_ALL:
        if (cat_n >=1) {
            kit.unset_category_flag(cat_n);
        } else {
            if (force_flag) {
                kit.unset_all_flags();
            } else {
                cerr << "Specify -f (--force) to clear all flags" << endl << endl;
            }
        }
        break;
    case SHOW_GUI:
        Service service(*kit.get_dao());
        KitListGui gui(argc, argv, service);
#ifdef XML_DAO
        // When we're not using the database dbname, actually
        // contains the first non-option argument, which in this
        // case is the filename
        if (dbname.length() > 0)
            gui.open_file(dbname);
#endif
        gui.run();
        break;
    } // case

}
