/*

    This file is part of Kitlist, a program to maintain a simple list
    of items and assign items to one or more categories.

    Copyright (C) 2008,2009 Frank Dean

    Kitlist is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Kitlist is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Kitlist.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "kitmodel.hpp"
#include <algorithm>
#include <cassert>
#include <glib.h>
#include <iostream>
#include <iterator>


/**
 * \brief resets the state of each flag to it's default.
 *
 * The dirty, deleted and new flags are set to false.
 */
void GuiState::reset() {
    m_dirty = false;
    m_deleted = false;
    m_new = false;
}


ModelCategory::ModelCategory() : Category(), GuiState() {
    m_removed_children = new ItemMap;
    m_added_children = new ItemMap;
}


ModelCategory::~ModelCategory() {
    delete m_removed_children;
    delete m_added_children;
}


/**
 * \brief Resets the Category state.
 *
 * Removes any items flagged as deleted.  Sets the GuiState to it's
 * defaults and clears the lists of added and removed items.
 */
void ModelCategory::reset() {
    GuiState::reset();
    m_removed_children->clear();
    m_added_children->clear();
}


void ModelCategory::purge() {
    ItemIter i = m_items.begin();
    while (i != m_items.end()) {
        Item* item = *i;
        if (((ModelItem*) item)->is_deleted()) {
            ItemIter iterTemp = i;
            m_items.erase(i);
            i = iterTemp++;
        } else {
            i++;
        }
    }    
}


/**
 * \brief Adds the passed item to this ModelCategory.
 *
 * Also updates the lists of removed and added items.
 */
void ModelCategory::add_item(Item* item) {
    Category::add_item(item);
    // If the item was previously removed, just remove it from the removed list, so
    // it just won't be deleted.  Don't add it to the inserted list, as it must have
    // previously belonged.
    ItemMap::size_type n = m_removed_children->erase(item->get_id());
    if (n == 0) {
        /* std::pair<ItemMapIter, bool> r =*/
        m_added_children->insert(std::make_pair(item->get_id(), (ModelItem*) item));
    }
}


/**
 * \brief Removes the passed item from this ModelCategory.
 *
 * Also updates the lists of removed and added items.
 */
void ModelCategory::remove_item(Item* item) {
    Category::remove_item(item);
    // Much like adding an item, if the item has previously been added,
    // just remove that fact.
    ItemMap::size_type n = m_added_children->erase(item->get_id());
    if (n == 0) {
        m_removed_children->insert(std::make_pair(item->get_id(), (ModelItem*) item));
    }
}


/**
 * \brief Removes all the passed items from this ModelCategory.
 *
 * Also updates the lists of removed and added items.
 */
void ModelCategory::remove_items(ModelItemContainer* items) {
    for (ModelItemIter i = items->begin(); i != items->end(); ++i) {
        remove_item(*i);
    }
}


/**
 * Returns the list of items belonging to the Category.
 *
 * Items flagged as deleted are excluded.
 */
ModelItemContainer* ModelCategory::get_model_items() {
    ModelItemContainer* retval = new ModelItemContainer;
    for (ItemIter i = m_items.begin(); i != m_items.end(); ++i) {
        ModelItem* mi = (ModelItem*)*i;
        if (!mi->is_deleted())
            retval->push_back(mi);
    }
    return retval;
}


/**
 * \brief Returns the list of items belonging to the Category.
 *
 * Items flagged as deleted are excluded.
 */

ItemContainer* ModelCategory::get_items() {
    ItemContainer* retval = new ItemContainer;
    for (ItemIter i = m_items.begin(); i != m_items.end(); ++i) {
        ModelItem* mi = (ModelItem*)*i;
        if (!mi->is_deleted()) {
            retval->push_back(mi);
        }
    }
    return retval;
}


/**
 * \brief Returns the list of items belonging to the Category,
 * filtered by the passed functor.
 *
 * Items flagged as deleted are excluded.
 * \param functor if the operator() method returns true the item
 * is included in the returned list.
 */

ItemContainer* ModelCategory::get_items(ItemFunctor& functor) {
    ItemContainer* retval = new ItemContainer;
    for (ItemIter i = m_items.begin(); i != m_items.end(); ++i) {
        ModelItem* mi = (ModelItem*)*i;
        if (!mi->is_deleted() && functor(**i)) {
            retval->push_back(mi);
        }
    }
    return retval;
}


/**
 * \brief Creates an empty data model.
 */
KitModel::KitModel() : m_dirty(false), m_item_filter(ALL) {
    m_item_map = new ItemMap;
    m_category_map = new CategoryMap;
}


/**
 * \brief Destructor
 *
 * Deletes all items and categories belonging to the model.
 */
KitModel::~KitModel() {
    for (ItemMapIter i = m_item_map->begin(); i != m_item_map->end(); ++i) {
        delete (*i).second;
        (*i).second = 0;
    }
    m_item_map->clear();
    for (CategoryMapIter i = m_category_map->begin(); i != m_category_map->end(); ++i) {
        delete (*i).second;
        (*i).second = 0;
    }
    m_category_map->clear();
    delete m_item_map;
    delete m_category_map;
}


/**
 * \brief Calls the provided slot for each item in the map.
 *
 */
void KitModel::foreach_item_iter(const SlotForeachModelItemIter& slot) {
    for (ItemMapIter i = m_item_map->begin(); i != m_item_map->end(); ++i) {
        if (slot(i))
            break;
    }    
}


/**
 * \brief Calls the provided slot for each item in the map.
 *
 */
void KitModel::foreach_item(const SlotForeachModelItem& slot) {
    for (ItemMapIter i = m_item_map->begin(); i != m_item_map->end(); ++i) {
        if (slot(*(*i).second))
            break;
    }    
}


/**
 * \brief Calls the provided slot for each category in the map.
 *
 */
void KitModel::foreach_category_iter(const SlotForeachCategoryIter& slot) {
    for (CategoryMapIter i = m_category_map->begin(); i != m_category_map->end(); ++i) {
        if (slot(i))
            break;
    }    
}


/**
 * \brief Calls the provided slot for each category in the map.
 *
 */
void KitModel::foreach_category(const SlotForeachCategory& slot) {
    for (CategoryMapIter i = m_category_map->begin(); i != m_category_map->end(); ++i) {
        if (slot(*(*i).second))
            break;
    }    
}


/**
 * \brief Finds a Category by it's unique ID.
 */
ModelCategory* KitModel::find_category(long id) {
    ModelCategory* retval = NULL;
    CategoryMapIter i = m_category_map->find(id);
    if (i != m_category_map->end()) {
        retval = (*i).second;
    }
    return retval;
}


/**
 * \brief Finds an item by it's unique ID.
 */
ModelItem* KitModel::find_item(long id) {
    ModelItem* retval = NULL;
    ItemMapIter i = m_item_map->find(id);
    if (i != m_item_map->end()) {
        retval = (*i).second;
    }
    return retval;
}


// ModelCategoryContainer* KitModel::getCategories() {
//     ModelCategoryContainer* retval = new ModelCategoryContainer;
//     for (CategoryMapIter i = m_category_map->begin(); i != m_category_map->end(); ++i) {
//         retval->push_back((*i).second);
//     }
//     return retval;
// }


/**
 * \brief Returns a list of all categories.
 */
CategoryContainer* KitModel::get_categories() {
    CategoryContainer* retval = new CategoryContainer;
    for (CategoryMapIter i = m_category_map->begin(); i != m_category_map->end(); ++i) {
        ModelCategory* cat = (*i).second;
        if (!cat->is_deleted()) {
            retval->push_back(cat);
        }
    }
    return retval;
}


// ModelItemContainer* KitModel::getAllItems() {
//     ModelItemContainer* retval = new ModelItemContainer;
//     for (ItemMapIter i = m_item_map->begin(); i != m_item_map->end(); ++i) {
//         retval->push_back((*i).second);
//     }
//     return retval;
// }


/**
 * \brief Returns a list of all items.
 *
 * Excluded items are excluded from the list.  The caller is
 * responsible for deleting the returned item list.
 */
ItemContainer* KitModel::get_all_items() {
    ItemContainer* retval = new ItemContainer;
    for (ItemMapIter i = m_item_map->begin(); i != m_item_map->end(); ++i) {
        if (!((*i).second)->is_deleted()) {
            retval->push_back((*i).second);
        }
    }
    return retval;
}


/**
 * \brief Returns a list of all items, filtered by the passed
 * functor.
 *
 * Excluded items are excluded from the list.  The caller is
 * responsible for deleting the returned item list.
 */
ItemContainer* KitModel::get_all_items(ItemFunctor& functor) {
    ItemContainer* retval = new ItemContainer;
    for (ItemMapIter i = m_item_map->begin(); i != m_item_map->end(); ++i) {
        if (!((*i).second)->is_deleted() && functor(*(*i).second)) {
            retval->push_back((*i).second);
        }
    }
    return retval;
}


/**
 * \brief Add a category to the model.
 *
 * The category is included in the model's map.
 */
void KitModel::add_category(ModelCategory* category) {
    m_category_map->insert(std::make_pair(category->get_id(), category));
}


/**
 * \brief Adds an item to the model.
 *
 * The item is included in the model's map.
 */
void KitModel::add_item(ModelItem* item) {
    m_item_map->insert(std::make_pair(item->get_id(), item));
}


/**
 * \brief Adds an item to the model and associates it with the
 * specified category.
 *
 * The category must have already been added to the model's map.
 */
void KitModel::add_item(ModelItem* item, long cat_id) {
    std::pair<long, ModelItem*> pair = std::make_pair(item->get_id(), item);
    m_item_map->insert(pair);
    ModelCategory* c = find_category(cat_id);
    if (c) {
        c->add_item(item);
    } else
        g_warning("category id %d was not found - unable to associate item", cat_id);
}


/**
 * \brief Copies items to the specified category.
 *
 * Only copies those items that do not already exist in the target
 * category.
 */
void KitModel::copy_items(const ModelItemContainer& items, long cat_id) {
    assert(cat_id != -1);
    ModelCategory* c = find_category(cat_id);
    if (c) {
        c->set_dirty(true);
        // Make a copy of the list of source items requested for insertion
        // so that we can sort them
        ModelItemContainer sourceItems(items);
        // Create a list of ModelItems rather than Items
        // The Category class should be holding ModelItems
        ModelItemContainer targetItems;
        for (ItemIter i = c->m_items.begin(); i != c->m_items.end(); ++i) {
            targetItems.push_back((ModelItem*) (*i));
        }
        // Each list sorted by pointer references - duplicate items will have the same
        // pointer references
        sort(sourceItems.begin(), sourceItems.end()/*, ModelItemCompareId()*/);
        // To hold the unique list of non-duplicate items
        ModelItemContainer newItems;
        sort(targetItems.begin(), targetItems.end()/*, ModelItemCompareId()*/);
        std::set_difference(
            sourceItems.begin(),
            sourceItems.end(),
            targetItems.begin(),
            targetItems.end(),
            std::back_insert_iterator<ModelItemContainer>(newItems)
            );
        // newItems now contains only those items to be inserted that do not already exist
        // Don't just add them, need to use category's add_item method to update list of added and removed
        // c->m_items.insert(c->m_items.end(), newItems.begin(), newItems.end());
        // Also add these to ModelCategory's map of added items
        for (ModelItemIter i = newItems.begin(); i != newItems.end(); ++i) {
            c->add_item(*i);
        }
    } else {
        g_error("Warning: category id %d was not found - unable to copy items",
                cat_id);
    }
}


/**
 * \brief Applies the current filter.
 *
 * \param checked The checked/ticked state of the item being filtered.
 * \return true if the item should be included.
 */
bool KitModel::filter(bool checked) {
    switch (m_item_filter) {
    case ALL : return true;
    case CHECKED : return checked;
    case UNCHECKED : return !checked;
    }
    return true;
}


/**
 * \brief Resets all contained objects to their default states.
 *
 * This method needs to be called after load or save operations to
 * ensure all the dirty, deleted and new flags of each object are
 * reset.
 *
 * After a save operation, KitModel::purge() should be called before
 * this method.
 */
void KitModel::reset() {
    for (CategoryMapIter i = m_category_map->begin(); i != m_category_map->end(); ++i) {
        ModelCategory* cat = (*i).second;
        if (!cat->is_deleted())
            cat->reset();
    }
    for (ItemMapIter i = m_item_map->begin(); i != m_item_map->end(); ++i) {
        ModelItem* item = (*i).second;
        if (!item->is_deleted())
            item->reset();
    }
    m_dirty = false;
}

/**
 * \brief Purges deleted categories and items from the model.
 *
 * Typically, this method is called after a save operation, but before
 * calling KitModel::reset()
 */
void KitModel::purge() {
    CategoryMapIter c = m_category_map->begin();
    while (c != m_category_map->end()) {
        ModelCategory* cat = (*c).second;
        if (cat->is_deleted()) {
            CategoryMapIter iterTemp = c;
            iterTemp++;
            m_category_map->erase(c);
            c = iterTemp;
            delete cat;
        } else {
            cat->purge();
            c++;
        }
    }
    ItemMapIter i = m_item_map->begin();
    while (i != m_item_map->end()) {
        ModelItem* item = (*i).second;
        if (item->is_deleted()) {
            ItemMapIter iterTemp = i;
            iterTemp++;
            m_item_map->erase(i);
            i = iterTemp;
            delete item;
        } else {
            i++;
        }
    }
}
