/*

    This file is part of Kitlist, a program to maintain a simple list
    of items and assign items to one or more categories.

    Copyright (C) 2008,2009 Frank Dean

    Kitlist is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Kitlist is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Kitlist.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef KIT_LIST_DAO_H
#define KIT_LIST_DAO_H 1

#include "category.hpp"
#include "item.hpp"
#include "kitmodel.hpp"
#include <string>


/**
 * Options for selecting all items, only checked items, or only
 * unchecked items.
 */
enum item_choice {
    ALL_ITEMS,
    CHECKED_ITEMS,
    UNCHECKED_ITEMS };


/**
 * \brief Defines the methods that an implementation of this class
 * must implement.
 */
class KitListDao {
protected :
    /// Optional parameter indicating whether to include verbose
    /// output to STDOUT.  Defaults to false.
    int m_verbose_flag; 
public :
    /**
     * \brief Constructor which will use default database connection
     * parameters.
     *
     * \param verbose Optional parameter indicating whether to include
     * verbose output to STDOUT.  Defaults to false.
     */
    KitListDao(int verbose = 0) : m_verbose_flag(verbose) {}

    virtual ~KitListDao() {}

    /**
     * \brief Loads the data model.
     *
     * The data model holds a rich graph of objects, representing the
     * entire list of categories and items.
     * \see KitModel
     */
    virtual KitModel* get_model() = 0;

    /**
     * \brief Saves the current data model.
     *
     * Note: The data model will only be saved if it is dirty.
     * Further, only items that are individually flagged as dirty will
     * be saved.
     * \see GuiState
     */
    virtual void save_model(KitModel* model) = 0;

    /**
     * Indicates whether verbose output should be written to STDOUT.
     */
    void set_verbose(int verbose_flag) {
        m_verbose_flag = verbose_flag;
    }

    /**
     * Indicates whether verbose output should be written to STDOUT.
     */
    int is_verbose() { return m_verbose_flag; };

    /**
     * \brief Loads a category.
     * \param choice Which items to load for the category.  One of
     * ALL_ITEMS, CHECKED_ITEMS or UNCHECKED_ITEMS.
     */
    virtual Category* get_category(long cat_id, item_choice choice = ALL_ITEMS) = 0;

    /**
     * \brief Returns a list of all items.
     * \param choice Which items to load.  One of ALL_ITEMS,
     * CHECKED_ITEMS or UNCHECKED_ITEMS.
     */
    virtual ItemContainer* get_all_items(item_choice choice = ALL_ITEMS) = 0;

    /**
     * Creates a new item.
     * \param name The name of the new item.
     */
    virtual long add_item(const std::string name) = 0;

    /**
     * Creates a new item and associates it with a category.
     * \param name The name of the new item.
     */
    virtual long add_item(const std::string name, long cat_id) = 0;

    /**
     * \brief Copies items from one category to another.
     *
     * Optionally, only checked or unchecked items are copied.
     * \param from_cat_id The ID of the source category.
     * \param to_cat_id The ID of the target category.
     * \param choice One of ALL_ITEMS, CHECKED_ITEMS or UNCHECKED_ITEMS.
     */
    virtual void append_items_to_category(long to_cat_id, long from_cat_id = -1, item_choice choice = ALL_ITEMS) = 0;

    /**
     * \brief Associates an existing item with an existing category.
     *
     * \param id The Item ID
     * \param cat_id The Category ID
     */
    virtual void associate_item_with_category(long id, long cat_id) = 0;

    /**
     * Returns a list of all categories.
     */
    virtual CategoryContainer get_categories() = 0;

    /**
     * \brief Creates a new category.
     * \param name the name of the new category.
     */
    virtual long new_category(const std::string name) = 0;

    /**
     * Deletes an item by it's ID.
     * \param id the ID of the item to delete.
     */
    virtual void delete_item(long id) = 0;

    /**
     * \brief Persists the state of the 'checked' flag of each item.
     */
    virtual void update_item_checked_state(ItemContainer& items) = 0;

    /**
     * Un-associates the specified item from the specified category.
     *
     * \param id The Item id.
     * \param cat_id The Category id.
     */
    virtual void remove_item_from_category(long id, long cat_id) = 0;

    /**
     * Returns the next unused unique id for items.
     */
    virtual long get_next_item_id() = 0;

    /**
     * Returns the next unused unique id for categories.
     */
    virtual long get_next_category_id() = 0;

    /**
     * Deletes a category.
     */
    virtual void delete_category(long id) = 0;

    /**
     * Sets the checked flag for an item.
     * \param id The id of the item to change.
     */
    virtual void set_item_flag(long id) = 0;

    /**
     * Clears the checked flag for an item.
     */
    virtual void unset_item_flag(long id) = 0;

    /**
     * Checks/ticks all items in the specified Category.
     */
    virtual void set_category_flag(long id) = 0;

    /**
     * Unchecks/unticks all items in the specified Category.
     */
    virtual void unset_category_flag(long id) = 0;

    /**
     * Checks/ticks all items.
     */
    virtual void set_all_flags() = 0;

    /**
     * Unchecks/unticks all items.
     */
    virtual void unset_all_flags() = 0;

    /**
     * \brief Indicates whether the implementation of the data model
     * requires a filename.
     *
     * Some persistence models may require a filename to be
     * chosen, others (e.g. database) may be defined through
     * another mechanism.  If a filename has not been set, then
     * fire up save-as instead.
     *
     * \return Always returns false.
     */
    virtual bool require_filename() { return false; }
};
#endif //KIT_LIST_DAO_H
