// Copyright 2015 Keybase, Inc. All rights reserved. Use of
// this source code is governed by the included BSD license.

// +build linux

package install

import (
	"io/ioutil"
	"os"
	"path"

	"github.com/keybase/client/go/libkb"
)

// Similar to the Brew install on OSX, the Linux install happens in two steps.
// First, the system package manager installs all the binaries as root. Second,
// an autostart file needs to be written to the user's home dir, so that
// Keybase launches when that user logs in. The second step is done the first
// time the user starts Keybase.
//
// ".desktop" files and the ~/.config/autostart directory are part of the
// freedesktop.org set of standards, which the popular desktop environments
// like Gnome and KDE all support. See
// http://standards.freedesktop.org/desktop-entry-spec/latest/.

const autostartFileText = `# This file is autogenerated when Keybase starts. Keybase will recreate it if you
# delete it, so to disable autostart you should comment out the Exec line below.
[Desktop Entry]
Name=Keybase autostart
Type=Application
NoDisplay=true
Exec=run_keybase
`

func autostartDir(g *libkb.GlobalContext) string {
	// strip off the "keybase" folder on the end of the config dir
	return path.Join(g.Env.GetConfigDir(), "..", "autostart")
}

func autostartFilePath(g *libkb.GlobalContext) string {
	return path.Join(autostartDir(g), "keybase_autostart.desktop")
}

func AutoInstall(g *libkb.GlobalContext, _ string, _ bool) ( /* newProc */ bool, error) {
	// If the desktop file already exists, short circuit.
	if _, err := os.Stat(autostartFilePath(g)); err == nil {
		return false, nil
	}
	err := os.MkdirAll(autostartDir(g), 0755)
	if err != nil {
		return false, err
	}
	return false, ioutil.WriteFile(autostartFilePath(g), []byte(autostartFileText), 0644)
}

func CheckIfValidLocation() error {
	return nil
}

func kbfsBinPath(runMode libkb.RunMode, binPath string) (string, error) {
	return kbfsBinPathDefault(runMode, binPath)
}

func RunAfterStartup(g *libkb.GlobalContext, isService bool) error {
	return nil
}
