# This file is a part of Julia. License is MIT: https://julialang.org/license

# Mapping from LaTeX math symbol to the corresponding Unicode codepoint.
# This is used for tab substitution in the REPL.

# The initial symbol listing was generated from the W3C symbol mapping file:
#         http://www.w3.org/Math/characters/unicode.xml
# by the following Julia script:
#=
using LightXML
xdoc = parse_file("unicode.xml")
latexsym = []
Ls = Set()
for c in child_nodes(root(xdoc))
    if name(c) == "character" && is_elementnode(c)
        ce = XMLElement(c)
        latex = nothing
        for el in ("AMS", "IEEE", "mathlatex", "latex")
            latex = find_element(ce, el)
            latex !== nothing && break
        end
        if latex !== nothing
            L = strip(content(latex))
            id = attribute(ce, "id")
            U = string(map(s -> Char(parse(Int, s, 16)),
                           split(id[2:end], "-"))...)
            if ismatch(r"^\\[A-Za-z]+$",L) && !isa(U,String)
                if L in Ls
                    println("# duplicated symbol $L ($id)")
                else
                    if U[1] == '\u22a5' # unicode.xml incorrectly uses \perp for \bot
                        L = "\\bot"
                    end
                    push!(latexsym, (L, U))
                    push!(Ls, L)
                end
            end
        end
    end
end
println("# ", length(latexsym), " symbols generated from unicode.xml")
for (L, U) in latexsym
    println("    \"$(escape_string(L))\" => \"$(escape_string(U))\",")
end
=#

# Additional symbols were generated from the unicode-math LaTeX package
# symbol listing, using the following script:
#=
fname = "unicode-math-table.tex"
isfile(fname) || download("http://mirror.math.ku.edu/tex-archive/macros/latex/contrib/unicode-math/$fname", fname)
const latex_strings = Set(values(Base.REPLCompletions.latex_symbols))
open(fname) do f
    for L in eachline(f)
        x = map(s -> rstrip(s, [' ','\t','\n']),
                split(replace(L, r"[{}\"]+", "\t"), "\t"))
        c = Char(parse(Int, x[2], 16))
        if (Base.is_id_char(c) || Base.isoperator(Symbol(c))) &&
           string(c) ∉ latex_strings && !isascii(c)
            tabcomname = escape_string(x[3])
            if startswith(tabcomname, "\\\\math")
                tabcomname = string("\\\\", tabcomname[7:end])
            end
            println("    \"", tabcomname, "\" => \"",
                    escape_string("$c"), "\",  # ", x[5])
        end
    end
end
=#

# Finally, we also add some symbols manually (at the top) as needed.

const latex_symbols = Dict(

# manual additions:

    "\\sqrt" => "\u221A",
    "\\cbrt" => "\u221B",
    "\\female" => "♀",
    "\\mars" => "♂",
    "\\pprime" => "″",
    "\\ppprime" => "‴",
    "\\pppprime" => "⁗",
    "\\backpprime" => "‶",
    "\\backppprime" => "‷",
    "\\xor" => "⊻",
    "\\iff" => "⟺",
    "\\implies" => "⟹",
    "\\impliedby" => "⟸",
    "\\to" => "→",

    # Superscripts
    "\\^0" => "⁰",
    "\\^1" => "¹",
    "\\^2" => "²",
    "\\^3" => "³",
    "\\^4" => "⁴",
    "\\^5" => "⁵",
    "\\^6" => "⁶",
    "\\^7" => "⁷",
    "\\^8" => "⁸",
    "\\^9" => "⁹",
    "\\^+" => "⁺",
    "\\^-" => "⁻",
    "\\^=" => "⁼",
    "\\^(" => "⁽",
    "\\^)" => "⁾",
    "\\^a" => "ᵃ",
    "\\^b" => "ᵇ",
    "\\^c" => "ᶜ",
    "\\^d" => "ᵈ",
    "\\^e" => "ᵉ",
    "\\^f" => "ᶠ",
    "\\^g" => "ᵍ",
    "\\^h" => "ʰ",
    "\\^i" => "ⁱ",
    "\\^j" => "ʲ",
    "\\^k" => "ᵏ",
    "\\^l" => "ˡ",
    "\\^m" => "ᵐ",
    "\\^n" => "ⁿ",
    "\\^o" => "ᵒ",
    "\\^p" => "ᵖ",
    "\\^r" => "ʳ",
    "\\^s" => "ˢ",
    "\\^t" => "ᵗ",
    "\\^u" => "ᵘ",
    "\\^v" => "ᵛ",
    "\\^w" => "ʷ",
    "\\^x" => "ˣ",
    "\\^y" => "ʸ",
    "\\^z" => "ᶻ",
    "\\^A" => "ᴬ",
    "\\^B" => "ᴮ",
    "\\^D" => "ᴰ",
    "\\^E" => "ᴱ",
    "\\^G" => "ᴳ",
    "\\^H" => "ᴴ",
    "\\^I" => "ᴵ",
    "\\^J" => "ᴶ",
    "\\^K" => "ᴷ",
    "\\^L" => "ᴸ",
    "\\^M" => "ᴹ",
    "\\^N" => "ᴺ",
    "\\^O" => "ᴼ",
    "\\^P" => "ᴾ",
    "\\^R" => "ᴿ",
    "\\^T" => "ᵀ",
    "\\^U" => "ᵁ",
    "\\^V" => "ⱽ",
    "\\^W" => "ᵂ",
    "\\^alpha" => "ᵅ",
    "\\^beta" => "ᵝ",
    "\\^gamma" => "ᵞ",
    "\\^delta" => "ᵟ",
    "\\^epsilon" => "ᵋ",
    "\\^theta" => "ᶿ",
    "\\^iota" => "ᶥ",
    "\\^phi" => "ᵠ",
    "\\^chi" => "ᵡ",
    "\\^Phi" => "ᶲ",

    # Subscripts
    "\\_0" => "₀",
    "\\_1" => "₁",
    "\\_2" => "₂",
    "\\_3" => "₃",
    "\\_4" => "₄",
    "\\_5" => "₅",
    "\\_6" => "₆",
    "\\_7" => "₇",
    "\\_8" => "₈",
    "\\_9" => "₉",
    "\\_+" => "₊",
    "\\_-" => "₋",
    "\\_=" => "₌",
    "\\_(" => "₍",
    "\\_)" => "₎",
    "\\_a" => "ₐ",
    "\\_e" => "ₑ",
    "\\_h" => "ₕ",
    "\\_i" => "ᵢ",
    "\\_j" => "ⱼ",
    "\\_k" => "ₖ",
    "\\_l" => "ₗ",
    "\\_m" => "ₘ",
    "\\_n" => "ₙ",
    "\\_o" => "ₒ",
    "\\_p" => "ₚ",
    "\\_r" => "ᵣ",
    "\\_s" => "ₛ",
    "\\_t" => "ₜ",
    "\\_u" => "ᵤ",
    "\\_v" => "ᵥ",
    "\\_x" => "ₓ",
    "\\_schwa" => "ₔ",
    "\\_beta" => "ᵦ",
    "\\_gamma" => "ᵧ",
    "\\_rho" => "ᵨ",
    "\\_phi" => "ᵩ",
    "\\_chi" => "ᵪ",

    # Misc. Math and Physics
    "\\ldots" => "…",
    "\\hbar" => "ħ",
    "\\del" => "∇",

    "\\sout" => "̶",# ulem package, same as Elzbar
    "\\euro" => "€",

# 732 symbols generated from unicode.xml
    "\\textexclamdown" => "¡",
    "\\sterling" => "£",
    "\\yen" => "¥",
    "\\textbrokenbar" => "¦",
    "\\S" => "§",
    "\\textasciidieresis" => "¨",
    "\\copyright" => "©",
    "\\textordfeminine" => "ª",
    "\\neg" => "¬",
    "\\circledR" => "®",
    "\\textasciimacron" => "¯",
    "\\degree" => "°",
    "\\pm" => "±",
    "\\textasciiacute" => "´",
    "\\P" => "¶",
    "\\cdotp" => "·",
    "\\textordmasculine" => "º",
    "\\textonequarter" => "¼",
    "\\textonehalf" => "½",
    "\\textthreequarters" => "¾",
    "\\textquestiondown" => "¿",
    "\\AA" => "Å",
    "\\AE" => "Æ",
    "\\DH" => "Ð",
    "\\times" => "×",
    "\\O" => "Ø",
    "\\TH" => "Þ",
    "\\ss" => "ß",
    "\\aa" => "å",
    "\\ae" => "æ",
    "\\eth" => "ð",
    "\\div" => "÷",
    "\\o" => "ø",
    "\\th" => "þ",
    "\\DJ" => "Đ",
    "\\dj" => "đ",
    "\\Elzxh" => "ħ",
    "\\imath" => "ı",
    "\\L" => "Ł",
    "\\l" => "ł",
    "\\NG" => "Ŋ",
    "\\ng" => "ŋ",
    "\\OE" => "Œ",
    "\\oe" => "œ",
    "\\texthvlig" => "ƕ",
    "\\textnrleg" => "ƞ",
    "\\textdoublepipe" => "ǂ",
    "\\Elztrna" => "ɐ",
    "\\Elztrnsa" => "ɒ",
    "\\Elzopeno" => "ɔ",
    "\\Elzrtld" => "ɖ",
    "\\Elzschwa" => "ə",
    "\\varepsilon" => "ε",
    "\\Elzpgamma" => "ɣ",
    "\\Elzpbgam" => "ɤ",
    "\\Elztrnh" => "ɥ",
    "\\Elzbtdl" => "ɬ",
    "\\Elzrtll" => "ɭ",
    "\\Elztrnm" => "ɯ",
    "\\Elztrnmlr" => "ɰ",
    "\\Elzltlmr" => "ɱ",
    "\\Elzltln" => "ɲ",
    "\\Elzrtln" => "ɳ",
    "\\Elzclomeg" => "ɷ",
    "\\textphi" => "ɸ",
    "\\Elztrnr" => "ɹ",
    "\\Elztrnrl" => "ɺ",
    "\\Elzrttrnr" => "ɻ",
    "\\Elzrl" => "ɼ",
    "\\Elzrtlr" => "ɽ",
    "\\Elzfhr" => "ɾ",
    "\\Elzrtls" => "ʂ",
    "\\Elzesh" => "ʃ",
    "\\Elztrnt" => "ʇ",
    "\\Elzrtlt" => "ʈ",
    "\\Elzpupsil" => "ʊ",
    "\\Elzpscrv" => "ʋ",
    "\\Elzinvv" => "ʌ",
    "\\Elzinvw" => "ʍ",
    "\\Elztrny" => "ʎ",
    "\\Elzrtlz" => "ʐ",
    "\\Elzyogh" => "ʒ",
    "\\Elzglst" => "ʔ",
    "\\Elzreglst" => "ʕ",
    "\\Elzinglst" => "ʖ",
    "\\textturnk" => "ʞ",
    "\\Elzdyogh" => "ʤ",
    "\\Elztesh" => "ʧ",
    "\\rasp" => "ʼ",
    "\\textasciicaron" => "ˇ",
    "\\Elzverts" => "ˈ",
    "\\Elzverti" => "ˌ",
    "\\Elzlmrk" => "ː",
    "\\Elzhlmrk" => "ˑ",
    "\\Elzsbrhr" => "˒",
    "\\Elzsblhr" => "˓",
    "\\Elzrais" => "˔",
    "\\Elzlow" => "˕",
    "\\u" => "˘",
    "\\texttildelow" => "˜",
    "\\grave" => "̀",
    "\\acute" => "́",
    "\\hat" => "̂",
    "\\tilde" => "̃",
    "\\bar" => "̄",
    "\\breve" => "̆",
    "\\dot" => "̇",
    "\\ddot" => "̈",
    "\\ocirc" => "̊",
    "\\H" => "̋",
    "\\check" => "̌",
    "\\Elzpalh" => "̡",
    "\\Elzrh" => "̢",
    "\\c" => "̧",
    "\\k" => "̨",
    "\\Elzsbbrg" => "̪",
    "\\Elzxl" => "̵",
    "\\Elzbar" => "̶",
    "\\Alpha" => "Α",
    "\\Beta" => "Β",
    "\\Gamma" => "Γ",
    "\\Delta" => "Δ",
    "\\Epsilon" => "Ε",
    "\\Zeta" => "Ζ",
    "\\Eta" => "Η",
    "\\Theta" => "Θ",
    "\\Iota" => "Ι",
    "\\Kappa" => "Κ",
    "\\Lambda" => "Λ",
    "\\Xi" => "Ξ",
    "\\Pi" => "Π",
    "\\Rho" => "Ρ",
    "\\Sigma" => "Σ",
    "\\Tau" => "Τ",
    "\\Upsilon" => "Υ",
    "\\Phi" => "Φ",
    "\\Chi" => "Χ",
    "\\Psi" => "Ψ",
    "\\Omega" => "Ω",
    "\\alpha" => "α",
    "\\beta" => "β",
    "\\gamma" => "γ",
    "\\delta" => "δ",
    "\\zeta" => "ζ",
    "\\eta" => "η",
    "\\theta" => "θ",
    "\\iota" => "ι",
    "\\kappa" => "κ",
    "\\lambda" => "λ",
    "\\mu" => "μ",
    "\\nu" => "ν",
    "\\xi" => "ξ",
    "\\pi" => "π",
    "\\rho" => "ρ",
    "\\varsigma" => "ς",
    "\\sigma" => "σ",
    "\\tau" => "τ",
    "\\upsilon" => "υ",
    "\\varphi" => "φ",
    "\\chi" => "χ",
    "\\psi" => "ψ",
    "\\omega" => "ω",
    "\\vartheta" => "ϑ",
    "\\phi" => "ϕ",
    "\\varpi" => "ϖ",
    "\\Stigma" => "Ϛ",
    "\\Digamma" => "Ϝ",
    "\\digamma" => "ϝ",
    "\\Koppa" => "Ϟ",
    "\\Sampi" => "Ϡ",
    "\\varkappa" => "ϰ",
    "\\varrho" => "ϱ",
    "\\textTheta" => "ϴ",
    "\\epsilon" => "ϵ",
    "\\backepsilon" => "϶",
    "\\enspace" => " ",
    "\\quad" => " ",
    "\\thickspace" => " ",
    "\\thinspace" => " ",
    "\\hspace" => " ",
    "\\endash" => "–",
    "\\emdash" => "—",
    "\\Vert" => "‖",
    "\\lq" => "‘",
    "\\rq" => "’",
    "\\Elzreapos" => "‛",
    "\\textquotedblleft" => "“",
    "\\textquotedblright" => "”",
    "\\dagger" => "†",
    "\\ddagger" => "‡",
    "\\bullet" => "•",
    "\\dots" => "…",
    "\\textperthousand" => "‰",
    "\\textpertenthousand" => "‱",
    "\\prime" => "′",
    "\\backprime" => "‵",
    "\\guilsinglleft" => "‹",
    "\\guilsinglright" => "›",
    "\\nolinebreak" => "\u2060",
    "\\Elzpes" => "₧",
    "\\dddot" => "⃛",
    "\\ddddot" => "⃜",
    "\\hslash" => "ℏ",
    "\\Im" => "ℑ",
    "\\ell" => "ℓ",
    "\\textnumero" => "№",
    "\\wp" => "℘",
    "\\Re" => "ℜ",
    "\\Elzxrat" => "℞",
    "\\texttrademark" => "™",
    "\\mho" => "℧",
    "\\aleph" => "ℵ",
    "\\beth" => "ℶ",
    "\\gimel" => "ℷ",
    "\\daleth" => "ℸ",
    "\\BbbPi" => "ℿ",
    "\\bbsum" => "⅀",
    "\\Game" => "⅁",
    "\\leftarrow" => "←",
    "\\uparrow" => "↑",
    "\\rightarrow" => "→",
    "\\downarrow" => "↓",
    "\\leftrightarrow" => "↔",
    "\\updownarrow" => "↕",
    "\\nwarrow" => "↖",
    "\\nearrow" => "↗",
    "\\searrow" => "↘",
    "\\swarrow" => "↙",
    "\\nleftarrow" => "↚",
    "\\nrightarrow" => "↛",
    "\\leftsquigarrow" => "↜",
    "\\rightsquigarrow" => "↝",
    "\\twoheadleftarrow" => "↞",
    "\\twoheadrightarrow" => "↠",
    "\\leftarrowtail" => "↢",
    "\\rightarrowtail" => "↣",
    "\\mapsto" => "↦",
    "\\hookleftarrow" => "↩",
    "\\hookrightarrow" => "↪",
    "\\looparrowleft" => "↫",
    "\\looparrowright" => "↬",
    "\\leftrightsquigarrow" => "↭",
    "\\nleftrightarrow" => "↮",
    "\\Lsh" => "↰",
    "\\Rsh" => "↱",
    "\\curvearrowleft" => "↶",
    "\\curvearrowright" => "↷",
    "\\circlearrowleft" => "↺",
    "\\circlearrowright" => "↻",
    "\\leftharpoonup" => "↼",
    "\\leftharpoondown" => "↽",
    "\\upharpoonleft" => "↾",
    "\\upharpoonright" => "↿",
    "\\rightharpoonup" => "⇀",
    "\\rightharpoondown" => "⇁",
    "\\downharpoonright" => "⇂",
    "\\downharpoonleft" => "⇃",
    "\\rightleftarrows" => "⇄",
    "\\dblarrowupdown" => "⇅",
    "\\leftrightarrows" => "⇆",
    "\\leftleftarrows" => "⇇",
    "\\upuparrows" => "⇈",
    "\\rightrightarrows" => "⇉",
    "\\downdownarrows" => "⇊",
    "\\leftrightharpoons" => "⇋",
    "\\rightleftharpoons" => "⇌",
    "\\nLeftarrow" => "⇍",
    "\\nRightarrow" => "⇏",
    "\\Leftarrow" => "⇐",
    "\\Uparrow" => "⇑",
    "\\Rightarrow" => "⇒",
    "\\Downarrow" => "⇓",
    "\\Leftrightarrow" => "⇔",
    "\\Updownarrow" => "⇕",
    "\\Lleftarrow" => "⇚",
    "\\Rrightarrow" => "⇛",
    "\\DownArrowUpArrow" => "⇵",
    "\\leftarrowtriangle" => "⇽",
    "\\rightarrowtriangle" => "⇾",
    "\\forall" => "∀",
    "\\complement" => "∁",
    "\\partial" => "∂",
    "\\exists" => "∃",
    "\\nexists" => "∄",
    "\\varnothing" => "∅",
    "\\emptyset" => "∅",
    "\\nabla" => "∇",
    "\\in" => "∈",
    "\\notin" => "∉",
    "\\ni" => "∋",
    "\\prod" => "∏",
    "\\coprod" => "∐",
    "\\sum" => "∑",
    "\\minus" => "−",
    "\\mp" => "∓",
    "\\dotplus" => "∔",
    "\\setminus" => "∖",
    "\\ast" => "∗",
    "\\circ" => "∘",
    "\\surd" => "√",
    "\\propto" => "∝",
    "\\infty" => "∞",
    "\\rightangle" => "∟",
    "\\angle" => "∠",
    "\\measuredangle" => "∡",
    "\\sphericalangle" => "∢",
    "\\mid" => "∣",
    "\\nmid" => "∤",
    "\\parallel" => "∥",
    "\\nparallel" => "∦",
    "\\wedge" => "∧",
    "\\vee" => "∨",
    "\\cap" => "∩",
    "\\cup" => "∪",
    "\\int" => "∫",
    "\\iint" => "∬",
    "\\iiint" => "∭",
    "\\oint" => "∮",
    "\\oiint" => "∯",
    "\\oiiint" => "∰",
    "\\clwintegral" => "∱",
    "\\therefore" => "∴",
    "\\because" => "∵",
    "\\Colon" => "∷",
    "\\dotminus" => "∸",
    "\\kernelcontraction" => "∻",
    "\\sim" => "∼",
    "\\backsim" => "∽",
    "\\lazysinv" => "∾",
    "\\wr" => "≀",
    "\\nsim" => "≁",
    "\\eqsim" => "≂",
    "\\neqsim" => "≂̸",
    "\\simeq" => "≃",
    "\\nsime" => "≄",
    "\\cong" => "≅",
    "\\approxnotequal" => "≆",
    "\\ncong" => "≇",
    "\\approx" => "≈",
    "\\napprox" => "≉",
    "\\approxeq" => "≊",
    "\\tildetrpl" => "≋",
    "\\allequal" => "≌",
    "\\asymp" => "≍",
    "\\Bumpeq" => "≎",
    "\\nBumpeq" => "≎̸",
    "\\bumpeq" => "≏",
    "\\nbumpeq" => "≏̸",
    "\\doteq" => "≐",
    "\\Doteq" => "≑",
    "\\fallingdotseq" => "≒",
    "\\risingdotseq" => "≓",
    "\\coloneq" => "≔",
    "\\eqcolon" => "≕",
    "\\eqcirc" => "≖",
    "\\circeq" => "≗",
    "\\wedgeq" => "≙",
    "\\starequal" => "≛",
    "\\triangleq" => "≜",
    "\\questeq" => "≟",
    "\\ne" => "≠",
    "\\equiv" => "≡",
    "\\nequiv" => "≢",
    "\\le" => "≤",
    "\\ge" => "≥",
    "\\leqq" => "≦",
    "\\geqq" => "≧",
    "\\lneqq" => "≨",
    "\\lvertneqq" => "≨︀",
    "\\gneqq" => "≩",
    "\\gvertneqq" => "≩︀",
    "\\ll" => "≪",
    "\\NotLessLess" => "≪̸",
    "\\gg" => "≫",
    "\\NotGreaterGreater" => "≫̸",
    "\\between" => "≬",
    "\\nless" => "≮",
    "\\ngtr" => "≯",
    "\\nleq" => "≰",
    "\\ngeq" => "≱",
    "\\lesssim" => "≲",
    "\\gtrsim" => "≳",
    "\\lessgtr" => "≶",
    "\\gtrless" => "≷",
    "\\notlessgreater" => "≸",
    "\\notgreaterless" => "≹",
    "\\prec" => "≺",
    "\\succ" => "≻",
    "\\preccurlyeq" => "≼",
    "\\succcurlyeq" => "≽",
    "\\precsim" => "≾",
    "\\nprecsim" => "≾̸",
    "\\succsim" => "≿",
    "\\nsuccsim" => "≿̸",
    "\\nprec" => "⊀",
    "\\nsucc" => "⊁",
    "\\subset" => "⊂",
    "\\supset" => "⊃",
    "\\nsubset" => "⊄",
    "\\nsupset" => "⊅",
    "\\subseteq" => "⊆",
    "\\supseteq" => "⊇",
    "\\nsubseteq" => "⊈",
    "\\nsupseteq" => "⊉",
    "\\subsetneq" => "⊊",
    "\\varsubsetneqq" => "⊊︀",
    "\\supsetneq" => "⊋",
    "\\varsupsetneq" => "⊋︀",
    "\\cupdot" => "⊍",
    "\\uplus" => "⊎",
    "\\sqsubset" => "⊏",
    "\\NotSquareSubset" => "⊏̸",
    "\\sqsupset" => "⊐",
    "\\NotSquareSuperset" => "⊐̸",
    "\\sqsubseteq" => "⊑",
    "\\sqsupseteq" => "⊒",
    "\\sqcap" => "⊓",
    "\\sqcup" => "⊔",
    "\\oplus" => "⊕",
    "\\ominus" => "⊖",
    "\\otimes" => "⊗",
    "\\oslash" => "⊘",
    "\\odot" => "⊙",
    "\\circledcirc" => "⊚",
    "\\circledast" => "⊛",
    "\\circleddash" => "⊝",
    "\\boxplus" => "⊞",
    "\\boxminus" => "⊟",
    "\\boxtimes" => "⊠",
    "\\boxdot" => "⊡",
    "\\vdash" => "⊢",
    "\\dashv" => "⊣",
    "\\top" => "⊤",
    "\\bot" => "⊥",
    "\\models" => "⊧",
    "\\vDash" => "⊨",
    "\\Vdash" => "⊩",
    "\\Vvdash" => "⊪",
    "\\VDash" => "⊫",
    "\\nvdash" => "⊬",
    "\\nvDash" => "⊭",
    "\\nVdash" => "⊮",
    "\\nVDash" => "⊯",
    "\\vartriangleleft" => "⊲",
    "\\vartriangleright" => "⊳",
    "\\trianglelefteq" => "⊴",
    "\\trianglerighteq" => "⊵",
    "\\original" => "⊶",
    "\\image" => "⊷",
    "\\multimap" => "⊸",
    "\\hermitconjmatrix" => "⊹",
    "\\intercal" => "⊺",
    "\\veebar" => "⊻",
    "\\rightanglearc" => "⊾",
    "\\bigwedge" => "⋀",
    "\\bigvee" => "⋁",
    "\\bigcap" => "⋂",
    "\\bigcup" => "⋃",
    "\\diamond" => "⋄",
    "\\cdot" => "⋅",
    "\\star" => "⋆",
    "\\divideontimes" => "⋇",
    "\\bowtie" => "⋈",
    "\\ltimes" => "⋉",
    "\\rtimes" => "⋊",
    "\\leftthreetimes" => "⋋",
    "\\rightthreetimes" => "⋌",
    "\\backsimeq" => "⋍",
    "\\curlyvee" => "⋎",
    "\\curlywedge" => "⋏",
    "\\Subset" => "⋐",
    "\\Supset" => "⋑",
    "\\Cap" => "⋒",
    "\\Cup" => "⋓",
    "\\pitchfork" => "⋔",
    "\\lessdot" => "⋖",
    "\\gtrdot" => "⋗",
    "\\verymuchless" => "⋘",
    "\\ggg" => "⋙",
    "\\lesseqgtr" => "⋚",
    "\\gtreqless" => "⋛",
    "\\curlyeqprec" => "⋞",
    "\\curlyeqsucc" => "⋟",
    "\\Elzsqspne" => "⋥",
    "\\lnsim" => "⋦",
    "\\gnsim" => "⋧",
    "\\precnsim" => "⋨",
    "\\succnsim" => "⋩",
    "\\ntriangleleft" => "⋪",
    "\\ntriangleright" => "⋫",
    "\\ntrianglelefteq" => "⋬",
    "\\ntrianglerighteq" => "⋭",
    "\\vdots" => "⋮",
    "\\cdots" => "⋯",
    "\\adots" => "⋰",
    "\\ddots" => "⋱",
    "\\barwedge" => "⌅",
    "\\lceil" => "⌈",
    "\\rceil" => "⌉",
    "\\lfloor" => "⌊",
    "\\rfloor" => "⌋",
    "\\recorder" => "⌕",
    "\\ulcorner" => "⌜",
    "\\urcorner" => "⌝",
    "\\llcorner" => "⌞",
    "\\lrcorner" => "⌟",
    "\\frown" => "⌢",
    "\\smile" => "⌣",
    "\\langle" => "⟨",
    "\\rangle" => "⟩",
    "\\obar" => "⌽",
    "\\Elzdlcorn" => "⎣",
    "\\lmoustache" => "⎰",
    "\\rmoustache" => "⎱",
    "\\textvisiblespace" => "␣",
    "\\circledS" => "Ⓢ",
    "\\Elzdshfnc" => "┆",
    "\\Elzsqfnw" => "┙",
    "\\diagup" => "╱",
    "\\diagdown" => "╲",
    "\\blacksquare" => "■",
    "\\square" => "□",
    "\\Elzvrecto" => "▯",
    "\\bigtriangleup" => "△",
    "\\blacktriangle" => "▴",
    "\\vartriangle" => "▵",
    "\\blacktriangleright" => "▸",
    "\\triangleright" => "▹",
    "\\bigtriangledown" => "▽",
    "\\blacktriangledown" => "▾",
    "\\triangledown" => "▿",
    "\\blacktriangleleft" => "◂",
    "\\triangleleft" => "◃",
    "\\lozenge" => "◊",
    "\\bigcirc" => "○",
    "\\Elzcirfl" => "◐",
    "\\Elzcirfr" => "◑",
    "\\Elzcirfb" => "◒",
    "\\Elzrvbull" => "◘",
    "\\Elzsqfl" => "◧",
    "\\Elzsqfr" => "◨",
    "\\Elzsqfse" => "◪",
    "\\bigstar" => "★",
    "\\rightmoon" => "☾",
    "\\mercury" => "☿",
    "\\venus" => "♀",
    "\\male" => "♂",
    "\\jupiter" => "♃",
    "\\saturn" => "♄",
    "\\uranus" => "♅",
    "\\neptune" => "♆",
    "\\pluto" => "♇",
    "\\aries" => "♈",
    "\\taurus" => "♉",
    "\\gemini" => "♊",
    "\\cancer" => "♋",
    "\\leo" => "♌",
    "\\virgo" => "♍",
    "\\libra" => "♎",
    "\\scorpio" => "♏",
    "\\sagittarius" => "♐",
    "\\capricornus" => "♑",
    "\\aquarius" => "♒",
    "\\pisces" => "♓",
    "\\spadesuit" => "♠",
    "\\heartsuit" => "♡",
    "\\diamondsuit" => "♢",
    "\\clubsuit" => "♣",
    "\\quarternote" => "♩",
    "\\eighthnote" => "♪",
    "\\flat" => "♭",
    "\\natural" => "♮",
    "\\sharp" => "♯",
    "\\checkmark" => "✓",
    "\\maltese" => "✠",
    "\\longleftarrow" => "⟵",
    "\\longrightarrow" => "⟶",
    "\\longleftrightarrow" => "⟷",
    "\\Longleftarrow" => "⟸",
    "\\Longrightarrow" => "⟹",
    "\\Longleftrightarrow" => "⟺",
    "\\longmapsto" => "⟼",
    "\\Mapsfrom" => "⤆",
    "\\Mapsto" => "⤇",
    "\\Uuparrow" => "⤊",
    "\\Ddownarrow" => "⤋",
    "\\bkarow" => "⤍",
    "\\dbkarow" => "⤏",
    "\\drbkarrow" => "⤐",
    "\\UpArrowBar" => "⤒",
    "\\DownArrowBar" => "⤓",
    "\\twoheadrightarrowtail" => "⤖",
    "\\hksearow" => "⤥",
    "\\hkswarow" => "⤦",
    "\\tona" => "⤧",
    "\\toea" => "⤨",
    "\\tosa" => "⤩",
    "\\towa" => "⤪",
    "\\rdiagovfdiag" => "⤫",
    "\\fdiagovrdiag" => "⤬",
    "\\seovnearrow" => "⤭",
    "\\neovsearrow" => "⤮",
    "\\fdiagovnearrow" => "⤯",
    "\\rdiagovsearrow" => "⤰",
    "\\neovnwarrow" => "⤱",
    "\\nwovnearrow" => "⤲",
    "\\ElzRlarr" => "⥂",
    "\\ElzrLarr" => "⥄",
    "\\Elzrarrx" => "⥇",
    "\\LeftRightVector" => "⥎",
    "\\RightUpDownVector" => "⥏",
    "\\DownLeftRightVector" => "⥐",
    "\\LeftUpDownVector" => "⥑",
    "\\LeftVectorBar" => "⥒",
    "\\RightVectorBar" => "⥓",
    "\\RightUpVectorBar" => "⥔",
    "\\RightDownVectorBar" => "⥕",
    "\\DownLeftVectorBar" => "⥖",
    "\\DownRightVectorBar" => "⥗",
    "\\LeftUpVectorBar" => "⥘",
    "\\LeftDownVectorBar" => "⥙",
    "\\LeftTeeVector" => "⥚",
    "\\RightTeeVector" => "⥛",
    "\\RightUpTeeVector" => "⥜",
    "\\RightDownTeeVector" => "⥝",
    "\\DownLeftTeeVector" => "⥞",
    "\\DownRightTeeVector" => "⥟",
    "\\LeftUpTeeVector" => "⥠",
    "\\LeftDownTeeVector" => "⥡",
    "\\UpEquilibrium" => "⥮",
    "\\ReverseUpEquilibrium" => "⥯",
    "\\RoundImplies" => "⥰",
    "\\Vvert" => "⦀",
    "\\Elroang" => "⦆",
    "\\Elzddfnc" => "⦙",
    "\\Angle" => "⦜",
    "\\Elzlpargt" => "⦠",
    "\\obslash" => "⦸",
    "\\boxdiag" => "⧄",
    "\\boxbslash" => "⧅",
    "\\boxast" => "⧆",
    "\\boxcircle" => "⧇",
    "\\ElzLap" => "⧊",
    "\\Elzdefas" => "⧋",
    "\\LeftTriangleBar" => "⧏",
    "\\NotLeftTriangleBar" => "⧏̸",
    "\\RightTriangleBar" => "⧐",
    "\\NotRightTriangleBar" => "⧐̸",
    "\\dualmap" => "⧟",
    "\\shuffle" => "⧢",
    "\\blacklozenge" => "⧫",
    "\\RuleDelayed" => "⧴",
    "\\bigodot" => "⨀",
    "\\bigoplus" => "⨁",
    "\\bigotimes" => "⨂",
    "\\bigcupdot" => "⨃",
    "\\biguplus" => "⨄",
    "\\bigsqcap" => "⨅",
    "\\bigsqcup" => "⨆",
    "\\conjquant" => "⨇",
    "\\disjquant" => "⨈",
    "\\bigtimes" => "⨉",
    "\\iiiint" => "⨌",
    "\\intbar" => "⨍",
    "\\intBar" => "⨎",
    "\\clockoint" => "⨏",
    "\\sqrint" => "⨖",
    "\\intx" => "⨘",
    "\\intcap" => "⨙",
    "\\intcup" => "⨚",
    "\\upint" => "⨛",
    "\\lowint" => "⨜",
    "\\plusdot" => "⨥",
    "\\minusdot" => "⨪",
    "\\ElzTimes" => "⨯",
    "\\btimes" => "⨲",
    "\\intprod" => "⨼",
    "\\intprodr" => "⨽",
    "\\amalg" => "⨿",
    "\\ElzAnd" => "⩓",
    "\\ElzOr" => "⩔",
    "\\ElOr" => "⩖",
    "\\perspcorrespond" => "⩞",
    "\\Elzminhat" => "⩟",
    "\\Equal" => "⩵",
    "\\ddotseq" => "⩷",
    "\\leqslant" => "⩽",
    "\\nleqslant" => "⩽̸",
    "\\geqslant" => "⩾",
    "\\ngeqslant" => "⩾̸",
    "\\lessapprox" => "⪅",
    "\\gtrapprox" => "⪆",
    "\\lneq" => "⪇",
    "\\gneq" => "⪈",
    "\\lnapprox" => "⪉",
    "\\gnapprox" => "⪊",
    "\\lesseqqgtr" => "⪋",
    "\\gtreqqless" => "⪌",
    "\\eqslantless" => "⪕",
    "\\eqslantgtr" => "⪖",
    "\\NestedLessLess" => "⪡",
    "\\NotNestedLessLess" => "⪡̸",
    "\\NestedGreaterGreater" => "⪢",
    "\\NotNestedGreaterGreater" => "⪢̸",
    "\\partialmeetcontraction" => "⪣",
    "\\bumpeqq" => "⪮",
    "\\preceq" => "⪯",
    "\\npreceq" => "⪯̸",
    "\\succeq" => "⪰",
    "\\nsucceq" => "⪰̸",
    "\\precneqq" => "⪵",
    "\\succneqq" => "⪶",
    "\\precapprox" => "⪷",
    "\\succapprox" => "⪸",
    "\\precnapprox" => "⪹",
    "\\succnapprox" => "⪺",
    "\\subseteqq" => "⫅",
    "\\nsubseteqq" => "⫅̸",
    "\\supseteqq" => "⫆",
    "\\nsupseteqq" => "⫆̸",
    "\\subsetneqq" => "⫋",
    "\\supsetneqq" => "⫌",
    "\\mlcp" => "⫛",
    "\\forks" => "⫝̸",
    "\\forksnot" => "⫝",
    "\\dashV" => "⫣",
    "\\Dashv" => "⫤",
    "\\interleave" => "⫴",
    "\\Elztdcol" => "⫶",
    "\\openbracketleft" => "⟦",
    "\\llbracket" => "⟦",
    "\\openbracketright" => "⟧",
    "\\rrbracket" => "⟧",
    "\\overbrace" => "⏞",
    "\\underbrace" => "⏟",

# 1607 symbols generated from unicode-math-table.tex:
    "\\Zbar" => "Ƶ",  # impedance (latin capital letter z with stroke)
    "\\overbar" => "̅",  # overbar embellishment
    "\\ovhook" => "̉",  # combining hook above
    "\\candra" => "̐",  # candrabindu (non-spacing)
    "\\oturnedcomma" => "̒",  # combining turned comma above
    "\\ocommatopright" => "̕",  # combining comma above right
    "\\droang" => "̚",  # left angle above (non-spacing)
    "\\wideutilde" => "̰",  # under tilde accent (multiple characters and non-spacing)
    "\\underbar" => "̱",  # combining macron below
    "\\not" => "̸",  # combining long solidus overlay
    "\\upMu" => "Μ",  # capital mu, greek
    "\\upNu" => "Ν",  # capital nu, greek
    "\\upOmicron" => "Ο",  # capital omicron, greek
    "\\upepsilon" => "ε",  # rounded small epsilon, greek
    "\\upomicron" => "ο",  # small omicron, greek
    "\\upvarbeta" => "ϐ",  # rounded small beta, greek
    "\\upoldKoppa" => "Ϙ",  # greek letter archaic koppa
    "\\upoldkoppa" => "ϙ",  # greek small letter archaic koppa
    "\\upstigma" => "ϛ",  # greek small letter stigma
    "\\upkoppa" => "ϟ",  # greek small letter koppa
    "\\upsampi" => "ϡ",  # greek small letter sampi
    "\\tieconcat" => "⁀",  # character tie, z notation sequence concatenation
    "\\leftharpoonaccent" => "⃐",  # combining left harpoon above
    "\\rightharpoonaccent" => "⃑",  # combining right harpoon above
    "\\vertoverlay" => "⃒",  # combining long vertical line overlay
    "\\overleftarrow" => "⃖",  # combining left arrow above
    "\\vec" => "⃗",  # combining right arrow above
    "\\enclosecircle" => "⃝",  # combining enclosing circle
    "\\enclosesquare" => "⃞",  # combining enclosing square
    "\\enclosediamond" => "⃟",  # combining enclosing diamond
    "\\overleftrightarrow" => "⃡",  # combining left right arrow above
    "\\enclosetriangle" => "⃤",  # combining enclosing upward pointing triangle
    "\\annuity" => "⃧",  # combining annuity symbol
    "\\threeunderdot" => "⃨",  # combining triple underdot
    "\\widebridgeabove" => "⃩",  # combining wide bridge above
    "\\underrightharpoondown" => "\u20ec",  # combining rightwards harpoon with barb downwards
    "\\underleftharpoondown" => "\u20ed",  # combining leftwards harpoon with barb downwards
    "\\underleftarrow" => "\u20ee",  # combining left arrow below
    "\\underrightarrow" => "\u20ef",  # combining right arrow below
    "\\asteraccent" => "\u20f0",  # combining asterisk above
    "\\BbbC" => "ℂ",  # /bbb c, open face c
    "\\Eulerconst" => "ℇ",  # euler constant
    "\\mscrg" => "ℊ",  # /scr g, script letter g
    "\\mscrH" => "ℋ",  # hamiltonian (script capital h)
    "\\mfrakH" => "ℌ",  # /frak h, upper case h
    "\\BbbH" => "ℍ",  # /bbb h, open face h
    "\\Planckconst" => "ℎ",  # planck constant
    "\\mscrI" => "ℐ",  # /scr i, script letter i
    "\\mscrL" => "ℒ",  # lagrangian (script capital l)
    "\\BbbN" => "ℕ",  # /bbb n, open face n
    "\\BbbP" => "ℙ",  # /bbb p, open face p
    "\\BbbQ" => "ℚ",  # /bbb q, open face q
    "\\mscrR" => "ℛ",  # /scr r, script letter r
    "\\BbbR" => "ℝ",  # /bbb r, open face r
    "\\BbbZ" => "ℤ",  # /bbb z, open face z
    "\\mfrakZ" => "ℨ",  # /frak z, upper case z
    "\\turnediota" => "℩",  # turned iota
    "\\Angstrom" => "Å",  # angstrom capital a, ring
    "\\mscrB" => "ℬ",  # bernoulli function (script capital b)
    "\\mfrakC" => "ℭ",  # black-letter capital c
    "\\mscre" => "ℯ",  # /scr e, script letter e
    "\\mscrE" => "ℰ",  # /scr e, script letter e
    "\\mscrF" => "ℱ",  # /scr f, script letter f
    "\\Finv" => "Ⅎ",  # turned capital f
    "\\mscrM" => "ℳ",  # physics m-matrix (script capital m)
    "\\mscro" => "ℴ",  # order of (script small o)
    "\\Bbbpi" => "\u213c",  # double-struck small pi
    "\\Bbbgamma" => "ℽ",  # double-struck small gamma
    "\\BbbGamma" => "ℾ",  # double-struck capital gamma
    "\\sansLturned" => "⅂",  # turned sans-serif capital l
    "\\sansLmirrored" => "⅃",  # reversed sans-serif capital l
    "\\Yup" => "⅄",  # turned sans-serif capital y
    "\\mitBbbD" => "ⅅ",  # double-struck italic capital d
    "\\mitBbbd" => "ⅆ",  # double-struck italic small d
    "\\mitBbbe" => "ⅇ",  # double-struck italic small e
    "\\mitBbbi" => "ⅈ",  # double-struck italic small i
    "\\mitBbbj" => "ⅉ",  # double-struck italic small j
    "\\PropertyLine" => "⅊",  # property line
    "\\upand" => "⅋",  # turned ampersand
    "\\twoheaduparrow" => "↟",  # up two-headed arrow
    "\\twoheaddownarrow" => "↡",  # down two-headed arrow
    "\\mapsfrom" => "↤",  # maps to, leftward
    "\\mapsup" => "↥",  # maps to, upward
    "\\mapsdown" => "↧",  # maps to, downward
    "\\updownarrowbar" => "↨",  # up down arrow with base (perpendicular)
    "\\downzigzagarrow" => "↯",  # downwards zigzag arrow
    "\\Ldsh" => "↲",  # left down angled arrow
    "\\Rdsh" => "↳",  # right down angled arrow
    "\\linefeed" => "↴",  # rightwards arrow with corner downwards
    "\\carriagereturn" => "↵",  # downwards arrow with corner leftward = carriage return
    "\\barovernorthwestarrow" => "↸",  # north west arrow to long bar
    "\\barleftarrowrightarrowbar" => "↹",  # leftwards arrow to bar over rightwards arrow to bar
    "\\nLeftrightarrow" => "⇎",  # not left and right double arrows
    "\\Nwarrow" => "⇖",  # nw pointing double arrow
    "\\Nearrow" => "⇗",  # ne pointing double arrow
    "\\Searrow" => "⇘",  # se pointing double arrow
    "\\Swarrow" => "⇙",  # sw pointing double arrow
    "\\leftsquigarrow" => "⇜",  # leftwards squiggle arrow
    "\\rightsquigarrow" => "⇝",  # rightwards squiggle arrow
    "\\nHuparrow" => "⇞",  # upwards arrow with double stroke
    "\\nHdownarrow" => "⇟",  # downwards arrow with double stroke
    "\\leftdasharrow" => "⇠",  # leftwards dashed arrow
    "\\updasharrow" => "⇡",  # upwards dashed arrow
    "\\rightdasharrow" => "⇢",  # rightwards dashed arrow
    "\\downdasharrow" => "⇣",  # downwards dashed arrow
    "\\barleftarrow" => "⇤",  # leftwards arrow to bar
    "\\rightarrowbar" => "⇥",  # rightwards arrow to bar
    "\\leftwhitearrow" => "⇦",  # leftwards white arrow
    "\\upwhitearrow" => "⇧",  # upwards white arrow
    "\\rightwhitearrow" => "⇨",  # rightwards white arrow
    "\\downwhitearrow" => "⇩",  # downwards white arrow
    "\\whitearrowupfrombar" => "⇪",  # upwards white arrow from bar
    "\\circleonrightarrow" => "⇴",  # right arrow with small circle
    "\\rightthreearrows" => "⇶",  # three rightwards arrows
    "\\nvleftarrow" => "⇷",  # leftwards arrow with vertical stroke
    "\\nvrightarrow" => "⇸",  # rightwards arrow with vertical stroke
    "\\nvleftrightarrow" => "⇹",  # left right arrow with vertical stroke
    "\\nVleftarrow" => "⇺",  # leftwards arrow with double vertical stroke
    "\\nVrightarrow" => "⇻",  # rightwards arrow with double vertical stroke
    "\\nVleftrightarrow" => "⇼",  # left right arrow with double vertical stroke
    "\\leftrightarrowtriangle" => "⇿",  # left right open-headed arrow
    "\\increment" => "∆",  # laplacian (delta; nabla\string^2)
    "\\smallin" => "∊",  # set membership (small set membership)
    "\\nni" => "∌",  # negated contains, variant
    "\\smallni" => "∍",  # /ni /owns r: contains (small contains as member)
    "\\QED" => "∎",  # end of proof
    "\\vysmblkcircle" => "∙",  # bullet operator
    "\\fourthroot" => "∜",  # fourth root
    "\\varointclockwise" => "∲",  # contour integral, clockwise
    "\\ointctrclockwise" => "∳",  # contour integral, anticlockwise
    "\\dotsminusdots" => "∺",  # minus with four dots, geometric properties
    "\\sinewave" => "∿",  # sine wave
    "\\arceq" => "≘",  # arc, equals; corresponds to
    "\\veeeq" => "≚",  # logical or, equals
    "\\eqdef" => "≝",  # equals by definition
    "\\measeq" => "≞",  # measured by (m over equals)
    "\\Equiv" => "≣",  # strict equivalence (4 lines)
    "\\nasymp" => "≭",  # not asymptotically equal to
    "\\nlesssim" => "≴",  # not less, similar
    "\\ngtrsim" => "≵",  # not greater, similar
    "\\circledequal" => "⊜",  # equal in circle
    "\\prurel" => "⊰",  # element precedes under relation
    "\\scurel" => "⊱",  # succeeds under relation
    "\\barwedge" => "⊼",  # bar, wedge (large wedge)
    "\\barvee" => "⊽",  # bar, vee (large vee)
    "\\varlrtriangle" => "⊿",  # right triangle
    "\\equalparallel" => "⋕",  # parallel, equal; equal or parallel
    "\\eqless" => "⋜",  # equal-or-less
    "\\eqgtr" => "⋝",  # equal-or-greater
    "\\npreccurlyeq" => "⋠",  # not precedes, curly equals
    "\\nsucccurlyeq" => "⋡",  # not succeeds, curly equals
    "\\nsqsubseteq" => "⋢",  # not, square subset, equals
    "\\nsqsupseteq" => "⋣",  # not, square superset, equals
    "\\sqsubsetneq" => "⋤",  # square subset, not equals
    "\\disin" => "⋲",  # element of with long horizontal stroke
    "\\varisins" => "⋳",  # element of with vertical bar at end of horizontal stroke
    "\\isins" => "⋴",  # small element of with vertical bar at end of horizontal stroke
    "\\isindot" => "⋵",  # element of with dot above
    "\\varisinobar" => "⋶",  # element of with overbar
    "\\isinobar" => "⋷",  # small element of with overbar
    "\\isinvb" => "⋸",  # element of with underbar
    "\\isinE" => "⋹",  # element of with two horizontal strokes
    "\\nisd" => "⋺",  # contains with long horizontal stroke
    "\\varnis" => "⋻",  # contains with vertical bar at end of horizontal stroke
    "\\nis" => "⋼",  # small contains with vertical bar at end of horizontal stroke
    "\\varniobar" => "⋽",  # contains with overbar
    "\\niobar" => "⋾",  # small contains with overbar
    "\\bagmember" => "⋿",  # z notation bag membership
    "\\diameter" => "⌀",  # diameter sign
    "\\house" => "⌂",  # house
    "\\vardoublebarwedge" => "⌆",  # /doublebarwedge b: logical and, double bar above [perspective (double bar over small wedge)]
    "\\invnot" => "⌐",  # reverse not
    "\\sqlozenge" => "⌑",  # square lozenge
    "\\profline" => "⌒",  # profile of a line
    "\\profsurf" => "⌓",  # profile of a surface
    "\\viewdata" => "⌗",  # viewdata square
    "\\turnednot" => "⌙",  # turned not sign
    "\\varhexagonlrbonds" => "⌬",  # six carbon ring, corner down, double bonds lower right etc
    "\\conictaper" => "⌲",  # conical taper
    "\\topbot" => "⌶",  # top and bottom
    "\\APLnotslash" => "⌿",  # solidus, bar through (apl functional symbol slash bar)
    "\\APLnotbackslash" => "⍀",  # apl functional symbol backslash bar
    "\\APLboxupcaret" => "⍓",  # boxed up caret
    "\\APLboxquestion" => "⍰",  # boxed question mark
    "\\hexagon" => "⎔",  # horizontal benzene ring [hexagon flat open]
    "\\overbracket" => "⎴",  # top square bracket
    "\\underbracket" => "⎵",  # bottom square bracket
    "\\bbrktbrk" => "⎶",  # bottom square bracket over top square bracket
    "\\sqrtbottom" => "⎷",  # radical symbol bottom
    "\\lvboxline" => "⎸",  # left vertical box line
    "\\rvboxline" => "⎹",  # right vertical box line
    "\\varcarriagereturn" => "⏎",  # return symbol
    "\\trapezium" => "\u23e2",  # white trapezium
    "\\benzenr" => "\u23e3",  # benzene ring with circle
    "\\strns" => "\u23e4",  # straightness
    "\\fltns" => "\u23e5",  # flatness
    "\\accurrent" => "\u23e6",  # ac current
    "\\elinters" => "\u23e7",  # electrical intersection
    "\\blanksymbol" => "␢",  # blank symbol
    "\\blockuphalf" => "▀",  # upper half block
    "\\blocklowhalf" => "▄",  # lower half block
    "\\blockfull" => "█",  # full block
    "\\blocklefthalf" => "▌",  # left half block
    "\\blockrighthalf" => "▐",  # right half block
    "\\blockqtrshaded" => "░",  # 25\% shaded block
    "\\blockhalfshaded" => "▒",  # 50\% shaded block
    "\\blockthreeqtrshaded" => "▓",  # 75\% shaded block
    "\\squoval" => "▢",  # white square with rounded corners
    "\\blackinwhitesquare" => "▣",  # white square containing black small square
    "\\squarehfill" => "▤",  # square, horizontal rule filled
    "\\squarevfill" => "▥",  # square, vertical rule filled
    "\\squarehvfill" => "▦",  # square with orthogonal crosshatch fill
    "\\squarenwsefill" => "▧",  # square, nw-to-se rule filled
    "\\squareneswfill" => "▨",  # square, ne-to-sw rule filled
    "\\squarecrossfill" => "▩",  # square with diagonal crosshatch fill
    "\\smblksquare" => "▪",  # /blacksquare - sq bullet, filled
    "\\smwhtsquare" => "▫",  # white small square
    "\\hrectangleblack" => "▬",  # black rectangle
    "\\hrectangle" => "▭",  # horizontal rectangle, open
    "\\vrectangleblack" => "▮",  # black vertical rectangle
    "\\parallelogramblack" => "▰",  # black parallelogram
    "\\parallelogram" => "▱",  # parallelogram, open
    "\\bigblacktriangleup" => "▲",  #    0x25b2 6 6d      black up-pointing triangle
    "\\blacktriangleright" => "▶",  # (large) right triangle, filled
    "\\blackpointerright" => "►",  # black right-pointing pointer
    "\\whitepointerright" => "▻",  # white right-pointing pointer
    "\\bigblacktriangledown" => "▼",  # big down triangle, filled
    "\\blacktriangleleft" => "◀",  # (large) left triangle, filled
    "\\blackpointerleft" => "◄",  # black left-pointing pointer
    "\\whitepointerleft" => "◅",  # white left-pointing pointer
    "\\mdlgblkdiamond" => "◆",  # black diamond
    "\\mdlgwhtdiamond" => "◇",  # white diamond; diamond, open
    "\\blackinwhitediamond" => "◈",  # white diamond containing black small diamond
    "\\fisheye" => "◉",  # fisheye
    "\\dottedcircle" => "◌",  # dotted circle
    "\\circlevertfill" => "◍",  # circle with vertical fill
    "\\bullseye" => "◎",  # bullseye
    "\\mdlgblkcircle" => "●",  # circle, filled
    "\\circletophalfblack" => "◓",  # circle, filled top half
    "\\circleurquadblack" => "◔",  # circle with upper right quadrant black
    "\\blackcircleulquadwhite" => "◕",  # circle with all but upper left quadrant black
    "\\blacklefthalfcircle" => "◖",  # left half black circle
    "\\blackrighthalfcircle" => "◗",  # right half black circle
    "\\inversewhitecircle" => "◙",  # inverse white circle
    "\\invwhiteupperhalfcircle" => "◚",  # upper half inverse white circle
    "\\invwhitelowerhalfcircle" => "◛",  # lower half inverse white circle
    "\\ularc" => "◜",  # upper left quadrant circular arc
    "\\urarc" => "◝",  # upper right quadrant circular arc
    "\\lrarc" => "◞",  # lower right quadrant circular arc
    "\\llarc" => "◟",  # lower left quadrant circular arc
    "\\topsemicircle" => "◠",  # upper half circle
    "\\botsemicircle" => "◡",  # lower half circle
    "\\lrblacktriangle" => "◢",  # lower right triangle, filled
    "\\llblacktriangle" => "◣",  # lower left triangle, filled
    "\\ulblacktriangle" => "◤",  # upper left triangle, filled
    "\\urblacktriangle" => "◥",  # upper right triangle, filled
    "\\smwhtcircle" => "◦",  # white bullet
    "\\squareulblack" => "◩",  # square, filled top left corner
    "\\boxbar" => "◫",  # vertical bar in box
    "\\trianglecdot" => "◬",  # triangle with centered dot
    "\\triangleleftblack" => "◭",  # up-pointing triangle with left half black
    "\\trianglerightblack" => "◮",  # up-pointing triangle with right half black
    "\\lgwhtcircle" => "◯",  # large circle
    "\\squareulquad" => "◰",  # white square with upper left quadrant
    "\\squarellquad" => "◱",  # white square with lower left quadrant
    "\\squarelrquad" => "◲",  # white square with lower right quadrant
    "\\squareurquad" => "◳",  # white square with upper right quadrant
    "\\circleulquad" => "◴",  # white circle with upper left quadrant
    "\\circlellquad" => "◵",  # white circle with lower left quadrant
    "\\circlelrquad" => "◶",  # white circle with lower right quadrant
    "\\circleurquad" => "◷",  # white circle with upper right quadrant
    "\\ultriangle" => "◸",  # upper left triangle
    "\\urtriangle" => "◹",  # upper right triangle
    "\\lltriangle" => "◺",  # lower left triangle
    "\\mdwhtsquare" => "◻",  # white medium square
    "\\mdblksquare" => "◼",  # black medium square
    "\\mdsmwhtsquare" => "◽",  # white medium small square
    "\\mdsmblksquare" => "◾",  # black medium small square
    "\\lrtriangle" => "◿",  # lower right triangle
    "\\bigwhitestar" => "☆",  # star, open
    "\\astrosun" => "☉",  # sun
    "\\danger" => "☡",  # dangerous bend (caution sign)
    "\\blacksmiley" => "☻",  # black smiling face
    "\\sun" => "☼",  # white sun with rays
    "\\rightmoon" => "☽",  # first quarter moon
    "\\varspadesuit" => "♤",  # spade, white (card suit)
    "\\varheartsuit" => "♥",  # filled heart (card suit)
    "\\vardiamondsuit" => "♦",  # filled diamond (card suit)
    "\\varclubsuit" => "♧",  # club, white (card suit)
    "\\twonotes" => "♫",  # beamed eighth notes
    "\\acidfree" => "\u267e",  # permanent paper sign
    "\\dicei" => "⚀",  # die face-1
    "\\diceii" => "⚁",  # die face-2
    "\\diceiii" => "⚂",  # die face-3
    "\\diceiv" => "⚃",  # die face-4
    "\\dicev" => "⚄",  # die face-5
    "\\dicevi" => "⚅",  # die face-6
    "\\circledrightdot" => "⚆",  # white circle with dot right
    "\\circledtwodots" => "⚇",  # white circle with two dots
    "\\blackcircledrightdot" => "⚈",  # black circle with white dot right
    "\\blackcircledtwodots" => "⚉",  # black circle with two white dots
    "\\Hermaphrodite" => "\u26a5",  # male and female sign
    "\\mdwhtcircle" => "\u26aa",  # medium white circle
    "\\mdblkcircle" => "\u26ab",  # medium black circle
    "\\mdsmwhtcircle" => "\u26ac",  # medium small white circle
    "\\neuter" => "\u26b2",  # neuter
    "\\circledstar" => "✪",  # circled white star
    "\\varstar" => "✶",  # six pointed black star
    "\\dingasterisk" => "✽",  # heavy teardrop-spoked asterisk
    "\\draftingarrow" => "➛",  # right arrow with bold head (drafting)
    "\\threedangle" => "\u27c0",  # three dimensional angle
    "\\whiteinwhitetriangle" => "\u27c1",  # white triangle containing small white triangle
    "\\perp" => "\u27c2",  # perpendicular
    "\\bsolhsub" => "\u27c8",  # reverse solidus preceding subset
    "\\suphsol" => "\u27c9",  # superset preceding solidus
    "\\wedgedot" => "⟑",  # and with dot
    "\\upin" => "⟒",  # element of opening upwards
    "\\bigbot" => "⟘",  # large up tack
    "\\bigtop" => "⟙",  # large down tack
    "\\UUparrow" => "⟰",  # upwards quadruple arrow
    "\\DDownarrow" => "⟱",  # downwards quadruple arrow
    "\\longmapsfrom" => "⟻",  # long leftwards arrow from bar
    "\\Longmapsfrom" => "⟽",  # long leftwards double arrow from bar
    "\\Longmapsto" => "⟾",  # long rightwards double arrow from bar
    "\\longrightsquigarrow" => "⟿",  # long rightwards squiggle arrow
    "\\nvtwoheadrightarrow" => "⤀",  # rightwards two-headed arrow with vertical stroke
    "\\nVtwoheadrightarrow" => "⤁",  # rightwards two-headed arrow with double vertical stroke
    "\\nvLeftarrow" => "⤂",  # leftwards double arrow with vertical stroke
    "\\nvRightarrow" => "⤃",  # rightwards double arrow with vertical stroke
    "\\nvLeftrightarrow" => "⤄",  # left right double arrow with vertical stroke
    "\\twoheadmapsto" => "⤅",  # rightwards two-headed arrow from bar
    "\\downarrowbarred" => "⤈",  # downwards arrow with horizontal stroke
    "\\uparrowbarred" => "⤉",  # upwards arrow with horizontal stroke
    "\\leftbkarrow" => "⤌",  # leftwards double dash arrow
    "\\leftdbkarrow" => "⤎",  # leftwards triple dash arrow
    "\\rightdotarrow" => "⤑",  # rightwards arrow with dotted stem
    "\\nvrightarrowtail" => "⤔",  # rightwards arrow with tail with vertical stroke
    "\\nVrightarrowtail" => "⤕",  # rightwards arrow with tail with double vertical stroke
    "\\nvtwoheadrightarrowtail" => "⤗",  # rightwards two-headed arrow with tail with vertical stroke
    "\\nVtwoheadrightarrowtail" => "⤘",  # rightwards two-headed arrow with tail with double vertical stroke
    "\\diamondleftarrow" => "⤝",  # leftwards arrow to black diamond
    "\\rightarrowdiamond" => "⤞",  # rightwards arrow to black diamond
    "\\diamondleftarrowbar" => "⤟",  # leftwards arrow from bar to black diamond
    "\\barrightarrowdiamond" => "⤠",  # rightwards arrow from bar to black diamond
    "\\rightarrowplus" => "⥅",  # rightwards arrow with plus below
    "\\leftarrowplus" => "⥆",  # leftwards arrow with plus below
    "\\leftrightarrowcircle" => "⥈",  # left right arrow through small circle
    "\\twoheaduparrowcircle" => "⥉",  # upwards two-headed arrow from small circle
    "\\leftrightharpoonupdown" => "⥊",  # left barb up right barb down harpoon
    "\\leftrightharpoondownup" => "⥋",  # left barb down right barb up harpoon
    "\\updownharpoonrightleft" => "⥌",  # up barb right down barb left harpoon
    "\\updownharpoonleftright" => "⥍",  # up barb left down barb right harpoon
    "\\leftharpoonsupdown" => "⥢",  # leftwards harpoon with barb up above leftwards harpoon with barb down
    "\\upharpoonsleftright" => "⥣",  # upwards harpoon with barb left beside upwards harpoon with barb right
    "\\rightharpoonsupdown" => "⥤",  # rightwards harpoon with barb up above rightwards harpoon with barb down
    "\\downharpoonsleftright" => "⥥",  # downwards harpoon with barb left beside downwards harpoon with barb right
    "\\leftrightharpoonsup" => "⥦",  # leftwards harpoon with barb up above rightwards harpoon with barb up
    "\\leftrightharpoonsdown" => "⥧",  # leftwards harpoon with barb down above rightwards harpoon with barb down
    "\\rightleftharpoonsup" => "⥨",  # rightwards harpoon with barb up above leftwards harpoon with barb up
    "\\rightleftharpoonsdown" => "⥩",  # rightwards harpoon with barb down above leftwards harpoon with barb down
    "\\leftharpoonupdash" => "⥪",  # leftwards harpoon with barb up above long dash
    "\\dashleftharpoondown" => "⥫",  # leftwards harpoon with barb down below long dash
    "\\rightharpoonupdash" => "⥬",  # rightwards harpoon with barb up above long dash
    "\\dashrightharpoondown" => "⥭",  # rightwards harpoon with barb down below long dash
    "\\measuredangleleft" => "⦛",  # measured angle opening left
    "\\rightanglemdot" => "⦝",  # measured right angle with dot
    "\\angles" => "⦞",  # angle with s inside
    "\\angdnr" => "⦟",  # acute angle
    "\\sphericalangleup" => "⦡",  # spherical angle opening up
    "\\turnangle" => "⦢",  # turned angle
    "\\revangle" => "⦣",  # reversed angle
    "\\angleubar" => "⦤",  # angle with underbar
    "\\revangleubar" => "⦥",  # reversed angle with underbar
    "\\wideangledown" => "⦦",  # oblique angle opening up
    "\\wideangleup" => "⦧",  # oblique angle opening down
    "\\measanglerutone" => "⦨",  # measured angle with open arm ending in arrow pointing up and right
    "\\measanglelutonw" => "⦩",  # measured angle with open arm ending in arrow pointing up and left
    "\\measanglerdtose" => "⦪",  # measured angle with open arm ending in arrow pointing down and right
    "\\measangleldtosw" => "⦫",  # measured angle with open arm ending in arrow pointing down and left
    "\\measangleurtone" => "⦬",  # measured angle with open arm ending in arrow pointing right and up
    "\\measangleultonw" => "⦭",  # measured angle with open arm ending in arrow pointing left and up
    "\\measangledrtose" => "⦮",  # measured angle with open arm ending in arrow pointing right and down
    "\\measangledltosw" => "⦯",  # measured angle with open arm ending in arrow pointing left and down
    "\\revemptyset" => "⦰",  # reversed empty set
    "\\emptysetobar" => "⦱",  # empty set with overbar
    "\\emptysetocirc" => "⦲",  # empty set with small circle above
    "\\emptysetoarr" => "⦳",  # empty set with right arrow above
    "\\emptysetoarrl" => "⦴",  # empty set with left arrow above
    "\\circledparallel" => "⦷",  # circled parallel
    "\\odotslashdot" => "⦼",  # circled anticlockwise-rotated division sign
    "\\circledwhitebullet" => "⦾",  # circled white bullet
    "\\circledbullet" => "⦿",  # circled bullet
    "\\olessthan" => "⧀",  # circled less-than
    "\\ogreaterthan" => "⧁",  # circled greater-than
    "\\lrtriangleeq" => "⧡",  # increases as
    "\\eparsl" => "⧣",  # equals sign and slanted parallel
    "\\smeparsl" => "⧤",  # equals sign and slanted parallel with tilde above
    "\\eqvparsl" => "⧥",  # identical to and slanted parallel
    "\\dsol" => "⧶",  # solidus with overbar
    "\\rsolbar" => "⧷",  # reverse solidus with horizontal stroke
    "\\doubleplus" => "⧺",  # double plus
    "\\tripleplus" => "⧻",  # triple plus
    "\\modtwosum" => "⨊",  # modulo two sum
    "\\sumint" => "⨋",  # summation with integral
    "\\cirfnint" => "⨐",  # circulation function
    "\\awint" => "⨑",  # anticlockwise integration
    "\\rppolint" => "⨒",  # line integration with rectangular path around pole
    "\\scpolint" => "⨓",  # line integration with semicircular path around pole
    "\\npolint" => "⨔",  # line integration not including the pole
    "\\pointint" => "⨕",  # integral around a point operator
    "\\ringplus" => "⨢",  # plus sign with small circle above
    "\\plushat" => "⨣",  # plus sign with circumflex accent above
    "\\simplus" => "⨤",  # plus sign with tilde above
    "\\plussim" => "⨦",  # plus sign with tilde below
    "\\plussubtwo" => "⨧",  # plus sign with subscript two
    "\\plustrif" => "⨨",  # plus sign with black triangle
    "\\commaminus" => "⨩",  # minus sign with comma above
    "\\minusfdots" => "⨫",  # minus sign with falling dots
    "\\minusrdots" => "⨬",  # minus sign with rising dots
    "\\opluslhrim" => "⨭",  # plus sign in left half circle
    "\\oplusrhrim" => "⨮",  # plus sign in right half circle
    "\\dottimes" => "⨰",  # multiplication sign with dot above
    "\\timesbar" => "⨱",  # multiplication sign with underbar
    "\\smashtimes" => "⨳",  # smash product
    "\\otimeslhrim" => "⨴",  # multiplication sign in left half circle
    "\\otimesrhrim" => "⨵",  # multiplication sign in right half circle
    "\\otimeshat" => "⨶",  # circled multiplication sign with circumflex accent
    "\\Otimes" => "⨷",  # multiplication sign in double circle
    "\\odiv" => "⨸",  # circled division sign
    "\\triangleplus" => "⨹",  # plus sign in triangle
    "\\triangleminus" => "⨺",  # minus sign in triangle
    "\\triangletimes" => "⨻",  # multiplication sign in triangle
    "\\capdot" => "⩀",  # intersection with dot
    "\\uminus" => "⩁",  # union with minus sign
    "\\barcup" => "⩂",  # union with overbar
    "\\barcap" => "⩃",  # intersection with overbar
    "\\capwedge" => "⩄",  # intersection with logical and
    "\\cupvee" => "⩅",  # union with logical or
    "\\twocups" => "⩊",  # union beside and joined with union
    "\\twocaps" => "⩋",  # intersection beside and joined with intersection
    "\\closedvarcup" => "⩌",  # closed union with serifs
    "\\closedvarcap" => "⩍",  # closed intersection with serifs
    "\\Sqcap" => "⩎",  # double square intersection
    "\\Sqcup" => "⩏",  # double square union
    "\\closedvarcupsmashprod" => "⩐",  # closed union with serifs and smash product
    "\\wedgeodot" => "⩑",  # logical and with dot above
    "\\veeodot" => "⩒",  # logical or with dot above
    "\\wedgeonwedge" => "⩕",  # two intersecting logical and
    "\\bigslopedvee" => "⩗",  # sloping large or
    "\\bigslopedwedge" => "⩘",  # sloping large and
    "\\wedgemidvert" => "⩚",  # logical and with middle stem
    "\\veemidvert" => "⩛",  # logical or with middle stem
    "\\midbarwedge" => "⩜",  # ogical and with horizontal dash
    "\\midbarvee" => "⩝",  # logical or with horizontal dash
    "\\wedgedoublebar" => "⩠",  # logical and with double underbar
    "\\varveebar" => "⩡",  # small vee with underbar
    "\\doublebarvee" => "⩢",  # logical or with double overbar
    "\\veedoublebar" => "⩣",  # logical or with double underbar
    "\\eqdot" => "⩦",  # equals sign with dot below
    "\\dotequiv" => "⩧",  # identical with dot above
    "\\dotsim" => "⩪",  # tilde operator with dot above
    "\\simrdots" => "⩫",  # tilde operator with rising dots
    "\\simminussim" => "⩬",  # similar minus similar
    "\\congdot" => "⩭",  # congruent with dot above
    "\\asteq" => "⩮",  # equals with asterisk
    "\\hatapprox" => "⩯",  # almost equal to with circumflex accent
    "\\approxeqq" => "⩰",  # approximately equal or equal to
    "\\eqqplus" => "⩱",  # equals sign above plus sign
    "\\pluseqq" => "⩲",  # plus sign above equals sign
    "\\eqqsim" => "⩳",  # equals sign above tilde operator
    "\\Coloneq" => "⩴",  # double colon equal
    "\\eqeqeq" => "⩶",  # three consecutive equals signs
    "\\equivDD" => "⩸",  # equivalent with four dots above
    "\\ltcir" => "⩹",  # less-than with circle inside
    "\\gtcir" => "⩺",  # greater-than with circle inside
    "\\ltquest" => "⩻",  # less-than with question mark above
    "\\gtquest" => "⩼",  # greater-than with question mark above
    "\\lesdot" => "⩿",  # less-than or slanted equal to with dot inside
    "\\gesdot" => "⪀",  # greater-than or slanted equal to with dot inside
    "\\lesdoto" => "⪁",  # less-than or slanted equal to with dot above
    "\\gesdoto" => "⪂",  # greater-than or slanted equal to with dot above
    "\\lesdotor" => "⪃",  # less-than or slanted equal to with dot above right
    "\\gesdotol" => "⪄",  # greater-than or slanted equal to with dot above left
    "\\lsime" => "⪍",  # less-than above similar or equal
    "\\gsime" => "⪎",  # greater-than above similar or equal
    "\\lsimg" => "⪏",  # less-than above similar above greater-than
    "\\gsiml" => "⪐",  # greater-than above similar above less-than
    "\\lgE" => "⪑",  # less-than above greater-than above double-line equal
    "\\glE" => "⪒",  # greater-than above less-than above double-line equal
    "\\lesges" => "⪓",  # less-than above slanted equal above greater-than above slanted equal
    "\\gesles" => "⪔",  # greater-than above slanted equal above less-than above slanted equal
    "\\elsdot" => "⪗",  # slanted equal to or less-than with dot inside
    "\\egsdot" => "⪘",  # slanted equal to or greater-than with dot inside
    "\\eqqless" => "⪙",  # double-line equal to or less-than
    "\\eqqgtr" => "⪚",  # double-line equal to or greater-than
    "\\eqqslantless" => "⪛",  # double-line slanted equal to or less-than
    "\\eqqslantgtr" => "⪜",  # double-line slanted equal to or greater-than
    "\\simless" => "⪝",  # similar or less-than
    "\\simgtr" => "⪞",  # similar or greater-than
    "\\simlE" => "⪟",  # similar above less-than above equals sign
    "\\simgE" => "⪠",  # similar above greater-than above equals sign
    "\\glj" => "⪤",  # greater-than overlapping less-than
    "\\gla" => "⪥",  # greater-than beside less-than
    "\\ltcc" => "⪦",  # less-than closed by curve
    "\\gtcc" => "⪧",  # greater-than closed by curve
    "\\lescc" => "⪨",  # less-than closed by curve above slanted equal
    "\\gescc" => "⪩",  # greater-than closed by curve above slanted equal
    "\\smt" => "⪪",  # smaller than
    "\\lat" => "⪫",  # larger than
    "\\smte" => "⪬",  # smaller than or equal to
    "\\late" => "⪭",  # larger than or equal to
    "\\precneq" => "⪱",  # precedes above single-line not equal to
    "\\succneq" => "⪲",  # succeeds above single-line not equal to
    "\\preceqq" => "⪳",  # precedes above equals sign
    "\\succeqq" => "⪴",  # succeeds above equals sign
    "\\Prec" => "⪻",  # double precedes
    "\\Succ" => "⪼",  # double succeeds
    "\\subsetdot" => "⪽",  # subset with dot
    "\\supsetdot" => "⪾",  # superset with dot
    "\\subsetplus" => "⪿",  # subset with plus sign below
    "\\supsetplus" => "⫀",  # superset with plus sign below
    "\\submult" => "⫁",  # subset with multiplication sign below
    "\\supmult" => "⫂",  # superset with multiplication sign below
    "\\subedot" => "⫃",  # subset of or equal to with dot above
    "\\supedot" => "⫄",  # superset of or equal to with dot above
    "\\subsim" => "⫇",  # subset of above tilde operator
    "\\supsim" => "⫈",  # superset of above tilde operator
    "\\subsetapprox" => "⫉",  # subset of above almost equal to
    "\\supsetapprox" => "⫊",  # superset of above almost equal to
    "\\lsqhook" => "⫍",  # square left open box operator
    "\\rsqhook" => "⫎",  # square right open box operator
    "\\csub" => "⫏",  # closed subset
    "\\csup" => "⫐",  # closed superset
    "\\csube" => "⫑",  # closed subset or equal to
    "\\csupe" => "⫒",  # closed superset or equal to
    "\\subsup" => "⫓",  # subset above superset
    "\\supsub" => "⫔",  # superset above subset
    "\\subsub" => "⫕",  # subset above subset
    "\\supsup" => "⫖",  # superset above superset
    "\\suphsub" => "⫗",  # superset beside subset
    "\\supdsub" => "⫘",  # superset beside and joined by dash with subset
    "\\forkv" => "⫙",  # element of opening downwards
    "\\lllnest" => "⫷",  # stacked very much less-than
    "\\gggnest" => "⫸",  # stacked very much greater-than
    "\\leqqslant" => "⫹",  # double-line slanted less-than or equal to
    "\\geqqslant" => "⫺",  # double-line slanted greater-than or equal to
    "\\squaretopblack" => "\u2b12",  # square with top half black
    "\\squarebotblack" => "\u2b13",  # square with bottom half black
    "\\squareurblack" => "\u2b14",  # square with upper right diagonal half black
    "\\squarellblack" => "\u2b15",  # square with lower left diagonal half black
    "\\diamondleftblack" => "\u2b16",  # diamond with left half black
    "\\diamondrightblack" => "\u2b17",  # diamond with right half black
    "\\diamondtopblack" => "\u2b18",  # diamond with top half black
    "\\diamondbotblack" => "\u2b19",  # diamond with bottom half black
    "\\dottedsquare" => "\u2b1a",  # dotted square
    "\\lgblksquare" => "\u2b1b",  # black large square
    "\\lgwhtsquare" => "\u2b1c",  # white large square
    "\\vysmblksquare" => "\u2b1d",  # black very small square
    "\\vysmwhtsquare" => "\u2b1e",  # white very small square
    "\\pentagonblack" => "\u2b1f",  # black pentagon
    "\\pentagon" => "\u2b20",  # white pentagon
    "\\varhexagon" => "\u2b21",  # white hexagon
    "\\varhexagonblack" => "\u2b22",  # black hexagon
    "\\hexagonblack" => "\u2b23",  # horizontal black hexagon
    "\\lgblkcircle" => "\u2b24",  # black large circle
    "\\mdblkdiamond" => "\u2b25",  # black medium diamond
    "\\mdwhtdiamond" => "\u2b26",  # white medium diamond
    "\\mdblklozenge" => "\u2b27",  # black medium lozenge
    "\\mdwhtlozenge" => "\u2b28",  # white medium lozenge
    "\\smblkdiamond" => "\u2b29",  # black small diamond
    "\\smblklozenge" => "\u2b2a",  # black small lozenge
    "\\smwhtlozenge" => "\u2b2b",  # white small lozenge
    "\\blkhorzoval" => "\u2b2c",  # black horizontal ellipse
    "\\whthorzoval" => "\u2b2d",  # white horizontal ellipse
    "\\blkvertoval" => "\u2b2e",  # black vertical ellipse
    "\\whtvertoval" => "\u2b2f",  # white vertical ellipse
    "\\circleonleftarrow" => "\u2b30",  # left arrow with small circle
    "\\leftthreearrows" => "\u2b31",  # three leftwards arrows
    "\\leftarrowonoplus" => "\u2b32",  # left arrow with circled plus
    "\\longleftsquigarrow" => "\u2b33",  # long leftwards squiggle arrow
    "\\nvtwoheadleftarrow" => "\u2b34",  # leftwards two-headed arrow with vertical stroke
    "\\nVtwoheadleftarrow" => "\u2b35",  # leftwards two-headed arrow with double vertical stroke
    "\\twoheadmapsfrom" => "\u2b36",  # leftwards two-headed arrow from bar
    "\\twoheadleftdbkarrow" => "\u2b37",  # leftwards two-headed triple-dash arrow
    "\\leftdotarrow" => "\u2b38",  # leftwards arrow with dotted stem
    "\\nvleftarrowtail" => "\u2b39",  # leftwards arrow with tail with vertical stroke
    "\\nVleftarrowtail" => "\u2b3a",  # leftwards arrow with tail with double vertical stroke
    "\\twoheadleftarrowtail" => "\u2b3b",  # leftwards two-headed arrow with tail
    "\\nvtwoheadleftarrowtail" => "\u2b3c",  # leftwards two-headed arrow with tail with vertical stroke
    "\\nVtwoheadleftarrowtail" => "\u2b3d",  # leftwards two-headed arrow with tail with double vertical stroke
    "\\leftarrowx" => "\u2b3e",  # leftwards arrow through x
    "\\leftcurvedarrow" => "\u2b3f",  # wave arrow pointing directly left
    "\\equalleftarrow" => "\u2b40",  # equals sign above leftwards arrow
    "\\bsimilarleftarrow" => "\u2b41",  # reverse tilde operator above leftwards arrow
    "\\leftarrowbackapprox" => "\u2b42",  # leftwards arrow above reverse almost equal to
    "\\rightarrowgtr" => "\u2b43",  # rightwards arrow through greater-than
    "\\rightarrowsupset" => "\u2b44",  # rightwards arrow through subset
    "\\LLeftarrow" => "\u2b45",  # leftwards quadruple arrow
    "\\RRightarrow" => "\u2b46",  # rightwards quadruple arrow
    "\\bsimilarrightarrow" => "\u2b47",  # reverse tilde operator above rightwards arrow
    "\\rightarrowbackapprox" => "\u2b48",  # rightwards arrow above reverse almost equal to
    "\\similarleftarrow" => "\u2b49",  # tilde operator above leftwards arrow
    "\\leftarrowapprox" => "\u2b4a",  # leftwards arrow above almost equal to
    "\\leftarrowbsimilar" => "\u2b4b",  # leftwards arrow above reverse tilde operator
    "\\rightarrowbsimilar" => "\u2b4c",  # righttwards arrow above reverse tilde operator
    "\\medwhitestar" => "\u2b50",  # white medium star
    "\\medblackstar" => "\u2b51",  # black medium star
    "\\smwhitestar" => "\u2b52",  # white small star
    "\\rightpentagonblack" => "\u2b53",  # black right-pointing pentagon
    "\\rightpentagon" => "\u2b54",  # white right-pointing pentagon
    "\\postalmark" => "〒",  # postal mark
    "\\mbfA" => "𝐀",  # mathematical bold capital a
    "\\mbfB" => "𝐁",  # mathematical bold capital b
    "\\mbfC" => "𝐂",  # mathematical bold capital c
    "\\mbfD" => "𝐃",  # mathematical bold capital d
    "\\mbfE" => "𝐄",  # mathematical bold capital e
    "\\mbfF" => "𝐅",  # mathematical bold capital f
    "\\mbfG" => "𝐆",  # mathematical bold capital g
    "\\mbfH" => "𝐇",  # mathematical bold capital h
    "\\mbfI" => "𝐈",  # mathematical bold capital i
    "\\mbfJ" => "𝐉",  # mathematical bold capital j
    "\\mbfK" => "𝐊",  # mathematical bold capital k
    "\\mbfL" => "𝐋",  # mathematical bold capital l
    "\\mbfM" => "𝐌",  # mathematical bold capital m
    "\\mbfN" => "𝐍",  # mathematical bold capital n
    "\\mbfO" => "𝐎",  # mathematical bold capital o
    "\\mbfP" => "𝐏",  # mathematical bold capital p
    "\\mbfQ" => "𝐐",  # mathematical bold capital q
    "\\mbfR" => "𝐑",  # mathematical bold capital r
    "\\mbfS" => "𝐒",  # mathematical bold capital s
    "\\mbfT" => "𝐓",  # mathematical bold capital t
    "\\mbfU" => "𝐔",  # mathematical bold capital u
    "\\mbfV" => "𝐕",  # mathematical bold capital v
    "\\mbfW" => "𝐖",  # mathematical bold capital w
    "\\mbfX" => "𝐗",  # mathematical bold capital x
    "\\mbfY" => "𝐘",  # mathematical bold capital y
    "\\mbfZ" => "𝐙",  # mathematical bold capital z
    "\\mbfa" => "𝐚",  # mathematical bold small a
    "\\mbfb" => "𝐛",  # mathematical bold small b
    "\\mbfc" => "𝐜",  # mathematical bold small c
    "\\mbfd" => "𝐝",  # mathematical bold small d
    "\\mbfe" => "𝐞",  # mathematical bold small e
    "\\mbff" => "𝐟",  # mathematical bold small f
    "\\mbfg" => "𝐠",  # mathematical bold small g
    "\\mbfh" => "𝐡",  # mathematical bold small h
    "\\mbfi" => "𝐢",  # mathematical bold small i
    "\\mbfj" => "𝐣",  # mathematical bold small j
    "\\mbfk" => "𝐤",  # mathematical bold small k
    "\\mbfl" => "𝐥",  # mathematical bold small l
    "\\mbfm" => "𝐦",  # mathematical bold small m
    "\\mbfn" => "𝐧",  # mathematical bold small n
    "\\mbfo" => "𝐨",  # mathematical bold small o
    "\\mbfp" => "𝐩",  # mathematical bold small p
    "\\mbfq" => "𝐪",  # mathematical bold small q
    "\\mbfr" => "𝐫",  # mathematical bold small r
    "\\mbfs" => "𝐬",  # mathematical bold small s
    "\\mbft" => "𝐭",  # mathematical bold small t
    "\\mbfu" => "𝐮",  # mathematical bold small u
    "\\mbfv" => "𝐯",  # mathematical bold small v
    "\\mbfw" => "𝐰",  # mathematical bold small w
    "\\mbfx" => "𝐱",  # mathematical bold small x
    "\\mbfy" => "𝐲",  # mathematical bold small y
    "\\mbfz" => "𝐳",  # mathematical bold small z
    "\\mitA" => "𝐴",  # mathematical italic capital a
    "\\mitB" => "𝐵",  # mathematical italic capital b
    "\\mitC" => "𝐶",  # mathematical italic capital c
    "\\mitD" => "𝐷",  # mathematical italic capital d
    "\\mitE" => "𝐸",  # mathematical italic capital e
    "\\mitF" => "𝐹",  # mathematical italic capital f
    "\\mitG" => "𝐺",  # mathematical italic capital g
    "\\mitH" => "𝐻",  # mathematical italic capital h
    "\\mitI" => "𝐼",  # mathematical italic capital i
    "\\mitJ" => "𝐽",  # mathematical italic capital j
    "\\mitK" => "𝐾",  # mathematical italic capital k
    "\\mitL" => "𝐿",  # mathematical italic capital l
    "\\mitM" => "𝑀",  # mathematical italic capital m
    "\\mitN" => "𝑁",  # mathematical italic capital n
    "\\mitO" => "𝑂",  # mathematical italic capital o
    "\\mitP" => "𝑃",  # mathematical italic capital p
    "\\mitQ" => "𝑄",  # mathematical italic capital q
    "\\mitR" => "𝑅",  # mathematical italic capital r
    "\\mitS" => "𝑆",  # mathematical italic capital s
    "\\mitT" => "𝑇",  # mathematical italic capital t
    "\\mitU" => "𝑈",  # mathematical italic capital u
    "\\mitV" => "𝑉",  # mathematical italic capital v
    "\\mitW" => "𝑊",  # mathematical italic capital w
    "\\mitX" => "𝑋",  # mathematical italic capital x
    "\\mitY" => "𝑌",  # mathematical italic capital y
    "\\mitZ" => "𝑍",  # mathematical italic capital z
    "\\mita" => "𝑎",  # mathematical italic small a
    "\\mitb" => "𝑏",  # mathematical italic small b
    "\\mitc" => "𝑐",  # mathematical italic small c
    "\\mitd" => "𝑑",  # mathematical italic small d
    "\\mite" => "𝑒",  # mathematical italic small e
    "\\mitf" => "𝑓",  # mathematical italic small f
    "\\mitg" => "𝑔",  # mathematical italic small g
    "\\miti" => "𝑖",  # mathematical italic small i
    "\\mitj" => "𝑗",  # mathematical italic small j
    "\\mitk" => "𝑘",  # mathematical italic small k
    "\\mitl" => "𝑙",  # mathematical italic small l
    "\\mitm" => "𝑚",  # mathematical italic small m
    "\\mitn" => "𝑛",  # mathematical italic small n
    "\\mito" => "𝑜",  # mathematical italic small o
    "\\mitp" => "𝑝",  # mathematical italic small p
    "\\mitq" => "𝑞",  # mathematical italic small q
    "\\mitr" => "𝑟",  # mathematical italic small r
    "\\mits" => "𝑠",  # mathematical italic small s
    "\\mitt" => "𝑡",  # mathematical italic small t
    "\\mitu" => "𝑢",  # mathematical italic small u
    "\\mitv" => "𝑣",  # mathematical italic small v
    "\\mitw" => "𝑤",  # mathematical italic small w
    "\\mitx" => "𝑥",  # mathematical italic small x
    "\\mity" => "𝑦",  # mathematical italic small y
    "\\mitz" => "𝑧",  # mathematical italic small z
    "\\mbfitA" => "𝑨",  # mathematical bold italic capital a
    "\\mbfitB" => "𝑩",  # mathematical bold italic capital b
    "\\mbfitC" => "𝑪",  # mathematical bold italic capital c
    "\\mbfitD" => "𝑫",  # mathematical bold italic capital d
    "\\mbfitE" => "𝑬",  # mathematical bold italic capital e
    "\\mbfitF" => "𝑭",  # mathematical bold italic capital f
    "\\mbfitG" => "𝑮",  # mathematical bold italic capital g
    "\\mbfitH" => "𝑯",  # mathematical bold italic capital h
    "\\mbfitI" => "𝑰",  # mathematical bold italic capital i
    "\\mbfitJ" => "𝑱",  # mathematical bold italic capital j
    "\\mbfitK" => "𝑲",  # mathematical bold italic capital k
    "\\mbfitL" => "𝑳",  # mathematical bold italic capital l
    "\\mbfitM" => "𝑴",  # mathematical bold italic capital m
    "\\mbfitN" => "𝑵",  # mathematical bold italic capital n
    "\\mbfitO" => "𝑶",  # mathematical bold italic capital o
    "\\mbfitP" => "𝑷",  # mathematical bold italic capital p
    "\\mbfitQ" => "𝑸",  # mathematical bold italic capital q
    "\\mbfitR" => "𝑹",  # mathematical bold italic capital r
    "\\mbfitS" => "𝑺",  # mathematical bold italic capital s
    "\\mbfitT" => "𝑻",  # mathematical bold italic capital t
    "\\mbfitU" => "𝑼",  # mathematical bold italic capital u
    "\\mbfitV" => "𝑽",  # mathematical bold italic capital v
    "\\mbfitW" => "𝑾",  # mathematical bold italic capital w
    "\\mbfitX" => "𝑿",  # mathematical bold italic capital x
    "\\mbfitY" => "𝒀",  # mathematical bold italic capital y
    "\\mbfitZ" => "𝒁",  # mathematical bold italic capital z
    "\\mbfita" => "𝒂",  # mathematical bold italic small a
    "\\mbfitb" => "𝒃",  # mathematical bold italic small b
    "\\mbfitc" => "𝒄",  # mathematical bold italic small c
    "\\mbfitd" => "𝒅",  # mathematical bold italic small d
    "\\mbfite" => "𝒆",  # mathematical bold italic small e
    "\\mbfitf" => "𝒇",  # mathematical bold italic small f
    "\\mbfitg" => "𝒈",  # mathematical bold italic small g
    "\\mbfith" => "𝒉",  # mathematical bold italic small h
    "\\mbfiti" => "𝒊",  # mathematical bold italic small i
    "\\mbfitj" => "𝒋",  # mathematical bold italic small j
    "\\mbfitk" => "𝒌",  # mathematical bold italic small k
    "\\mbfitl" => "𝒍",  # mathematical bold italic small l
    "\\mbfitm" => "𝒎",  # mathematical bold italic small m
    "\\mbfitn" => "𝒏",  # mathematical bold italic small n
    "\\mbfito" => "𝒐",  # mathematical bold italic small o
    "\\mbfitp" => "𝒑",  # mathematical bold italic small p
    "\\mbfitq" => "𝒒",  # mathematical bold italic small q
    "\\mbfitr" => "𝒓",  # mathematical bold italic small r
    "\\mbfits" => "𝒔",  # mathematical bold italic small s
    "\\mbfitt" => "𝒕",  # mathematical bold italic small t
    "\\mbfitu" => "𝒖",  # mathematical bold italic small u
    "\\mbfitv" => "𝒗",  # mathematical bold italic small v
    "\\mbfitw" => "𝒘",  # mathematical bold italic small w
    "\\mbfitx" => "𝒙",  # mathematical bold italic small x
    "\\mbfity" => "𝒚",  # mathematical bold italic small y
    "\\mbfitz" => "𝒛",  # mathematical bold italic small z
    "\\mscrA" => "𝒜",  # mathematical script capital a
    "\\mscrC" => "𝒞",  # mathematical script capital c
    "\\mscrD" => "𝒟",  # mathematical script capital d
    "\\mscrG" => "𝒢",  # mathematical script capital g
    "\\mscrJ" => "𝒥",  # mathematical script capital j
    "\\mscrK" => "𝒦",  # mathematical script capital k
    "\\mscrN" => "𝒩",  # mathematical script capital n
    "\\mscrO" => "𝒪",  # mathematical script capital o
    "\\mscrP" => "𝒫",  # mathematical script capital p
    "\\mscrQ" => "𝒬",  # mathematical script capital q
    "\\mscrS" => "𝒮",  # mathematical script capital s
    "\\mscrT" => "𝒯",  # mathematical script capital t
    "\\mscrU" => "𝒰",  # mathematical script capital u
    "\\mscrV" => "𝒱",  # mathematical script capital v
    "\\mscrW" => "𝒲",  # mathematical script capital w
    "\\mscrX" => "𝒳",  # mathematical script capital x
    "\\mscrY" => "𝒴",  # mathematical script capital y
    "\\mscrZ" => "𝒵",  # mathematical script capital z
    "\\mscra" => "𝒶",  # mathematical script small a
    "\\mscrb" => "𝒷",  # mathematical script small b
    "\\mscrc" => "𝒸",  # mathematical script small c
    "\\mscrd" => "𝒹",  # mathematical script small d
    "\\mscrf" => "𝒻",  # mathematical script small f
    "\\mscrh" => "𝒽",  # mathematical script small h
    "\\mscri" => "𝒾",  # mathematical script small i
    "\\mscrj" => "𝒿",  # mathematical script small j
    "\\mscrk" => "𝓀",  # mathematical script small k
    "\\mscrl" => "\U1d4c1",  # mathematical script small l
    "\\mscrm" => "𝓂",  # mathematical script small m
    "\\mscrn" => "𝓃",  # mathematical script small n
    "\\mscrp" => "𝓅",  # mathematical script small p
    "\\mscrq" => "𝓆",  # mathematical script small q
    "\\mscrr" => "𝓇",  # mathematical script small r
    "\\mscrs" => "𝓈",  # mathematical script small s
    "\\mscrt" => "𝓉",  # mathematical script small t
    "\\mscru" => "𝓊",  # mathematical script small u
    "\\mscrv" => "𝓋",  # mathematical script small v
    "\\mscrw" => "𝓌",  # mathematical script small w
    "\\mscrx" => "𝓍",  # mathematical script small x
    "\\mscry" => "𝓎",  # mathematical script small y
    "\\mscrz" => "𝓏",  # mathematical script small z
    "\\mbfscrA" => "𝓐",  # mathematical bold script capital a
    "\\mbfscrB" => "𝓑",  # mathematical bold script capital b
    "\\mbfscrC" => "𝓒",  # mathematical bold script capital c
    "\\mbfscrD" => "𝓓",  # mathematical bold script capital d
    "\\mbfscrE" => "𝓔",  # mathematical bold script capital e
    "\\mbfscrF" => "𝓕",  # mathematical bold script capital f
    "\\mbfscrG" => "𝓖",  # mathematical bold script capital g
    "\\mbfscrH" => "𝓗",  # mathematical bold script capital h
    "\\mbfscrI" => "𝓘",  # mathematical bold script capital i
    "\\mbfscrJ" => "𝓙",  # mathematical bold script capital j
    "\\mbfscrK" => "𝓚",  # mathematical bold script capital k
    "\\mbfscrL" => "𝓛",  # mathematical bold script capital l
    "\\mbfscrM" => "𝓜",  # mathematical bold script capital m
    "\\mbfscrN" => "𝓝",  # mathematical bold script capital n
    "\\mbfscrO" => "𝓞",  # mathematical bold script capital o
    "\\mbfscrP" => "𝓟",  # mathematical bold script capital p
    "\\mbfscrQ" => "𝓠",  # mathematical bold script capital q
    "\\mbfscrR" => "𝓡",  # mathematical bold script capital r
    "\\mbfscrS" => "𝓢",  # mathematical bold script capital s
    "\\mbfscrT" => "𝓣",  # mathematical bold script capital t
    "\\mbfscrU" => "𝓤",  # mathematical bold script capital u
    "\\mbfscrV" => "𝓥",  # mathematical bold script capital v
    "\\mbfscrW" => "𝓦",  # mathematical bold script capital w
    "\\mbfscrX" => "𝓧",  # mathematical bold script capital x
    "\\mbfscrY" => "𝓨",  # mathematical bold script capital y
    "\\mbfscrZ" => "𝓩",  # mathematical bold script capital z
    "\\mbfscra" => "𝓪",  # mathematical bold script small a
    "\\mbfscrb" => "𝓫",  # mathematical bold script small b
    "\\mbfscrc" => "𝓬",  # mathematical bold script small c
    "\\mbfscrd" => "𝓭",  # mathematical bold script small d
    "\\mbfscre" => "𝓮",  # mathematical bold script small e
    "\\mbfscrf" => "𝓯",  # mathematical bold script small f
    "\\mbfscrg" => "𝓰",  # mathematical bold script small g
    "\\mbfscrh" => "𝓱",  # mathematical bold script small h
    "\\mbfscri" => "𝓲",  # mathematical bold script small i
    "\\mbfscrj" => "𝓳",  # mathematical bold script small j
    "\\mbfscrk" => "𝓴",  # mathematical bold script small k
    "\\mbfscrl" => "𝓵",  # mathematical bold script small l
    "\\mbfscrm" => "𝓶",  # mathematical bold script small m
    "\\mbfscrn" => "𝓷",  # mathematical bold script small n
    "\\mbfscro" => "𝓸",  # mathematical bold script small o
    "\\mbfscrp" => "𝓹",  # mathematical bold script small p
    "\\mbfscrq" => "𝓺",  # mathematical bold script small q
    "\\mbfscrr" => "𝓻",  # mathematical bold script small r
    "\\mbfscrs" => "𝓼",  # mathematical bold script small s
    "\\mbfscrt" => "𝓽",  # mathematical bold script small t
    "\\mbfscru" => "𝓾",  # mathematical bold script small u
    "\\mbfscrv" => "𝓿",  # mathematical bold script small v
    "\\mbfscrw" => "𝔀",  # mathematical bold script small w
    "\\mbfscrx" => "𝔁",  # mathematical bold script small x
    "\\mbfscry" => "𝔂",  # mathematical bold script small y
    "\\mbfscrz" => "𝔃",  # mathematical bold script small z
    "\\mfrakA" => "𝔄",  # mathematical fraktur capital a
    "\\mfrakB" => "𝔅",  # mathematical fraktur capital b
    "\\mfrakD" => "𝔇",  # mathematical fraktur capital d
    "\\mfrakE" => "𝔈",  # mathematical fraktur capital e
    "\\mfrakF" => "𝔉",  # mathematical fraktur capital f
    "\\mfrakG" => "𝔊",  # mathematical fraktur capital g
    "\\mfrakJ" => "𝔍",  # mathematical fraktur capital j
    "\\mfrakK" => "𝔎",  # mathematical fraktur capital k
    "\\mfrakL" => "𝔏",  # mathematical fraktur capital l
    "\\mfrakM" => "𝔐",  # mathematical fraktur capital m
    "\\mfrakN" => "𝔑",  # mathematical fraktur capital n
    "\\mfrakO" => "𝔒",  # mathematical fraktur capital o
    "\\mfrakP" => "𝔓",  # mathematical fraktur capital p
    "\\mfrakQ" => "𝔔",  # mathematical fraktur capital q
    "\\mfrakS" => "𝔖",  # mathematical fraktur capital s
    "\\mfrakT" => "𝔗",  # mathematical fraktur capital t
    "\\mfrakU" => "𝔘",  # mathematical fraktur capital u
    "\\mfrakV" => "𝔙",  # mathematical fraktur capital v
    "\\mfrakW" => "𝔚",  # mathematical fraktur capital w
    "\\mfrakX" => "𝔛",  # mathematical fraktur capital x
    "\\mfrakY" => "𝔜",  # mathematical fraktur capital y
    "\\mfraka" => "𝔞",  # mathematical fraktur small a
    "\\mfrakb" => "𝔟",  # mathematical fraktur small b
    "\\mfrakc" => "𝔠",  # mathematical fraktur small c
    "\\mfrakd" => "𝔡",  # mathematical fraktur small d
    "\\mfrake" => "𝔢",  # mathematical fraktur small e
    "\\mfrakf" => "𝔣",  # mathematical fraktur small f
    "\\mfrakg" => "𝔤",  # mathematical fraktur small g
    "\\mfrakh" => "𝔥",  # mathematical fraktur small h
    "\\mfraki" => "𝔦",  # mathematical fraktur small i
    "\\mfrakj" => "𝔧",  # mathematical fraktur small j
    "\\mfrakk" => "𝔨",  # mathematical fraktur small k
    "\\mfrakl" => "𝔩",  # mathematical fraktur small l
    "\\mfrakm" => "𝔪",  # mathematical fraktur small m
    "\\mfrakn" => "𝔫",  # mathematical fraktur small n
    "\\mfrako" => "𝔬",  # mathematical fraktur small o
    "\\mfrakp" => "𝔭",  # mathematical fraktur small p
    "\\mfrakq" => "𝔮",  # mathematical fraktur small q
    "\\mfrakr" => "𝔯",  # mathematical fraktur small r
    "\\mfraks" => "𝔰",  # mathematical fraktur small s
    "\\mfrakt" => "𝔱",  # mathematical fraktur small t
    "\\mfraku" => "𝔲",  # mathematical fraktur small u
    "\\mfrakv" => "𝔳",  # mathematical fraktur small v
    "\\mfrakw" => "𝔴",  # mathematical fraktur small w
    "\\mfrakx" => "𝔵",  # mathematical fraktur small x
    "\\mfraky" => "𝔶",  # mathematical fraktur small y
    "\\mfrakz" => "𝔷",  # mathematical fraktur small z
    "\\BbbA" => "𝔸",  # mathematical double-struck capital a
    "\\BbbB" => "𝔹",  # mathematical double-struck capital b
    "\\BbbD" => "𝔻",  # mathematical double-struck capital d
    "\\BbbE" => "𝔼",  # mathematical double-struck capital e
    "\\BbbF" => "𝔽",  # mathematical double-struck capital f
    "\\BbbG" => "𝔾",  # mathematical double-struck capital g
    "\\BbbI" => "𝕀",  # mathematical double-struck capital i
    "\\BbbJ" => "𝕁",  # mathematical double-struck capital j
    "\\BbbK" => "𝕂",  # mathematical double-struck capital k
    "\\BbbL" => "𝕃",  # mathematical double-struck capital l
    "\\BbbM" => "𝕄",  # mathematical double-struck capital m
    "\\BbbO" => "𝕆",  # mathematical double-struck capital o
    "\\BbbS" => "𝕊",  # mathematical double-struck capital s
    "\\BbbT" => "𝕋",  # mathematical double-struck capital t
    "\\BbbU" => "𝕌",  # mathematical double-struck capital u
    "\\BbbV" => "𝕍",  # mathematical double-struck capital v
    "\\BbbW" => "𝕎",  # mathematical double-struck capital w
    "\\BbbX" => "𝕏",  # mathematical double-struck capital x
    "\\BbbY" => "𝕐",  # mathematical double-struck capital y
    "\\Bbba" => "𝕒",  # mathematical double-struck small a
    "\\Bbbb" => "𝕓",  # mathematical double-struck small b
    "\\Bbbc" => "𝕔",  # mathematical double-struck small c
    "\\Bbbd" => "𝕕",  # mathematical double-struck small d
    "\\Bbbe" => "𝕖",  # mathematical double-struck small e
    "\\Bbbf" => "𝕗",  # mathematical double-struck small f
    "\\Bbbg" => "𝕘",  # mathematical double-struck small g
    "\\Bbbh" => "𝕙",  # mathematical double-struck small h
    "\\Bbbi" => "𝕚",  # mathematical double-struck small i
    "\\Bbbj" => "𝕛",  # mathematical double-struck small j
    "\\Bbbk" => "𝕜",  # mathematical double-struck small k
    "\\Bbbl" => "𝕝",  # mathematical double-struck small l
    "\\Bbbm" => "𝕞",  # mathematical double-struck small m
    "\\Bbbn" => "𝕟",  # mathematical double-struck small n
    "\\Bbbo" => "𝕠",  # mathematical double-struck small o
    "\\Bbbp" => "𝕡",  # mathematical double-struck small p
    "\\Bbbq" => "𝕢",  # mathematical double-struck small q
    "\\Bbbr" => "𝕣",  # mathematical double-struck small r
    "\\Bbbs" => "𝕤",  # mathematical double-struck small s
    "\\Bbbt" => "𝕥",  # mathematical double-struck small t
    "\\Bbbu" => "𝕦",  # mathematical double-struck small u
    "\\Bbbv" => "𝕧",  # mathematical double-struck small v
    "\\Bbbw" => "𝕨",  # mathematical double-struck small w
    "\\Bbbx" => "𝕩",  # mathematical double-struck small x
    "\\Bbby" => "𝕪",  # mathematical double-struck small y
    "\\Bbbz" => "𝕫",  # mathematical double-struck small z
    "\\mbffrakA" => "𝕬",  # mathematical bold fraktur capital a
    "\\mbffrakB" => "𝕭",  # mathematical bold fraktur capital b
    "\\mbffrakC" => "𝕮",  # mathematical bold fraktur capital c
    "\\mbffrakD" => "𝕯",  # mathematical bold fraktur capital d
    "\\mbffrakE" => "𝕰",  # mathematical bold fraktur capital e
    "\\mbffrakF" => "𝕱",  # mathematical bold fraktur capital f
    "\\mbffrakG" => "𝕲",  # mathematical bold fraktur capital g
    "\\mbffrakH" => "𝕳",  # mathematical bold fraktur capital h
    "\\mbffrakI" => "𝕴",  # mathematical bold fraktur capital i
    "\\mbffrakJ" => "𝕵",  # mathematical bold fraktur capital j
    "\\mbffrakK" => "𝕶",  # mathematical bold fraktur capital k
    "\\mbffrakL" => "𝕷",  # mathematical bold fraktur capital l
    "\\mbffrakM" => "𝕸",  # mathematical bold fraktur capital m
    "\\mbffrakN" => "𝕹",  # mathematical bold fraktur capital n
    "\\mbffrakO" => "𝕺",  # mathematical bold fraktur capital o
    "\\mbffrakP" => "𝕻",  # mathematical bold fraktur capital p
    "\\mbffrakQ" => "𝕼",  # mathematical bold fraktur capital q
    "\\mbffrakR" => "𝕽",  # mathematical bold fraktur capital r
    "\\mbffrakS" => "𝕾",  # mathematical bold fraktur capital s
    "\\mbffrakT" => "𝕿",  # mathematical bold fraktur capital t
    "\\mbffrakU" => "𝖀",  # mathematical bold fraktur capital u
    "\\mbffrakV" => "𝖁",  # mathematical bold fraktur capital v
    "\\mbffrakW" => "𝖂",  # mathematical bold fraktur capital w
    "\\mbffrakX" => "𝖃",  # mathematical bold fraktur capital x
    "\\mbffrakY" => "𝖄",  # mathematical bold fraktur capital y
    "\\mbffrakZ" => "𝖅",  # mathematical bold fraktur capital z
    "\\mbffraka" => "𝖆",  # mathematical bold fraktur small a
    "\\mbffrakb" => "𝖇",  # mathematical bold fraktur small b
    "\\mbffrakc" => "𝖈",  # mathematical bold fraktur small c
    "\\mbffrakd" => "𝖉",  # mathematical bold fraktur small d
    "\\mbffrake" => "𝖊",  # mathematical bold fraktur small e
    "\\mbffrakf" => "𝖋",  # mathematical bold fraktur small f
    "\\mbffrakg" => "𝖌",  # mathematical bold fraktur small g
    "\\mbffrakh" => "𝖍",  # mathematical bold fraktur small h
    "\\mbffraki" => "𝖎",  # mathematical bold fraktur small i
    "\\mbffrakj" => "𝖏",  # mathematical bold fraktur small j
    "\\mbffrakk" => "𝖐",  # mathematical bold fraktur small k
    "\\mbffrakl" => "𝖑",  # mathematical bold fraktur small l
    "\\mbffrakm" => "𝖒",  # mathematical bold fraktur small m
    "\\mbffrakn" => "𝖓",  # mathematical bold fraktur small n
    "\\mbffrako" => "𝖔",  # mathematical bold fraktur small o
    "\\mbffrakp" => "𝖕",  # mathematical bold fraktur small p
    "\\mbffrakq" => "𝖖",  # mathematical bold fraktur small q
    "\\mbffrakr" => "𝖗",  # mathematical bold fraktur small r
    "\\mbffraks" => "𝖘",  # mathematical bold fraktur small s
    "\\mbffrakt" => "𝖙",  # mathematical bold fraktur small t
    "\\mbffraku" => "𝖚",  # mathematical bold fraktur small u
    "\\mbffrakv" => "𝖛",  # mathematical bold fraktur small v
    "\\mbffrakw" => "𝖜",  # mathematical bold fraktur small w
    "\\mbffrakx" => "𝖝",  # mathematical bold fraktur small x
    "\\mbffraky" => "𝖞",  # mathematical bold fraktur small y
    "\\mbffrakz" => "𝖟",  # mathematical bold fraktur small z
    "\\msansA" => "𝖠",  # mathematical sans-serif capital a
    "\\msansB" => "𝖡",  # mathematical sans-serif capital b
    "\\msansC" => "𝖢",  # mathematical sans-serif capital c
    "\\msansD" => "𝖣",  # mathematical sans-serif capital d
    "\\msansE" => "𝖤",  # mathematical sans-serif capital e
    "\\msansF" => "𝖥",  # mathematical sans-serif capital f
    "\\msansG" => "𝖦",  # mathematical sans-serif capital g
    "\\msansH" => "𝖧",  # mathematical sans-serif capital h
    "\\msansI" => "𝖨",  # mathematical sans-serif capital i
    "\\msansJ" => "𝖩",  # mathematical sans-serif capital j
    "\\msansK" => "𝖪",  # mathematical sans-serif capital k
    "\\msansL" => "𝖫",  # mathematical sans-serif capital l
    "\\msansM" => "𝖬",  # mathematical sans-serif capital m
    "\\msansN" => "𝖭",  # mathematical sans-serif capital n
    "\\msansO" => "𝖮",  # mathematical sans-serif capital o
    "\\msansP" => "𝖯",  # mathematical sans-serif capital p
    "\\msansQ" => "𝖰",  # mathematical sans-serif capital q
    "\\msansR" => "𝖱",  # mathematical sans-serif capital r
    "\\msansS" => "𝖲",  # mathematical sans-serif capital s
    "\\msansT" => "𝖳",  # mathematical sans-serif capital t
    "\\msansU" => "𝖴",  # mathematical sans-serif capital u
    "\\msansV" => "𝖵",  # mathematical sans-serif capital v
    "\\msansW" => "𝖶",  # mathematical sans-serif capital w
    "\\msansX" => "𝖷",  # mathematical sans-serif capital x
    "\\msansY" => "𝖸",  # mathematical sans-serif capital y
    "\\msansZ" => "𝖹",  # mathematical sans-serif capital z
    "\\msansa" => "𝖺",  # mathematical sans-serif small a
    "\\msansb" => "𝖻",  # mathematical sans-serif small b
    "\\msansc" => "𝖼",  # mathematical sans-serif small c
    "\\msansd" => "𝖽",  # mathematical sans-serif small d
    "\\msanse" => "𝖾",  # mathematical sans-serif small e
    "\\msansf" => "𝖿",  # mathematical sans-serif small f
    "\\msansg" => "𝗀",  # mathematical sans-serif small g
    "\\msansh" => "𝗁",  # mathematical sans-serif small h
    "\\msansi" => "𝗂",  # mathematical sans-serif small i
    "\\msansj" => "𝗃",  # mathematical sans-serif small j
    "\\msansk" => "𝗄",  # mathematical sans-serif small k
    "\\msansl" => "𝗅",  # mathematical sans-serif small l
    "\\msansm" => "𝗆",  # mathematical sans-serif small m
    "\\msansn" => "𝗇",  # mathematical sans-serif small n
    "\\msanso" => "𝗈",  # mathematical sans-serif small o
    "\\msansp" => "𝗉",  # mathematical sans-serif small p
    "\\msansq" => "𝗊",  # mathematical sans-serif small q
    "\\msansr" => "𝗋",  # mathematical sans-serif small r
    "\\msanss" => "𝗌",  # mathematical sans-serif small s
    "\\msanst" => "𝗍",  # mathematical sans-serif small t
    "\\msansu" => "𝗎",  # mathematical sans-serif small u
    "\\msansv" => "𝗏",  # mathematical sans-serif small v
    "\\msansw" => "𝗐",  # mathematical sans-serif small w
    "\\msansx" => "𝗑",  # mathematical sans-serif small x
    "\\msansy" => "𝗒",  # mathematical sans-serif small y
    "\\msansz" => "𝗓",  # mathematical sans-serif small z
    "\\mbfsansA" => "𝗔",  # mathematical sans-serif bold capital a
    "\\mbfsansB" => "𝗕",  # mathematical sans-serif bold capital b
    "\\mbfsansC" => "𝗖",  # mathematical sans-serif bold capital c
    "\\mbfsansD" => "𝗗",  # mathematical sans-serif bold capital d
    "\\mbfsansE" => "𝗘",  # mathematical sans-serif bold capital e
    "\\mbfsansF" => "𝗙",  # mathematical sans-serif bold capital f
    "\\mbfsansG" => "𝗚",  # mathematical sans-serif bold capital g
    "\\mbfsansH" => "𝗛",  # mathematical sans-serif bold capital h
    "\\mbfsansI" => "𝗜",  # mathematical sans-serif bold capital i
    "\\mbfsansJ" => "𝗝",  # mathematical sans-serif bold capital j
    "\\mbfsansK" => "𝗞",  # mathematical sans-serif bold capital k
    "\\mbfsansL" => "𝗟",  # mathematical sans-serif bold capital l
    "\\mbfsansM" => "𝗠",  # mathematical sans-serif bold capital m
    "\\mbfsansN" => "𝗡",  # mathematical sans-serif bold capital n
    "\\mbfsansO" => "𝗢",  # mathematical sans-serif bold capital o
    "\\mbfsansP" => "𝗣",  # mathematical sans-serif bold capital p
    "\\mbfsansQ" => "𝗤",  # mathematical sans-serif bold capital q
    "\\mbfsansR" => "𝗥",  # mathematical sans-serif bold capital r
    "\\mbfsansS" => "𝗦",  # mathematical sans-serif bold capital s
    "\\mbfsansT" => "𝗧",  # mathematical sans-serif bold capital t
    "\\mbfsansU" => "𝗨",  # mathematical sans-serif bold capital u
    "\\mbfsansV" => "𝗩",  # mathematical sans-serif bold capital v
    "\\mbfsansW" => "𝗪",  # mathematical sans-serif bold capital w
    "\\mbfsansX" => "𝗫",  # mathematical sans-serif bold capital x
    "\\mbfsansY" => "𝗬",  # mathematical sans-serif bold capital y
    "\\mbfsansZ" => "𝗭",  # mathematical sans-serif bold capital z
    "\\mbfsansa" => "𝗮",  # mathematical sans-serif bold small a
    "\\mbfsansb" => "𝗯",  # mathematical sans-serif bold small b
    "\\mbfsansc" => "𝗰",  # mathematical sans-serif bold small c
    "\\mbfsansd" => "𝗱",  # mathematical sans-serif bold small d
    "\\mbfsanse" => "𝗲",  # mathematical sans-serif bold small e
    "\\mbfsansf" => "𝗳",  # mathematical sans-serif bold small f
    "\\mbfsansg" => "𝗴",  # mathematical sans-serif bold small g
    "\\mbfsansh" => "𝗵",  # mathematical sans-serif bold small h
    "\\mbfsansi" => "𝗶",  # mathematical sans-serif bold small i
    "\\mbfsansj" => "𝗷",  # mathematical sans-serif bold small j
    "\\mbfsansk" => "𝗸",  # mathematical sans-serif bold small k
    "\\mbfsansl" => "𝗹",  # mathematical sans-serif bold small l
    "\\mbfsansm" => "𝗺",  # mathematical sans-serif bold small m
    "\\mbfsansn" => "𝗻",  # mathematical sans-serif bold small n
    "\\mbfsanso" => "𝗼",  # mathematical sans-serif bold small o
    "\\mbfsansp" => "𝗽",  # mathematical sans-serif bold small p
    "\\mbfsansq" => "𝗾",  # mathematical sans-serif bold small q
    "\\mbfsansr" => "𝗿",  # mathematical sans-serif bold small r
    "\\mbfsanss" => "𝘀",  # mathematical sans-serif bold small s
    "\\mbfsanst" => "𝘁",  # mathematical sans-serif bold small t
    "\\mbfsansu" => "𝘂",  # mathematical sans-serif bold small u
    "\\mbfsansv" => "𝘃",  # mathematical sans-serif bold small v
    "\\mbfsansw" => "𝘄",  # mathematical sans-serif bold small w
    "\\mbfsansx" => "𝘅",  # mathematical sans-serif bold small x
    "\\mbfsansy" => "𝘆",  # mathematical sans-serif bold small y
    "\\mbfsansz" => "𝘇",  # mathematical sans-serif bold small z
    "\\mitsansA" => "𝘈",  # mathematical sans-serif italic capital a
    "\\mitsansB" => "𝘉",  # mathematical sans-serif italic capital b
    "\\mitsansC" => "𝘊",  # mathematical sans-serif italic capital c
    "\\mitsansD" => "𝘋",  # mathematical sans-serif italic capital d
    "\\mitsansE" => "𝘌",  # mathematical sans-serif italic capital e
    "\\mitsansF" => "𝘍",  # mathematical sans-serif italic capital f
    "\\mitsansG" => "𝘎",  # mathematical sans-serif italic capital g
    "\\mitsansH" => "𝘏",  # mathematical sans-serif italic capital h
    "\\mitsansI" => "𝘐",  # mathematical sans-serif italic capital i
    "\\mitsansJ" => "𝘑",  # mathematical sans-serif italic capital j
    "\\mitsansK" => "𝘒",  # mathematical sans-serif italic capital k
    "\\mitsansL" => "𝘓",  # mathematical sans-serif italic capital l
    "\\mitsansM" => "𝘔",  # mathematical sans-serif italic capital m
    "\\mitsansN" => "𝘕",  # mathematical sans-serif italic capital n
    "\\mitsansO" => "𝘖",  # mathematical sans-serif italic capital o
    "\\mitsansP" => "𝘗",  # mathematical sans-serif italic capital p
    "\\mitsansQ" => "𝘘",  # mathematical sans-serif italic capital q
    "\\mitsansR" => "𝘙",  # mathematical sans-serif italic capital r
    "\\mitsansS" => "𝘚",  # mathematical sans-serif italic capital s
    "\\mitsansT" => "𝘛",  # mathematical sans-serif italic capital t
    "\\mitsansU" => "𝘜",  # mathematical sans-serif italic capital u
    "\\mitsansV" => "𝘝",  # mathematical sans-serif italic capital v
    "\\mitsansW" => "𝘞",  # mathematical sans-serif italic capital w
    "\\mitsansX" => "𝘟",  # mathematical sans-serif italic capital x
    "\\mitsansY" => "𝘠",  # mathematical sans-serif italic capital y
    "\\mitsansZ" => "𝘡",  # mathematical sans-serif italic capital z
    "\\mitsansa" => "𝘢",  # mathematical sans-serif italic small a
    "\\mitsansb" => "𝘣",  # mathematical sans-serif italic small b
    "\\mitsansc" => "𝘤",  # mathematical sans-serif italic small c
    "\\mitsansd" => "𝘥",  # mathematical sans-serif italic small d
    "\\mitsanse" => "𝘦",  # mathematical sans-serif italic small e
    "\\mitsansf" => "𝘧",  # mathematical sans-serif italic small f
    "\\mitsansg" => "𝘨",  # mathematical sans-serif italic small g
    "\\mitsansh" => "𝘩",  # mathematical sans-serif italic small h
    "\\mitsansi" => "𝘪",  # mathematical sans-serif italic small i
    "\\mitsansj" => "𝘫",  # mathematical sans-serif italic small j
    "\\mitsansk" => "𝘬",  # mathematical sans-serif italic small k
    "\\mitsansl" => "𝘭",  # mathematical sans-serif italic small l
    "\\mitsansm" => "𝘮",  # mathematical sans-serif italic small m
    "\\mitsansn" => "𝘯",  # mathematical sans-serif italic small n
    "\\mitsanso" => "𝘰",  # mathematical sans-serif italic small o
    "\\mitsansp" => "𝘱",  # mathematical sans-serif italic small p
    "\\mitsansq" => "𝘲",  # mathematical sans-serif italic small q
    "\\mitsansr" => "𝘳",  # mathematical sans-serif italic small r
    "\\mitsanss" => "𝘴",  # mathematical sans-serif italic small s
    "\\mitsanst" => "𝘵",  # mathematical sans-serif italic small t
    "\\mitsansu" => "𝘶",  # mathematical sans-serif italic small u
    "\\mitsansv" => "𝘷",  # mathematical sans-serif italic small v
    "\\mitsansw" => "𝘸",  # mathematical sans-serif italic small w
    "\\mitsansx" => "𝘹",  # mathematical sans-serif italic small x
    "\\mitsansy" => "𝘺",  # mathematical sans-serif italic small y
    "\\mitsansz" => "𝘻",  # mathematical sans-serif italic small z
    "\\mbfitsansA" => "𝘼",  # mathematical sans-serif bold italic capital a
    "\\mbfitsansB" => "𝘽",  # mathematical sans-serif bold italic capital b
    "\\mbfitsansC" => "𝘾",  # mathematical sans-serif bold italic capital c
    "\\mbfitsansD" => "𝘿",  # mathematical sans-serif bold italic capital d
    "\\mbfitsansE" => "𝙀",  # mathematical sans-serif bold italic capital e
    "\\mbfitsansF" => "𝙁",  # mathematical sans-serif bold italic capital f
    "\\mbfitsansG" => "𝙂",  # mathematical sans-serif bold italic capital g
    "\\mbfitsansH" => "𝙃",  # mathematical sans-serif bold italic capital h
    "\\mbfitsansI" => "𝙄",  # mathematical sans-serif bold italic capital i
    "\\mbfitsansJ" => "𝙅",  # mathematical sans-serif bold italic capital j
    "\\mbfitsansK" => "𝙆",  # mathematical sans-serif bold italic capital k
    "\\mbfitsansL" => "𝙇",  # mathematical sans-serif bold italic capital l
    "\\mbfitsansM" => "𝙈",  # mathematical sans-serif bold italic capital m
    "\\mbfitsansN" => "𝙉",  # mathematical sans-serif bold italic capital n
    "\\mbfitsansO" => "𝙊",  # mathematical sans-serif bold italic capital o
    "\\mbfitsansP" => "𝙋",  # mathematical sans-serif bold italic capital p
    "\\mbfitsansQ" => "𝙌",  # mathematical sans-serif bold italic capital q
    "\\mbfitsansR" => "𝙍",  # mathematical sans-serif bold italic capital r
    "\\mbfitsansS" => "𝙎",  # mathematical sans-serif bold italic capital s
    "\\mbfitsansT" => "𝙏",  # mathematical sans-serif bold italic capital t
    "\\mbfitsansU" => "𝙐",  # mathematical sans-serif bold italic capital u
    "\\mbfitsansV" => "𝙑",  # mathematical sans-serif bold italic capital v
    "\\mbfitsansW" => "𝙒",  # mathematical sans-serif bold italic capital w
    "\\mbfitsansX" => "𝙓",  # mathematical sans-serif bold italic capital x
    "\\mbfitsansY" => "𝙔",  # mathematical sans-serif bold italic capital y
    "\\mbfitsansZ" => "𝙕",  # mathematical sans-serif bold italic capital z
    "\\mbfitsansa" => "𝙖",  # mathematical sans-serif bold italic small a
    "\\mbfitsansb" => "𝙗",  # mathematical sans-serif bold italic small b
    "\\mbfitsansc" => "𝙘",  # mathematical sans-serif bold italic small c
    "\\mbfitsansd" => "𝙙",  # mathematical sans-serif bold italic small d
    "\\mbfitsanse" => "𝙚",  # mathematical sans-serif bold italic small e
    "\\mbfitsansf" => "𝙛",  # mathematical sans-serif bold italic small f
    "\\mbfitsansg" => "𝙜",  # mathematical sans-serif bold italic small g
    "\\mbfitsansh" => "𝙝",  # mathematical sans-serif bold italic small h
    "\\mbfitsansi" => "𝙞",  # mathematical sans-serif bold italic small i
    "\\mbfitsansj" => "𝙟",  # mathematical sans-serif bold italic small j
    "\\mbfitsansk" => "𝙠",  # mathematical sans-serif bold italic small k
    "\\mbfitsansl" => "𝙡",  # mathematical sans-serif bold italic small l
    "\\mbfitsansm" => "𝙢",  # mathematical sans-serif bold italic small m
    "\\mbfitsansn" => "𝙣",  # mathematical sans-serif bold italic small n
    "\\mbfitsanso" => "𝙤",  # mathematical sans-serif bold italic small o
    "\\mbfitsansp" => "𝙥",  # mathematical sans-serif bold italic small p
    "\\mbfitsansq" => "𝙦",  # mathematical sans-serif bold italic small q
    "\\mbfitsansr" => "𝙧",  # mathematical sans-serif bold italic small r
    "\\mbfitsanss" => "𝙨",  # mathematical sans-serif bold italic small s
    "\\mbfitsanst" => "𝙩",  # mathematical sans-serif bold italic small t
    "\\mbfitsansu" => "𝙪",  # mathematical sans-serif bold italic small u
    "\\mbfitsansv" => "𝙫",  # mathematical sans-serif bold italic small v
    "\\mbfitsansw" => "𝙬",  # mathematical sans-serif bold italic small w
    "\\mbfitsansx" => "𝙭",  # mathematical sans-serif bold italic small x
    "\\mbfitsansy" => "𝙮",  # mathematical sans-serif bold italic small y
    "\\mbfitsansz" => "𝙯",  # mathematical sans-serif bold italic small z
    "\\mttA" => "𝙰",  # mathematical monospace capital a
    "\\mttB" => "𝙱",  # mathematical monospace capital b
    "\\mttC" => "𝙲",  # mathematical monospace capital c
    "\\mttD" => "𝙳",  # mathematical monospace capital d
    "\\mttE" => "𝙴",  # mathematical monospace capital e
    "\\mttF" => "𝙵",  # mathematical monospace capital f
    "\\mttG" => "𝙶",  # mathematical monospace capital g
    "\\mttH" => "𝙷",  # mathematical monospace capital h
    "\\mttI" => "𝙸",  # mathematical monospace capital i
    "\\mttJ" => "𝙹",  # mathematical monospace capital j
    "\\mttK" => "𝙺",  # mathematical monospace capital k
    "\\mttL" => "𝙻",  # mathematical monospace capital l
    "\\mttM" => "𝙼",  # mathematical monospace capital m
    "\\mttN" => "𝙽",  # mathematical monospace capital n
    "\\mttO" => "𝙾",  # mathematical monospace capital o
    "\\mttP" => "𝙿",  # mathematical monospace capital p
    "\\mttQ" => "𝚀",  # mathematical monospace capital q
    "\\mttR" => "𝚁",  # mathematical monospace capital r
    "\\mttS" => "𝚂",  # mathematical monospace capital s
    "\\mttT" => "𝚃",  # mathematical monospace capital t
    "\\mttU" => "𝚄",  # mathematical monospace capital u
    "\\mttV" => "𝚅",  # mathematical monospace capital v
    "\\mttW" => "𝚆",  # mathematical monospace capital w
    "\\mttX" => "𝚇",  # mathematical monospace capital x
    "\\mttY" => "𝚈",  # mathematical monospace capital y
    "\\mttZ" => "𝚉",  # mathematical monospace capital z
    "\\mtta" => "𝚊",  # mathematical monospace small a
    "\\mttb" => "𝚋",  # mathematical monospace small b
    "\\mttc" => "𝚌",  # mathematical monospace small c
    "\\mttd" => "𝚍",  # mathematical monospace small d
    "\\mtte" => "𝚎",  # mathematical monospace small e
    "\\mttf" => "𝚏",  # mathematical monospace small f
    "\\mttg" => "𝚐",  # mathematical monospace small g
    "\\mtth" => "𝚑",  # mathematical monospace small h
    "\\mtti" => "𝚒",  # mathematical monospace small i
    "\\mttj" => "𝚓",  # mathematical monospace small j
    "\\mttk" => "𝚔",  # mathematical monospace small k
    "\\mttl" => "𝚕",  # mathematical monospace small l
    "\\mttm" => "𝚖",  # mathematical monospace small m
    "\\mttn" => "𝚗",  # mathematical monospace small n
    "\\mtto" => "𝚘",  # mathematical monospace small o
    "\\mttp" => "𝚙",  # mathematical monospace small p
    "\\mttq" => "𝚚",  # mathematical monospace small q
    "\\mttr" => "𝚛",  # mathematical monospace small r
    "\\mtts" => "𝚜",  # mathematical monospace small s
    "\\mttt" => "𝚝",  # mathematical monospace small t
    "\\mttu" => "𝚞",  # mathematical monospace small u
    "\\mttv" => "𝚟",  # mathematical monospace small v
    "\\mttw" => "𝚠",  # mathematical monospace small w
    "\\mttx" => "𝚡",  # mathematical monospace small x
    "\\mtty" => "𝚢",  # mathematical monospace small y
    "\\mttz" => "𝚣",  # mathematical monospace small z
    "\\imath" => "\U1d6a4",  # mathematical italic small dotless i
    "\\jmath" => "\U1d6a5",  # mathematical italic small dotless j
    "\\mbfAlpha" => "𝚨",  # mathematical bold capital alpha
    "\\mbfBeta" => "𝚩",  # mathematical bold capital beta
    "\\mbfGamma" => "𝚪",  # mathematical bold capital gamma
    "\\mbfDelta" => "𝚫",  # mathematical bold capital delta
    "\\mbfEpsilon" => "𝚬",  # mathematical bold capital epsilon
    "\\mbfZeta" => "𝚭",  # mathematical bold capital zeta
    "\\mbfEta" => "𝚮",  # mathematical bold capital eta
    "\\mbfTheta" => "𝚯",  # mathematical bold capital theta
    "\\mbfIota" => "𝚰",  # mathematical bold capital iota
    "\\mbfKappa" => "𝚱",  # mathematical bold capital kappa
    "\\mbfLambda" => "𝚲",  # mathematical bold capital lambda
    "\\mbfMu" => "𝚳",  # mathematical bold capital mu
    "\\mbfNu" => "𝚴",  # mathematical bold capital nu
    "\\mbfXi" => "𝚵",  # mathematical bold capital xi
    "\\mbfOmicron" => "𝚶",  # mathematical bold capital omicron
    "\\mbfPi" => "𝚷",  # mathematical bold capital pi
    "\\mbfRho" => "𝚸",  # mathematical bold capital rho
    "\\mbfvarTheta" => "𝚹",  # mathematical bold capital theta symbol
    "\\mbfSigma" => "𝚺",  # mathematical bold capital sigma
    "\\mbfTau" => "𝚻",  # mathematical bold capital tau
    "\\mbfUpsilon" => "𝚼",  # mathematical bold capital upsilon
    "\\mbfPhi" => "𝚽",  # mathematical bold capital phi
    "\\mbfChi" => "𝚾",  # mathematical bold capital chi
    "\\mbfPsi" => "𝚿",  # mathematical bold capital psi
    "\\mbfOmega" => "𝛀",  # mathematical bold capital omega
    "\\mbfnabla" => "𝛁",  # mathematical bold nabla
    "\\mbfalpha" => "𝛂",  # mathematical bold small alpha
    "\\mbfbeta" => "𝛃",  # mathematical bold small beta
    "\\mbfgamma" => "𝛄",  # mathematical bold small gamma
    "\\mbfdelta" => "𝛅",  # mathematical bold small delta
    "\\mbfepsilon" => "𝛆",  # mathematical bold small epsilon
    "\\mbfzeta" => "𝛇",  # mathematical bold small zeta
    "\\mbfeta" => "𝛈",  # mathematical bold small eta
    "\\mbftheta" => "𝛉",  # mathematical bold small theta
    "\\mbfiota" => "𝛊",  # mathematical bold small iota
    "\\mbfkappa" => "𝛋",  # mathematical bold small kappa
    "\\mbflambda" => "𝛌",  # mathematical bold small lambda
    "\\mbfmu" => "𝛍",  # mathematical bold small mu
    "\\mbfnu" => "𝛎",  # mathematical bold small nu
    "\\mbfxi" => "𝛏",  # mathematical bold small xi
    "\\mbfomicron" => "𝛐",  # mathematical bold small omicron
    "\\mbfpi" => "𝛑",  # mathematical bold small pi
    "\\mbfrho" => "𝛒",  # mathematical bold small rho
    "\\mbfvarsigma" => "𝛓",  # mathematical bold small final sigma
    "\\mbfsigma" => "𝛔",  # mathematical bold small sigma
    "\\mbftau" => "𝛕",  # mathematical bold small tau
    "\\mbfupsilon" => "𝛖",  # mathematical bold small upsilon
    "\\mbfvarphi" => "𝛗",  # mathematical bold small phi
    "\\mbfchi" => "𝛘",  # mathematical bold small chi
    "\\mbfpsi" => "𝛙",  # mathematical bold small psi
    "\\mbfomega" => "𝛚",  # mathematical bold small omega
    "\\mbfpartial" => "𝛛",  # mathematical bold partial differential
    "\\mbfvarepsilon" => "𝛜",  # mathematical bold epsilon symbol
    "\\mbfvartheta" => "𝛝",  # mathematical bold theta symbol
    "\\mbfvarkappa" => "𝛞",  # mathematical bold kappa symbol
    "\\mbfphi" => "𝛟",  # mathematical bold phi symbol
    "\\mbfvarrho" => "𝛠",  # mathematical bold rho symbol
    "\\mbfvarpi" => "𝛡",  # mathematical bold pi symbol
    "\\mitAlpha" => "𝛢",  # mathematical italic capital alpha
    "\\mitBeta" => "𝛣",  # mathematical italic capital beta
    "\\mitGamma" => "𝛤",  # mathematical italic capital gamma
    "\\mitDelta" => "𝛥",  # mathematical italic capital delta
    "\\mitEpsilon" => "𝛦",  # mathematical italic capital epsilon
    "\\mitZeta" => "𝛧",  # mathematical italic capital zeta
    "\\mitEta" => "𝛨",  # mathematical italic capital eta
    "\\mitTheta" => "𝛩",  # mathematical italic capital theta
    "\\mitIota" => "𝛪",  # mathematical italic capital iota
    "\\mitKappa" => "𝛫",  # mathematical italic capital kappa
    "\\mitLambda" => "𝛬",  # mathematical italic capital lambda
    "\\mitMu" => "𝛭",  # mathematical italic capital mu
    "\\mitNu" => "𝛮",  # mathematical italic capital nu
    "\\mitXi" => "𝛯",  # mathematical italic capital xi
    "\\mitOmicron" => "𝛰",  # mathematical italic capital omicron
    "\\mitPi" => "𝛱",  # mathematical italic capital pi
    "\\mitRho" => "𝛲",  # mathematical italic capital rho
    "\\mitvarTheta" => "𝛳",  # mathematical italic capital theta symbol
    "\\mitSigma" => "𝛴",  # mathematical italic capital sigma
    "\\mitTau" => "𝛵",  # mathematical italic capital tau
    "\\mitUpsilon" => "𝛶",  # mathematical italic capital upsilon
    "\\mitPhi" => "𝛷",  # mathematical italic capital phi
    "\\mitChi" => "𝛸",  # mathematical italic capital chi
    "\\mitPsi" => "𝛹",  # mathematical italic capital psi
    "\\mitOmega" => "𝛺",  # mathematical italic capital omega
    "\\mitnabla" => "𝛻",  # mathematical italic nabla
    "\\mitalpha" => "𝛼",  # mathematical italic small alpha
    "\\mitbeta" => "𝛽",  # mathematical italic small beta
    "\\mitgamma" => "𝛾",  # mathematical italic small gamma
    "\\mitdelta" => "𝛿",  # mathematical italic small delta
    "\\mitepsilon" => "𝜀",  # mathematical italic small epsilon
    "\\mitzeta" => "𝜁",  # mathematical italic small zeta
    "\\miteta" => "𝜂",  # mathematical italic small eta
    "\\mittheta" => "𝜃",  # mathematical italic small theta
    "\\mitiota" => "𝜄",  # mathematical italic small iota
    "\\mitkappa" => "𝜅",  # mathematical italic small kappa
    "\\mitlambda" => "𝜆",  # mathematical italic small lambda
    "\\mitmu" => "𝜇",  # mathematical italic small mu
    "\\mitnu" => "𝜈",  # mathematical italic small nu
    "\\mitxi" => "𝜉",  # mathematical italic small xi
    "\\mitomicron" => "𝜊",  # mathematical italic small omicron
    "\\mitpi" => "𝜋",  # mathematical italic small pi
    "\\mitrho" => "𝜌",  # mathematical italic small rho
    "\\mitvarsigma" => "𝜍",  # mathematical italic small final sigma
    "\\mitsigma" => "𝜎",  # mathematical italic small sigma
    "\\mittau" => "𝜏",  # mathematical italic small tau
    "\\mitupsilon" => "𝜐",  # mathematical italic small upsilon
    "\\mitphi" => "𝜑",  # mathematical italic small phi
    "\\mitchi" => "𝜒",  # mathematical italic small chi
    "\\mitpsi" => "𝜓",  # mathematical italic small psi
    "\\mitomega" => "𝜔",  # mathematical italic small omega
    "\\mitpartial" => "𝜕",  # mathematical italic partial differential
    "\\mitvarepsilon" => "𝜖",  # mathematical italic epsilon symbol
    "\\mitvartheta" => "𝜗",  # mathematical italic theta symbol
    "\\mitvarkappa" => "𝜘",  # mathematical italic kappa symbol
    "\\mitvarphi" => "𝜙",  # mathematical italic phi symbol
    "\\mitvarrho" => "𝜚",  # mathematical italic rho symbol
    "\\mitvarpi" => "𝜛",  # mathematical italic pi symbol
    "\\mbfitAlpha" => "𝜜",  # mathematical bold italic capital alpha
    "\\mbfitBeta" => "𝜝",  # mathematical bold italic capital beta
    "\\mbfitGamma" => "𝜞",  # mathematical bold italic capital gamma
    "\\mbfitDelta" => "𝜟",  # mathematical bold italic capital delta
    "\\mbfitEpsilon" => "𝜠",  # mathematical bold italic capital epsilon
    "\\mbfitZeta" => "𝜡",  # mathematical bold italic capital zeta
    "\\mbfitEta" => "𝜢",  # mathematical bold italic capital eta
    "\\mbfitTheta" => "𝜣",  # mathematical bold italic capital theta
    "\\mbfitIota" => "𝜤",  # mathematical bold italic capital iota
    "\\mbfitKappa" => "𝜥",  # mathematical bold italic capital kappa
    "\\mbfitLambda" => "𝜦",  # mathematical bold italic capital lambda
    "\\mbfitMu" => "𝜧",  # mathematical bold italic capital mu
    "\\mbfitNu" => "𝜨",  # mathematical bold italic capital nu
    "\\mbfitXi" => "𝜩",  # mathematical bold italic capital xi
    "\\mbfitOmicron" => "𝜪",  # mathematical bold italic capital omicron
    "\\mbfitPi" => "𝜫",  # mathematical bold italic capital pi
    "\\mbfitRho" => "𝜬",  # mathematical bold italic capital rho
    "\\mbfitvarTheta" => "𝜭",  # mathematical bold italic capital theta symbol
    "\\mbfitSigma" => "𝜮",  # mathematical bold italic capital sigma
    "\\mbfitTau" => "𝜯",  # mathematical bold italic capital tau
    "\\mbfitUpsilon" => "𝜰",  # mathematical bold italic capital upsilon
    "\\mbfitPhi" => "𝜱",  # mathematical bold italic capital phi
    "\\mbfitChi" => "𝜲",  # mathematical bold italic capital chi
    "\\mbfitPsi" => "𝜳",  # mathematical bold italic capital psi
    "\\mbfitOmega" => "𝜴",  # mathematical bold italic capital omega
    "\\mbfitnabla" => "𝜵",  # mathematical bold italic nabla
    "\\mbfitalpha" => "𝜶",  # mathematical bold italic small alpha
    "\\mbfitbeta" => "𝜷",  # mathematical bold italic small beta
    "\\mbfitgamma" => "𝜸",  # mathematical bold italic small gamma
    "\\mbfitdelta" => "𝜹",  # mathematical bold italic small delta
    "\\mbfitepsilon" => "𝜺",  # mathematical bold italic small epsilon
    "\\mbfitzeta" => "𝜻",  # mathematical bold italic small zeta
    "\\mbfiteta" => "𝜼",  # mathematical bold italic small eta
    "\\mbfittheta" => "𝜽",  # mathematical bold italic small theta
    "\\mbfitiota" => "𝜾",  # mathematical bold italic small iota
    "\\mbfitkappa" => "𝜿",  # mathematical bold italic small kappa
    "\\mbfitlambda" => "𝝀",  # mathematical bold italic small lambda
    "\\mbfitmu" => "𝝁",  # mathematical bold italic small mu
    "\\mbfitnu" => "𝝂",  # mathematical bold italic small nu
    "\\mbfitxi" => "𝝃",  # mathematical bold italic small xi
    "\\mbfitomicron" => "𝝄",  # mathematical bold italic small omicron
    "\\mbfitpi" => "𝝅",  # mathematical bold italic small pi
    "\\mbfitrho" => "𝝆",  # mathematical bold italic small rho
    "\\mbfitvarsigma" => "𝝇",  # mathematical bold italic small final sigma
    "\\mbfitsigma" => "𝝈",  # mathematical bold italic small sigma
    "\\mbfittau" => "𝝉",  # mathematical bold italic small tau
    "\\mbfitupsilon" => "𝝊",  # mathematical bold italic small upsilon
    "\\mbfitphi" => "𝝋",  # mathematical bold italic small phi
    "\\mbfitchi" => "𝝌",  # mathematical bold italic small chi
    "\\mbfitpsi" => "𝝍",  # mathematical bold italic small psi
    "\\mbfitomega" => "𝝎",  # mathematical bold italic small omega
    "\\mbfitpartial" => "𝝏",  # mathematical bold italic partial differential
    "\\mbfitvarepsilon" => "𝝐",  # mathematical bold italic epsilon symbol
    "\\mbfitvartheta" => "𝝑",  # mathematical bold italic theta symbol
    "\\mbfitvarkappa" => "𝝒",  # mathematical bold italic kappa symbol
    "\\mbfitvarphi" => "𝝓",  # mathematical bold italic phi symbol
    "\\mbfitvarrho" => "𝝔",  # mathematical bold italic rho symbol
    "\\mbfitvarpi" => "𝝕",  # mathematical bold italic pi symbol
    "\\mbfsansAlpha" => "𝝖",  # mathematical sans-serif bold capital alpha
    "\\mbfsansBeta" => "𝝗",  # mathematical sans-serif bold capital beta
    "\\mbfsansGamma" => "𝝘",  # mathematical sans-serif bold capital gamma
    "\\mbfsansDelta" => "𝝙",  # mathematical sans-serif bold capital delta
    "\\mbfsansEpsilon" => "𝝚",  # mathematical sans-serif bold capital epsilon
    "\\mbfsansZeta" => "𝝛",  # mathematical sans-serif bold capital zeta
    "\\mbfsansEta" => "𝝜",  # mathematical sans-serif bold capital eta
    "\\mbfsansTheta" => "𝝝",  # mathematical sans-serif bold capital theta
    "\\mbfsansIota" => "𝝞",  # mathematical sans-serif bold capital iota
    "\\mbfsansKappa" => "𝝟",  # mathematical sans-serif bold capital kappa
    "\\mbfsansLambda" => "𝝠",  # mathematical sans-serif bold capital lambda
    "\\mbfsansMu" => "𝝡",  # mathematical sans-serif bold capital mu
    "\\mbfsansNu" => "𝝢",  # mathematical sans-serif bold capital nu
    "\\mbfsansXi" => "𝝣",  # mathematical sans-serif bold capital xi
    "\\mbfsansOmicron" => "𝝤",  # mathematical sans-serif bold capital omicron
    "\\mbfsansPi" => "𝝥",  # mathematical sans-serif bold capital pi
    "\\mbfsansRho" => "𝝦",  # mathematical sans-serif bold capital rho
    "\\mbfsansvarTheta" => "𝝧",  # mathematical sans-serif bold capital theta symbol
    "\\mbfsansSigma" => "𝝨",  # mathematical sans-serif bold capital sigma
    "\\mbfsansTau" => "𝝩",  # mathematical sans-serif bold capital tau
    "\\mbfsansUpsilon" => "𝝪",  # mathematical sans-serif bold capital upsilon
    "\\mbfsansPhi" => "𝝫",  # mathematical sans-serif bold capital phi
    "\\mbfsansChi" => "𝝬",  # mathematical sans-serif bold capital chi
    "\\mbfsansPsi" => "𝝭",  # mathematical sans-serif bold capital psi
    "\\mbfsansOmega" => "𝝮",  # mathematical sans-serif bold capital omega
    "\\mbfsansnabla" => "𝝯",  # mathematical sans-serif bold nabla
    "\\mbfsansalpha" => "𝝰",  # mathematical sans-serif bold small alpha
    "\\mbfsansbeta" => "𝝱",  # mathematical sans-serif bold small beta
    "\\mbfsansgamma" => "𝝲",  # mathematical sans-serif bold small gamma
    "\\mbfsansdelta" => "𝝳",  # mathematical sans-serif bold small delta
    "\\mbfsansepsilon" => "𝝴",  # mathematical sans-serif bold small epsilon
    "\\mbfsanszeta" => "𝝵",  # mathematical sans-serif bold small zeta
    "\\mbfsanseta" => "𝝶",  # mathematical sans-serif bold small eta
    "\\mbfsanstheta" => "𝝷",  # mathematical sans-serif bold small theta
    "\\mbfsansiota" => "𝝸",  # mathematical sans-serif bold small iota
    "\\mbfsanskappa" => "𝝹",  # mathematical sans-serif bold small kappa
    "\\mbfsanslambda" => "𝝺",  # mathematical sans-serif bold small lambda
    "\\mbfsansmu" => "𝝻",  # mathematical sans-serif bold small mu
    "\\mbfsansnu" => "𝝼",  # mathematical sans-serif bold small nu
    "\\mbfsansxi" => "𝝽",  # mathematical sans-serif bold small xi
    "\\mbfsansomicron" => "𝝾",  # mathematical sans-serif bold small omicron
    "\\mbfsanspi" => "𝝿",  # mathematical sans-serif bold small pi
    "\\mbfsansrho" => "𝞀",  # mathematical sans-serif bold small rho
    "\\mbfsansvarsigma" => "𝞁",  # mathematical sans-serif bold small final sigma
    "\\mbfsanssigma" => "𝞂",  # mathematical sans-serif bold small sigma
    "\\mbfsanstau" => "𝞃",  # mathematical sans-serif bold small tau
    "\\mbfsansupsilon" => "𝞄",  # mathematical sans-serif bold small upsilon
    "\\mbfsansphi" => "𝞅",  # mathematical sans-serif bold small phi
    "\\mbfsanschi" => "𝞆",  # mathematical sans-serif bold small chi
    "\\mbfsanspsi" => "𝞇",  # mathematical sans-serif bold small psi
    "\\mbfsansomega" => "𝞈",  # mathematical sans-serif bold small omega
    "\\mbfsanspartial" => "𝞉",  # mathematical sans-serif bold partial differential
    "\\mbfsansvarepsilon" => "𝞊",  # mathematical sans-serif bold epsilon symbol
    "\\mbfsansvartheta" => "𝞋",  # mathematical sans-serif bold theta symbol
    "\\mbfsansvarkappa" => "𝞌",  # mathematical sans-serif bold kappa symbol
    "\\mbfsansvarphi" => "𝞍",  # mathematical sans-serif bold phi symbol
    "\\mbfsansvarrho" => "𝞎",  # mathematical sans-serif bold rho symbol
    "\\mbfsansvarpi" => "𝞏",  # mathematical sans-serif bold pi symbol
    "\\mbfitsansAlpha" => "𝞐",  # mathematical sans-serif bold italic capital alpha
    "\\mbfitsansBeta" => "𝞑",  # mathematical sans-serif bold italic capital beta
    "\\mbfitsansGamma" => "𝞒",  # mathematical sans-serif bold italic capital gamma
    "\\mbfitsansDelta" => "𝞓",  # mathematical sans-serif bold italic capital delta
    "\\mbfitsansEpsilon" => "𝞔",  # mathematical sans-serif bold italic capital epsilon
    "\\mbfitsansZeta" => "𝞕",  # mathematical sans-serif bold italic capital zeta
    "\\mbfitsansEta" => "𝞖",  # mathematical sans-serif bold italic capital eta
    "\\mbfitsansTheta" => "𝞗",  # mathematical sans-serif bold italic capital theta
    "\\mbfitsansIota" => "𝞘",  # mathematical sans-serif bold italic capital iota
    "\\mbfitsansKappa" => "𝞙",  # mathematical sans-serif bold italic capital kappa
    "\\mbfitsansLambda" => "𝞚",  # mathematical sans-serif bold italic capital lambda
    "\\mbfitsansMu" => "𝞛",  # mathematical sans-serif bold italic capital mu
    "\\mbfitsansNu" => "𝞜",  # mathematical sans-serif bold italic capital nu
    "\\mbfitsansXi" => "𝞝",  # mathematical sans-serif bold italic capital xi
    "\\mbfitsansOmicron" => "𝞞",  # mathematical sans-serif bold italic capital omicron
    "\\mbfitsansPi" => "𝞟",  # mathematical sans-serif bold italic capital pi
    "\\mbfitsansRho" => "𝞠",  # mathematical sans-serif bold italic capital rho
    "\\mbfitsansvarTheta" => "𝞡",  # mathematical sans-serif bold italic capital theta symbol
    "\\mbfitsansSigma" => "𝞢",  # mathematical sans-serif bold italic capital sigma
    "\\mbfitsansTau" => "𝞣",  # mathematical sans-serif bold italic capital tau
    "\\mbfitsansUpsilon" => "𝞤",  # mathematical sans-serif bold italic capital upsilon
    "\\mbfitsansPhi" => "𝞥",  # mathematical sans-serif bold italic capital phi
    "\\mbfitsansChi" => "𝞦",  # mathematical sans-serif bold italic capital chi
    "\\mbfitsansPsi" => "𝞧",  # mathematical sans-serif bold italic capital psi
    "\\mbfitsansOmega" => "𝞨",  # mathematical sans-serif bold italic capital omega
    "\\mbfitsansnabla" => "𝞩",  # mathematical sans-serif bold italic nabla
    "\\mbfitsansalpha" => "𝞪",  # mathematical sans-serif bold italic small alpha
    "\\mbfitsansbeta" => "𝞫",  # mathematical sans-serif bold italic small beta
    "\\mbfitsansgamma" => "𝞬",  # mathematical sans-serif bold italic small gamma
    "\\mbfitsansdelta" => "𝞭",  # mathematical sans-serif bold italic small delta
    "\\mbfitsansepsilon" => "𝞮",  # mathematical sans-serif bold italic small epsilon
    "\\mbfitsanszeta" => "𝞯",  # mathematical sans-serif bold italic small zeta
    "\\mbfitsanseta" => "𝞰",  # mathematical sans-serif bold italic small eta
    "\\mbfitsanstheta" => "𝞱",  # mathematical sans-serif bold italic small theta
    "\\mbfitsansiota" => "𝞲",  # mathematical sans-serif bold italic small iota
    "\\mbfitsanskappa" => "𝞳",  # mathematical sans-serif bold italic small kappa
    "\\mbfitsanslambda" => "𝞴",  # mathematical sans-serif bold italic small lambda
    "\\mbfitsansmu" => "𝞵",  # mathematical sans-serif bold italic small mu
    "\\mbfitsansnu" => "𝞶",  # mathematical sans-serif bold italic small nu
    "\\mbfitsansxi" => "𝞷",  # mathematical sans-serif bold italic small xi
    "\\mbfitsansomicron" => "𝞸",  # mathematical sans-serif bold italic small omicron
    "\\mbfitsanspi" => "𝞹",  # mathematical sans-serif bold italic small pi
    "\\mbfitsansrho" => "𝞺",  # mathematical sans-serif bold italic small rho
    "\\mbfitsansvarsigma" => "𝞻",  # mathematical sans-serif bold italic small final sigma
    "\\mbfitsanssigma" => "𝞼",  # mathematical sans-serif bold italic small sigma
    "\\mbfitsanstau" => "𝞽",  # mathematical sans-serif bold italic small tau
    "\\mbfitsansupsilon" => "𝞾",  # mathematical sans-serif bold italic small upsilon
    "\\mbfitsansphi" => "𝞿",  # mathematical sans-serif bold italic small phi
    "\\mbfitsanschi" => "𝟀",  # mathematical sans-serif bold italic small chi
    "\\mbfitsanspsi" => "𝟁",  # mathematical sans-serif bold italic small psi
    "\\mbfitsansomega" => "𝟂",  # mathematical sans-serif bold italic small omega
    "\\mbfitsanspartial" => "𝟃",  # mathematical sans-serif bold italic partial differential
    "\\mbfitsansvarepsilon" => "𝟄",  # mathematical sans-serif bold italic epsilon symbol
    "\\mbfitsansvartheta" => "𝟅",  # mathematical sans-serif bold italic theta symbol
    "\\mbfitsansvarkappa" => "𝟆",  # mathematical sans-serif bold italic kappa symbol
    "\\mbfitsansvarphi" => "𝟇",  # mathematical sans-serif bold italic phi symbol
    "\\mbfitsansvarrho" => "𝟈",  # mathematical sans-serif bold italic rho symbol
    "\\mbfitsansvarpi" => "𝟉",  # mathematical sans-serif bold italic pi symbol
    "\\mbfDigamma" => "\U1d7ca",  # mathematical bold capital digamma
    "\\mbfdigamma" => "\U1d7cb",  # mathematical bold small digamma
    "\\mbfzero" => "𝟎",  # mathematical bold digit 0
    "\\mbfone" => "𝟏",  # mathematical bold digit 1
    "\\mbftwo" => "𝟐",  # mathematical bold digit 2
    "\\mbfthree" => "𝟑",  # mathematical bold digit 3
    "\\mbffour" => "𝟒",  # mathematical bold digit 4
    "\\mbffive" => "𝟓",  # mathematical bold digit 5
    "\\mbfsix" => "𝟔",  # mathematical bold digit 6
    "\\mbfseven" => "𝟕",  # mathematical bold digit 7
    "\\mbfeight" => "𝟖",  # mathematical bold digit 8
    "\\mbfnine" => "𝟗",  # mathematical bold digit 9
    "\\Bbbzero" => "𝟘",  # mathematical double-struck digit 0
    "\\Bbbone" => "𝟙",  # mathematical double-struck digit 1
    "\\Bbbtwo" => "𝟚",  # mathematical double-struck digit 2
    "\\Bbbthree" => "𝟛",  # mathematical double-struck digit 3
    "\\Bbbfour" => "𝟜",  # mathematical double-struck digit 4
    "\\Bbbfive" => "𝟝",  # mathematical double-struck digit 5
    "\\Bbbsix" => "𝟞",  # mathematical double-struck digit 6
    "\\Bbbseven" => "𝟟",  # mathematical double-struck digit 7
    "\\Bbbeight" => "𝟠",  # mathematical double-struck digit 8
    "\\Bbbnine" => "𝟡",  # mathematical double-struck digit 9
    "\\msanszero" => "𝟢",  # mathematical sans-serif digit 0
    "\\msansone" => "𝟣",  # mathematical sans-serif digit 1
    "\\msanstwo" => "𝟤",  # mathematical sans-serif digit 2
    "\\msansthree" => "𝟥",  # mathematical sans-serif digit 3
    "\\msansfour" => "𝟦",  # mathematical sans-serif digit 4
    "\\msansfive" => "𝟧",  # mathematical sans-serif digit 5
    "\\msanssix" => "𝟨",  # mathematical sans-serif digit 6
    "\\msansseven" => "𝟩",  # mathematical sans-serif digit 7
    "\\msanseight" => "𝟪",  # mathematical sans-serif digit 8
    "\\msansnine" => "𝟫",  # mathematical sans-serif digit 9
    "\\mbfsanszero" => "𝟬",  # mathematical sans-serif bold digit 0
    "\\mbfsansone" => "𝟭",  # mathematical sans-serif bold digit 1
    "\\mbfsanstwo" => "𝟮",  # mathematical sans-serif bold digit 2
    "\\mbfsansthree" => "𝟯",  # mathematical sans-serif bold digit 3
    "\\mbfsansfour" => "𝟰",  # mathematical sans-serif bold digit 4
    "\\mbfsansfive" => "𝟱",  # mathematical sans-serif bold digit 5
    "\\mbfsanssix" => "𝟲",  # mathematical sans-serif bold digit 6
    "\\mbfsansseven" => "𝟳",  # mathematical sans-serif bold digit 7
    "\\mbfsanseight" => "𝟴",  # mathematical sans-serif bold digit 8
    "\\mbfsansnine" => "𝟵",  # mathematical sans-serif bold digit 9
    "\\mttzero" => "𝟶",  # mathematical monospace digit 0
    "\\mttone" => "𝟷",  # mathematical monospace digit 1
    "\\mtttwo" => "𝟸",  # mathematical monospace digit 2
    "\\mttthree" => "𝟹",  # mathematical monospace digit 3
    "\\mttfour" => "𝟺",  # mathematical monospace digit 4
    "\\mttfive" => "𝟻",  # mathematical monospace digit 5
    "\\mttsix" => "𝟼",  # mathematical monospace digit 6
    "\\mttseven" => "𝟽",  # mathematical monospace digit 7
    "\\mtteight" => "𝟾",  # mathematical monospace digit 8
    "\\mttnine" => "𝟿",  # mathematical monospace digit 9

    "\\triangleright" => "▷",  # (large) right triangle, open; z notation range restriction
    "\\triangleleft" => "◁",  # (large) left triangle, open; z notation domain restriction
    "\\leftouterjoin" => "⟕",  # left outer join
    "\\rightouterjoin" => "⟖",  # right outer join
    "\\fullouterjoin" => "⟗",  # full outer join
    "\\Join" => "⨝",  # join
    "\\underbar" => "̲",  # combining low line
    "\\underleftrightarrow" => "͍",  # underleftrightarrow accent
    "\\leftwavearrow" => "↜",  # left arrow-wavy
    "\\rightwavearrow" => "↝",  # right arrow-wavy
    "\\varbarwedge" => "⌅",  # /barwedge b: logical and, bar above [projective (bar over small wedge)]
    "\\smallblacktriangleright" => "▸",  # right triangle, filled
    "\\smallblacktriangleleft" => "◂",  # left triangle, filled
    "\\leftmoon" => "☾",  # last quarter moon
    "\\smalltriangleright" => "▹",  # right triangle, open
    "\\smalltriangleleft" => "◃",  # left triangle, open

)
