/* luks.c
 *
 * hashkill - a hash cracking tool
 * Copyright (C) 2010 Milen Rangelov <gat3way@gat3way.eu>
 *
 * This software is Copyright (c) 2013 Dhiru Kholia <dhiru at openwall.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#if FMT_EXTERNS_H
extern struct fmt_main fmt_luks;
#elif FMT_REGISTERS_H
john_register_one(&fmt_luks);
#else

#if AC_BUILT
#include "autoconfig.h"
#else
#define _LARGEFILE64_SOURCE 1
#endif
#include "jumbo.h" // large file support
#include "os.h"
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <errno.h>
#include "stdint.h"
#include <stdlib.h>
#include <sys/types.h>
#include <openssl/aes.h>
#include "sha.h"
#include "sha2.h"
#include <string.h>
#include "arch.h"
#include "johnswap.h"
#include "misc.h"
#include "common.h"
#include "formats.h"
#include "params.h"
#include "options.h"
#include "memory.h"
#include "base64.h"
#include "gladman_pwd2key.h"
#ifdef _OPENMP
static int omp_t = 1;
#include <omp.h>
#define OMP_SCALE               1
#endif
#include "memdbg.h"

#define LUKS_MAGIC_L        6
#define LUKS_CIPHERNAME_L   32
#define LUKS_CIPHERMODE_L   32
#define LUKS_HASHSPEC_L     32
#define UUID_STRING_L       40
#define LUKS_DIGESTSIZE 20
#define LUKS_SALTSIZE 32
#define LUKS_NUMKEYS 8

#define FORMAT_LABEL		"LUKS"
#define FORMAT_NAME		""
#define ALGORITHM_NAME		"PBKDF2-SHA1 32/" ARCH_BITS_STR
#define BENCHMARK_COMMENT	""
#define PLAINTEXT_LENGTH  	125
#define BENCHMARK_LENGTH	-1
#define BINARY_SIZE		LUKS_DIGESTSIZE
#define BINARY_ALIGN		4
#define SALT_SIZE		sizeof(struct custom_salt)
#define SALT_ALIGN			4
#define MIN_KEYS_PER_CRYPT	1
#define MAX_KEYS_PER_CRYPT	1

#if ARCH_LITTLE_ENDIAN
#define john_htonl(x) ((((x)>>24) & 0xffL) | (((x)>>8) & 0xff00L) | \
		(((x)<<8) & 0xff0000L) | (((x)<<24) & 0xff000000L))

#define john_ntohl(x) ((((x)>>24) & 0xffL) | (((x)>>8) & 0xff00L) | \
		(((x)<<8) & 0xff0000L) | (((x)<<24) & 0xff000000L))
#else
#define john_htonl(x) (x)
#define john_ntohl(x) (x)
#endif

static struct fmt_tests luks_tests[] = {
#ifndef _MSC_VER
	{"$luks$1$592$4c554b53babe000161657300000000000000000000000000000000000000000000000000000000006362632d65737369763a73686132353600000000000000000000000000000000736861310000000000000000000000000000000000000000000000000000000000000408000000104f386b50df3fcd9132589a934851faaff16709ff628ed0b628b0d7151b3600c0b3f95d8404a8b35fdf5dd6b6ff10f4c352fde11900010b1762663664393836622d633836352d343261622d616534662d6165313336633938383735360000000000ac71f3000430f5d9c39e349b48d7cf1771d9c152840b389a4353ff186436ec75cc397529ed40260000000800000fa00000dead0000000000000000000000000000000000000000000000000000000000000000000000000000008800000fa00000dead0000000000000000000000000000000000000000000000000000000000000000000000000000010800000fa00000dead0000000000000000000000000000000000000000000000000000000000000000000000000000018800000fa00000dead0000000000000000000000000000000000000000000000000000000000000000000000000000020800000fa00000dead0000000000000000000000000000000000000000000000000000000000000000000000000000028800000fa00000dead0000000000000000000000000000000000000000000000000000000000000000000000000000030800000fa00000dead0000000000000000000000000000000000000000000000000000000000000000000000000000038800000fa0$64000$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$4f386b50df3fcd9132589a934851faaff16709ff", "!@#$%^&*"},
#endif
	{NULL}
};

/* taken from LUKS on disk format specification */
struct luks_phdr {
	char magic[LUKS_MAGIC_L];
	uint16_t version;
	char cipherName[LUKS_CIPHERNAME_L];
	char cipherMode[LUKS_CIPHERMODE_L];
	char hashSpec[LUKS_HASHSPEC_L];
	uint32_t payloadOffset;
	uint32_t keyBytes;
	char mkDigest[LUKS_DIGESTSIZE];
	char mkDigestSalt[LUKS_SALTSIZE];
	uint32_t mkDigestIterations;
	char uuid[UUID_STRING_L];
	struct {
		uint32_t active;
		uint32_t passwordIterations;
		char passwordSalt[LUKS_SALTSIZE];
		uint32_t keyMaterialOffset;
		uint32_t stripes;
	} keyblock[LUKS_NUMKEYS];
};

static struct custom_salt {
	struct luks_phdr myphdr;
	int loaded;
	unsigned char *cipherbuf;
	int afsize;
	int bestslot;
	int bestiter;
	char path[8192];
} *cur_salt;

static void XORblock(char *src1, char *src2, char *dst, int n)
{
	int j;

	for (j = 0; j < n; j++)
		dst[j] = src1[j] ^ src2[j];
}

static int diffuse(unsigned char *src, unsigned char *dst, int size)
{
	uint32_t i;
	uint32_t IV;		/* host byte order independent hash IV */
	SHA_CTX ctx;
	int fullblocks = (size) / 20;
	int padding = size % 20;

	for (i = 0; i < fullblocks; i++) {
		IV = john_htonl(i);
		SHA1_Init(&ctx);
		SHA1_Update(&ctx, &IV, 4);
		SHA1_Update(&ctx, src + 20 * i, 20);
		SHA1_Final(dst + 20 * i, &ctx);
	}

	if (padding) {
		IV = john_htonl(fullblocks);
		SHA1_Init(&ctx);
		SHA1_Update(&ctx, &IV, 4);
		SHA1_Update(&ctx, src + 20 * fullblocks, padding);
		SHA1_Final(dst + 20 * fullblocks, &ctx);
	}
	return 0;
}

static int AF_merge(unsigned char *src, unsigned char *dst, int afsize,
    int stripes)
{
	int i;
	char *bufblock;
	int blocksize = afsize / stripes;

	bufblock = mem_calloc(blocksize + 20);

	for (i = 0; i < (stripes - 1); i++) {
		XORblock((char *) (src + (blocksize * i)), bufblock, bufblock,
		    blocksize);
		diffuse((unsigned char *) bufblock, (unsigned char *) bufblock,
		    blocksize);
	}
	XORblock((char *) (src + blocksize * (stripes - 1)), bufblock,
	    (char *) dst, blocksize);

	MEM_FREE(bufblock);
	return 0;
}

static int af_sectors(int blocksize, int blocknumbers)
{
	int af_size;

	af_size = blocksize * blocknumbers;
	af_size = (af_size + 511) / 512;
	af_size *= 512;
	return af_size;
}


static void decrypt_aes_cbc_essiv(unsigned char *src, unsigned char *dst,
    unsigned char *key, int size, struct custom_salt *cs)
{
	AES_KEY aeskey;
	unsigned char essiv[16];
	unsigned char essivhash[32];
	int a;
	SHA256_CTX ctx;
	unsigned char sectorbuf[16];
	unsigned char zeroiv[16];

	for (a = 0; a < (size / 512); a++) {
		SHA256_Init(&ctx);
		SHA256_Update(&ctx, key, john_ntohl(cs->myphdr.keyBytes));
		SHA256_Final(essivhash, &ctx);
		memset(sectorbuf, 0, 16);
		memset(zeroiv, 0, 16);
		memset(essiv, 0, 16);
#if ARCH_LITTLE_ENDIAN
		memcpy(sectorbuf, &a, 4);
#else
		{ int b = JOHNSWAP(a); memcpy(sectorbuf, &b, 4); }
#endif
		AES_set_encrypt_key(essivhash, 256, &aeskey);
		AES_cbc_encrypt(sectorbuf, essiv, 16, &aeskey, zeroiv, AES_ENCRYPT);
		AES_set_decrypt_key(key, john_ntohl(cs->myphdr.keyBytes)*8, &aeskey);
		AES_cbc_encrypt((src+a*512), (dst+a*512), 512, &aeskey, essiv, AES_DECRYPT);
	}
}

static int hash_plugin_parse_hash(char *filename, struct custom_salt *cs, int is_critical)
{
	FILE *myfile;
	int readbytes;

	myfile = jtr_fopen(filename, "rb");

	if (!myfile) {
		fprintf(stderr, "\n%s : %s!\n", filename, strerror(errno));
		return -1;
	}

	// can this go over 4gb?
	cs->cipherbuf = mem_alloc_tiny(cs->afsize + 1, MEM_ALIGN_NONE);
	if (!cs->cipherbuf)
		goto bad;
	// printf(">>> %d\n", cs->afsize);
	readbytes = fread(cs->cipherbuf, cs->afsize, 1, myfile);

	if (readbytes < 0) {
		//free(cs->cipherbuf);
		fprintf(stderr, "%s : unable to read required data\n",
			filename);
		goto bad;
	}
	fclose(myfile);
	return 0;

bad:
	fclose(myfile);
	if (is_critical) {
		fprintf(stderr, "\nLUKS plug-in is unable to continue due to errors!\n");
		exit(-1);
	}
	return -1;
}


static char (*saved_key)[PLAINTEXT_LENGTH + 1];
static ARCH_WORD_32 (*crypt_out)[BINARY_SIZE / sizeof(ARCH_WORD_32)];

static void init(struct fmt_main *self)
{
	static int warned = 0;
//	extern struct fmt_main fmt_luks;
#ifdef _OPENMP
	omp_t = omp_get_max_threads();
	self->params.min_keys_per_crypt *= omp_t;
	omp_t *= OMP_SCALE;
	self->params.max_keys_per_crypt *= omp_t;
#endif
	saved_key = mem_calloc_tiny(sizeof(*saved_key) *
			self->params.max_keys_per_crypt, MEM_ALIGN_WORD);
	crypt_out = mem_calloc_tiny(sizeof(*crypt_out) * self->params.max_keys_per_crypt, MEM_ALIGN_WORD);

/*
 * LUKS format will need to be redesigned to address the issues mentioned in 
 * https://github.com/magnumripper/JohnTheRipper/issues/557.
 * This will require a change in john's hash representation for LUKS format.
 * The redesign will happen after the next official jumbo release.
 * To avoid having to support the current LUKS hash representation forever,
 * just print a warning that the hash representation will change in future releases.
 *
 * So far, no "official" jumbo release supports the LUKS format, currently only
 * users of bleeding-jumbo may have used LUKS format. These users should be able
 * to re-run luks2john and retry the passwords that have been stored for the current LUKS hashes
 * once the redesign of john's LUKS format implementation has been completed.)
 */
	if (!options.listconf && !(options.flags & FLG_TEST_CHK) && warned++ == 0) {
		fprintf(stderr,
		        "WARNING, LUKS format hash representation will change in future releases,\n"
		        "see doc/README.LUKS\n"); // FIXME: address github issue #557 after 1.8.0-jumbo-1
		fflush(stderr);
	}

//	 This printf will 'help' debug a system that truncates that monster hash, but does not cause compiler to die.
//	printf ("length=%d end=%s\n", strlen(fmt_luks.params.tests[0].ciphertext), &((fmt_luks.params.tests[0].ciphertext)[strlen(fmt_luks.params.tests[0].ciphertext)-30]));
}

static int ishex(char *q)
{
	while (atoi16[ARCH_INDEX(*q)] != 0x7F)
		q++;
	return !*q;
}

static int valid(char *ciphertext, struct fmt_main *self)
{
	char *ctcopy;
	char *keeptr;
	char *p, *q;
	int is_inlined;
	int res;
	static struct custom_salt cs;

	if (strncmp(ciphertext, "$luks$", 6) != 0)
		return 0;
	ctcopy = strdup(ciphertext);
	keeptr = ctcopy;
	ctcopy += 6;
	if ((p = strtok(ctcopy, "$")) == NULL)	/* is_inlined */
		goto err;
	is_inlined = atoi(p);

	if (is_inlined) {
		if ((p = strtok(NULL, "$")) == NULL)
			goto err;
		res = atoi(p);
		if (res != sizeof(struct luks_phdr))
			goto err;
		if ((p = strtok(NULL, "$")) == NULL)
			goto err;
		if (res * 2 != strlen(p))
			goto err;
		if ((p = strtok(NULL, "$")) == NULL)
			goto err;
		res = atoi(p);
		if ((p = strtok(NULL, "$")) == NULL)
			goto err;
		if ((p = strtok(NULL, "$")) == NULL)
			goto err;
		if (strlen(p) != LUKS_DIGESTSIZE * 2)
			goto err;
		if (!ishex(p))
			goto err;
	}
	else {
		if ((p = strtok(NULL, "$")) == NULL)	/* path */
			goto err;
		q = p;
		if ((p = strtok(NULL, "$")) == NULL)	/* mkDigest */
			goto err;

		/* more tests */
		if (hash_plugin_parse_hash(q, &cs, 0) != 0) {
			MEM_FREE(cs.cipherbuf);
			return 0;
		}
		MEM_FREE(cs.cipherbuf);
	}

	MEM_FREE(keeptr);
	return 1;
err:
	MEM_FREE(keeptr);
	return 0;
}

static void *get_salt(char *ciphertext)
{
	char *ctcopy = strdup(ciphertext);
	char *keeptr = ctcopy;
	char *p;
	int is_inlined;
	int res;
	int i;
	int cnt;
	unsigned char *out;
	static struct custom_salt cs;
	unsigned int bestiter = 0xFFFFFFFF;
	size_t size;
	ctcopy += 6;

	memset(&cs, 0, sizeof(cs));
	out = (unsigned char*)&cs.myphdr;

	p = strtok(ctcopy, "$");
	is_inlined = atoi(p);

	/* common handling */
	p = strtok(NULL, "$");
	res = atoi(p);
	assert(res == sizeof(struct luks_phdr));
	p = strtok(NULL, "$");
	for (i = 0; i < res; i++) {
		out[i] = (atoi16[ARCH_INDEX(*p)] << 4) | atoi16[ARCH_INDEX(p[1])];
		p += 2;
	}
	p = strtok(NULL, "$");
	res = atoi(p);

	cs.afsize = af_sectors(john_ntohl(cs.myphdr.keyBytes),
			john_ntohl(cs.myphdr.keyblock[cs.bestslot].stripes));

	assert(res == cs.afsize);

	if (is_inlined) {
		p = strtok(NULL, "$");
		size = (strlen(p) + 20) / 4 * 3 + 1;
		cs.cipherbuf = mem_alloc_tiny(size, MEM_ALIGN_NONE);
		base64_decode(p, strlen(p), (char*)cs.cipherbuf);
	}
	else {
		p = strtok(NULL, "$");
		p = strtok(NULL, "$");
		strcpy(cs.path, p);
		hash_plugin_parse_hash(cs.path, &cs, 1);
	}
	for (cnt = 0; cnt < LUKS_NUMKEYS; cnt++) {
			if ((john_ntohl(cs.myphdr.keyblock[cnt].passwordIterations) < bestiter)
			&& (john_ntohl(cs.myphdr.keyblock[cnt].passwordIterations) > 1) &&
			(john_ntohl(cs.myphdr.keyblock[cnt].active) == 0x00ac71f3)) {
				cs.bestslot = cnt;
				cs.bestiter =
				john_ntohl(cs.myphdr.keyblock[cnt].passwordIterations);
			}
	}
	MEM_FREE(keeptr);
	return (void *)&cs;
}

static void *get_binary(char *ciphertext)
{
	static union {
		unsigned char c[LUKS_DIGESTSIZE];
		ARCH_WORD dummy;
	} buf;
	unsigned char *out = buf.c;
	char *p;
	int i;
	p = strrchr(ciphertext, '$') + 1;
	for (i = 0; i < LUKS_DIGESTSIZE; i++) {
		out[i] =
		    (atoi16[ARCH_INDEX(*p)] << 4) |
		    atoi16[ARCH_INDEX(p[1])];
		p += 2;
	}

	return out;
}

static int get_hash_0(int index) { return crypt_out[index][0] & 0xf; }
static int get_hash_1(int index) { return crypt_out[index][0] & 0xff; }
static int get_hash_2(int index) { return crypt_out[index][0] & 0xfff; }
static int get_hash_3(int index) { return crypt_out[index][0] & 0xffff; }
static int get_hash_4(int index) { return crypt_out[index][0] & 0xfffff; }
static int get_hash_5(int index) { return crypt_out[index][0] & 0xffffff; }
static int get_hash_6(int index) { return crypt_out[index][0] & 0x7ffffff; }

static void set_salt(void *salt)
{
	cur_salt = (struct custom_salt *)salt;
}

static int crypt_all(int *pcount, struct db_salt *salt)
{
	int count = *pcount;
	int index = 0;

#ifdef _OPENMP
#pragma omp parallel for
	for (index = 0; index < count; index++)
#endif
	{
		unsigned char keycandidate[255];
		unsigned char masterkeycandidate[255];
		unsigned char *af_decrypted = mem_alloc(cur_salt->afsize + 20);
		char *password = saved_key[index];
		int iterations = cur_salt->bestiter;
		int dklen = john_ntohl(cur_salt->myphdr.keyBytes);

		// printf("itertations %d %d %d\n", iterations, dklen, cur_salt->bestslot);

		// Get pbkdf2 of the password to obtain decryption key
		derive_key((const uint8_t*)password, strlen(password),
			(const uint8_t*)(cur_salt->myphdr.keyblock[cur_salt->bestslot].passwordSalt),
			LUKS_SALTSIZE,
			iterations,
			keycandidate,
			dklen);

		// Decrypt the blocksi
		decrypt_aes_cbc_essiv(cur_salt->cipherbuf, af_decrypted, keycandidate, cur_salt->afsize, cur_salt);
		// AFMerge the blocks
		AF_merge(af_decrypted, masterkeycandidate, cur_salt->afsize,
		john_ntohl(cur_salt->myphdr.keyblock[cur_salt->bestslot].stripes));
		// pbkdf2 again
		derive_key(masterkeycandidate,
			john_ntohl(cur_salt->myphdr.keyBytes),
			(const uint8_t*)cur_salt->myphdr.mkDigestSalt,
			LUKS_SALTSIZE,
			john_ntohl(cur_salt->myphdr.mkDigestIterations),
			(unsigned char*)crypt_out[index],
			LUKS_DIGESTSIZE);
		MEM_FREE(af_decrypted);
	}
	return count;
}

static int cmp_all(void *binary, int count)
{
	int index = 0;
#ifdef _OPENMP
	for (; index < count; index++)
#endif
		if (!memcmp(binary, crypt_out[index], LUKS_DIGESTSIZE))
			return 1;
	return 0;
}

static int cmp_one(void *binary, int index)
{
	return !memcmp(binary, crypt_out[index], LUKS_DIGESTSIZE);
}

static int cmp_exact(char *source, int index)
{
	return 1;
}

static void luks_set_key(char *key, int index)
{
	int saved_key_length = strlen(key);
	if (saved_key_length > PLAINTEXT_LENGTH)
		saved_key_length = PLAINTEXT_LENGTH;
	memcpy(saved_key[index], key, saved_key_length);
	saved_key[index][saved_key_length] = 0;
}

static char *get_key(int index)
{
	return saved_key[index];
}

struct fmt_main fmt_luks = {
	{
		FORMAT_LABEL,
		FORMAT_NAME,
		ALGORITHM_NAME,
		BENCHMARK_COMMENT,
		BENCHMARK_LENGTH,
		PLAINTEXT_LENGTH,
		BINARY_SIZE,
		BINARY_ALIGN,
		SALT_SIZE,
		SALT_ALIGN,
		MIN_KEYS_PER_CRYPT,
		MAX_KEYS_PER_CRYPT,
		FMT_CASE | FMT_8_BIT | FMT_OMP,
#if FMT_MAIN_VERSION > 11
		{ NULL },
#endif
		luks_tests
	}, {
		init,
		fmt_default_done,
		fmt_default_reset,
		fmt_default_prepare,
		valid,
		fmt_default_split,
		get_binary,
		get_salt,
#if FMT_MAIN_VERSION > 11
		{ NULL },
#endif
		fmt_default_source,
		{
			fmt_default_binary_hash_0,
			fmt_default_binary_hash_1,
			fmt_default_binary_hash_2,
			fmt_default_binary_hash_3,
			fmt_default_binary_hash_4,
			fmt_default_binary_hash_5,
			fmt_default_binary_hash_6
		},
		fmt_default_salt_hash,
		set_salt,
		luks_set_key,
		get_key,
		fmt_default_clear_keys,
		crypt_all,
		{
			get_hash_0,
			get_hash_1,
			get_hash_2,
			get_hash_3,
			get_hash_4,
			get_hash_5,
			get_hash_6
		},
		cmp_all,
		cmp_one,
		cmp_exact
	}
};

#endif /* plugin stanza */
