open Lexing
open Util

module IH = Data.IntCols.Hash
type 'a inthash = 'a IH.t


(* ------------------------------------------------------------
 * Positions
 * ------------------------------------------------------------ *)

let nowhere = {pos_fname = "" ; pos_lnum = 0 ; pos_bol = 0; pos_cnum = 0}
let pos_is_nowhere pos = (pos.pos_fname = "")

let lexstate_set_file_line lexbuf fname line charnum = 
	lexbuf.lex_curr_p <- 
		{pos_fname = fname; pos_lnum = line; 
		 pos_bol = charnum; pos_cnum = charnum}
	
let lexstate_newline lexbuf =
	let pos = lexbuf.lex_curr_p in
	lexbuf.lex_curr_p <- {pos with
		pos_lnum = pos.pos_lnum + 1;
		pos_bol = pos.pos_cnum}


(* ------------------------------------------------------------
 * A Span of positions
 * ------------------------------------------------------------ *)

type span = {pstart : Lexing.position; pend : Lexing.position}

let span_is_nowhere s = pos_is_nowhere s.pstart

let mkspan pstart pend = 
	if pos_is_nowhere pstart then {pstart = pend; pend = pend} else
	if pos_is_nowhere pend then {pstart = pstart; pend = pstart} else
	{pstart = pstart; pend = pend}

let spanspan startspan endspan = 
	if span_is_nowhere endspan then startspan else
	if span_is_nowhere startspan then endspan else
	{pstart = startspan.pstart; pend = endspan.pend}


(* ------------------------------------------------------------
 * Parsed Tokens
 * ------------------------------------------------------------ *)

type lextoken = {mutable whitespace : string; what : string; tspan : span}

let whitespace : string list ref = ref []

let empty_token = {whitespace = ""; what = ""; tspan = mkspan nowhere nowhere}
let token_is_null token = (token.what = "")

let inputtokens_rev : lextoken list ref = ref []   (* in reverse order *)
let thisline_rev : lextoken list ref = ref []
let inputlines_rev : lextoken list list ref = ref []
let prevline_rev : lextoken list ref = ref []

let get_inputlines () = List.rev !inputlines_rev
let get_inputtokens () = List.rev !inputtokens_rev

let chop_newlines s =
	if String.contains s '\n' then
		let linestart = String.rindex s '\n' in
		let s = String.sub s (linestart + 1)
			(String.length s - linestart - 1) in
		s
	else s


let print_token t = 
	print_string (strip_tabs (chop_newlines t.whitespace)); 
	print_string t.what
let print_current_line () =
	let line = if !thisline_rev <> [] 
		then !thisline_rev else !prevline_rev in
	List.iter print_token (List.rev line); print_endline ""

let lt_what lt = lt.what
let lt_white lt = lt.whitespace
let lt_span lt = lt.tspan

let laststart = ref nowhere
let lastend = ref nowhere


let current_span () = mkspan !laststart !lastend

let track_tokens = ref true

let gather_whitespace () = 
	String.concat "" (List.rev !whitespace)

let mktoken lexbuf = 
	laststart := lexbuf.lex_start_p;
	lastend := lexbuf.lex_curr_p;
	if !track_tokens then begin
		let tspan = mkspan lexbuf.lex_start_p lexbuf.lex_curr_p in
		let lexeme = Lexing.lexeme lexbuf in
		let whitestr = gather_whitespace () in
		let token = {whitespace = whitestr; what = lexeme; tspan = tspan} in 
		whitespace := [];
		inputtokens_rev := token :: !inputtokens_rev;
		thisline_rev := token :: !thisline_rev;
		token
	end else empty_token

let dotoken lexbuf = ignore (mktoken lexbuf)

let tokenlexeme lexbuf = dotoken lexbuf; Lexing.lexeme lexbuf

let add_whitespace lexbuf = if !track_tokens then 
	whitespace := Lexing.lexeme lexbuf :: !whitespace
	
let newline lexbuf = 
	lexstate_newline lexbuf;
	inputlines_rev := (List.rev !thisline_rev) :: !inputlines_rev;
	if !thisline_rev <> [] then prevline_rev := !thisline_rev;
	thisline_rev := []

let reset_inputtokens () = 
	thisline_rev := []; 
	inputlines_rev := [];
	inputtokens_rev := [];
	prevline_rev := [];
	whitespace := [];
	laststart := nowhere;
	lastend := nowhere;
	track_tokens := true
	
	

