#ifndef DRIVEN_CAVITY_H
#define DRIVEN_CAVITY_H

#include "partition_info.h"

template <class V>
struct vector_traits {
  typedef typename V::value_type value_type;
};

template <class T>
struct vector_traits<T*> {
  typedef T value_type;
};


//number of variables per point
#define NUM_OF_SV 4

#define U(i)     4*(i)
#define V(i)     4*(i)+1
#define Omega(i) 4*(i)+2
#define Temp(i)  4*(i)+3

struct eval {
  eval(const Info& _info) : info(_info) {}

  template <class VecX, class VecF>
  void operator()(const VecX& x , VecF& f) const {
    _evaluate(x, f);
  }
  
  //x and f are both global indexed here.
  template <class VecX, class VecF>
  void _evaluate(const VecX& x , VecF& f) const {

    int local_nx_begin = info.local_nx_begin;
    int local_nx_end   = info.local_nx_end;
    int local_ny_begin = info.local_ny_begin;
    int local_ny_end   = info.local_ny_end;
    int i, j;
    
    int nx = info.nx;
    int ny = info.ny;
    int local_nx = nx;

    const double grashof = info.grashof;
    const double prandtl = info.prandtl;
    const double lid     = info.lidvelocity;
    const double dhx = nx-1;
    const double dhy = ny-1;
    const double hx = 1.0/double(nx-1);
    const double hy = 1.0/double(ny-1);
    const double hxdhy = hx*dhy;
    const double hydhx = hy*dhx;

    int gxm = local_nx; 

    /* Test whether we are on the bottom edge of the global array */
    if (local_ny_begin == 0) {
      /* bottom edge */
      for (i=local_nx_begin; i<local_nx_end; i++) {
	int row = i;
	f[U(row)]     = x[U(row)];
	f[V(row)]     = x[V(row)];
	f[Omega(row)] = x[Omega(row)] + (x[U(row+gxm)] - x[U(row)])*dhy;
	f[Temp(row)]  = x[Temp(row)]-x[Temp(row+gxm)];
      }
    }
    
    /* Test whether we are on the top edge of the global array */
    if (local_ny_end == ny) {
      for (i=local_nx_begin; i<local_nx_end; i++) {
	int row = i + (ny - 1) * local_nx;;

        f[U(row)]     = x[U(row)] - lid;
        f[V(row)]     = x[V(row)];
        f[Omega(row)] = x[Omega(row)] + (x[U(row)] - x[U(row-gxm)])*dhy; 
	f[Temp(row)]  = x[Temp(row)]-x[Temp(row-gxm)];
      }
    }

    /* Test whether we are on the left edge of the global array */
    if (local_nx_begin == 0) {
      /* left edge */
      for (i=local_ny_begin; i<local_ny_end; i++) {
	int row = i * local_nx;
	f[U(row)]     = x[U(row)];
	f[V(row)]     = x[V(row)];
	f[Omega(row)] = x[Omega(row)] - (x[V(row+1)] - x[V(row)])*dhx; 
	f[Temp(row)]  = x[Temp(row)];
      }
    }

    /* Test whether we are on the right edge of the global array */
    if (local_nx_end == nx) {
      for (i=local_ny_begin; i<local_ny_end; i++) {
	int row = (nx-1) + i * local_nx;
	f[U(row)]     = x[U(row)];
	f[V(row)]     = x[V(row)];
	f[Omega(row)] = x[Omega(row)] - (x[V(row)] - x[V(row-1)])*dhx; 
	f[Temp(row)]  = x[Temp(row)] - (double)(grashof>0);
      }
    }

    if (local_nx_begin == 0)
      ++local_nx_begin;
    if (local_nx_end == nx)
      --local_nx_end;
    if (local_ny_begin == 0)
      ++local_ny_begin;
    if (local_ny_end == ny)
      --local_ny_end;


    const double half = 0.5;
    const double two  = 2.0;

    typedef typename vector_traits<VecX>::value_type T;
    T u, uxx, uyy, vx, avx, vxp, vxm, vy, avy, vyp, vym;

    for (j=local_ny_begin; j<local_ny_end; ++j) {
      int row_base = j * nx;
      for (i=local_nx_begin; i<local_nx_end; ++i) {

	int row = row_base + i;

	/* U velocity */
        u          = x[U(row)];
        uxx        = (two*u - x[U(row-1)] - x[U(row+1)])*hydhx;
        uyy        = (two*u - x[U(row-gxm)] - x[U(row+gxm)])*hxdhy;
        f[U(row)]  = uxx + uyy - half*(x[Omega(row+gxm)]-x[Omega(row-gxm)])*hx;

	/* V velocity */
        u          = x[V(row)];
        uxx        = (two*u - x[V(row-1)] - x[V(row+1)])*hydhx;
        uyy        = (two*u - x[V(row-gxm)] - x[V(row+gxm)])*hxdhy;
        f[V(row)]  = uxx + uyy + half*(x[Omega(row+1)]-x[Omega(row-1)])*hy;

	/*
	  convective coefficients for upwinding
        */
	vx = x[U(row)]; avx = std::abs(vx);
        vxp = half*(vx+avx); vxm = half*(vx-avx);
	vy = x[V(row)]; avy = std::abs(vy);
        vyp = half*(vy+avy); vym = half*(vy-avy);

	/* Omega */
        u          = x[Omega(row)];
        uxx        = (two*u - x[Omega(row-1)] - x[Omega(row+1)])*hydhx;
        uyy        = (two*u - x[Omega(row-gxm)] - x[Omega(row+gxm)])*hxdhy;
	f[Omega(row)] = uxx + uyy + 
			(vxp*(u - x[Omega(row-1)]) +
			  vxm*(x[Omega(row+1)] - u)) * hy +
			(vyp*(u - x[Omega(row-gxm)]) +
			  vym*(x[Omega(row+gxm)] - u)) * hx -
			half * grashof * (x[Temp(row+1)] - x[Temp(row-1)]) * hy;

	/*
	  convective coefficients for upwinding
        */
	vx = x[U(row)]; avx = std::abs(vx);
        vxp = half*(vx+avx); vxm = half*(vx-avx);
	vy = x[V(row)]; avy = std::abs(vy);
        vyp = half*(vy+avy); vym = half*(vy-avy);

        /* Temperature */
        u             = x[Temp(row)];
        uxx           = (two*u - x[Temp(row-1)] - x[Temp(row+1)])*hydhx;
        uyy           = (two*u - x[Temp(row-gxm)] - x[Temp(row+gxm)])*hxdhy;
	f[Temp(row)] =  uxx + uyy  + prandtl * (
			(vxp*(u - x[Temp(row-1)]) +
			  vxm*(x[Temp(row+1)] - u)) * hy +
		        (vyp*(u - x[Temp(row-gxm)]) +
		       	  vym*(x[Temp(row+gxm)] - u)) * hx);

      }
    }  
  }
  
protected:
  Info info;
};


#undef U
#undef V
#undef Omega
#undef Temp


#endif //DRIVEN_CAVITY_H
