// -*- c++ -*-
//
//=======================================================================
// Copyright (C) 1997-2001
// Authors: Andrew Lumsdaine <lums@osl.iu.edu> 
//          Lie-Quan Lee     <llee@osl.iu.edu>
//
// This file is part of the Iterative Template Library
//
// You should have received a copy of the License Agreement for the
// Iterative Template Library along with the software;  see the
// file LICENSE.  
//
// Permission to modify the code and to distribute modified code is
// granted, provided the text of this NOTICE is retained, a notice that
// the code was modified is included with the above COPYRIGHT NOTICE and
// with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
// file is distributed with the modified code.
//
// LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
// By way of example, but not limitation, Licensor MAKES NO
// REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
// PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
// OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
// OR OTHER RIGHTS.
//=======================================================================
//
//                                                                           
//
//===========================================================================

#ifndef ITL_SCALED1D_H
#define ITL_SCALED1D_H

#include <iterator>

namespace itl {

  //: scale iterator
  //!category: iterators, adaptors
  //!component: type
  //
  // The scale iterator is an adaptor which multiplies the
  // value of the underlying element by some scalar as
  // they are access (through the dereference operator).
  // Scale iterators are somewhat different from most in
  // that they are always considered to be a constant iterator
  // whether or not the underlying elements are mutable.
  //
  // Typically users will not need to use scale iterator
  // directly. It is really just an implementation detail
  // of the scaled1D container.
  //
  //!definition: scale_iterator.h
  //!tparam: RandomAccessIterator - The underlying iterator
  //!tparam: T - The type of the scalar to multiply by
  //!models: RandomAccessIterator
  //!typereqs: T must be convertible to RandomAccessIterator's value_type
  //!typereqs: RandomAccessIterator's value_type must be a model of Ring
  template <class RandomAccessIterator, class T>
  class scale_iterator {
    typedef scale_iterator<RandomAccessIterator, T>    self;
    typedef std::iterator_traits<RandomAccessIterator> IterTraits;
  public:
    //: The value type
    typedef typename IterTraits::value_type            value_type;
#if !defined ( _MSVCPP_ )
    //: The difference type  
    typedef typename IterTraits::difference_type       difference_type;
    //: The pointer type
    typedef typename IterTraits::pointer               pointer;          
#else
    typedef typename IterTraits::distance_type         difference_type;
    typedef difference_type                            distance_type;
    typedef value_type*                                pointer;
#endif
    //: The iterator category
    typedef typename IterTraits::iterator_category       iterator_category;
    
    typedef difference_type                              Distance;
    typedef RandomAccessIterator                         iterator_type;
    //: The reference type
    typedef value_type                                   reference;
    typedef value_type                                   const_reference;
    
    //: The default constructor
    //!wheredef: Trivial Iterator
    inline scale_iterator() : alpha(0) { }
    
    //!wheredef: scale_iterator
    inline scale_iterator(const RandomAccessIterator& x)
      : current(x), alpha(1) { }
    
    //: Normal constructor
    //!wheredef: scale_iterator
    inline scale_iterator(const RandomAccessIterator& x, const value_type& a)
      : current(x), alpha(a) { }
    
    //: Copy constructor
    //!wheredef: Trivial Iterator
    inline scale_iterator(const self& x)
      : current(x.current), alpha(x.alpha) { }
    
    //: Convert to base iterator
    //!wheredef: scale_iterator
    inline operator RandomAccessIterator() { return current; }
    
    //: Access base iterator
    //!wheredef: scale_iterator
    inline RandomAccessIterator base() const { return current; }
    
    //: Dereference (and scale)
    //!wheredef: Trivial Iterator
    inline value_type operator*() const { return alpha * *current; }
    
    //: Preincrement
    //!wheredef: Forward Iterator
    inline self& operator++ () { ++current; return *this; }
    
    //: Postincrement
    //!wheredef: Forward Iterator
    inline self operator++ (int) { self tmp = *this; ++current; return tmp; }
    
    //: Preincrement
    //!wheredef: Bidirectional Iterator
    inline self& operator-- () { --current; return *this; }
    
    //: Postincrement
    //!wheredef: Bidirectional Iterator
    inline self operator-- (int) { self tmp = *this; --current; return tmp; }
    
    //: Iterator addition
    //!wheredef: Random Access Iterator
    inline self operator+ (Distance n) const {
      self c = current;
      c += n;
      return self(c.current, alpha);
    }
    
    //: Advance a distance
    //!wheredef: Random Access Iterator
    inline self& operator+= (Distance n) { current += n; return *this; }
    
    //: Subtract a distance
    //!wheredef: Random Access Iterator
    inline self operator- (Distance n) const {
      return self (current - n, alpha);
    }
    
    inline difference_type operator- (const self& x) const {
      return current - x.current;
    }
    
    //: Retreat a distance
    //!wheredef: Random Access Iterator
    inline self& operator-= (Distance n) { current -= n; return *this; }
    
    //: Access at an offset
    inline value_type operator[] (Distance n) const {
      return alpha * *(current + n);
    }
    //: Equality
    //!wheredef: Trivial Iterator
    inline bool operator==(const self& x) const 
    { return current == x.current; }
    
    //: Inequality
    //!wheredef: Trivial Iterator
    inline bool operator!=(const self& x) const 
    { return current != x.current; }

    //: Less than
    //!wheredef: Random Access Iterator
    inline bool operator<(const self& x) const 
    { return current < x.current; }  

  protected:
    RandomAccessIterator current;
    T alpha;
  };
  
  


  //: Scaled Container
  //!category: containers, adaptors
  //!component: type
  //
  //  This class in not meant to be used directly. Instead it is
  //  created automatically when the <TT>scaled(x,alpha)</TT> function is
  //  invoked. See the documentation for "Shortcut for Creating A
  //  Scaled Vector".  This vector type is READ ONLY therefore there
  //  are only const versions of things ie. there is no iterator
  //  typedef, just const_iterator.
  //  
  //!definition: scaled1D.h
  //!tparam: RandomAccessContainerRef - The type of underlying container
  //!models: RandomAccessRefContainerRef
  template <class RandomAccessIterator>
  class scaled1D {
    typedef RandomAccessIterator Iter;
    typedef scaled1D<Iter> self;
  public:
    /**@name Type Definitions */
    
    //: The value type
    typedef typename std::iterator_traits<Iter>::value_type value_type;
    //: The unsigned integral type for dimensions and indices
    typedef int size_type;
    
    //: The iterator type (do not use this)
    typedef scale_iterator<Iter, value_type> iterator;
    
    //: The const iterator type
    typedef scale_iterator<Iter, value_type> const_iterator;
    
    //: The const reverse iterator type
    typedef std::reverse_iterator<const_iterator> const_reverse_iterator;
    
    //: The pointer to the value type
    typedef typename const_iterator::pointer pointer;
    
    //: The reference type
    typedef typename const_iterator::reference reference;
    
    /* not really a reference, scale_iterator uses the value_type */
    //: The const reference type
    typedef typename const_iterator::reference const_reference;
    //: The difference type
    typedef typename const_iterator::difference_type difference_type;
    
    /**@name Constructors */
    //: Default constructor
    inline scaled1D() { }
    
    //: Normal constructor
    inline scaled1D(const Iter& b, const Iter& e, value_type scale_) 
      : rep_b(b), rep_e(e), scale(scale_) { }
    
    /* Why need that?
       inline scaled1D(const Iter& r, value_type scale_, do_scaled) 
       : rep(r), scale(scale_) { }
    */
#if 0
    //: Copy constructor
    inline scaled1D(const self& x) 
      : rep_b(x.rep_b), rep_e(x.rep_e), scale(x.scale) { }
#endif
    
    //: Assignment operator
    inline self& operator=(const self& x) {
      rep_b = x.rep_b; rep_e=x.rep_e; scale = x.scale; return *this;
    }

    //: Destructor
    inline ~scaled1D() { }

    //: Return a const iterator pointing to the beginning of the vector
    //!wheredef: Container
    inline const_iterator begin() const {
      return const_iterator(rep_b, scale); 
    }
    //: Return a const iterator pointing past the end of the vector
    //!wheredef: Container
    inline const_iterator end() const {
      return const_iterator(rep_e, scale); 
    }
    //: Return a const reverse iterator pointing to the last element of the vector
    //!wheredef: Reversible Container
    inline const_reverse_iterator rbegin() const {
      return const_reverse_iterator(end());
    }
    //: Return a const reverse iterator pointing past the end of the vector
    //!wheredef: Reversible Container
    inline const_reverse_iterator rend() const {
      return const_reverse_iterator(begin());
    }
    //: Return a const reference to the element with index i
    inline const_reference operator[](size_type i) const { 
      return rep_b[i] * scale; 
    }
    
    //: Return the size of the vector
    //!wheredef: Container  
    inline size_type size() const { return rep_e-rep_b; }
    
  protected:
    Iter rep_b;
    Iter rep_e;
    value_type scale;
  };

 
  //: Shortcut for Creating a Scaled Argument
  //   This function can be used to scale arguments in MTL
  //   functions. For example, to perform the vector addition operation
  //   z <- a x + b y one would do the following:
  //   <pre>
  //   mtl::add(scaled(x, alpha), scaled(y, beta), z);
  //   </pre>
  //   The actual multiplication by alpha and beta is done within the
  //   algorithm, so the performance is the same as if the add()
  //   function had parameters for alpha and beta.  The
  //   <TT>scaled()</TT> function can be used with any vector or matrix
  //   argument in MTL functions.  Do not confuse this function with
  //   <TT>mtl::scale()</TT> which are stand-alone functions.
  //!category: containers
  //!component: function
  //!definition: scaled1D.h
  //!typereqs: T must be convertible to Scalable's value_type
  //!typereqs: Scalable's value_type must be a model of Ring
  //!complexity: compile time and adds a single multiplication
  //!complexity: to each element access inside of any algorithm
  //!example: y_ax_y.cc
  template <class Scalable, class T> 
  inline
  scaled1D<typename Scalable::const_iterator>
  scaled(const Scalable& A, const T& alpha)
  {
    typedef scaled1D<typename Scalable::const_iterator> scaled_type;
    return scaled_type(A.begin(), A.end(), alpha);
  }
  
  
} /* namespace itl */


#endif
