//===========================================================================
//  CVS Information:                                                         
//                                                                           
//     $RCSfile: bicgstab.cpp,v $  $Revision: 1.3 $  $State: Exp $ 
//     $Author: llee $  $Date: 2001/10/18 14:08:31 $ 
//     $Locker:  $ 
//---------------------------------------------------------------------------
//                                                                           
// DESCRIPTION                                                               
//                                                                           
//---------------------------------------------------------------------------
//                                                                           
// LICENSE AGREEMENT                                                         
// Copyright 1997, University of Notre Dame.
// Authors: Andrew Lumsdaine, Lie-Quan Lee
//
// This file is part of the Iterative Template Library
//
// You should have received a copy of the License Agreement for the
// Iterative Template Library along with the software;  see the
// file LICENSE.  If not, contact Office of Research, University of Notre
// Dame, Notre Dame, IN  46556.
//
// Permission to modify the code and to distribute modified code is
// granted, provided the text of this NOTICE is retained, a notice that
// the code was modified is included with the above COPYRIGHT NOTICE and
// with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
// file is distributed with the modified code.
//
// LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
// By way of example, but not limitation, Licensor MAKES NO
// REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
// PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
// OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
// OR OTHER RIGHTS.
//---------------------------------------------------------------------------
//                                                                           
// REVISION HISTORY:                                                         
//                                                                           
// $Log: bicgstab.cpp,v $
// Revision 1.3  2001/10/18 14:08:31  llee
// re-organize the directory structures
//
// Revision 1.2  2000/07/27 15:50:20  llee1
// for vc++
//
// Revision 1.1  2000/07/26 21:49:13  llee1
// change file extension from .cc to .cpp
//
// Revision 1.3  2000/07/18 14:30:44  llee1
// *** empty log message ***
//
// Revision 1.2  2000/07/17 15:44:04  llee1
// *** empty log message ***
//
//===========================================================================

#include <mtl/matrix.h>
#include <mtl/mtl.h>
#include <mtl/utils.h>

#include <itl/interface/mtl.h>
#include <itl/krylov/bicgstab.h>
#include <itl/preconditioner/ssor.h>

/*

  In thsi example, we show how to use bicgstab algorithm, the output should be:

iteration 0: resid 2.23607
iteration 0: resid 1.77927
iteration 1: resid 1.36397
iteration 1: resid 1.05124
iteration 2: resid 0.0696221
iteration 2: resid 0.184695
iteration 3: resid 0.0234955
iteration 3: resid 2.71963e-09
finished! error code = 0
3 iterations
2.71963e-09 is actual final residual.
1.21626e-09 is actual relative tolerance achieved.
Relative tol: 1e-06  Absolute tol: 0
     1      2      0      0      3   x          0.205847  =       1
     4      5      0      6      0   x         0.0419363  =       1
     0      7      8      0      9   x         -0.178049  =       1
     0      0     10     11     12   x       -0.00551162  =       1
     0      0     13      0     14   x           0.23676  =       1
*/

typedef  double Type;

//begin
typedef mtl::matrix< Type, mtl::rectangle<>, 
		     mtl::array< mtl::compressed<> >, 
		     mtl::row_major >::type Matrix;
//end

int main (int argc, char* argv[]) 
{
  using std::cout;
  using std::endl;

  int max_iter = 50;
  //begin
  Matrix A(5, 5);
  //end
  A(0, 0) = 1.0;
  A(0, 1) = 2.0;
  A(0, 4) = 3.0;
  A(1, 0) = 4.0;
  A(1, 1) = 5.0;
  A(1, 3) = 6.0;
  A(2, 1) = 7.0;
  A(2, 2) = 8.0;
  A(2, 4) = 9.0;
  A(3, 2) = 10.;
  A(3, 3) = 11.;
  A(3, 4) = 12.;
  A(4, 2) = 13.; 
  A(4, 4) = 14.;
  //begin
  mtl::dense1D<Type> x(A.nrows(), Type(0));
  mtl::dense1D<Type> b(A.ncols());
  for (mtl::dense1D<Type>::iterator i=b.begin(); i!=b.end(); i++)
    *i = 1.;
  //SSOR preconditioner
  itl::SSOR<Matrix> precond(A);
  //iteration
  itl::noisy_iteration<double> iter(b, max_iter, 1e-6);
  //bicgstab algorithm
  itl::bicgstab(A, x, b, precond(), iter);
  //end

  //verify the result
  mtl::dense1D<Type> b1(A.ncols());
  itl::mult(A, x, b1);
  itl::add(b1, itl::scaled(b, -1.), b1);

  if ( itl::two_norm(b1) < 1.e-6) { //output
    for (int i=0; i<5; i++) {
      for (int j=0; j<5; j++) {
	cout.width(6);
	//!!! it is not recommended to use A(i,j) for large sparse matrices
	cout << A(i, j) << " ";
      }
      cout << "  x  ";
      cout.width(16);
      cout << x[i] << "  =  ";
      cout.width(6);
      cout << b[i] << endl;
    }
  } else {
    cout << "Residual " << iter.resid() << endl;
  }

  return 0;
}


