// --------------------------------------------------------------------
// PDF Resources
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2016  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "iperesources.h"

using namespace ipe;

// --------------------------------------------------------------------

/*! \class ipe::PdfResources
 * \ingroup base
 * \brief All the resources needed by the text objects in the document.
 */

void PdfResources::add(int num, PdfFile *file)
{
  if (object(num)) // already present
    return;
  std::unique_ptr<const PdfObj> obj = file->take(num);
  if (!obj)
    return;  // no such object
  const PdfObj *q = obj.get();
  iObjects[num] = std::move(obj);
  addIndirect(q, file);
  iEmbedSequence.push_back(num); // after all its dependencies!
}

void PdfResources::addIndirect(const PdfObj *q, PdfFile *file)
{
  if (q->array()) {
    const PdfArray *arr = q->array();
    for (int i = 0; i < arr->count(); ++i)
      addIndirect(arr->obj(i, nullptr), file);
  } else if (q->dict()) {
    const PdfDict *dict = q->dict();
    for (int i = 0; i < dict->count(); ++i)
      addIndirect(dict->value(i), file);
  } else if (q->ref())
    add(q->ref()->value(), file);
}

const PdfObj *PdfResources::object(int num) const noexcept
{
  auto got = iObjects.find(num);
  if (got != iObjects.end())
    return got->second.get();
  else
    return nullptr;
}

const PdfDict *PdfResources::get(String key) const noexcept
{
  auto it = std::find_if(iItems.begin(), iItems.end(),
			 [key](const Item &item)
			 { return item.iKey == key; } );
  if (it == iItems.end())
    return nullptr;
  else
    return it->iDict;
}

const PdfDict *PdfResources::getFont(int id) const noexcept
{
  auto it = iFontObjects.find(id);
  if (it != iFontObjects.end()) {
    const PdfObj *obj = object(it->second);
    if (obj && obj->dict())
      return obj->dict();
  }
  return nullptr;
}

bool PdfResources::collect(const PdfDict *dict, PdfFile *file)
{
  const PdfObj *res = dict->get("Resources", file);
  if (!res || !res->dict()) {
    ipeDebug("No /Resources in XForm.");
    return false;
  }
  /* A resource is a dictionary, like this:
    /Font << /F8 9 0 R /F10 18 0 R >>
    /ProcSet [ /PDF /Text ]
  */
  for (int i = 0; i < res->dict()->count(); ++i) {
    String key = res->dict()->key(i);
    const PdfObj *obj = res->dict()->value(i);
    if (key == "ProcSet") {
      // ipeDebug("ProcSet should be set to %s", obj->repr().z());
    } else {
      const PdfDict *rd = obj->dict();
      if (!rd) {
	ipeDebug("Resource is not a dictionary");
	return false;
      }
      auto it = std::find_if(iItems.begin(), iItems.end(),
			     [key](const Item &item)
			     { return item.iKey == key; } );
      PdfDict *d = nullptr;
      if (it == iItems.end()) {
	Item item;
	item.iKey = key;
	d = item.iDict = new PdfDict;
	iItems.push_back(item);
      } else
	d = it->iDict;
      for (int j = 0; j < rd->count(); ++j) {
	String elKey = rd->key(j);
	const PdfObj *el = rd->value(j);
	// ipeDebug("Inserting %s: %s -> %s",
	// key.z(), elKey.z(), el->repr().z());
	const PdfObj *prev = d->get(elKey, nullptr);
	if (prev) {
	  // TODO: check if old and new element are the same!
	} else {
	  if (el->name())
	    d->add(elKey, new PdfName(el->name()->value()));
	  else if (el->number())
	    d->add(elKey, new PdfNumber(el->number()->value()));
	  else if (el->ref()) {
	    int ref = el->ref()->value();
	    d->add(elKey, new PdfRef(ref));
	    add(ref, file);  // take all dependencies from file
	    if (key == "Font") {
	      if (elKey[0] != 'F') {
		ipeDebug("Weird font name: %s", elKey.z());
		return false;
	      }
	      int fontNumber = Lex(elKey.substr(1)).getInt();
	      iFontObjects[fontNumber] = ref;
	    }
	  } else {
	    ipeDebug("Surprising type in resource: %s", el->repr().z());
	    return false;
	  }
	}
      }
    }
  }
  return true;
}

// --------------------------------------------------------------------
