// --------------------------------------------------------------------
// Platform dependent methods
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2015  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipebase.h"
#include "ipeattributes.h"

#ifdef WIN32
#include <windows.h>
#include <shlobj.h>
#include <direct.h>
#else
#include <sys/wait.h>
#include <xlocale.h>
#endif
#include <cstdlib>
#include <sys/types.h>
#include <sys/stat.h>
#include <cstdarg>
#include <unistd.h>
#include <clocale>

using namespace ipe;

// --------------------------------------------------------------------

/*! \class ipe::Platform
  \ingroup base
  \brief Platform dependent methods.
*/

//! Return the Ipelib version.
/*! This is available as a function so that one can verify what
  version of Ipelib one has actually linked with (as opposed to the
  header files used during compilation).
*/
int Platform::libVersion()
{
  return IPELIB_VERSION;
}

// --------------------------------------------------------------------

static bool initialized = false;
static bool showDebug = false;
static Platform::DebugHandler debugHandler = 0;
#ifdef WIN32
// not yet available in MINGW RT library
// _locale_t ipeLocale;
#else
locale_t ipeLocale;
#endif

#ifdef WIN32
// Counterpart to Platform::wideToUtf8
void Platform::utf8ToWide(const char *utf8, std::vector<wchar_t> &wbuf)
{
  int rw = MultiByteToWideChar(CP_UTF8, 0, utf8, -1, NULL, 0);
  wbuf.resize(rw);
  MultiByteToWideChar(CP_UTF8, 0, utf8, -1, &wbuf[0], rw);
}
#endif

static void debugHandlerImpl(const char *msg)
{
  if (showDebug) {
    fprintf(stderr, "%s\n", msg);
#ifdef WIN32
    fflush(stderr);
    OutputDebugStringA(msg);
#endif
  }
}

static void cleanup_repository()
{
#ifdef WIN32
  // _free_locale(ipeLocale);
#else
  freelocale(ipeLocale);
#endif
  Repository::cleanup();
}

//! Initialize Ipelib.
/*! This method must be called before Ipelib is used.

  It creates a LC_NUMERIC locale set to 'C', which is necessary for
  correct loading and saving of Ipe objects.  The method also checks
  that the correct version of Ipelib is loaded, and aborts with an
  error message if the version is not correct.  Also enables ipeDebug
  messages if environment variable IPEDEBUG is defined.  (You can
  override this using setDebug).
*/
void Platform::initLib(int version)
{
  if (initialized)
    return;
  initialized = true;
  showDebug = false;
  if (getenv("IPEDEBUG") != 0)
    showDebug = true;
  debugHandler = debugHandlerImpl;
#ifdef WIN32
  // ipeLocale = ::_create_locale(LC_NUMERIC, "C");
#else
  ipeLocale = newlocale(LC_NUMERIC_MASK, "C", NULL);
#endif
  atexit(cleanup_repository);
#ifndef WIN32
  if (version == IPELIB_VERSION)
    return;
  fprintf(stderr,
	  "Ipetoipe has been compiled with header files for Ipelib %d\n"
	  "but is dynamically linked against libipe %d.\n"
	  "Check with 'ldd' which libipe is being loaded, and "
	  "replace it by the correct version or set LD_LIBRARY_PATH.\n"
	  , version, IPELIB_VERSION);
  exit(99);
#endif
}

//! Enable or disable display of ipeDebug messages.
void Platform::setDebug(bool debug)
{
  showDebug = debug;
}

// --------------------------------------------------------------------

void ipeDebug(const char *msg, ...)
{
  if (debugHandler) {
    char buf[8196];
    va_list ap;
    va_start(ap, msg);
    std::vsprintf(buf, msg, ap);
    va_end(ap);
    debugHandler(buf);
  }
}

// --------------------------------------------------------------------

//! Return correct path separator for this platform.
char Platform::pathSeparator()
{
#ifdef WIN32
  return '\\';
#else
  return '/';
#endif
}

//! Returns current working directory.
/*! Returns empty string if something fails. */
String Platform::currentDirectory()
{
#ifdef WIN32
  wchar_t *buffer;
  if ((buffer = _wgetcwd(0, 0)) == 0)
    return String();
  return Platform::wideToUtf8(buffer);
#else
  char buffer[1024];
  if (getcwd(buffer, 1024) != buffer)
    return String();
  return String(buffer);
#endif
}

#ifdef IPEBUNDLE
String Platform::ipeDir(const char *suffix)
{
#ifdef WIN32
  wchar_t exename[OFS_MAXPATHNAME];
  GetModuleFileNameW(0, exename, OFS_MAXPATHNAME);
  String exe = Platform::wideToUtf8(exename);
#else
#endif
  int i = exe.rfind(pathSeparator());
  if (i >= 0) {
    i = exe.left(i).rfind(pathSeparator());
    if (i >= 0)
      return exe.left(i + 1) + suffix;
  }
  ipeDebug("WARNING: Cannot find location for '%s'", suffix);
  return suffix; // this should better not happen
}
#endif

#ifndef WIN32
static String dotIpe()
{
  const char *home = getenv("HOME");
  if (!home)
    return String();
  String res = String(home) + "/.ipe";
  if (!Platform::fileExists(res) && mkdir(res.z(), 0700) != 0)
    return String();
  return res + "/";
}
#endif

//! Returns directory for running Latex.
/*! The directory is created if it does not exist.  Returns an empty
  string if the directory cannot be found or cannot be created.
  The directory returned ends in the path separator.
 */
String Platform::latexDirectory()
{
#ifdef WIN32
  String latexDir;
  const wchar_t *p = _wgetenv(L"IPELATEXDIR");
  if (getenv("IPEWINE"))
    latexDir = "z:\\tmp\\ipelatex";
  else if (p) {
    latexDir = Platform::wideToUtf8(p);
  } else {
    wchar_t szPath[MAX_PATH];
    if (SUCCEEDED(SHGetFolderPathW(NULL, CSIDL_LOCAL_APPDATA,
				   NULL, 0, szPath))) {
      latexDir = Platform::wideToUtf8(szPath) + "\\ipe";
    } else {
      p = _wgetenv(L"LOCALAPPDATA");
      if (p)
	latexDir = Platform::wideToUtf8(p) + "\\ipe";
      else
	latexDir = ipeDir("latexrun");
    }
  }
  if (latexDir.right(1) == "\\")
    latexDir = latexDir.left(latexDir.size() - 1);
  if (!fileExists(latexDir)) {
    if (Platform::mkdir(latexDir.z()) != 0)
      return String();
  }
  latexDir += "\\";
  return latexDir;
#else
  const char *p = getenv("IPELATEXDIR");
  String latexDir;
  if (p) {
    latexDir = p;
    if (latexDir.right(1) == "/")
      latexDir = latexDir.left(latexDir.size() - 1);
  } else {
    latexDir = dotIpe() + "latexrun";
  }
  if (!fileExists(latexDir) && mkdir(latexDir.z(), 0700) != 0)
    return String();
  latexDir += "/";
  return latexDir;
#endif
}

//! Returns filename of fontmap.
String Platform::fontmapFile()
{
  const char *p = getenv("IPEFONTMAP");
  if (p)
    return String(p);
#ifdef IPEBUNDLE
  return ipeDir("data\\fontmap.xml");
#else
  return IPEFONTMAP;
#endif
}

//! Determine whether file exists.
bool Platform::fileExists(String fname)
{
#ifdef WIN32
  std::vector<wchar_t> wname;
  utf8ToWide(fname.z(), wname);
  return (_waccess(&wname[0], F_OK) == 0);
#else
  return (access(fname.z(), F_OK) == 0);
#endif
}

//! Read entire file into string.
/*! Returns an empty string if file cannot be found or read.
  There is no way to distinguish an empty file from this. */
String Platform::readFile(String fname)
{
  std::FILE *file = Platform::fopen(fname.z(), "rb");
  if (!file)
    return String();
  String s;
  int ch;
  while ((ch = std::fgetc(file)) != EOF)
    s.append(ch);
  std::fclose(file);
  return s;
}

//! Runs pdflatex on file text.tex in given directory.
int Platform::runPdfLatex(String dir)
{
#ifdef WIN32
  const char *wine = getenv("IPEWINE");
  if (wine) {
    int secs = Lex(wine).getInt();
    std::system("/bin/sh -c \"cd /tmp/ipelatex; rm -f ipetemp.log;"
		"pdflatex ipetemp.tex\"");
    fprintf(stderr, "Waiting %d seconds for pdflatex\n", secs);
    Sleep(secs * 1000);
    return 0;
  } else {
    String s = dir + "runlatex.bat";
    std::FILE *f = Platform::fopen(s.z(), "wb");
    if (!f)
      return -1;

    if (dir.size() > 2 && dir[1] == ':')
      fprintf(f, "%s\r\n", dir.substr(0, 2).z());

    // CMD.EXE input needs to be encoded in "OEM codepage",
    // which can be different from "Windows codepage"
    std::vector<wchar_t> wbuf;
    utf8ToWide(dir.z(), wbuf);
    Buffer oemDir(2 * wbuf.size() + 1);
    CharToOemW(&wbuf[0], oemDir.data());

    fprintf(f, "cd \"%s\"\r\n", oemDir.data());
    fprintf(f, "pdflatex ipetemp.tex\r\n");
    std::fclose(f);

    // Declare and initialize process blocks
    PROCESS_INFORMATION processInformation;
    STARTUPINFOW startupInfo;

    memset(&processInformation, 0, sizeof(processInformation));
    memset(&startupInfo, 0, sizeof(startupInfo));
    startupInfo.cb = sizeof(startupInfo);

    // Call the executable program
    s = String("cmd /c call \"") + dir + String("runlatex.bat\"");
    utf8ToWide(s.z(), wbuf);

    int result = CreateProcessW(NULL, &wbuf[0], NULL, NULL, FALSE,
				NORMAL_PRIORITY_CLASS|CREATE_NO_WINDOW,
				NULL, NULL, &startupInfo, &processInformation);
    if (result == 0)
      return -1;  // failure to create process

    // Wait until child process exits.
    WaitForSingleObject(processInformation.hProcess, INFINITE);

    // Close process and thread handles.
    CloseHandle(processInformation.hProcess);
    CloseHandle(processInformation.hThread);
    return 0;
  }
#else
  String s("cd ");
  s += dir;
  s += "; rm -f ipetemp.log; pdflatex ipetemp.tex > /dev/null";
  int result = std::system(s.z());
  if (result != -1)
    result = WEXITSTATUS(result);
  return result;
#endif
}

#ifdef WIN32
FILE *Platform::fopen(const char *fname, const char *mode)
{
  std::vector<wchar_t> wname;
  utf8ToWide(fname, wname);
  std::vector<wchar_t> wmode;
  utf8ToWide(mode, wmode);
  return _wfopen(&wname[0], &wmode[0]);
}

int Platform::mkdir(const char *dname)
{
  std::vector<wchar_t> wname;
  utf8ToWide(dname, wname);
  return _wmkdir(&wname[0]);
}

struct Codepage {
  UINT cp;
  const char *name;
};

static const Codepage codepages[] = {
  { 28591, "iso-8859-1" },
  { 51949, "euc-kr" },
  { 51932, "euc-jp" },
  { 28592, "iso-8859-2" },
  { 21866, "koi8-u" },
  { 20866, "koi8-r" },
  { 932, "shift_jis" },
  { 0, 0 }
};

// See http://msdn.microsoft.com/en-us/library/
//  windows/desktop/dd317756%28v=vs.85%29.aspx
static UINT codepage(const String &encoding)
{
  const Codepage *cp = codepages;
  while (cp->name) {
    if (encoding == cp->name)
      return cp->cp;
    ++cp;
  }
  String s("You requested the character encoding '");
  s += encoding;
  s += "' for Latex conversion.\n"
    "There is no mapping for this encoding in Ipe.\n"
    "This may be a bug, please report it.";
  MessageBoxA(NULL, s.z(), "Error", MB_ICONEXCLAMATION | MB_OK);
  return CP_UTF8;
}

String Platform::winConv(const String &utf8, const String &encoding)
{
  UINT cp = codepage(encoding);
  int rw = MultiByteToWideChar(CP_UTF8, 0, utf8.z(), -1, NULL, 0);
  wchar_t wbuf[rw];
  MultiByteToWideChar(CP_UTF8, 0, utf8.z(), -1, wbuf, rw);
  int rm = WideCharToMultiByte(cp, 0, wbuf, -1, NULL, 0, NULL, NULL);
  char multi[rm];
  WideCharToMultiByte(cp, 0, wbuf, -1, multi, rm, NULL, NULL);
  return String(multi);
}

String Platform::wideToUtf8(const wchar_t *wbuf)
{
  int rm = WideCharToMultiByte(CP_UTF8, 0, wbuf, -1, NULL, 0, NULL, NULL);
  char multi[rm];
  WideCharToMultiByte(CP_UTF8, 0, wbuf, -1, multi, rm, NULL, NULL);
  return String(multi);
}

#endif

// --------------------------------------------------------------------

double Platform::toDouble(String s)
{
#ifdef WIN32
  return strtod(s.z(), 0);
  // return _strtod_l(s.z(), 0, ipeLocale);
#else
  return strtod_l(s.z(), 0, ipeLocale);
#endif
}

void ipeAssertionFailed(const char *file, int line, const char *assertion)
{
  fprintf(stderr, "Assertion failed on line #%d (%s): '%s'\n",
	  line, file, assertion);
  abort();
  // exit(-1);
}

// --------------------------------------------------------------------
