var spdx = require('spdx');
var correct = require('spdx-correct');

var validResult = {
  validForNewPackages: true,
  validForOldPackages: true
};

var genericWarning = (
  'license should be ' +
  'a valid SPDX license expression without "LicenseRef", ' +
  '"UNLICENSED", or ' +
  '"SEE LICENSE IN <filename>"'
);

var fileReferenceRE = /^SEE LICEN[CS]E IN (.+)$/;

function startsWith(prefix, string) {
  return string.slice(0, prefix.length) === prefix;
}

function usesLicenseRef(ast) {
  if (ast.hasOwnProperty('license')) {
    var license = ast.license;
    return (
      startsWith('LicenseRef', license) ||
      startsWith('DocumentRef', license)
    );
  } else {
    return (
      usesLicenseRef(ast.left) ||
      usesLicenseRef(ast.right)
    );
  }
}

module.exports = function(argument) {
  var ast;

  try {
    ast = spdx.parse(argument);
  } catch (e) {
    if (
      argument === 'UNLICENSED' ||
      argument === 'UNLICENCED' ||
      fileReferenceRE.test(argument)
    ) {
      return validResult;
    } else {
      var result = {
        validForOldPackages: false,
        validForNewPackages: false,
        warnings: [genericWarning]
      };
      var corrected = correct(argument);
      if (corrected) {
        result.warnings.push(
          'license is similar to the valid expression "' + corrected + '"'
        );
      }
      return result;
    }
  }

  if (usesLicenseRef(ast)) {
    return {
      validForNewPackages: false,
      validForOldPackages: false,
      warnings: [genericWarning]
    };
  } else {
    return validResult;
  }
};
