subroutine astro_observatory_command(line,error)
  use gbl_message
  use gkernel_interfaces
  use imager_interfaces, except_this=>astro_observatory_command
  use ast_params
  use ast_astro
  !---------------------------------------------------------------------
  ! @ private
  ! Support routine for command
  ! ARRAY name
  ! or
  ! ARRAY Long Lat Altitude [Sun-Avoidance]
  !---------------------------------------------------------------------
  character(len=*), intent(in)  :: line   !
  logical,          intent(out) :: error  !
  ! Global
  character(len=*), parameter :: rname='ARRAY' 
  integer(kind=4), parameter :: mobs=10   ! 38
  ! Data
  character(len=12) :: knownobs(mobs)
!  data knownobs /  &
!    'BURE','PDBI','NOEMA','PICOVELETA','VELETA','30M',  &  ! IRAM ones first
!    'ACA','ALMA','APEX','ATF','CARMA','CSO','EFFELSBERG','FAST','FCRAO','GBT',  &
!    'GI2T','HAYSTACK','HHT','IOTA','JCMT','KITTPEAK','KOSMA','LASILLA',         &
!    'MAUNAKEA','MEDICINA','NRO','NOBEYAMA','PARANAL','PTI','SEST','SMA','SMT',  &
!    'TRAO','VLA','VLT','WESTFORD','YEBES' /
  data knownobs /'ACA','ALMA','NOEMA','PARANAL','VLT','VLTI','IOTA','PTI','VLA','SMA'/ 
  !
  character(len=12) :: argum, previous
  integer(kind=4) :: nc,ier,nk,year,acycle
  character(len=24) :: clon,clat
  real(kind=8) :: alti,slim
  integer(kind=4) :: date(7)
  !
  previous=obsname
  error = .false.
  if (sic_narg(0).eq.0) then
    !
    ! No argument == decode GAG_ASTRO_OBS
    !
    ier = sic_getlog('GAG_ASTRO_OBS',argum)
    if (ier.ne.0) then
      call astro_message(seve%e,rname,  &
        'Failure in translation of GAG_ASTRO_OBS logical')
      error = .true.
      return
    endif
  endif
  call sic_ke (line,0,1,argum,nc,.false.,error)
  if (error) return
  if (argum.eq.'?') then
    ! sic_ambigs_list required to keep "observatory ?" working with the use of sic_ambigs_sub
    call sic_ambigs_list(rname,seve%i,'Choices are:',knownobs)
    return
  endif
  call sic_ambigs_sub(rname,argum,obsname,nk,knownobs,mobs,error)
  if (error) return
  if (nk.ne.0) then
    !name of the observatory is recognized
    call astro_observatory_byname (obsname,error)
    if (error)  return
    call astro_message(seve%i,rname,'Selected '//trim(obsname)//' observatory')
  else
    !
    ! Do not recognize the observatory name == decode coordinates
    call sic_ch(line,0,1,clon,nc,.true.,error)
    if (error)  return
    call sic_ch(line,0,2,clat,nc,.true.,error)
    if (error)  return
    call sic_r8(line,0,3,alti,.true.,error)
    if (error) return
    slim = 30.d0
    call sic_r8(line,0,4,slim,.false.,error)
    if (error) return
    call astro_observatory_bychcoords(clon,clat,alti,slim,error)
    if (error)  return
  endif
  ! Make sure the Ephemeris are available
  call astro_init
  !
  call utc (date)            ! read UTC from system clock
  call do_astro_time(jnow_utc,d_ut1,d_tdt,error)
  !
  config_count = 0
end subroutine astro_observatory_command
!
subroutine astro_observatory_byname(arg,error)
  use gbl_message
  use gkernel_interfaces
  use ast_astro
  !---------------------------------------------------------------------
  ! @ private
  !  Set the Astro observatory given its name
  !---------------------------------------------------------------------
  character(len=*), intent(in)    :: arg    ! Telescope name
  logical,          intent(inout) :: error  ! Logical error flag
  ! Local
  character(len=*), parameter :: rname='OBSERVATORY'
  character(len=4) :: teles  ! Comparison is based on 4 characters
                             ! Hide this "detail" here
  real(4) :: diam, beamsize, factor
  integer :: ier
  logical :: sim_exist
  !
  teles = arg  ! arg can be smaller or larger than 4 characters!
  freq = 100.d0
  sim_exist = sic_varexist('SIM%FNU')
  if (sim_exist) call sic_get_dble('SIM%FNU',freq,error)
  !
  config_prefix = 'astro_stations'
  call gwcs_observatory(arg,lonlat,altitude,slimit,diam,error)
  factor = 1.13   ! Default beamsize in Lambda/Diam
  select case (teles)
  case ('ALMA')
    config_prefix = 'alma-current-'
    call astro_message(seve%i,rname,'Available configurations 1 to 10')
    factor = 1.13
  case ('ACA')
    config_prefix = 'aca-current-'
    call astro_message(seve%i,rname,'Available configurations I and NS')
    factor = 1.13
  case ('NOEM')
    config_prefix = 'noema-12'
    call astro_message(seve%i,rname,'Available configurations A, B, C, D')
    factor = 1.22
  case ('VLA','JVLA') 
    freq = 10.d0
    config_prefix = 'vla-'
    call astro_message(seve%i,rname,'Available configurations A, BnA, B, CnB, C, DnC, D')
  case ('PARA','VLT ','VLTI')
    call astro_message(seve%i,rname,  &
      &  'Frequency set to 299.792458 THz = 1 micron')
    freq = 299792.458d0
  case ('IOTA')
    call astro_message(seve%i,rname,  &
      &  'Frequency set to 187.370286 THz = 1.6 micron')
    freq = 187370.286d0
  case ('PTI ')
    call astro_message(seve%i,rname,'Frequency set to 136.269299 THz = '//  &
       & '2.2 micron')
    freq = 136269.299d0
  case ('GI2T')
    call astro_message(seve%i,rname,'Frequency set to 500 THz = '//  &
       & '0.6 micron')
    freq = 499653.8397d0
  end select
  ier = sic_setlog('astro_stations',config_prefix)
  !
  if (sim_exist) then
    beamsize = factor*(299792458D0/freq/1d9)/diam*180*3600/acos(-1d0)
    call sic_let_dble('SIM%FNU',freq,error)
    call sic_let_real('SIM%PRIMARY',beamsize,error)
  endif
end subroutine astro_observatory_byname
!
subroutine astro_observatory_bychcoords(clon,clat,alt,slim,error)
  use gbl_message
  use gkernel_interfaces
  use imager_interfaces, only : astro_observatory_byr8coords
  !---------------------------------------------------------------------
  ! @ private
  !  Set Astro observatory given its longitude/latitude (sexagesimal
  ! strings), altitude and avoidance
  !---------------------------------------------------------------------
  character(len=*), intent(in)    :: clon   ! Telescope longitude (sexagesimal string)
  character(len=*), intent(in)    :: clat   ! Telescope latitude (sexagesimal string)
  real(kind=8),     intent(in)    :: alt    ! Altitude [km]
  real(kind=8),     intent(in)    :: slim   ! Sun avoidance
  logical,          intent(inout) :: error  ! Logical error flag
  ! Local
  integer(kind=4) :: llon,llat
  real(kind=8) :: lon,lat
  !
  llon = lenc(clon)
  call sic_sexa(clon(1:llon),llon,lon,error)
  if (error) return
  !
  llat = lenc(clat)
  call sic_sexa(clat(1:llat),llat,lat,error)
  if (error) return
  !
  call astro_observatory_byr8coords(lon,lat,alt,slim,error)
  if (error)  return
  !
end subroutine astro_observatory_bychcoords
!
subroutine astro_observatory_byr8coords(lon,lat,alt,slim,error)
  use gbl_message
  use gkernel_interfaces
  use ast_astro
  !---------------------------------------------------------------------
  ! @ private
  !  Set Astro observatory given its longitude/latitude (float values),
  ! altitude and avoidance
  !---------------------------------------------------------------------
  real(kind=8), intent(in)    :: lon    ! Telescope longitude [deg]
  real(kind=8), intent(in)    :: lat    ! Telescope latitude [deg]
  real(kind=8), intent(in)    :: alt    ! Altitude [km]
  real(kind=8), intent(in)    :: slim   ! Sun avoidance
  logical,      intent(inout) :: error  ! Logical error flag
  !
  real(8) :: xyz(3)
  character(len=message_length) :: mess
  !
  if (abs(lat).gt.90.d0) then
    call astro_message(seve%e,'OBSERVATORY','Latitude must be < 90 degrees')
    error = .true.
    return
  endif
  !
  lonlat(1) = lon
  lonlat(2) = lat
  altitude = alt
  slimit = slim
  obsname = "User defined"
  !
  call gwcs_lonlat2geo(lon,lat,alt,xyz)
  !
  write(mess,'(A,3(1X,F0.8))') 'XYZ',xyz
  call astro_message(seve%i,'OBSERVATORY',mess)
end subroutine astro_observatory_byr8coords
!
subroutine astro_message(seve,comm,mess)
  use imager_interfaces, only : map_message
  integer :: seve
  character(len=*) :: comm
  character(len=*) :: mess
  !
  call map_message(seve,comm,mess)
end subroutine astro_message
