import stat
import time
import glob
import sys
import os
import re

import iguanaIR

#output "constants"
LOG_FATAL  = 0
LOG_ERROR  = 1
LOG_WARN   = 2
LOG_ALWAYS = 2.5
LOG_NORMAL = 3
LOG_INFO   = 4
LOG_DEBUG  = 5

msgPrefixes = [
    "FATAL: ",
    "ERROR: ",
    "WARNING: ",
    "",
    "INFO: ",
    "DEBUG: "
]

# other constants
FLASHER_VERSION = 0xFF00

# global variables
currentLevel = LOG_NORMAL
logFile = None
usb_ir = None
loader = None
body = None

#Global Message Flag -- only print can't list devices once
MessageCannotlistdevice = False

warningMesg = """WARNING! WARNING! WARNING!

This application is used to reflash your IguanaWorks USB IR transceiver. 
While we believe this process is safe, there is always the possibility
of breaking the device. We recommend only reflashing your device if
you are having a specific problem with your device and IguanaWorks
support recommends reflashing your device. Otherwise, use at your own
risk. Non-working devices can be reflashed by IguanaWorks for a nominal
fee. You may press ctrl-c to break out of this application, but DO NOT
STOP THE PROGRAM WHILE IT IS WRITING THE FLASH! 

<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>
No other applications may use the device when reflashing the device! 
Before proceeding, stop LIRC and any other applications that use the 
device.
<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>

It is also a good idea to unplug and replug your device after the
reflasher completes.  In some cases certain functionality (reception
for example) will not work until the device is powered off and then
on.
"""

def dieCleanly(level = None):
    """Exit the application with proper cleanup."""

    if level == None:
        level = LOG_ERROR

    #exit with appropriate value
    if level == LOG_FATAL:
        sys.exit(1)
    sys.exit(0)


def message(level, msg):
    """Print a message to a certain debug level"""
    retval = None

    if level <= currentLevel or level == LOG_ALWAYS:
        out = sys.stdout

        # if logfile is open print to it instead
        if logFile == "-":
            out = sys.log
        elif level <= LOG_WARN:
            out = sys.stderr

        retval = msgPrefixes[int(level + 0.5)] + msg
        out.write(retval)
        retval = len(retval)

    if level <= LOG_FATAL:
        dieCleanly(level)

    return retval


def printUsage(msg = None):
    usage = "Usage: " + sys.argv[0] + " [OPTION]..." + """

--blank-pages : Write out blank pages in addition to code pages.

--body : Specify the file to read the body from.

--dry-run : Do not write any "dangerous" pages.

--force : Write the firmware regardless of the device version.

--flash-multiple : Loop flashing and testing multiple devices in a row.

-h
--help : Print this usage message.

--keep-loader : Keep the loader regardless of version detection.

--loader : Specify the file to read the loader from.

-l
--log-file : Specify a log to receive all messages.

--old-firmware : Specify the file to read old (monolithic) firmware from.

-q
--quiet : Decrease verbosity.

-v
--verbose : Increase verbosity.
"""

    if msg != None:
        message(LOG_FATAL, msg + usage)
    message(LOG_ALWAYS, usage)
    dieCleanly(LOG_ALWAYS)

def listDevices(path = iguanaIR.IGSOCK_NAME):
    global MessageCannotlistdevice
    devices = []
    try:
#        print os.path.split(path)[0]
#        print os.listdir(path)
#        print glob.glob(path + '*')
        for device in os.listdir(path):
            if not stat.S_ISLNK(os.lstat(os.path.join(path, device)).st_mode):
                devices.append(device)
    except Exception, inst:
        devices = ['0']
        if MessageCannotlistdevice==False:
            message(LOG_WARN,"""Guessing device "0", failed to list devices: \n%s\n""" % (inst) )
            MessageCannotlistdevice=True

    return devices

_conn = None
def connect(timeout = 2, willChange = False, quiet = False):
    global _conn

    # check every quarter second
    sleepTime = 0.25
    timeoutStart = timeout

    start = None
    while timeout > 0:
        # list the possible devices
        devices = listDevices()
        devices.sort()

        # no change, so lets just run on
        if len(devices) == 0:
            pass
        elif start is None:
            start = devices[0]
            if not willChange:
                break
        elif len(devices) == 1 and \
             start and start != devices[0]:
            break

        # delay and count down the timeout
        time.sleep(sleepTime)
        timeout -= sleepTime

    # make sure there is a device detected
    devices = listDevices()
    if len(devices) == 0:
        if not quiet:
            message(LOG_ERROR,
                    "Device not found.  Plug the device in and try again.\n")
            raise KeyboardInterrupt()
    # connect to the device
    else:
        if len(devices) > 1 and not quiet:
            message(LOG_WARN,
                    "Multiple devices found, using '%s'\n" % devices[0])
        _conn = iguanaIR.connect(devices[0])


def reconnect():
    global _conn
    if _conn is not None:
        iguanaIR.close(_conn)
        _conn = None

# used detect the version before we try to write it
def deviceTransaction(type, data = '', quiet = False):
    global _conn

    # connect on demand
    if _conn is None:
        connect()

    retval = False
    req = iguanaIR.createRequest(type, data)
    if not iguanaIR.writeRequest(req, _conn):
        if not quiet:
            message(LOG_ERROR, 'Failed to write packet. %s\n' % _conn)
    else:
        resp = iguanaIR.readResponse(_conn, 3000)
        if resp is None:
            if not quiet:
                message(LOG_ERROR, "No response received.\n")
        elif type == iguanaIR.IG_DEV_GETVERSION:
            if not iguanaIR.responseIsError(resp):
                data = iguanaIR.removeData(resp)
                retval = ord(data[0]) + (ord(data[1]) << 8)
        elif iguanaIR.responseIsError(resp):
            if not quiet:
                # it is alright to get 0x0 on a reset (TODO: why get anything?)
                if type != iguanaIR.IG_DEV_RESET or iguanaIR.code(resp) != 0:
                    message(LOG_ERROR, 'Error response code: 0x%s\n' % iguanaIR.code(resp))
        elif type == iguanaIR.IG_DEV_GETFEATURES:
            retval = ord(iguanaIR.removeData(resp)[0])
        elif type == iguanaIR.IG_DEV_IDSOFF or \
             type == iguanaIR.IG_DEV_IDSON or \
             type == iguanaIR.IG_DEV_SETID:
            retval = True
        else:
            retval = iguanaIR.removeData(resp)

    return retval

def replugWarning():
    message(LOG_WARN, "Communication temporarily lost....\n")
    message(LOG_NORMAL, "  Please unplug and replug the device before continuing.\n    Press enter to continue.\n")
    sys.stdin.readline()

def deviceVersion(target = None):
    count = 0
    version = None
    while not version:
        # usually be quiet, but complain before failure
        version = deviceTransaction(iguanaIR.IG_DEV_GETVERSION,
                                    quiet = count < 4)
        if not version:
            count += 1
            if count == 3:
                replugWarning()
                count = 0
            else:
                time.sleep(0.20)
            message(LOG_INFO, "Attempting to connect to device....\n")
            connect(quiet = True)

    # just print the version of the target device
    message(LOG_INFO, "Found device version 0x%x\n" % version)

    if target and target != version:
        if not UNSAFE:
            message(LOG_NORMAL, "Dry run completed.\n")
            raise KeyboardInterrupt
        else:
            message(LOG_FATAL, "Incorrect version (0x%4.4x != 0x%4.4x) found.  Reflashing has failed.\n" % (version, target))

    return version

def deviceFeatures():
    return deviceTransaction(iguanaIR.IG_DEV_GETFEATURES,
                             quiet = True)

def appendPage(start, page, pages):
    if page:
        for x in page:
            if x != '30':
                pages.append({ 'start'  : start,
                               'bytes'  : page })
                break

def readPages(input):
    pages = []

    # read pages from the input file into the pages array
    lineNum = 0
    start = None
    page = []
    for line in input:
        line = line.strip()

        lineNum += 1
        if line:
            # parse .hex files from the compiler
            if line[0] == ':':
                lead = line[0:3]
                address = int(line[3:7], 16)
                pad = line[7:9]
                body = line[9:-2]
                checksum = line[-2:]

                if address == 0 and lineNum != 1:
                    break

                bytes = []
                for x in range(len(body) / 2):
                    bytes.append(body[x * 2:x * 2 + 2])
                appendPage(address, bytes[0:64], pages)
                appendPage(address + 64, bytes[64:], pages)

                if pad != '00':
                    message(LOG_FATAL, "pad is not 00: %s\n" % pad)
            # parse the output of the read command from the programmer
            else:
                bytes = line.split()
                if len(bytes) != 17:
                    message(LOG_WARN, "Ignoring line: %s\n" % line)
                else:
                    address = int(bytes.pop(0)[:-1], 16)
                    if address % 64 == 0:
                        # save any old page
                        appendPage(start, page, pages)

                        # prepare for the next page
                        start = address
                        page = bytes
                    else:
                        page.extend(bytes)

    # save the last page
    if start is not None:
        appendPage(start, page, pages)

    message(LOG_INFO, 'Found %d pages that contain data.\n' % len(pages))

    return pages

def preparePages(pages):
    # compute the iguana 'writeBlock' data for each page
    for page in pages:
        # compute the checksum right before we need it
        chksum = 0
        for x in page['bytes']:
            chksum += int(x, 16)
        page['chksum'] = chksum

        # construct 68 data bytes we need
        page['data'] = chr(page['start'] / 64)
        if UNSAFE:
            page['data'] += chr(0x42)
        else:
            page['data'] += chr(0)
        page['data'] += chr(page['chksum'] / 0x100) + \
                        chr(page['chksum'] % 0x100)
        for byte in page['bytes']:
            page['data'] += chr(int(byte, 16))

    # we have to write the pages in reverse to avoid messing up the jump
    # table before we're ready, so reverse the page order.
    pages.reverse()

def writePageToDevice(page):
    # write the page to the device
#TODO: DEBUG:
#    deviceTransaction(iguanaIR.IG_DEV_RECVON)
    result = deviceTransaction(iguanaIR.IG_DEV_WRITEBLOCK, page['data'])
    if result is None or result is False:
        message(LOG_FATAL, "deviceTransaction failed.\n")
    elif len(result) > 0:
        chksum = 0
        for x in struct.unpack('B' * len(result), result):
            chksum = (chksum << 8) + x
#        print '%x' % chksum, len(result), [result]
        if chksum != page['chksum']:
            message(LOG_FATAL, "checksum does not match: 0x%x != 0x%x on page %s\n" % (chksum, page['chksum'], page['start'] / 64))

def writePagesToDevice(pages):
    # need to know the starting version
    version = deviceVersion()

    # write each page out
    count = 0
    for page in pages:
        count += 1

        if page['start'] == 0 and not UNSAFE:
            message(LOG_WARN, "Not writing page 0 during dry run.\n")
            continue

        # note that we're writing a page
        if os.isatty(sys.stdout.fileno()):
            sys.stdout.write('\r')
        sys.stdout.write("    Writing page %d/%d (offset %d)" % (count, len(pages), page['start'] / 64))

        if os.isatty(sys.stdout.fileno()):
            sys.stdout.write('  ')
        else:
            sys.stdout.write('\n')
        sys.stdout.flush()

        writePageToDevice(page)

        # if we're in an older device delay and reconnect
        if version <= 4:
            connect(5, True)
    sys.stdout.write('\n')

    # reset the device (it may have already)
    if version > 4:
        message(LOG_INFO, "Rebooting the device.\n")
        deviceTransaction(iguanaIR.IG_DEV_RESET)

def writeHexFile(filename, fullFlash = False, setFeatures = False):
    blanks = []

    # read the firmware memory map in
    pages = readPages(open(filename, 'r'))

    if fullFlash:
        blanks = range(128)
        for page in pages:
            blanks.remove(page['start'] / 64)

    # place the features if necessary (and possible)
    if setFeatures and \
       (targetVersion >= 0x0101 or targetVersion == 0x0000):
        message(LOG_INFO, "Setting features to 0x%x\n" % setFeatures)
        for page in pages:
            if page['start'] == 0xC00:
                page['bytes'][61] = '%2.2x' % setFeatures

    # write out the code pages
    preparePages(pages)
    writePagesToDevice(pages)

    # prepare and write out the blank pages if requested
    pages = []
    for page in blanks:
        pages.append({'start' : page * 64,
                      'bytes' : ['30'] * 64 })
    if pages:
        sys.stdout.write('Blanking pages on the device:\n')
        preparePages(pages)
        writePagesToDevice(pages)

def checkFileExists(path):
    try:
        os.stat(path)
    except OSError, inst:
        if inst[0] != errno.ENOENT:
            raise
        return False
    return True

def listHexFiles(type, devOK = False, hexDir = None):
    if hexDir is None:
        testdir = os.path.split(sys.path[0])
        rootdir = sys.path[0]
        if len(testdir) == 2 and testdir[1] == 'library.zip':
            rootdir = testdir[0]
        hexDir = os.path.join(rootdir, 'hex')

    retval = glob.glob(os.path.join(hexDir, '%s-*.hex' % type))
    if not devOK:
        for item in glob.glob(os.path.join(hexDir, '%s-0.hex' % type)):
            retval.remove(item)
    return retval

def findHexFile(type, target = None, mustFind = True, devOK = False):
    latest = None
    latestVersion = None

    for hex in listHexFiles(type, devOK):
        # ensure the file really exists
        if checkFileExists(hex):
            version = int(hex.rsplit('-',1)[-1].split('.', 1)[0])
            # see if we can take any version
            if target is None:
                # see if we found a newer version
                if latest is None or version > latestVersion:
                    latestVersion = version
                    latest = hex
            # searching for a specific version
            elif version == target:
                latest = hex
                break
    else:        
        if target is None and devOK:
            best = glob.glob(os.path.join(rootdir,
                                          'hex', '%s-0.hex' % type))
            if len(best) == 1 and checkFileExists(best[0]):
                latest = best[0]
    
    if latest is None and mustFind:
        if target is not None:
            message(LOG_FATAL, "Failed to find %s version %d\nFailed to find hex files, is the hex directory missing?" % (type, target))
        else:
            message(LOG_FATAL, "Failed to find any %s version\nFailed to find hex files, is the hex directory missing?" % (type))
            
        
        
    return latest

def hexVersion(filename):
    return int(re.search('.*?(\d+)\.hex$', filename).group(1))

def reflashDevice(features, keepLoader):
    wrote = False
    id = None

    # not necessary when only writing the body
    version = deviceVersion()
    if (not usb_ir and \
        (version >> 8) == hexVersion(loader) and hexVersion(loader) != 0) or \
       (usb_ir and version == hexVersion(usb_ir)):
        keepLoader = not forceFlash

    # get the id so we can reset it at the end
    if not forceFlash and (version & 0xFF) != 0x00:
        id = deviceTransaction(iguanaIR.IG_DEV_GETID, quiet = True)
    # write a single page at the end of the device to avoid id problems
    page = { 'start' : 8128, 'bytes' : ['30'] * 64 }
    page['bytes'][0] = '7F'
    preparePages([page])
    writePageToDevice(page)

    # write the reflasher if necessary but first detect the version
    # before we try to write it
    if not keepLoader and deviceVersion() != FLASHER_VERSION:
        sys.stdout.write('  Writing reflasher to the device:\n')
        writeHexFile(os.path.join(sys.path[0], 'hex', 'reflasher.hex'))
        deviceVersion(FLASHER_VERSION)
        wrote = True

    # put on the usb_ir if that's the final target version
    if usb_ir is not None:
        if keepLoader:
            message(LOG_NORMAL, "Keeping the previously installed firmware.\n")
        else:
            sys.stdout.write('  Writing usb_ir to the device:\n')
            writeHexFile(usb_ir, fullFlash)
            deviceVersion(hexVersion(usb_ir))
            wrote = True
    # put on the boot loader and the correct body
    else:
        version = deviceVersion()
        if keepLoader and body is not None:
            message(LOG_NORMAL, "Keeping the previously installed loader.\n")
        else:
            sys.stdout.write('  Writing loader to the device:\n')
            blank = fullFlash
            if body is None:
                blank = True
            writeHexFile(loader, blank)
            version = deviceVersion(hexVersion(loader) << 8)
            wrote = True

        if body is not None:
            sys.stdout.write('  Writing body to the device:\n')
            checkingIDs = ord(deviceTransaction(iguanaIR.IG_DEV_IDSTATE)) != 0
            if deviceTransaction(iguanaIR.IG_DEV_IDSOFF):
                writeHexFile(body, setFeatures = features)
                deviceVersion((targetVersion & 0xFF00) | hexVersion(body))
                if checkingIDs:
                    deviceTransaction(iguanaIR.IG_DEV_IDSON)
            wrote = True

    if id:
        message(LOG_NORMAL, "Attempting to reset device id to '%s'.\n" % id)
        deviceTransaction(iguanaIR.IG_DEV_SETID, id)
    else:
        message(LOG_NORMAL, "Clearing device id.\n")
        deviceTransaction(iguanaIR.IG_DEV_SETID, '')

    if wrote and deviceVersion() < 0x0100:
        replugWarning()

def checkReceiver():
    # read 40 signals to test the receiver
    deviceTransaction(iguanaIR.IG_DEV_RECVON)
    count = 0
    while count < 40:
        packet = iguanaIR.readResponse(_conn, 1000)
        if packet is None:
            break

        data = iguanaIR.removeData(packet)
        format = 'I' * (len(data) / 4)
        for signal in struct.unpack(format, data):
            if signal != 0x25555:
                count += 1

            type = 'space: '
            if signal & iguanaIR.IG_PULSE_BIT:
                type = 'pulse: '
            message(LOG_INFO,
                    type + "%d\n" % (signal & iguanaIR.IG_PULSE_MASK))
    else:
        deviceTransaction(iguanaIR.IG_DEV_RECVOFF)
        return True

    return False

def sendPanasonicPower():
    # send a signal on all active channels
    panasonicPower = """pulse 3498
space 1685
pulse 448
space 362
pulse 448
space 1237
pulse 448
space 384
pulse 448
space 362
pulse 448
space 362
pulse 448
space 362
pulse 448
space 384
pulse 448
space 362
pulse 448
space 362
pulse 448
space 362
pulse 448
space 362
pulse 448
space 384
pulse 448
space 362
pulse 448
space 1237
pulse 448
space 362
pulse 448
space 362
pulse 448
space 362
pulse 448
space 362
pulse 448
space 384
pulse 448
space 362
pulse 448
space 362
pulse 448
space 362
pulse 448
space 384
pulse 448
space 1237
pulse 448
space 362
pulse 448
space 384
pulse 448
space 362
pulse 448
space 362
pulse 448
space 362
pulse 448
space 384
pulse 448
space 362
pulse 448
space 362
pulse 448
space 1237
pulse 448
space 362
pulse 448
space 1237
pulse 448
space 1237
pulse 448
space 1237
pulse 448
space 1237
pulse 448
space 362
pulse 448
space 384
pulse 448
space 1237
pulse 448
space 362
pulse 448
space 1237
pulse 448
space 1237
pulse 448
space 1237
pulse 448
space 1237
pulse 448
space 362
pulse 448
space 1237
pulse 448
"""
    panasonicPower = """pulse 3498
space 3157
pulse 981
space 2368
pulse 1002
space 2325
pulse 1002
space 618
pulse 1002
space 2325
pulse 874
space 2474
pulse 853
space 768
pulse 874
space 768
pulse 874
space 2474
pulse 874
space 768
pulse 874
space 768
pulse 853
space 768
pulse 874
space 768
pulse 874
space 768
pulse 874
space 2474
pulse 874
space 768
pulse 874
space 768
pulse 874
space 2474
pulse 874
space 2474
pulse 874
space 768
pulse 874
space 2474
pulse 874
space 2474
pulse 853
space 2496
pulse 832
"""

    output = tempfile.NamedTemporaryFile()
    output.write(panasonicPower)
    output.flush()
    (count, data) = iguanaIR.readPulseFile(output.name)
    output.close()

    deviceTransaction(iguanaIR.IG_DEV_SEND, data)


