#!/usr/bin/env python

import unittest

import os

from HyperEstraier import *

class ConditionTest(unittest.TestCase):
	def setUp(self):
		self.cond = Condition()
	
	def testConstantVariables(self):
		self.cond.CONDSURE
		self.cond.CONDUSU
		self.cond.CONDFAST
		self.cond.CONDAGIT
		self.cond.CONDNOIDF
		self.cond.CONDSIMPLE

	def testSetPhrase(self):
		self.cond.set_phrase("foo")
	
	def testAddAttr(self):
		self.cond.add_attr('@uri ISTREW .txt')
	
	def testSetOrder(self):
		self.cond.set_order('@id')
	
	def testSetMax(self):
		self.cond.set_max(30)
	
	def testSetOptions(self):
		self.cond.set_options(Condition.CONDSURE)

class DocumentTest(unittest.TestCase):
	def setUp(self):
		self.doc = Document()
	
	def testAddAttr(self):
		self.doc.add_attr('@uri', 'http://tokuhirom.dnsalias.org/')
	
	def testAddText(self):
		self.doc.add_text('Hyper Estraier')
		self.doc.add_text('QDBM')
	
	def testAddHiddenText(self):
		self.doc.add_hidden_text('a full-text search system for communities')
	
	def testID(self):
		self.doc.id()
	
	def testAttrNames(self):
		self.doc.attr_names()
	
	def testAttr(self):
		self.assertEqual(self.doc.attr('@foo'), None)
		self.doc.add_attr('@foo', 'bar')
		self.assertEqual(self.doc.attr('@foo'), 'bar')
	
	def testTexts(self):
		self.assertEqual(list(self.doc.texts()), [])
		self.doc.add_text('foo')
		self.doc.add_text('bar')
		self.assertEqual(list(self.doc.texts()), ['foo', 'bar'])
	
	"""
	def testCatTexts(self):
		self.assertEqual(self.doc.cat_texts(), '')
		self.doc.add_text('foo')
		self.doc.add_text('bar')
		self.assertEqual(self.doc.cat_texts(), 'foobar')
		"""
	
	def testDumpDraft(self):
		self.doc.add_text('foo')
		self.doc.add_text('bar')
		self.assertEqual(self.doc.dump_draft(), "\nfoo\nbar\n")
	
	def testMakeSnippet(self):
		self.doc.add_text('foo')
		self.doc.add_text('bar')
		self.doc.add_text('baz')
		self.assertEqual(
				self.doc.make_snippet(['foo', 'baz'], 30, 30, 20),
				'foo\tfoo\n bar \nbaz\tbaz\n\n')
	
	def testScanWords(self):
		self.doc.add_text('foo')
		self.assertEqual(self.doc.scan_words('foo'), True)
		self.assertEqual(self.doc.scan_words('bar'), False)

class DatabaseStaticTest(unittest.TestCase):
	def testConstantVariables(self):
		Database.ERRNOERR
		Database.ERRINVAL
		Database.ERRACCES
		Database.ERRLOCK
		Database.ERRDB
		Database.ERRIO
		Database.ERRNOITEM
		Database.ERRMISC
		Database.DBREADER
		Database.DBWRITER
		Database.DBCREAT
		Database.DBTRUNC
		Database.DBNOLCK
		Database.DBLCKNB
		Database.DBPERFNG
		Database.PDCLEAN
		Database.ODCLEAN
		Database.OPTNOPURGE
		Database.OPTNODBOPT
		Database.GDNOATTR
		Database.GDNOTEXT
	
	def testErrMsg(self):
		Database.err_msg(0)

class DatabaseWriterTest(unittest.TestCase):
	def setUp(self):
		self.db = Database()
		self.assertEqual(self.db.open('casket',
				Database.DBWRITER | Database.DBCREAT),
				True)
		self.doc = Document()
		self.doc.add_attr('@uri', 'http://www.example.com/')
		self.doc.add_text('foo')
		self.assertEqual(self.db.put_doc(self.doc, 0), True)

	def testError(self):
		self.assertEqual(self.db.error(), Database.ERRNOERR)
		self.db.close()

	def testFatal(self):
		self.assertEqual(self.db.fatal(), False)
		self.db.close()

	def testFlush(self):
		self.assertEqual(self.db.flush(0), True)
		self.db.close()

	def testSync(self):
		self.assertEqual(self.db.sync(), True)
		self.db.close()
	
	def testOptimize(self):
		self.assertEqual(self.db.optimize(0), True)
		self.db.close()
	
	def testPutDoc(self):
		doc = Document()
		doc.add_attr('@uri', 'http://www.example.com/')
		doc.add_text('foo')
		self.assertEqual(self.db.put_doc(doc, 0), True)
		self.db.close()
	
	def testOutDoc(self):
		self.assertEqual(self.db.out_doc(self.doc.id(), 0), True)
		self.db.close()
	
	def testGetDoc(self):
		self.assertNotEqual(self.db.get_doc(self.doc.id(), 0), None)
		self.db.close()

	def testUriToID(self):
		self.assertEqual(self.db.uri_to_id('http://www.example.com/'), self.doc.id())
		self.db.close()
	
	def testEtchDoc(self):
		self.assertEqual(self.db.etch_doc(self.doc, 10).keys(), ['foo'])
		self.db.close()
	
	def testIterInit(self):
		self.assertEqual(self.db.iter_init(), True)
		self.db.close()
	
	def testIterNext(self):
		self.assertEqual(self.db.iter_init(), True)
		self.db.iter_next()
		self.db.close()
	
	def testName(self):
		self.assertEqual(self.db.name(), 'casket')
		self.db.close()
	
	def testDocNum(self):
		self.assertEqual(self.db.doc_num(), 1)
		self.db.close()

	def testWordNum(self):
		self.assertEqual(self.db.word_num(), 1)
		self.db.close()

	def testSize(self):
		self.assert_(self.db.size()>0)
		self.db.close()
	
	def testSearch(self):
		cond = Condition()
		cond.set_phrase('foo')
		self.assertNotEqual(len(list(self.db.search(cond, 0))), 0)
		self.db.close()
	
	def testSetCacheSize(self):
		self.db.set_cache_size(100, 100, 100)
		self.db.close()
	
	def testSetSpecialCache(self):
		self.db.set_special_cache('hoge', 100)
		self.db.close()

if __name__ == '__main__':
	unittest.main()

