/***************************************************************
 *
 * Copyright (C) 1990-2007, Condor Team, Computer Sciences Department,
 * University of Wisconsin-Madison, WI.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License.  You may
 * obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***************************************************************/


#ifndef _CONDOR_TRANSFER_QUEUE_H
#define _CONDOR_TRANSFER_QUEUE_H


#include "dc_service.h"
#include "dc_transfer_queue.h"
#include "simplelist.h"
#include "reli_sock.h"
#include "generic_stats.h"

class IOStats {
 public:
	stats_entry_sum_ema_rate<double> bytes_sent;
	stats_entry_sum_ema_rate<double> bytes_received;
	stats_entry_sum_ema_rate<double> file_read;
	stats_entry_sum_ema_rate<double> file_write;
	stats_entry_sum_ema_rate<double> net_read;
	stats_entry_sum_ema_rate<double> net_write;
	stats_entry_abs<double> upload_MB_waiting;
	stats_entry_abs<double> download_MB_waiting;

	void Add(IOStats &s);
	void ConfigureEMAHorizons(classy_counted_ptr<stats_ema_config> config);
};

// transfer queue server's representation of a client
class TransferQueueRequest {
 public:
	TransferQueueRequest(ReliSock *sock,filesize_t sandbox_size,char const *fname,char const *jobid,char const *queue_user,bool downloading,time_t max_queue_age);
	~TransferQueueRequest();

	char const *Description();

	bool SendGoAhead(XFER_QUEUE_ENUM go_ahead=XFER_QUEUE_GO_AHEAD,char const *reason=NULL);

	bool ReadReport(class TransferQueueManager *manager);

	ReliSock *m_sock;
	MyString m_queue_user;   // Name of file transfer queue user. (TRANSFER_QUEUE_USER_EXPR)
	std::string m_up_down_queue_user; // queue user prefixed by "U" or "D" for upload/download
	MyString m_jobid;   // For information purposes, the job associated with
	                    // this file transfer.
	double m_sandbox_size_MB;
	MyString m_fname;   // File this client originally requested to transfer.
	                    // In current implementation, it may silently move on
	                    // to a different file without notifying us.
	bool m_downloading; // true if client wants to download a file; o.w. upload
	bool m_gave_go_ahead; // true if we told this client to go ahead

	time_t m_max_queue_age; // clean transfer from queue after this time
	                        // 0 indicates no limit
	time_t m_time_born;
	time_t m_time_go_ahead;

	MyString m_description; // buffer for Description()
};

class TransferQueueManager: public Service {
 public:
	TransferQueueManager();
	~TransferQueueManager();

	void InitAndReconfig();

	void RegisterHandlers();

	int HandleRequest(int cmd,Stream *stream);

	int HandleReport( Stream *sock );

		// Iterate through queue, pruning disconnected clients, and
		// giving the go ahead to those that deserve it.
	void CheckTransferQueue();

		// This is called to register a future call to CheckTransferQueue.
	void TransferQueueChanged();

	bool GetContactInfo(char const *command_sock_addr, std::string &contact_str);

	int GetNumUploading() { return m_uploading; }
	int GetNumDownloading() { return m_downloading; }
	int GetNumWaitingToUpload() { return m_waiting_to_upload; }
	int GetNumWaitingToDownload() { return m_waiting_to_download; }
	int GetUploadWaitTime() { return m_upload_wait_time; }
	int GetDownloadWaitTime() { return m_download_wait_time; }
	int GetMaxUploading() { return m_max_uploads; }
	int GetMaxDownloading() { return m_max_downloads; }

	void publish(ClassAd *ad);
	void publish(ClassAd *ad, char const *publish_config);
	void publish(ClassAd *ad,int pubflags);

	void AddRecentIOStats(IOStats &s,const std::string up_down_queue_user);
 private:
	SimpleList<TransferQueueRequest *> m_xfer_queue;
	int m_max_uploads;   // 0 if unlimited
	int m_max_downloads; // 0 if unlimited
	time_t m_default_max_queue_age; // 0 if unlimited

	bool m_throttle_disk_load;
	double m_disk_load_low_throttle;
	double m_disk_load_high_throttle;
	int m_throttle_disk_load_max_concurrency;
	time_t m_throttle_disk_load_incremented;
	time_t m_throttle_disk_load_increment_wait;
	std::string m_disk_throttle_short_horizon;
	std::string m_disk_throttle_long_horizon;

	int m_check_queue_timer;

	int m_uploading;
	int m_downloading;
	int m_waiting_to_upload;
	int m_waiting_to_download;
	int m_upload_wait_time;
	int m_download_wait_time;

	stats_entry_abs<int> m_max_uploading_stat;
	stats_entry_abs<int> m_max_downloading_stat;
	stats_entry_abs<int> m_uploading_stat;
	stats_entry_abs<int> m_downloading_stat;
	stats_entry_abs<int> m_waiting_to_upload_stat;
	stats_entry_abs<int> m_waiting_to_download_stat;
	stats_entry_abs<int> m_upload_wait_time_stat;
	stats_entry_abs<int> m_download_wait_time_stat;

	stats_entry_abs<double> m_disk_throttle_low_stat;
	stats_entry_abs<double> m_disk_throttle_high_stat;
	stats_entry_abs<int> m_disk_throttle_limit_stat;
	stats_entry_ema<double> m_disk_throttle_excess;
	stats_entry_ema<double> m_disk_throttle_shortfall;

	unsigned int m_round_robin_counter; // increments each time we send GoAhead to a client

	class TransferQueueUser {
	public:
		TransferQueueUser(): running(0), idle(0), recency(0) {}
		bool Stale(unsigned int stale_recency);
		unsigned int running;
		unsigned int idle;
		unsigned int recency; // round robin counter at time of last GoAhead
		IOStats iostats;
	};
	typedef std::map< std::string,TransferQueueUser > QueueUserMap;
	QueueUserMap m_queue_users;      // key = up_down_queue_user, value = TransferQueueUser record

	unsigned int m_round_robin_garbage_counter;
	time_t m_round_robin_garbage_time;

	IOStats m_iostats;
	int m_update_iostats_interval;
	int m_update_iostats_timer;
	classy_counted_ptr<stats_ema_config> ema_config;
	StatisticsPool m_stat_pool;
	int m_publish_flags;

	bool AddRequest( TransferQueueRequest *client );
	void RemoveRequest( TransferQueueRequest *client );

	TransferQueueUser &GetUserRec(const std::string &user);
	void SetRoundRobinRecency(const std::string &user);
	void CollectUserRecGarbage(ClassAd *unpublish_ad);
	void ClearRoundRobinRecency();
	void ClearTransferCounts();
	void UpdateIOStats();
	void IOStatsChanged();
	void RegisterStats(char const *user,IOStats &iostats,bool unregister=false,ClassAd *unpublish_ad=NULL);
	void UnregisterStats(char const *user,IOStats &iostats,ClassAd *unpublish_ad) {
		RegisterStats(user,iostats,true,unpublish_ad);
	}

	void parseThrottleConfig(char const *config_param,bool &enable_throttle,double &low,double &high,std::string &throttle_short_horizon,std::string &throttle_long_horizon,time_t &throttle_increment_wait);

};


#endif
