/*
 * Hrsh2csv is a command line utility for converting PDB desktop files created by
 * the Hours (http://hours.sourceforge.net) Palm application on a handheld into
 * files in the CSV (comma separated value) format.
 * 
 * Copyright (C) 2002 Peter Novotnik
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/** 
 * @file pdblib.c
   @brief contains the implementation of function declarations in pdblib.h

   authors      donpedro <peternov1@gmx.de>
                Eric Santonacci <Eric.Santonacci@talc.fr>
   $Id: pdblib.c,v 1.4 2003/05/02 21:47:01 geppetto Exp $
*/


#include <stdlib.h>
#include "pdblib.h"
#include "globals.h"




/**
 * loads the appInfo of a database
 * @param dbFile - input database file
 * @param appInfoID - address in the file where to read the app info block
 * @param size - size of the info block (how much to read from dbFile)
 * @return pointer to the block (allocated in memory) or NULL on failure
 * <br><b>NOTE:</b> you must free the memory allocated by this function with free()
 */
void * pdbLoadAppInfoType( FILE * dbFile, unsigned int appInfoID, size_t size )
{
	void * infoP;

	if( !size )
		return (NULL);

	infoP = (void *)malloc( size );
	if( !infoP )
		return (NULL);

	if( fseek( dbFile, appInfoID, SEEK_SET ) )
	{
		free( infoP );
		return (NULL);
	}

	if( fread( infoP, sizeof( char ), size, dbFile ) == 0 )
	{
		free( infoP );
		return (NULL);
	}

	return (infoP);
}






/**
 * allocates an PDBHeaderType objects and tries to initialize it. 
 * the information is being retrieved from the file database
 * <br><b>this is called as the very first operation on the dbFile 
 * as the appropriate data is at the beginning of the file</b>
 *	@param dbFile - file to read from
 * @return NULL - on failure, else - pointer to the allocated object
 */
PDBHeaderType * pdbLoadDBHeader( FILE * dbFile )
{
	PDBHeaderType * headerP = NULL;

	/* allocate a new header struct */
	if( (headerP = (PDBHeaderType *)malloc( sizeof( PDBHeaderType ) )) == NULL )
	{
		fprintf( stderr, "can\'t allocate new header structure" );
		exit( EXIT_FAILURE );
	}


  /* attempt to retrieve DB header from file */
	if( fread( &headerP->name, sizeof( char ), 32, dbFile ) == 0 )
	{
		fprintf( stderr, "can\'t get name of database.\n" );
		free( headerP );
		return (NULL);
	}


	/* a try to allocate buffer for the PDBHeaderType */
	if( fread( &headerP->attributes, sizeof( headerP->attributes ), 1, dbFile ) == 0 )
	{
		fprintf( stderr, "can\'t get attributes of database.\n" );
		free( headerP );
		return (NULL);
	}
#ifdef __MAC2PC_CONVERSION__
	headerP->attributes = MAC2PC_SHORT( headerP->attributes );/* WARNING: Intel specific! */
#endif //__MAC2PC_CONVERSION__
		
	
	/* retrieve the version of the database */
	if( fread( &headerP->version, sizeof( short ), 1, dbFile ) == 0 )
	{
		fprintf( stderr, "can\'t get version of database.\n" );
		free( headerP );
		return (NULL);
	}
#ifdef __MAC2PC_CONVERSION__
	headerP->version = MAC2PC_SHORT( headerP->version ); /* WARNING: Intel specific! */
#endif //__MAC2PC_CONVERSION__


	/* retrieve the createDate of the database */
	if( fread( &headerP->creationDate, sizeof( int ), 1, dbFile ) == 0 )
	{
		fprintf( stderr, "can\'t get creation date of database.\n" );
		free( headerP );
		return (NULL);
	}
#ifdef __MAC2PC_CONVERSION__
	headerP->creationDate = MAC2PC_INT( headerP->creationDate ); /* WARNING: Intel specific! */
#endif //__MAC2PC_CONVERSION__

	/* retrieve the last modificationDate of the database */
	if( fread( &headerP->modificationDate, sizeof( int ), 1, dbFile ) == 0 )
	{
		fprintf( stderr, "can\'t get modification date of database.\n" );
		free( headerP );
		return (NULL);
	}
#ifdef __MAC2PC_CONVERSION__
	headerP->modificationDate = MAC2PC_INT( headerP->modificationDate ); /* WARNING: Intel specific */
#endif //__MAC2PC_CONVERSION__

	
	/* retrieve the last backup date of the database */
	if( fread( &headerP->lastBackupDate, sizeof( int ), 1, dbFile ) == 0 )
	{
		fprintf( stderr, "can\'t get last backup date of database.\n" );
		free( headerP );
		return (NULL);
	}
#ifdef __MAC2PC_CONVERSION__
	headerP->lastBackupDate = MAC2PC_INT( headerP->lastBackupDate ); /* WARNING: Intel specific */
#endif //__MAC2PC_CONVERSION__


	/* retrieve the modificationNumber of the database that is incremented each time there is a change in it */
	if( fread( &headerP->modificationNumber, sizeof( int ), 1, dbFile ) == 0 )
	{
		fprintf( stderr, "can\'t get modification number of database.\n" );
		free( headerP );
		return (NULL);
	}
#ifdef __MAC2PC_CONVERSION__
	headerP->modificationNumber = MAC2PC_INT( headerP->modificationNumber ); /* WARNING: Intel specific */
#endif //__MAC2PC_CONVERSION__

	
	/* retrieve the address of the application info block if there is any */
	if( fread( &headerP->appInfoID, sizeof( unsigned int ), 1, dbFile ) == 0 )
	{
		fprintf( stderr, "can\'t get appInfoID of database.\n" );
		free( headerP );
		return (NULL);
	}
#ifdef __MAC2PC_CONVERSION__
	headerP->appInfoID = MAC2PC_INT( headerP->appInfoID ); /* WARNING: Intel specific */ 
#endif //__MAC2PC_CONVERSION__


	/* retrieve the address of the sort block info id.
	 *
	 * WARNING: 
	 * IN THE CURRENT IMPLEMENTATION OF THE HOTSYNC PROCESS THE
	 * SORT BLOCK OF A DATABASE IS NOT BACKED UP !!!
	 */
	if( fread( &headerP->sortInfoID, sizeof( int ), 1, dbFile ) == 0 )
	{
		fprintf( stderr, "can\'t get sortInfoID of database.\n" );
		free( headerP );
		return (NULL);
	}
#ifdef __MAC2PC_CONVERSION__
	headerP->sortInfoID = MAC2PC_INT( headerP->sortInfoID ); /* WARNING: Intel specific */ 
#endif //__MAC2PC_CONVERSION__

	
	/* retrieve the TYPE of the database; number represented through four characters */
	if( fread( &headerP->type, sizeof( char ), 4, dbFile ) == 0 )
	{
		fprintf( stderr, "can\'t get type of database.\n" );
		free( headerP );
		return NULL;
	}


	/* retrieve the CREATOR ID of the database; number represented through four characters */
	if( fread( &headerP->creator, sizeof( char ), 4, dbFile ) == 0 )
	{
		fprintf( stderr, "can\'t get creator of database.\n" );
		free( headerP );
		return NULL;
	}


	/* retrieve the seed for uniqueID */
	/* i don't know excactly for what this is used */
	if( fread( &headerP->uniqueIDSeed, sizeof( int ), 1, dbFile ) == 0 )
	{
		fprintf( stderr, "can\'t get uniqueIDSeed of database.\n" );
		free( headerP );
		return (NULL);
	}
#ifdef __MAC2PC_CONVERSION__
	headerP->uniqueIDSeed = MAC2PC_INT( headerP->uniqueIDSeed ); /* WARNING: Intel specific */ 
#endif //__MAC2PC_CONVERSION__

	/* header completly read :-) */
	return (headerP);
}






/**
 * loads the standard data that each record holds and fills up our buffer for it
 * @param dbFile - input database file
 *	@param records - pointer to an array to be filled up
 * @return -1 - on error, else - number of records
 * <br><b>if the __MAC2PC_CONVERSION__ macro is defined this function performs the
 * conversion of the byte order</b>
 */
int pdbLoadDBRecords( FILE *dbFile, PDBRecordType **records )
{
	int i;
	int nextRecordListID = 0;
	unsigned short numRecords = 0;

	if( fread( &nextRecordListID, sizeof( int ), 1, dbFile ) == 0 )
		return (-1);
	if( fread( &numRecords, sizeof( unsigned short ), 1, dbFile ) == 0 )
		return (-1);

#ifdef __MAC2PC_CONVERSION__
	numRecords = (unsigned short)MAC2PC_SHORT( numRecords ); /* WARNING: Intel specific */
#endif //__MAC2PC_CONVERSION__

	if( !numRecords )
		return (0);

	(*records) = malloc(sizeof( PDBRecordType ) * numRecords);
	if( !(*records) )
		return (-1);

	for( i = 0; i < numRecords; i++ )
	{
		if( fread( &((*records)[i].localChunkID), sizeof( unsigned int ), 1, dbFile ) == 0 )
			return (-1);

#ifdef __MAC2PC_CONVERSION__
		(*records)[i].localChunkID = MAC2PC_INT( (*records)[i].localChunkID ); /* WARNING: Intel specific */
#endif //__MAC2PC_CONVERSION__
		if( fread( &((*records)[i].attributes), sizeof( char ), 1, dbFile ) == 0 )
			return (-1);

		if( fread( &((*records)[i].uniqueID), sizeof( char ), 3, dbFile ) == 0 )
			return (-1);
	}

	return (numRecords);
}







/**
 * given an index this function loads a record from the input file and returns a pointer to the allocated (record)object
 * @param dbFile - input database file
 * @param recIndex - index of the record
 * @param records - pointer to an array holding the standart record information
 * @param numRecs - number of records in the records array
 */
void * pdbLoadDBRecord( FILE *dbFile, int recIndex, PDBRecordType * records, int numRecs )
{
	int recLength = 0;
	long fileOffset = 0;
	void *recPtr;

	/* should actually not happen ... but one never knows */
	if( recIndex < 0 || recIndex >= numRecs )
		return (NULL);

	if( recIndex == (numRecs - 1) )
	{
		fseek( dbFile, 0, SEEK_END );
		recLength = ftell( dbFile ) - records[recIndex].localChunkID;
		fseek( dbFile, fileOffset, SEEK_SET );
	}
	else
		recLength = records[recIndex + 1].localChunkID - records[recIndex].localChunkID;

	if( (recPtr = malloc( recLength )) == NULL )
		return (NULL);
	
	fseek( dbFile, (long)records[recIndex].localChunkID, SEEK_SET );
	if( fread( recPtr, 1, recLength, dbFile ) == 0 )
	{
		free( recPtr );
		return (NULL);
	}

	return (recPtr);
}






/**
 * does all necessary things to walk through the file and calls for each record our passed callback function.
 * before starting the loop the parse each record the prolog callback function is invoked
 * after we passed the walktrough the passed epilog function is called
 * @param dbFile - palm pilot database desktop file
 * @param outfile - file the data should be written to
 * @param appInfoSize - size of the appInfoBlock in bytes
 * @param prologFunc - if not NULL this function will be called before entering a loop to go through all records
 * <br><b> if this function returns a different value to pdbErrNone the we will immediatelly return</b>
 * @param parseRecordFunc - if not NULL this function will be called for each record
 * @param epilogFunc - this function if not NULL is called after all records has been parsed
 * @return pdbErrNone - on success, pdbErrIvalidParam, pdbErrGeneralError, pdbErrCreatorDifferent, pdbErrTypeDifferent - on failure
 * <br>
 * NOTE:            this function sets the current position of the dbFile initially
 *                  to 0 and then walks though the file
 */

int pdbConvertDB( FILE * dbFile, FILE * outfile, pdbGetDBInfoTypeSize getInfoSize, pdbPrologParsingRecordsFunc prologFunc, pdbConvertRecordFunc parseRecordFunc, pdbEpilogParsingRecordsFunc epilogFunc )
{
   PDBHeaderType * headerP;
 	PDBRecordType * recordsP = NULL;
   void *  infoP;
   void *  recP;
   int     numRecs;
   int     i;
   int     ret_val = pdbErrNone;

   if( !dbFile || !outfile || !getInfoSize )
   {
      fprintf( stderr, ERROR_PREFIX " Invalid param.\n" );
      return (pdbErrInvalidParam);
   }

   if( fseek( dbFile, 0, SEEK_SET ) )
   {
      fprintf( stderr, ERROR_PREFIX " fseek() failed.\n" );
      return (pdbErrGeneralError);
   }

	if( (headerP = pdbLoadDBHeader( dbFile )) == NULL )
	{
		fprintf( stderr, ERROR_PREFIX " Not a PDB database.\n" );
		return (pdbErrGeneralError);
	}

	if( gVerbose )
		fprintf( stderr, "\
PalmPilot PDB database file:\n\
  name:       \'%s\'\n\
  type:       \'%c%c%c%c\'\n\
  creator:    \'%c%c%c%c\'\n",
											headerP->name, headerP->type[0], headerP->type[1], 
											headerP->type[2], headerP->type[3], 
											headerP->creator[0], headerP->creator[1], 
											headerP->creator[2], headerP->creator[3]
						);

	numRecs = pdbLoadDBRecords( dbFile, &recordsP );
	if( numRecs == -1 )
	{
		fprintf( stderr, ERROR_PREFIX " Not a PDB database.\n" );
		if( recordsP )
			free( recordsP );
		free( headerP );
		return (pdbErrGeneralError);
	}

	if( gVerbose )
    fprintf( stderr, "\
  numRecs:    %d\n\
  dbVersion:  %d\n", 
					numRecs, headerP->version );

	if( gVerbose )
		fprintf( stderr, "\
  appInfoID:  0x%08x\n", headerP->appInfoID );

	if( headerP->appInfoID )
		infoP = pdbLoadAppInfoType( dbFile, headerP->appInfoID, getInfoSize( headerP->version ) );
	else 
		infoP = NULL;

   if( prologFunc )
   {
      if( (ret_val = prologFunc( outfile, numRecs, headerP, infoP )) != pdbErrNone )
         return (ret_val);
   }

   if( parseRecordFunc )
   {
      for( i = 0; i < numRecs; i++ )
      {
         recP = pdbLoadDBRecord( dbFile, i, recordsP, numRecs );
         if( recP )
         {
            parseRecordFunc( outfile, i, recP, &(recordsP[i]), headerP, infoP );

            free( recP );
         }
      }
   }

	if( epilogFunc )
		epilogFunc( outfile, numRecs, headerP, infoP );

   if( infoP )
      free( infoP );
   free( recordsP );
   free( headerP );

   return (ret_val);
}
