/*
 * Hrsh2csv is a command line utility for converting PDB desktop files created by
 * the Hours (http://hours.sourceforge.net) Palm application on a handheld into
 * files in the CSV (comma separated value) format.
 *
 * Copyright (C) 2002 Peter Novotnik
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/* FOR BEST FORMATING OF THIS FILE SET TABSTOP TO 2 */

#include <stdio.h>
#include <stdlib.h>		/* because of EXIT definitions */
#include <string.h>		/* because of strcmp function */
#include "hrsh2csv.h"
#include "globals.h"

#ifndef DIR_SEPARATOR
#	ifdef __WIN32__
#		define DIR_SEPARATOR	'\\'
#	else
#		define DIR_SEPARATOR '/'
#	endif
#endif

enum  {	
				CONVERT,
				PRINT_DB_VERSION,
				PRINT_DB_INFO
			};

static int    validate_total_time_pattern( char * pattern );
static char * get_basename( char * progname );
static void 	print_version( char * prog_name );
static void 	print_help( char * prog_name );


/*
 * FUNCTION:		validate_total_time_pattern
 * DESCRIPTION:	this routine validates the passed pattern
 * PARAMETERS:	the pattern
 * RETURNS:			zero if the pattern is valid, otherwise non-zero
 */
static int validate_total_time_pattern( char * pattern )
{
	while( *pattern )
	{
		if( *pattern == '%' )
		{
			pattern++;
			if( *pattern != '%' && *pattern != 'H' && *pattern != 'M' )
				return 1;
		}
		pattern++;
	}

	return 0;
}

/* 
 * FUNCTION:		get_basename
 * DESCRIPTION: this routine extracts the base name of the passed filename
 * PARAMETERS:	the filename
 * RETURNS:			a pointer to next character after the last DIR_SEPARATOR or
 * 							a pointer to the first character if there is no DIR_SEPARATOR
 */
char * get_basename( char * progname )
{
	char * ret_val = progname;
	while( *progname )
	{
		if( *progname == DIR_SEPARATOR && *(progname+1) )
			ret_val = progname+1;
		progname++;
	}
	return (ret_val);
}

/*
 * FUNCTION:		print_version
 * DESCRIPTION:	prints the application version to stdout
 * PARAMETRRS: 	the name of this program
 * RETURNS:			nothing
 */
void print_version( char * prog_name )
{
	prog_name = get_basename( prog_name );

	printf( "%s %d.%d\n\
Copyright (C) 2003 Peter Novotnik <peternov1@gmx.de>\n\
\n\
This program comes with NO WARRANTY, to the extent permitted by law.\n\
You may redistribute it under the terms of the GNU General Public License;\n\
see the file named COPYING for details.\n", prog_name,
HRSH2CSV_MAJOR_VERSION, HRSH2CSV_MINOR_VERSION );
}

/*
 * FUNCTION:		print_help
 * DESCRIPTION:	prints a help message on how to use this program
 * PARAMETERS:	name of the program
 * RETURNS:			nothing
 */
void print_help( char * prog_name )
{
	prog_name = get_basename( prog_name );

	printf( "Usage: %s [OPTIONS] ... FILE\n\n", prog_name );
	printf( "`%s\' converts PDB files (typeID: \'DATA\', creatorID: \'HRSH\')\n\
created by the `Hours\' [http://hours.sourceforge.net] Palm\n\
application into ASCII files in the CSV (comma separated value) format.\n\n", prog_name );
	printf( "Options:\n\
      --help                  print this help, then exit\n\
      --version               print the program version, then exit\n\
\n\
      --dbversion             print the database version to stdout, then exit\n\
      --dbinfo                print database information to stdout, then exit\n\
                              this should work with any Palm database\n\
\n\
      --banner                print a banner after conversion\n\
      --no-header             supress the column titels\n\
      --only-ascii            replace non-ascii characters with a blank\n\
      --keep-dots             don\'t convert dots into a dash\n\
      --keep-newlines         don\'t replace newlines in record's description\n\
                              it is strongly recommended NOT to use this option\n\
      --date FORMAT           format of the date, defaults to `%%F\'\n\
                              lenght is limited to %d characters\n\
                              see strftime(3) for more information\n\
      --time FORMAT           define time format, defaults to `%%H:%%M\'\n\
                              see strftime(3) for more information\n\
                              lenght is limited to %d characters\n\
                              this effects the format of the begin and end time\n\
      --total-time FORMAT     define total hours format, defaults to `%%H:%%M\'\n\
                              FORMAT may hold the following options:\n\
                                %%H - will be replaced hours\n\
                                %%M - will be replaced minutes\n\
                                %%%% - is the `%%\' character\n\
                              length is limited to %d characters\n\
                              this effects the break and total times\n\
      --dont-enclose-values   don\'t enclose values with double quotes\n\
                              use this only if you know what you are doing\n\
  -s, --separator STRING      separator between values in a row, defaults to `,\'\n\
  -o, --output FILENAME       write output into FILENAME\n\
  -q, --quiet                 supress program\'s info messages\n\
", DATE_STRING_BUF_LEN-1, TIME_STRING_BUF_LEN-1, TOTAL_TIME_STRING_BUF_LEN-1 );

	printf( "\nIf `-o FILENAME\' is omitted, data is written to standard output.\n\n\
Report bugs to <peternov1@gmx.de>.\n" );
}

/*
 * FUNCTION:			main
 * DESCRIPTION:		checks the paramters and calls the converter
 * PARAMETERS:		see k&r bible
 * RETURNS:				EXIT_FAILURE on error
 * 								EXIT_SUCCESS on success
 */
int main( int argc, char ** argv )
{
	char * infile_name = NULL;
	char * outfile_name = NULL;
	FILE * input_file = NULL;
	FILE * output_file = NULL;
	int i;
	int operation = CONVERT;

#define PRINTF_TRY_HELP(name) 					fprintf( stderr, "Try `%s --help\' for more information.\n", name )
#define PRINTF_ARG_REQURIRED(name,arg) 	fprintf( stderr, "%s: Option `%s\' requires an argument.\n", name, arg )
#define CHECK_NEXT_ARG 									if( i+1 >= argc ) 											\
																				{																				\
																					PRINTF_ARG_REQURIRED(*argv,argv[i]);	\
																					PRINTF_TRY_HELP(*argv);								\
																					return (EXIT_FAILURE);								\
																				}

	/* parse the parameters */
	for( i = 1; i < argc; i++ )
	{
		if( *argv[i] == '-' )
		{
			if( !strcmp( argv[i], "-o" ) || !strcmp( argv[i], "--output" ) )
			{
				CHECK_NEXT_ARG
				outfile_name = argv[++i];
			}
			else if( !strcmp( argv[i], "--date" ) )
			{
				CHECK_NEXT_ARG
				gDatePattern = argv[++i];
			}
			else if( !strcmp( argv[i], "--time" ) )
			{
				CHECK_NEXT_ARG
				gTimePattern = argv[++i];
			}
			else if( !strcmp( argv[i], "-s" ) || !strcmp( argv[i], "--separator" ) )
			{
				CHECK_NEXT_ARG
				gSeparatorString = argv[++i];
			}
			else if( !strcmp( argv[i], "--total-time" ) )
			{
				CHECK_NEXT_ARG
				if( validate_total_time_pattern( argv[++i] ) )
				{
					fprintf( stderr, "%s: wrong format for --total-time: `%s\'\n'", *argv, argv[i] );
					PRINTF_TRY_HELP(*argv);
					return (EXIT_FAILURE);
				}
				gTotalTimePattern = argv[i];
			}
			else if( !strcmp( argv[i], "--help" ) )
			{
				print_help( *argv );
				return (EXIT_SUCCESS);
			}
			else if( !strcmp( argv[i], "--version" ) )
			{
				print_version( *argv );
				return (EXIT_SUCCESS);
			}
			else if( !strcmp( argv[i], "--dont-enclose-values" ) )
				gDontEncloseValues = 1;
			else if( !strcmp( argv[i], "--keep-newlines" ) )
				gDontRemoveNL = 1;
			else if( !strcmp( argv[i], "--keep-dots" ) )
				gDontConvert0x95 = 1;
			else if( !strcmp( argv[i], "--only-ascii" ) )
				gRemoveNonASCII = 1;
			else if( !strcmp( argv[i], "--no-header" ) )
				gPrintHeader = 0;
			else if( !strcmp( argv[i], "--banner" ) )
				gPrintBanner = 1;
			else if( !strcmp( argv[i] , "--dbversion" ) )
				operation = PRINT_DB_VERSION;
			else if( !strcmp( argv[i], "--dbinfo" ) )
				operation = PRINT_DB_INFO;
			else if( !strcmp( argv[i], "-q" ) || !strcmp( argv[i], "--quiet" ) )
				gVerbose = 0;
			else
			{
				fprintf( stderr, "%s: unknown option -- `%s\'\n", *argv, argv[i] );
				PRINTF_TRY_HELP(*argv);
				return (EXIT_SUCCESS);
			}
		}
		else
		{
			if( infile_name )
			{
				fprintf( stderr, "%s: can convert only one file\n", *argv );
				PRINTF_TRY_HELP(*argv);
				return (EXIT_SUCCESS);
			}
			infile_name = argv[i];
		}
	}

	if( !infile_name )
	{
		fprintf( stderr, "%s: What to convert?\n", *argv );
		PRINTF_TRY_HELP(*argv);
		return (EXIT_FAILURE);
	}

	if( !(input_file = fopen( infile_name, "rb" )) )
	{
		fprintf( stderr, "%s: error on opening intput database file %s\n", *argv, infile_name );
		return (EXIT_FAILURE);
	}

	if( operation == CONVERT )
	{
		if( outfile_name )
		{
			if( !(output_file = fopen( outfile_name, "w" )) )
			{
				fprintf( stderr, "%s error on opening/creating output file %s\n", *argv, outfile_name );
				return (EXIT_FAILURE);
			}
		}
		else
		{
			output_file = stdout;
		}

		if( gPrintBanner )
		{
			gARGC = argc;
			gARGV = argv;
		}

		pdbConvertDB( input_file, output_file, hrshGetSizeOfDBInfoType,
				(pdbPrologParsingRecordsFunc)hrshBeforeIterationCallback,
				(pdbConvertRecordFunc)hrshConvertRecord,
				(pdbEpilogParsingRecordsFunc)hrshAfterIterationCallback );

		if( output_file != stdout )
		{
			fclose( output_file );
		}
	} 
	else if( operation == PRINT_DB_VERSION )
		hrshPrintDBVersion( input_file );
	else if( operation == PRINT_DB_INFO )
		hrshPrintDBInfo( input_file );

	fclose( input_file );

	return (EXIT_SUCCESS);
}
