// Copyright (c) 2017, Lawrence Livermore National Security, LLC.
// Produced at the Lawrence Livermore National Laboratory (LLNL).
// Written by Cosmin G. Petra, petra1@llnl.gov.
// LLNL-CODE-742473. All rights reserved.
//
// This file is part of HiOp. For details, see https://github.com/LLNL/hiop. HiOp 
// is released under the BSD 3-clause license (https://opensource.org/licenses/BSD-3-Clause). 
// Please also read “Additional BSD Notice” below.
//
// Redistribution and use in source and binary forms, with or without modification, 
// are permitted provided that the following conditions are met:
// i. Redistributions of source code must retain the above copyright notice, this list 
// of conditions and the disclaimer below.
// ii. Redistributions in binary form must reproduce the above copyright notice, 
// this list of conditions and the disclaimer (as noted below) in the documentation and/or 
// other materials provided with the distribution.
// iii. Neither the name of the LLNS/LLNL nor the names of its contributors may be used to 
// endorse or promote products derived from this software without specific prior written 
// permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
// SHALL LAWRENCE LIVERMORE NATIONAL SECURITY, LLC, THE U.S. DEPARTMENT OF ENERGY OR 
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS 
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
// AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Additional BSD Notice
// 1. This notice is required to be provided under our contract with the U.S. Department 
// of Energy (DOE). This work was produced at Lawrence Livermore National Laboratory under 
// Contract No. DE-AC52-07NA27344 with the DOE.
// 2. Neither the United States Government nor Lawrence Livermore National Security, LLC 
// nor any of their employees, makes any warranty, express or implied, or assumes any 
// liability or responsibility for the accuracy, completeness, or usefulness of any 
// information, apparatus, product, or process disclosed, or represents that its use would
// not infringe privately-owned rights.
// 3. Also, reference herein to any specific commercial products, process, or services by 
// trade name, trademark, manufacturer or otherwise does not necessarily constitute or 
// imply its endorsement, recommendation, or favoring by the United States Government or 
// Lawrence Livermore National Security, LLC. The views and opinions of authors expressed 
// herein do not necessarily state or reflect those of the United States Government or 
// Lawrence Livermore National Security, LLC, and shall not be used for advertising or 
// product endorsement purposes.

#include "hiopHessianLowRank.hpp"
#include "LinAlgFactory.hpp"
#include "hiopVectorPar.hpp"

#include "hiop_blasdefs.hpp"

#ifdef HIOP_USE_MPI
#include "mpi.h"
#endif

//#include <unistd.h> //!remove me

#include <cassert>
#include <cstring>
#include <cmath>

#include <vector>
#include <algorithm>
using namespace std;

#define SIGMA_STRATEGY1 1
#define SIGMA_STRATEGY2 2
#define SIGMA_STRATEGY3 3
#define SIGMA_STRATEGY4 4
#define SIGMA_CONSTANT  5

namespace hiop
{

  hiopHessianLowRank::hiopHessianLowRank(hiopNlpDenseConstraints* nlp_in, int max_mem_len)
  : l_max(max_mem_len), l_curr(-1), sigma(1.), sigma0(1.), nlp(nlp_in), matrixChanged(false)
{
  DhInv = nlp->alloc_primal_vec();
  St_ = nlp->alloc_multivector_primal(0, l_max);
  Yt_ = St_->alloc_clone(); //faster than nlp->alloc_multivector_primal(...);
  //these are local
  L_  = LinearAlgebraFactory::create_matrix_dense("DEFAULT", 0, 0);
  D_  = LinearAlgebraFactory::create_vector("DEFAULT", 0);
  V_  = LinearAlgebraFactory::create_matrix_dense("DEFAULT", 0, 0);

  //the previous iteration's objects are set to NULL
  it_prev_ = new hiopIterate(nlp);
  grad_f_prev_ = nlp->alloc_primal_vec();
  Jac_c_prev_ = nlp->alloc_Jac_c();
  Jac_d_prev_ = nlp->alloc_Jac_d();

  //internal buffers for memory pool (none of them should be in n)
#ifdef HIOP_USE_MPI
  _buff_kxk    = new double[nlp->m() * nlp->m()];
  _buff_2lxk   = new double[nlp->m() * 2*l_max];
  _buff1_lxlx3 = new double[3*l_max*l_max];
  _buff2_lxlx3 = new double[3*l_max*l_max];
#else
   //not needed in non-MPI mode
  _buff_kxk  = NULL;
  _buff_2lxk = NULL;
  _buff1_lxlx3 = _buff2_lxlx3 = NULL;
#endif

  //auxiliary objects/buffers
  _S1=_Y1=NULL;
  _lxl_mat1=_kxl_mat1=_kx2l_mat1=NULL;
  _l_vec1 = _l_vec2 = _2l_vec1 = NULL;
  _n_vec1 = DhInv->alloc_clone();
  _n_vec2 = DhInv->alloc_clone();

  _V_work_vec=LinearAlgebraFactory::create_vector("DEFAULT", 0);
  _V_ipiv_vec=NULL; _V_ipiv_size=-1;

  
  sigma0 = nlp->options->GetNumeric("sigma0");
  sigma=sigma0;

  string sigma_strategy = nlp->options->GetString("sigma_update_strategy");
  transform(sigma_strategy.begin(), sigma_strategy.end(), sigma_strategy.begin(), ::tolower);
  sigma_update_strategy = SIGMA_STRATEGY3;
  if(sigma_strategy=="sty")
    sigma_update_strategy=SIGMA_STRATEGY1;
  else if(sigma_strategy=="sty_inv")
    sigma_update_strategy=SIGMA_STRATEGY2;
  else if(sigma_strategy=="snrm_ynrm")
    sigma_update_strategy=SIGMA_STRATEGY3;
  else if(sigma_strategy=="sty_srnm_ynrm")
    sigma_update_strategy=SIGMA_STRATEGY4;
  else if(sigma_strategy=="sigma0")
    sigma_update_strategy=SIGMA_CONSTANT;
  else assert(false && "sigma_update_strategy option not recognized");

  sigma_safe_min=1e-8;
  sigma_safe_max=1e+8;
  nlp->log->printf(hovScalars, "Hessian Low Rank: initial sigma is %g\n", sigma);
  nlp->log->printf(hovScalars, "Hessian Low Rank: sigma update strategy is %d [%s]\n", sigma_update_strategy, sigma_strategy.c_str());

  _Dx   = DhInv->alloc_clone();
#ifdef HIOP_DEEPCHECKS
  _Vmat = V_->alloc_clone();
#endif

  yk = nlp->alloc_primal_vec();
  sk = nlp->alloc_primal_vec();

}  

hiopHessianLowRank::~hiopHessianLowRank()
{
  if(DhInv) delete DhInv;
  delete _Dx;

  delete St_;
  delete Yt_;
  delete L_;
  delete D_;
  delete V_;
  if(yk) delete yk;
  if(sk) delete sk;
#ifdef HIOP_DEEPCHECKS
  delete _Vmat;
#endif


  delete it_prev_;
  delete grad_f_prev_;
  delete Jac_c_prev_;
  delete Jac_d_prev_;

  if(_buff_kxk)    delete[] _buff_kxk;
  if(_buff_2lxk)   delete[] _buff_2lxk;
  if(_buff1_lxlx3) delete[] _buff1_lxlx3;
  if(_buff2_lxlx3) delete[] _buff2_lxlx3;

  if(_S1) delete _S1;
  if(_Y1) delete _Y1;
  if(_lxl_mat1)    delete _lxl_mat1;
  if(_kxl_mat1)    delete _kxl_mat1; 
  if(_kx2l_mat1)   delete _kx2l_mat1;

  if(_l_vec1) delete _l_vec1;
  if(_l_vec2) delete _l_vec2;
  if(_n_vec1) delete _n_vec1;
  if(_n_vec2) delete _n_vec2;
  if(_2l_vec1) delete _2l_vec1;
  if(_V_ipiv_vec) delete[] _V_ipiv_vec;
  if(_V_work_vec) delete _V_work_vec;

  for(auto* it: a) {
    delete it;
  }

  for(auto* it: b) {
    delete it;
  }
}

void hiopHessianLowRank::alloc_for_limited_mem(const size_type& mem_length)
{
  //note: St_ and Yt_ always have l_curr rows
  if(l_curr == mem_length) {
    assert(D_->get_size() == l_curr);
    return;
  }
  delete D_;
  delete L_;
  delete Yt_;
  delete St_;
  St_ = nlp->alloc_multivector_primal(mem_length, l_max);
  Yt_ = St_->alloc_clone();

  //these are local
  L_  = LinearAlgebraFactory::create_matrix_dense("DEFAULT", mem_length, mem_length);
  D_  = LinearAlgebraFactory::create_vector("DEFAULT", mem_length);
}

bool hiopHessianLowRank::updateLogBarrierDiagonal(const hiopVector& Dx)
{
  DhInv->setToConstant(sigma);
  DhInv->axpy(1.0,Dx);
  _Dx->copyFrom(Dx);
#ifdef HIOP_DEEPCHECKS
  assert(DhInv->allPositive());
#endif
  DhInv->invert();
  nlp->log->write("hiopHessianLowRank: inverse diag DhInv:", *DhInv, hovMatrices);
  matrixChanged=true;
  return true;
}

#ifdef HIOP_DEEPCHECKS
void hiopHessianLowRank::print(FILE* f, hiopOutVerbosity v, const char* msg) const
{
  fprintf(f, "%s\n", msg);
#ifdef HIOP_DEEPCHECKS
  nlp->log->write("Dx", *_Dx, v);
#else
  fprintf(f, "Dx is not stored in this class, but it can be computed from Dx=DhInv^(1)-sigma");
#endif
  nlp->log->printf(v, "sigma=%22.16f;\n", sigma);
  nlp->log->write("DhInv", *DhInv, v);
  nlp->log->write("S_trans", *St_, v);
  nlp->log->write("Y_trans", *Yt_, v);

  fprintf(f, " [[Internal representation]]\n");
#ifdef HIOP_DEEPCHECKS
  nlp->log->write("V", *_Vmat, v);
#else
  fprintf(f, "V matrix is available at this point (only its LAPACK factorization). Print it in updateInternalBFGSRepresentation() instead, before factorizeV()\n");
#endif
  nlp->log->write("L", *L_, v);
  nlp->log->write("D", *D_, v);
}
#endif

#include <limits>

bool hiopHessianLowRank::update(const hiopIterate& it_curr, const hiopVector& grad_f_curr_,
				const hiopMatrix& Jac_c_curr_, const hiopMatrix& Jac_d_curr_)
{
  nlp->runStats.tmSolverInternal.start();

  const hiopMatrixDense& Jac_c_curr = dynamic_cast<const hiopMatrixDense&>(Jac_c_curr_);
  const hiopMatrixDense& Jac_d_curr = dynamic_cast<const hiopMatrixDense&>(Jac_d_curr_);

#ifdef HIOP_DEEPCHECKS
  assert(it_curr.zl->matchesPattern(nlp->get_ixl()));
  assert(it_curr.zu->matchesPattern(nlp->get_ixu()));
  assert(it_curr.sxl->matchesPattern(nlp->get_ixl()));
  assert(it_curr.sxu->matchesPattern(nlp->get_ixu()));
#endif
  //on first call l_curr=-1
  if(l_curr>=0) {
    size_type n=grad_f_curr_.get_size();
    //compute s_new = x_curr-x_prev
    hiopVector& s_new = new_n_vec1(n);
    s_new.copyFrom(*it_curr.x);
    s_new.axpy(-1.,*it_prev_->x);
    double s_infnorm=s_new.infnorm();
    if(s_infnorm>=100*std::numeric_limits<double>::epsilon()) { //norm of s not too small

      //compute y_new = \grad J(x_curr,\lambda_curr) - \grad J(x_prev, \lambda_curr) (yes, J(x_prev, \lambda_curr))
      //              = graf_f_curr-grad_f_prev + (Jac_c_curr-Jac_c_prev)yc_curr+ (Jac_d_curr-Jac_c_prev)yd_curr - zl_curr*s_new + zu_curr*s_new
      hiopVector& y_new = new_n_vec2(n);
      y_new.copyFrom(grad_f_curr_); 
      y_new.axpy(-1., *grad_f_prev_);
      Jac_c_curr.transTimesVec  (1.0, y_new, 1.0, *it_curr.yc);
      //!opt if nlp->Jac_c_isLinear no need for the multiplications
      Jac_c_prev_->transTimesVec(1.0, y_new,-1.0, *it_curr.yc);
      //!opt same here
      Jac_d_curr.transTimesVec  (1.0, y_new, 1.0, *it_curr.yd); 
      Jac_d_prev_->transTimesVec(1.0, y_new,-1.0, *it_curr.yd);
      
      double sTy = s_new.dotProductWith(y_new), s_nrm2=s_new.twonorm(), y_nrm2=y_new.twonorm();

#ifdef HIOP_DEEPCHECKS
      nlp->log->printf(hovLinAlgScalarsVerb, "hiopHessianLowRank: s^T*y=%20.14e ||s||=%20.14e ||y||=%20.14e\n", sTy, s_nrm2, y_nrm2);
      nlp->log->write("hiopHessianLowRank s_new",s_new, hovIteration);
      nlp->log->write("hiopHessianLowRank y_new",y_new, hovIteration);
#endif
      if(sTy>s_nrm2*y_nrm2*sqrt(std::numeric_limits<double>::epsilon())) { //sTy far away from zero

        if(l_max>0) {
          //compute the new row in L, update S and Y (either augment them or shift cols and add s_new and y_new)
          hiopVector& YTs = new_l_vec1(l_curr);
          Yt_->timesVec(0.0, YTs, 1.0, s_new);
          //update representation
          if(l_curr<l_max) {
            //just grow/augment the matrices
            St_->appendRow(s_new);
            Yt_->appendRow(y_new);
            growL(l_curr, l_max, YTs);
            growD(l_curr, l_max, sTy);
            l_curr++;
          } else {
            //shift
            St_->shiftRows(-1);
            Yt_->shiftRows(-1);
            St_->replaceRow(l_max-1, s_new);
            Yt_->replaceRow(l_max-1, y_new);
            updateL(YTs,sTy);
            updateD(sTy);
            l_curr=l_max;
          }
        } //end of l_max>0
#ifdef HIOP_DEEPCHECKS
        nlp->log->printf(hovMatrices, "\nhiopHessianLowRank: these are L and D from the BFGS compact representation\n");
        nlp->log->write("L", *L_, hovMatrices);
        nlp->log->write("D", *D_, hovMatrices);
        nlp->log->printf(hovMatrices, "\n");
#endif
        //update B0 (i.e., sigma)
        switch (sigma_update_strategy ) {
        case SIGMA_STRATEGY1:
          sigma=sTy/(s_nrm2*s_nrm2);
          break;
        case SIGMA_STRATEGY2:
          sigma=y_nrm2*y_nrm2/sTy;
          break;
        case SIGMA_STRATEGY3:
          sigma=sqrt(s_nrm2*s_nrm2 / y_nrm2 / y_nrm2);
          break;
        case SIGMA_STRATEGY4:
          sigma=0.5*(sTy/(s_nrm2*s_nrm2)+y_nrm2*y_nrm2/sTy);
          break;
        case SIGMA_CONSTANT:
          sigma=sigma0;
          break;
        default:
          assert(false && "Option value for sigma_update_strategy was not recognized.");
          break;
        } // else of the switch
        //safe guard it
        sigma=fmax(fmin(sigma_safe_max, sigma), sigma_safe_min);
        nlp->log->printf(hovLinAlgScalars, "hiopHessianLowRank: sigma was updated to %22.16e\n", sigma);
      } else { //sTy is too small or negative -> skip
        nlp->log->printf(hovLinAlgScalars, "hiopHessianLowRank: s^T*y=%12.6e not positive enough... skipping the Hessian update\n", sTy);
      }
    } else {// norm of s_new is too small -> skip
      nlp->log->printf(hovLinAlgScalars, "hiopHessianLowRank: ||s_new||=%12.6e too small... skipping the Hessian update\n", s_infnorm);
    }

    //save this stuff for next update
    it_prev_->copyFrom(it_curr);
    grad_f_prev_->copyFrom(grad_f_curr_);
    Jac_c_prev_->copyFrom(Jac_c_curr);
    Jac_d_prev_->copyFrom(Jac_d_curr);
    nlp->log->printf(hovLinAlgScalarsVerb, "hiopHessianLowRank: storing the iteration info as 'previous'\n", s_infnorm);

  } else {
    //this is the first optimization iterate, just save the iterate and exit
    it_prev_->copyFrom(it_curr);
    grad_f_prev_->copyFrom(grad_f_curr_);
    Jac_c_prev_->copyFrom(Jac_c_curr);
    Jac_d_prev_->copyFrom(Jac_d_curr);

    nlp->log->printf(hovLinAlgScalarsVerb, "HessianLowRank on first update, just saving iteration\n");

    l_curr++;
  }

  nlp->runStats.tmSolverInternal.stop();
  return true;
}

/* 
 * The dirty work to bring this^{-1} to the form
 * M = DhInv - DhInv*[B0*S Y] * V^{-1} * [ S^T*B0 ] *DhInv
 *                                       [ Y^T    ]
 * Namely it computes V, a symmetric 2lx2l given by
 *  V =  [S'*B0*(DhInv*B0-I)*S    -L+S'*B0*DhInv*Y ]
 *       [-L'+Y'*Dhinv*B0*S       +D+Y'*Dhinv*Y    ]
 * In this function V is factorized and it will hold the factors at the end of the function
 * Note that L, D, S, and Y are from the BFGS secant representation and are updated/computed in 'update'
 */
void hiopHessianLowRank::updateInternalBFGSRepresentation()
{
  size_type n=St_->n();
  size_type l=St_->m();

  //grow L,D, andV if needed
  if(L_->m()!=l) {
    delete L_;
    L_ = LinearAlgebraFactory::create_matrix_dense("DEFAULT", l, l);
  }
  if(D_->get_size()!=l) {
    delete D_;
    D_ = LinearAlgebraFactory::create_vector("DEFAULT", l);
  }
  if(V_->m()!=2*l) {
    delete V_;
    V_ = LinearAlgebraFactory::create_matrix_dense("DEFAULT", 2*l, 2*l);
  }

  //-- block (2,2)
  hiopMatrixDense& DpYtDhInvY = new_lxl_mat1(l);
  symmMatTimesDiagTimesMatTrans_local(0.0, DpYtDhInvY, 1.0,*Yt_,*DhInv);
#ifdef HIOP_USE_MPI
  const size_t buffsize=l*l*sizeof(double);
  memcpy(_buff1_lxlx3, DpYtDhInvY.local_data(), buffsize);
#else
  DpYtDhInvY.addDiagonal(1., *D_);
  V_->copyBlockFromMatrix(l,l,DpYtDhInvY);
#endif

  //-- block (1,2)
  hiopMatrixDense& StB0DhInvYmL = DpYtDhInvY; //just a rename
  hiopVector& B0DhInv = new_n_vec1(n);
  B0DhInv.copyFrom(*DhInv); B0DhInv.scale(sigma);
  matTimesDiagTimesMatTrans_local(StB0DhInvYmL, *St_, B0DhInv, *Yt_);
#ifdef HIOP_USE_MPI
  memcpy(_buff1_lxlx3+l*l, StB0DhInvYmL.local_data(), buffsize);
#else
  //substract L
  StB0DhInvYmL.addMatrix(-1.0, *L_);
  // (1,2) block in V
  V_->copyBlockFromMatrix(0,l,StB0DhInvYmL);
#endif

  //-- block (2,2)
  hiopVector& theDiag = B0DhInv; //just a rename, also reuses values
  theDiag.addConstant(-1.0); //at this point theDiag=DhInv*B0-I
  theDiag.scale(sigma);
  hiopMatrixDense& StDS = DpYtDhInvY; //a rename
  symmMatTimesDiagTimesMatTrans_local(0.0, StDS, 1.0, *St_, theDiag);
#ifdef HIOP_USE_MPI
  memcpy(_buff1_lxlx3+2*l*l, DpYtDhInvY.local_data(), buffsize);
#else
  V_->copyBlockFromMatrix(0,0,StDS);
#endif


#ifdef HIOP_USE_MPI
  int ierr;
  ierr = MPI_Allreduce(_buff1_lxlx3, _buff2_lxlx3, 3*l*l, MPI_DOUBLE, MPI_SUM, nlp->get_comm()); assert(ierr==MPI_SUCCESS);

  // - block (2,2)
  DpYtDhInvY.copyFrom(_buff2_lxlx3);
  DpYtDhInvY.addDiagonal(1., *D_);
  V_->copyBlockFromMatrix(l,l,DpYtDhInvY);

  // - block (1,2)
  StB0DhInvYmL.copyFrom(_buff2_lxlx3+l*l);
  StB0DhInvYmL.addMatrix(-1.0, *L_);
  V_->copyBlockFromMatrix(0,l,StB0DhInvYmL);

  // - block (1,1)
  StDS.copyFrom(_buff2_lxlx3+2*l*l);
  V_->copyBlockFromMatrix(0,0,StDS);
#endif
#ifdef HIOP_DEEPCHECKS
  delete _Vmat;
  _Vmat = V_->new_copy();
  _Vmat->overwriteLowerTriangleWithUpper();
#endif

  //finally, factorize V
  factorizeV();

  matrixChanged=false;
}

/* Solves this*x = res as x = this^{-1}*res
 * where 'this^{-1}' is
 * M = DhInv - DhInv*[B0*S Y] * V^{-1} * [ S^T*B0 ] *DhInv
 *                                       [ Y^T    ]
 *
 * M is is nxn, S,Y are nxl, V is upper triangular 2lx2l, and x is nx1
 * Remember we store Yt=Y^T and St=S^T
 */  
void hiopHessianLowRank::solve(const hiopVector& rhsx, hiopVector& x)
{
  if(matrixChanged) updateInternalBFGSRepresentation();

  size_type n=St_->n(), l=St_->m();
#ifdef HIOP_DEEPCHECKS
  assert(rhsx.get_size()==n);
  assert(x.get_size()==n);
  assert(DhInv->get_size()==n);
  assert(DhInv->isfinite_local() && "inf or nan entry detected");
  assert(rhsx.isfinite_local() && "inf or nan entry detected in rhs");
#endif

  //1. x = DhInv*res
  x.copyFrom(rhsx);
  x.componentMult(*DhInv);

  //2. stx= S^T*B0*DhInv*res and ytx=Y^T*DhInv*res
  hiopVector&stx=new_l_vec1(l), &ytx=new_l_vec2(l);
  stx.setToZero(); ytx.setToZero();
  Yt_->timesVec(0.0,ytx,1.0,x);

  hiopVector& B0DhInvx = new_n_vec1(n);
  B0DhInvx.copyFrom(x); //it contains DhInv*res
  B0DhInvx.scale(sigma); //B0*(DhInv*res) 
  St_->timesVec(0.0, stx, 1.0, B0DhInvx);

  //3. solve with V
  hiopVector& spart=stx; hiopVector& ypart=ytx;
  solveWithV(spart,ypart);

  //4. multiply with  DhInv*[B0*S Y], namely
  // result = DhInv*(B0*S*spart + Y*ypart)
  hiopVector&  result = new_n_vec1(n);
  St_->transTimesVec(0.0, result, 1.0, spart);
  result.scale(sigma);
  Yt_->transTimesVec(1.0, result, 1.0, ypart);
  result.componentMult(*DhInv);

  //5. x = first term - second term = x_computed_in_1 - result 
  x.axpy(-1.0,result);
#ifdef HIOP_DEEPCHECKS
  assert(x.isfinite_local() && "inf or nan entry detected in computed solution");
#endif
  
}

/* W = beta*W + alpha*X*inverse(this)*X^T (a more efficient version of solve)
 * where 'this^{-1}' is
 * M = DhInv + DhInv*[B0*S Y] * V^{-1} * [ S^T*B0 ] *DhInv
 *                                       [ Y^T    ]
 * W is kxk, S,Y are nxl, DhInv,B0 are n, V is 2lx2l
 * X is kxn
 */ 
void hiopHessianLowRank::
symMatTimesInverseTimesMatTrans(double beta,
                                hiopMatrixDense& W, 
				double alpha,
                                const hiopMatrixDense& X)
{
  if(matrixChanged) updateInternalBFGSRepresentation();

  size_type n=St_->n(), l=St_->m();
  size_type k=W.m(); 
  assert(X.m()==k);
  assert(X.n()==n);

#ifdef HIOP_DEEPCHECKS
   nlp->log->write("symMatTimesInverseTimesMatTrans: X is: ", X, hovMatrices);
#endif 

  //1. compute W=beta*W + alpha*X*DhInv*X'
#ifdef HIOP_USE_MPI
  if(0==nlp->get_rank())
    symmMatTimesDiagTimesMatTrans_local(beta,W,alpha,X,*DhInv);
  else
    symmMatTimesDiagTimesMatTrans_local(0.0, W,alpha,X,*DhInv);
  //W will be MPI_All_reduced later
#else
  symmMatTimesDiagTimesMatTrans_local(beta,W,alpha,X,*DhInv);
#endif
  //2. compute S1=X*DhInv*B0*S and Y1=X*DhInv*Y
  auto& S1=new_S1(X,*St_);
  auto& Y1=new_Y1(X,*Yt_); //both are kxl
  hiopVector& B0DhInv = new_n_vec1(n);
  B0DhInv.copyFrom(*DhInv); B0DhInv.scale(sigma);
  matTimesDiagTimesMatTrans_local(S1, X, B0DhInv, *St_);
  matTimesDiagTimesMatTrans_local(Y1, X, *DhInv,  *Yt_);

  //3. reduce W, S1, and Y1 (dimensions: kxk, kxl, kxl)
  hiopMatrixDense& S2Y2 = new_kx2l_mat1(k,l);  //Initialy S2Y2 = [Y1 S1]
  S2Y2.copyBlockFromMatrix(0,0,S1);
  S2Y2.copyBlockFromMatrix(0,l,Y1);
#ifdef HIOP_USE_MPI
  int ierr;
  ierr = MPI_Allreduce(S2Y2.local_data(), _buff_2lxk, 2*l*k, MPI_DOUBLE, MPI_SUM, nlp->get_comm()); assert(ierr==MPI_SUCCESS);
  ierr = MPI_Allreduce(W.local_data(),    _buff_kxk,  k*k,   MPI_DOUBLE, MPI_SUM, nlp->get_comm()); assert(ierr==MPI_SUCCESS);
  S2Y2.copyFrom(_buff_2lxk);
  W.copyFrom(_buff_kxk);
  //also copy S1 and Y1
  S1.copyFromMatrixBlock(S2Y2, 0,0);
  Y1.copyFromMatrixBlock(S2Y2, 0,l);
#endif
#ifdef HIOP_DEEPCHECKS
  nlp->log->write("symMatTimesInverseTimesMatTrans: W first term is: ", W, hovMatrices);
#endif 
  //4. [S2] = V \ [S1^T]
  //   [Y2]       [Y1^T]
  //S2Y2 is exactly [S1^T] when Fortran Lapack looks at it
  //                [Y1^T]
  hiopMatrixDense& RHS_fortran = S2Y2; 
  solveWithV(RHS_fortran);

  //5. W = W-alpha*[S1 Y1]*[S2^T] 
  //                       [Y2^T]
  S2Y2 = RHS_fortran;
  alpha = 0-alpha;
  hiopMatrixDense& S2=new_kxl_mat1(k,l);
  S2.copyFromMatrixBlock(S2Y2, 0, 0);
  S1.timesMatTrans_local(1.0, W, alpha, S2);

  hiopMatrixDense& Y2=S2;
  Y2.copyFromMatrixBlock(S2Y2, 0, l);
  Y1.timesMatTrans_local(1.0, W, alpha, Y2);

  //nlp->log->write("symMatTimesInverseTimesMatTrans: Y1 is : ", Y1, hovMatrices);
  //nlp->log->write("symMatTimesInverseTimesMatTrans: Y2 is : ", Y2, hovMatrices);
  //nlp->log->write("symMatTimesInverseTimesMatTrans: W is : ", W, hovMatrices);

  //we're done here

#ifdef HIOP_DEEPCHECKS
  nlp->log->write("symMatTimesInverseTimesMatTrans: final matrix is : ", W, hovMatrices);
#endif 

}


void hiopHessianLowRank::factorizeV()
{
  int N = V_->n();
  int lda = N;
  int info;
  if(N==0) {
    return;
  }

#ifdef HIOP_DEEPCHECKS
    nlp->log->write("factorizeV:  V is ", *V_, hovMatrices);
#endif

  char uplo='L'; //V is upper in C++ so it's lower in fortran

  if(_V_ipiv_vec==NULL) _V_ipiv_vec=new int[N];
  else if(_V_ipiv_size!=N) { delete[] _V_ipiv_vec; _V_ipiv_vec=new int[N]; _V_ipiv_size=N; }

  int lwork=-1;//inquire sizes
  double Vwork_tmp;
  DSYTRF(&uplo, &N, V_->local_data(), &lda, _V_ipiv_vec, &Vwork_tmp, &lwork, &info);
  assert(info==0);

  lwork=(int)Vwork_tmp;
  if(lwork != _V_work_vec->get_size()) {
    if(_V_work_vec!=NULL) {
      delete _V_work_vec;
    }
    _V_work_vec = LinearAlgebraFactory::create_vector("DEFAULT", lwork);
  } else assert(_V_work_vec);

  DSYTRF(&uplo, &N, V_->local_data(), &lda, _V_ipiv_vec, _V_work_vec->local_data(), &lwork, &info);
  
  if(info<0)
    nlp->log->printf(hovError, "hiopHessianLowRank::factorizeV error: %d argument to dsytrf has an illegal value\n", -info);
  else if(info>0)
    nlp->log->printf(hovError, "hiopHessianLowRank::factorizeV error: %d entry in the factorization's diagonal is exactly zero. Division by zero will occur if it a solve is attempted.\n", info);
  assert(info==0);
#ifdef HIOP_DEEPCHECKS
  nlp->log->write("factorizeV:  factors of V: ", *V_, hovMatrices);
#endif

}

void hiopHessianLowRank::solveWithV(hiopVector& rhs_s, hiopVector& rhs_y)
{
  int N = V_->n();
  if(N==0) {
    return;
  }

  int l = rhs_s.get_size();

#ifdef HIOP_DEEPCHECKS
  nlp->log->write("hiopHessianLowRank::solveWithV: RHS IN 's' part: ", rhs_s, hovMatrices);
  nlp->log->write("hiopHessianLowRank::solveWithV: RHS IN 'y' part: ", rhs_y, hovMatrices);
  hiopVector* rhs_saved= LinearAlgebraFactory::create_vector("DEFAULT", rhs_s.get_size()+rhs_y.get_size());
  rhs_saved->copyFromStarting(0, rhs_s);
  rhs_saved->copyFromStarting(l, rhs_y);
#endif

  int lda=N, one=1, info;
  char uplo='L'; 
#ifdef HIOP_DEEPCHECKS
  assert(N==rhs_s.get_size()+rhs_y.get_size());
#endif
  hiopVector& rhs=new_2l_vec1(l);
  rhs.copyFromStarting(0, rhs_s);
  rhs.copyFromStarting(l, rhs_y);

  DSYTRS(&uplo, &N, &one, V_->local_data(), &lda, _V_ipiv_vec, rhs.local_data(), &N, &info);

  if(info<0) nlp->log->printf(hovError, "hiopHessianLowRank::solveWithV error: %d argument to dsytrf has an illegal value\n", -info);
  assert(info==0);

  //copy back the solution
  rhs.copyToStarting(0,rhs_s);
  rhs.copyToStarting(l,rhs_y);

#ifdef HIOP_DEEPCHECKS
  nlp->log->write("solveWithV: SOL OUT 's' part: ", rhs_s, hovMatrices);
  nlp->log->write("solveWithV: SOL OUT 'y' part: ", rhs_y, hovMatrices);

  //residual calculation
  double nrmrhs=rhs_saved->infnorm();
  _Vmat->timesVec(1.0, *rhs_saved, -1.0, rhs);
  double nrmres=rhs_saved->infnorm();
  //nlp->log->printf(hovLinAlgScalars, "hiopHessianLowRank::solveWithV 1rhs: rel resid norm=%g\n", nrmres/(1+nrmrhs));
  nlp->log->printf(hovScalars, "hiopHessianLowRank::solveWithV 1rhs: rel resid norm=%g\n", nrmres/(1+nrmrhs));
  if(nrmres>1e-8)
    nlp->log->printf(hovWarning, "hiopHessianLowRank::solveWithV large residual=%g\n", nrmres);
  delete rhs_saved;
#endif

}

void hiopHessianLowRank::solveWithV(hiopMatrixDense& rhs)
{
  int N = V_->n();
  if(0==N) {
    return;
  }

#ifdef HIOP_DEEPCHECKS
  nlp->log->write("solveWithV: RHS IN: ", rhs, hovMatrices);
  hiopMatrixDense* rhs_saved = rhs.new_copy();
#endif

  //rhs is transpose in C++

  char uplo='L'; 
  int lda=N, ldb=N, nrhs=rhs.m(), info;
#ifdef HIOP_DEEPCHECKS
  assert(N==rhs.n()); 
#endif
  DSYTRS(&uplo, &N, &nrhs, V_->local_data(), &lda, _V_ipiv_vec, rhs.local_data(), &ldb, &info);

  if(info<0) nlp->log->printf(hovError, "hiopHessianLowRank::solveWithV error: %d argument to dsytrf has an illegal value\n", -info);
  assert(info==0);
#ifdef HIOP_DEEPCHECKS
  nlp->log->write("solveWithV: SOL OUT: ", rhs, hovMatrices);
  
  hiopMatrixDense& sol = rhs; //matrix of solutions
  /// TODO: get rid of these uses of specific hiopVector implementation
  hiopVector* x = LinearAlgebraFactory::create_vector("DEFAULT", rhs.n()); //again, keep in mind rhs is transposed
  hiopVector* r = LinearAlgebraFactory::create_vector("DEFAULT", rhs.n());

  double resnorm=0.0;
  for(int k=0; k<rhs.m(); k++) {
    rhs_saved->getRow(k, *r);
    sol.getRow(k,*x);
    double nrmrhs = r->infnorm();//nrmrhs=.0;
    _Vmat->timesVec(1.0, *r, -1.0, *x);
    double nrmres = r->infnorm();
    if(nrmres>1e-8)
      nlp->log->printf(hovWarning, "hiopHessianLowRank::solveWithV mult-rhs: rhs number %d has large resid norm=%g\n", k, nrmres);
    if(nrmres/(nrmrhs+1)>resnorm) resnorm=nrmres/(nrmrhs+1);
  }
  nlp->log->printf(hovLinAlgScalars, "hiopHessianLowRank::solveWithV mult-rhs: rel resid norm=%g\n", resnorm);
  delete x;
  delete r;
  delete rhs_saved;
#endif

}

void hiopHessianLowRank::growL(const int& lmem_curr, const int& lmem_max, const hiopVector& YTs)
{
  int l = L_->m();
#ifdef HIOP_DEEPCHECKS
  assert(l==L_->n());
  assert(lmem_curr==l);
  assert(lmem_max>=l);
#endif
  //newL = [   L     0]
  //       [ Y^T*s   0]
  hiopMatrixDense* newL = LinearAlgebraFactory::create_matrix_dense("DEFAULT", l+1, l+1);
  assert(newL);
  //copy from L to newL
  newL->copyBlockFromMatrix(0,0, *L_);

  double* newL_mat=newL->local_data(); //doing the rest here
  const double* YTs_vec=YTs.local_data_const();
  //for(int j=0; j<l; j++) newL_mat[l][j] = YTs_vec[j];
  for(int j=0; j<l; j++) newL_mat[l*(l+1)+j] = YTs_vec[j];

  //and the zero entries of the last column
  //for(int i=0; i<l+1; i++) newL_mat[i][l] = 0.0;
  for(int i=0; i<l+1; i++) newL_mat[i*(l+1)+l] = 0.0;

  //swap the pointers
  delete L_;
  L_ = newL;
}

void hiopHessianLowRank::growD(const int& lmem_curr, const int& lmem_max, const double& sTy)
{
  int l = D_->get_size();
  assert(l==lmem_curr);
  assert(lmem_max>=l);

  hiopVector* Dnew=LinearAlgebraFactory::create_vector("DEFAULT", l+1);
  double* Dnew_vec=Dnew->local_data();
  memcpy(Dnew_vec, D_->local_data_const(), l*sizeof(double));
  Dnew_vec[l]=sTy;

  delete D_;
  D_ = Dnew;
}

/* L_{ij} = s_{i-1}^T y_{j-1}, if i>j, otherwise zero. Here i,j = 0,1,...,l_curr-1
 * L_new = lift and shift L to the left; replace last row with [Yts;0]
 */
void hiopHessianLowRank::updateL(const hiopVector& YTs, const double& sTy)
{
  int l=YTs.get_size();
  assert(l==L_->m());
  assert(l==L_->n());
#ifdef HIOP_DEEPCHECKS
  assert(l_curr==l);
  assert(l_curr==l_max);
#endif
  const int lm1=l-1;
  double* L_mat=L_->local_data();
  const double* yts_vec=YTs.local_data_const();
  for(int i=1; i<lm1; i++) {
    for(int j=0; j<i; j++) {
      //L_mat[i][j] = L_mat[i+1][j+1];
      L_mat[i*l+j] = L_mat[(i+1)*l+j+1];
    }
  }
      

  //is this really needed?
  //for(int i=0; i<lm1; i++)
  //  L_mat[i][lm1]=0.0;

  //first entry in YTs corresponds to y_to_be_discarded_since_it_is_the_oldest'* s_new and is discarded
  for(int j=0; j<lm1; j++) {
    //L_mat[lm1][j]=yts_vec[j+1];
    L_mat[lm1*l+j]=yts_vec[j+1];
    
  }

  //L_mat[lm1][lm1]=0.0;
  L_mat[lm1*l+lm1]=0.0;
}
void hiopHessianLowRank::updateD(const double& sTy)
{
  int l=D_->get_size();
  double* D_vec = D_->local_data();
  for(int i=0; i<l-1; i++)
    D_vec[i]=D_vec[i+1];
  D_vec[l-1]=sTy;
}


hiopVector&  hiopHessianLowRank::new_l_vec1(int l)
{
  if(_l_vec1!=NULL && _l_vec1->get_size()==l) return *_l_vec1;
  
  if(_l_vec1!=NULL) {
    delete _l_vec1;
  }
  _l_vec1= LinearAlgebraFactory::create_vector("DEFAULT", l);
  return *_l_vec1;
}
hiopVector&  hiopHessianLowRank::new_l_vec2(int l)
{
  if(_l_vec2!=NULL && _l_vec2->get_size()==l) return *_l_vec2;
  
  if(_l_vec2!=NULL) {
    delete _l_vec2;
  }
  _l_vec2= LinearAlgebraFactory::create_vector("DEFAULT", l);
  return *_l_vec2;
}

hiopMatrixDense& hiopHessianLowRank::new_lxl_mat1(int l)
{
  if(_lxl_mat1!=NULL) {
    if( l==_lxl_mat1->m() ) {
      return *_lxl_mat1;
    } else {
      delete _lxl_mat1; 
      _lxl_mat1=NULL;
    }
  }
  _lxl_mat1 = LinearAlgebraFactory::create_matrix_dense("DEFAULT", l, l);
  return *_lxl_mat1;
}
hiopMatrixDense& hiopHessianLowRank::new_kx2l_mat1(int k, int l)
{
  int twol=2*l;
  if(NULL!=_kx2l_mat1) {
    assert(_kx2l_mat1->m()==k);
    if( twol==_kx2l_mat1->n() ) {
      return *_kx2l_mat1;
    } else {
      delete _kx2l_mat1; 
      _kx2l_mat1=NULL;
    }
  }
  _kx2l_mat1 = LinearAlgebraFactory::create_matrix_dense("DEFAULT", k, twol);
  
  return *_kx2l_mat1;
}
hiopMatrixDense& hiopHessianLowRank::new_kxl_mat1(int k, int l)
{
  if(_kxl_mat1!=NULL) {
    assert(_kxl_mat1->m()==k);
    if( l==_kxl_mat1->n() ) {
      return *_kxl_mat1;
    } else {
      delete _kxl_mat1; 
      _kxl_mat1=NULL;
    }
  }
  _kxl_mat1 = LinearAlgebraFactory::create_matrix_dense("DEFAULT", k, l);
  return *_kxl_mat1;
}
hiopMatrixDense& hiopHessianLowRank::new_S1(const hiopMatrixDense& X, const hiopMatrixDense& St)
{
  //S1 is X*some_diag*S  (kxl). Here St=S^T is lxn and X is kxn (l BFGS memory size, k number of constraints)
  size_type k=X.m(), l=St.m();
#ifdef HIOP_DEEPCHECKS
  assert(St.n()==X.n());
  if(_S1!=NULL) 
    assert(_S1->m()==k);
#endif
  if(NULL!=_S1 && _S1->n()!=l) { delete _S1; _S1=NULL; }
  
  if(NULL==_S1) _S1=LinearAlgebraFactory::create_matrix_dense("DEFAULT", k, l);

  return *_S1;
}

hiopMatrixDense& hiopHessianLowRank::new_Y1(const hiopMatrixDense& X, const hiopMatrixDense& Yt)
{
  //Y1 is X*somediag*Y (kxl). Here Yt=Y^T is lxn,  X is kxn
  size_type k=X.m(), l=Yt.m();
#ifdef HIOP_DEEPCHECKS
  assert(X.n()==Yt.n());
  if(_Y1!=NULL) assert(_Y1->m()==k);
#endif

  if(NULL!=_Y1 && _Y1->n()!=l) { delete _Y1; _Y1=NULL; }

  if(NULL==_Y1) _Y1=LinearAlgebraFactory::create_matrix_dense("DEFAULT", k, l);
  return *_Y1;
}
#ifdef HIOP_DEEPCHECKS

void hiopHessianLowRank::timesVec_noLogBarrierTerm(double beta, hiopVector& y, double alpha, const hiopVector&x)
{
  this->timesVecCmn(beta, y, alpha, x, false);
}

#endif //HIOP_DEEPCHECKS


void hiopHessianLowRank::
timesVecCmn(double beta, hiopVector& y, double alpha, const hiopVector& x, bool addLogTerm) const
{
  size_type n=St_->n();
  assert(l_curr==St_->m());
  assert(y.get_size()==n);
  assert(St_->get_local_size_n() == Yt_->get_local_size_n());

  //we have B+=B-B*s*B*s'/(s'*B*s)+yy'/(y'*s)
  //B0 is sigma*I. There is an additional diagonal log-barrier term _Dx

  bool print=false;
  if(print) {
    nlp->log->printf(hovMatrices, "---hiopHessianLowRank::timesVec \n");
    nlp->log->write("S=", *St_, hovMatrices);
    nlp->log->write("Y=", *Yt_, hovMatrices);
    nlp->log->write("DhInv=", *DhInv, hovMatrices);
    nlp->log->printf(hovMatrices, "sigma=%22.16e;  addLogTerm=%d;\n", sigma, addLogTerm);
    if(addLogTerm)
      nlp->log->write("Dx=", *_Dx, hovMatrices);
    nlp->log->printf(hovMatrices, "y=beta*y + alpha*this*x : beta=%g alpha=%g\n", beta, alpha);
    nlp->log->write("x_in=", x, hovMatrices);
    nlp->log->write("y_in=", y, hovMatrices);
  }

  //allocate and compute a_k and b_k
  //! make sure the pointers within these std::vectors are deallocated
  a.resize(l_curr, nullptr);
  b.resize(l_curr, nullptr);
  int n_local = Yt_->get_local_size_n();
  for(int k=0; k<l_curr; k++) {
    //bk=yk/sqrt(yk'*sk)
    yk->copyFrom(Yt_->local_data() + k*n_local);
    sk->copyFrom(St_->local_data() + k*n_local);
    double skTyk=yk->dotProductWith(*sk);
    
    if(skTyk < std::numeric_limits<double>::epsilon()) {
      nlp->log->printf(hovLinAlgScalars, "hiopHessianLowRank: ||s_k^T*y_k||=%12.6e too small..."
                       " set it to machine precision = %12.6e \n", skTyk, std::numeric_limits<double>::epsilon());
      skTyk = std::numeric_limits<double>::epsilon();
    }

    if(a[k] == nullptr && b[k] == nullptr) {
      b[k] = nlp->alloc_primal_vec();
      a[k] = nlp->alloc_primal_vec();
    }
    
    b[k]->copyFrom(*yk);
    b[k]->scale(1/sqrt(skTyk));

    //compute ak by an inner loop
    a[k]->copyFrom(*sk);
    a[k]->scale(sigma);

    for(int i=0; i<k; i++) {
      double biTsk = b[i]->dotProductWith(*sk);
      a[k]->axpy(+biTsk, *b[i]);
      double aiTsk = a[i]->dotProductWith(*sk);
      a[k]->axpy(-aiTsk, *a[i]);
    }
    double skTak = a[k]->dotProductWith(*sk);
    a[k]->scale(1/sqrt(skTak));
  }

  //now we have B= B_0 + sum{ bk bk' - ak ak' : k=0,1,...,l_curr-1} 
  //compute the product with x
  //y = beta*y+alpha*(B0+Dx)*x + alpha* sum { bk'x bk - ak'x ak : k=0,1,...,l_curr-1}
  y.scale(beta);
  if(addLogTerm) 
    y.axzpy(alpha,x,*_Dx);

  y.axpy(alpha*sigma, x); 

  for(int k=0; k<l_curr; k++) {
    double bkTx = b[k]->dotProductWith(x);
    double akTx = a[k]->dotProductWith(x);
    
    y.axpy( alpha*bkTx, *b[k]);
    y.axpy(-alpha*akTx, *a[k]);
  }

  if(print) {
    nlp->log->write("y_out=", y, hovMatrices);
  }

}

void hiopHessianLowRank::timesVec(double beta, hiopVector& y, double alpha, const hiopVector&x)
{
  this->timesVecCmn(beta, y, alpha, x);
}

void hiopHessianLowRank::timesVec(double beta, hiopVector& y, double alpha, const hiopVector&x) const
{
  this->timesVecCmn(beta, y, alpha, x);
}

/**************************************************************************
 * Internal helpers
 *************************************************************************/

/* symmetric multiplication W = beta*W + alpha*X*Diag*X^T 
 * W is kxk local, X is kxn distributed and Diag is n, distributed
 * The ops are perform locally. The reduce is done separately/externally to decrease comm
 */
void hiopHessianLowRank::
symmMatTimesDiagTimesMatTrans_local(double beta, hiopMatrixDense& W,
				    double alpha, const hiopMatrixDense& X,
				    const hiopVector& d)
{
  size_type k=W.m();
  size_type n_local=X.get_local_size_n();

  assert(X.m()==k);
    
#ifdef HIOP_DEEPCHECKS
  assert(W.n()==k);
  assert(d.get_size()==X.n());
  assert(d.get_local_size()==n_local);
#endif
  
  //#define chunk 512; //!opt
  const double *xi, *xj;
  double acc;
  double *Wdata=W.local_data();
  const double *Xdata=X.local_data_const();
  const double* dd=d.local_data_const();
  for(int i=0; i<k; i++) {
    //xi=Xdata[i];
    xi=Xdata+i*n_local;
    for(int j=i; j<k; j++) {
      //xj=Xdata[j];
      xj=Xdata+j*n_local;
      //compute W[i,j] = sum {X[i,p]*d[p]*X[j,p] : p=1,...,n_local}
      acc=0.0;
      for(size_type p=0; p<n_local; p++)
	acc += xi[p]*dd[p]*xj[p];

      //Wdata[i][j]=Wdata[j][i]=beta*Wdata[i][j]+alpha*acc;
      Wdata[i*k+j] = Wdata[j*k+i] = beta*Wdata[i*k+j]+alpha*acc;
    }
  }
}

/* W=S*D*X^T, where S is lxn, D is diag nxn, and X is kxn */
void hiopHessianLowRank::
matTimesDiagTimesMatTrans_local(hiopMatrixDense& W, const hiopMatrixDense& S, const hiopVector& d, const hiopMatrixDense& X)
{
#ifdef HIOP_DEEPCHECKS
  assert(S.n()==d.get_size());
  assert(S.n()==X.n());
#endif  
  int l=S.m(), n=d.get_local_size(), k=X.m();
  assert(X.get_local_size_n() == d.get_local_size());
  
  const double* Sdi;
  double* Wdi;
  const double* Xdj;
  double acc;
  double* Wd=W.local_data();
  const double* Sd=S.local_data_const();
  const double* Xd=X.local_data_const();
  const double *diag=d.local_data_const();
  //!opt
  for(int i=0;i<l; i++) {
    //Sdi=Sd[i]; Wdi=Wd[i];
    Sdi = Sd+i*n;
    Wdi = Wd+i*W.get_local_size_n();
    
    for(int j=0; j<k; j++) {
      //Xdj=Xd[j];
      Xdj = Xd+j*n;
      acc=0.;
      for(int p=0; p<n; p++) 
	//acc += Sdi[p]*diag[p]*Xdj[p];
        acc += Sdi[p]*diag[p]*Xdj[p];
      
      Wdi[j]=acc;
    }
  }
}
};
