
/*----------------------------------------------------------------------------+
 |                                                                            |
 |                  X10 Module Attributes for HEYU                            |
 |            Copyright 2004,2005,2006 Charles W. Sullivan                    |
 |                      All Right Reserved                                    |
 |                                                                            |
 |                                                                            |
 | This software is licensed free of charge for non-commercial distribution   |
 | and for personal and internal business use only.  Inclusion of this        |
 | software or any part thereof in a commercial product is prohibited         |
 | without the prior written permission of the author.  You may copy, use,    |
 | and distribute this software subject to the following restrictions:        |
 |                                                                            |
 |  1)	You may not charge money for it.                                      |
 |  2)	You may not remove or alter this license, copyright notice, or the    |
 |      included disclaimers.                                                 |
 |  3)	You may not claim you wrote it.                                       |
 |  4)	If you make improvements (or other changes), you are requested        |
 |	to send them to the official Heyu maintainer, Charles W. Sullivan     |
 |      (cwsulliv01@heyu.org), so there's a focal point for distributing      |
 |      improved versions.                                                    |
 |                                                                            |
 | As used herein, HEYU is a trademark of Daniel B. Suthers, while X10,       | 
 | CM11A, and ActiveHome are trademarks of X-10 (USA) Inc.                    |
 | SwitchLinc and LampLinc are trademarks of Smarthome, Inc.                  |
 | The author is not affiliated with any of these entities.                   |
 |                                                                            |
 | Charles W. Sullivan                                                        |
 | Greensboro, North Carolina                                                 |
 | Email: cwsulliv01@heyu.org                                                 |
 |                                                                            |
 | Disclaimers:                                                               |
 | THERE IS NO ASSURANCE THAT THIS SOFTWARE IS FREE OF DEFECTS AND IT MUST    |
 | NOT BE USED IN ANY SITUATION WHERE THERE IS ANY CHANCE THAT ITS            |
 | PERFORMANCE OR FAILURE TO PERFORM AS EXPECTED COULD RESULT IN LOSS OF      |
 | LIFE, INJURY TO PERSONS OR PROPERTY, FINANCIAL LOSS, OR LEGAL LIABILITY.   |
 |                                                                            |
 | TO THE EXTENT ALLOWED BY APPLICABLE LAW, THIS SOFTWARE IS PROVIDED "AS IS",|
 | WITH NO EXPRESS OR IMPLIED WARRANTY, INCLUDING, BUT NOT LIMITED TO, THE    |
 | IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.|
 |                                                                            |
 | IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW WILL THE AUTHOR BE LIABLE    |
 | FOR DAMAGES, INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL   |
 | DAMAGES ARISING OUT OF THE USE OR INABILITY TO USE THIS SOFTWARE EVEN IF   |
 | THE AUTHOR HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.            |
 |                                                                            |
 +----------------------------------------------------------------------------*/

#include <stdio.h>
#if defined(SYSV) || defined(FREEBSD) || defined(OPENBSD)
#include <string.h>
#else
#include <strings.h>
#endif
#include <stdlib.h>
#include "process.h"

/* Module attribute/response definitions */
#define  UNOFF     0x00000001     /* All Units Off */
#define  LION      0x00000002     /* All Lights ON */
#define  MON       0x00000004     /* Simple module ON */
#define  MOFF      0x00000008     /* Simple module OFF */
#define  DIM       0x00000010     /* Dim 1-22 */
#define  BRI       0x00000020     /* Bright 1-22 */
#define  BRIB4DIM  0x00000040     /* If OFF, full Brightness before dimming */
#define  LIOFF     0x00000080     /* All Lights OFF */
#define  STDX10    0x00000100     /* Standard X10 */
#define  EXT3SW    0x00000200     /* Extended codes Type 3 (switch only) */
#define  EXT3DIM   0x00000400     /* Extended codes Type 3 with preset dim 0-63 */
#define  PRESET    0x00000800     /* Preset 1-32 */
#define  STAON     0x00001000     /* Sends StatusOn ack */
#define  STAOFF    0x00002000     /* Sends StatusOff ack */
#define  STAREQ    0x00004000     /* Responds to Status Request */
#define  ALLON     0x00008000     /* All Units On */
#define  LIONFULL  0x00010000     /* All Lights ON -> Full brightness */
#define  ONFULL    0x00020000     /* ON -> Full (or option) brightness */
#define  ONFULLOFF 0x00040000     /* ON -> Full brightness if Off */
#define  RESUME    0x00080000     /* ON resumes previous level */
#define  TARG      0x00100000     /* Is a target instead of a module */
#define  EXC16     0x00200000     /* Last unit turns On (all others turn off) */
#define  EXC8      0x00400000     /* Last unit turns On (others in group of 8 turn off) */
#define  EXC4      0x00800000     /* Last unit turns On and others in group of 4 turn off) */
#define  EXT0      0x01000000     /* Extended code Type 0 shutter control */
#define  VDATA     0x02000000     /* Virtual data repository */
#define  OPTIONS   0x80000000     /* Allow module options in config file */

/* NOTE: Attributes STDX10, PRESET, and EXT3SW/EXT3DIM are */
/* mutually incompatible.                                  */

/* Module max dim levels */
#define  MXL0        0
#define  MXLS      210         /* Standard X10 modules 0-210 */
#define  MXLP       31         /* Preset modules 0-31 (zero-base) */
#define  MXLE       62         /* Extended code Type 3 dimmer modules 0-62 */
#define  MXLEA      63         /* Extended code Type 3 appliance modules 0,63 */
#define  MXLS0      25         /* Extended code Type 0 shutters 0-25 */
#define  MXLV      255         /* Virtual modules 0-255 */

/* Module type attributes.  To add a new module type,  */
/* "OR" together its attributes here and add an entry in the modules */
/* table below. (Keep the table uncluttered, for future additions.) */ 
#define  NOMATT      0
#define  BASIC       (UNOFF | MON | MOFF | ALLON)

#define  STDAM       (BASIC | STDX10)
#define  STDLM       (STDAM | DIM | BRI | BRIB4DIM | LION | ONFULLOFF)
#define  STDWS       (STDLM | LIOFF)
#define  AMS         (STDAM | STAON | STAOFF | STAREQ)
#define  LMS         (STDLM | STAON | STAOFF | STAREQ)
#define  SIREN       (STDAM | LION )
#define  REM2        (STDX10 | TARG | MON | MOFF)
#define  REM3        (STDX10 | TARG | MOFF | BRI | DIM) 
#define  REM4        (STDX10 | TARG | MON | MOFF | BRI | DIM) 
#define  REM6        (STDX10 | TARG | MON | MOFF | BRI | DIM | LION | UNOFF) 

#define  LM15A       (STDAM | LION | LIONFULL | LIOFF)

#define  AM14A       (BASIC | EXT3SW | ALLON | STAON | STAOFF | STAREQ)
#define  LM14A       (AM14A | EXT3DIM | DIM | BRI | LION | LIOFF | RESUME)

#define  SL1AM       (BASIC)
#define  SL2AM       (SL1AM | STAREQ)

#define  SL1LM       (BASIC | PRESET | DIM | BRI | LION | LIOFF | ONFULL | OPTIONS)
#define  SL2LM       (SL1LM | STAREQ)

#define  LL1LM       (BASIC | PRESET | DIM | BRI | LION | LIOFF | LIONFULL | ONFULL | OPTIONS)
#define  LL2LM       (LL1LM | STAREQ)
#define  AMEXC16     (STDAM | EXC16)
#define  AMEXC8      (STDAM | EXC8 )
#define  AMEXC4      (STDAM | EXC4 )
#define  CAMEXC4     (STDX10 | EXC4 | MON | MOFF)
#define  VIRT4       (TARG | MON | MOFF | BRI | DIM | OPTIONS)
#define  XSHUT0      (EXT0 | MON | MOFF | RESUME)
#define  VIRTUAL     (TARG | VDATA)

static struct modules_st {
  char           *label;  /* Case insensitive */
  int            maxlevel;
  unsigned long  cflags;
} modules[] = {
  {"NONE",        MXLS, NOMATT    },  /* Has no attributes */
  {"StdAM",       MXLS, STDAM     },  /* Standard X10 1-way Appliance Module */
  {"AM",          MXLS, STDAM     },  /* Standard X10 1-way Appliance Module */
  {"AM486",       MXLS, STDAM     },  /* Standard X10 1-way Appliance Module */
  {"AMS",         MXLS, AMS       },  /* 2-way Appliance Module */
  {"RR501",       MXLS, AMS       },  /* X10 Transceiver/Switch */
  {"StdLM",       MXLS, STDLM     },  /* Standard X10 1-way Lamp Module */
  {"LM",          MXLS, STDLM     },  /* Standard X10 1-way Lamp Module */
  {"LM465",       MXLS, STDLM     },  /* Standard X10 1-way Lamp Module */
  {"PLM03",       MXLS, STDLM     },  /* Standard X10 1-way Lamp Module */
  {"StdWS",       MXLS, STDWS     },  /* Standard X10 1-way Lamp Wall Switch */
  {"WS",          MXLS, STDWS     },  /* Standard X10 1-way Lamp Wall Switch */
  {"WS467",       MXLS, STDWS     },  /* Standard X10 1-way Lamp Wall Switch */
  {"LM15A",       MXLS, LM15A     },  /* X10 LM15A Socket Rocket */
  {"LMS",         MXLS, LMS       },  /* 2-way Lamp Module */
  {"AM14A",       MXLEA, AM14A    },  /* X10 2-way Appliance Module, 2-pin (AM14A) */
  {"AM15A",       MXLEA, AM14A    },  /* X10 2-way Appliance Module, 3-pin (AM15A) */
  {"PAM22",       MXLEA, AM14A    },  /* X10 2-way Appliance Module, 3-pin (AM15A) */
  {"LM14A",       MXLE, LM14A     },  /* X10 2-way Lamp Module (LM14A) */
  {"PLM21",       MXLE, LM14A     },  /* X10 2-way Lamp Module (LM14A) */
  {"SL1AM",       MXLP, SL1AM     },  /* SwitchLinc 1-way Switch */
  {"SL2AM",       MXLP, SL2AM     },  /* SwitchLinc 2-way Switch */
  {"SL1LM",       MXLP, SL1LM     },  /* SwitchLinc 1-way Lamp Module */
  {"SL2LM",       MXLP, SL2LM     },  /* SwitchLinc 2-way Lamp Module */
  {"SL2380W",     MXLP, SL2LM     },  /* SwitchLinc 2380W Dimmer */
  {"LL1LM",       MXLP, LL1LM     },  /* LampLinc 1-way Dimmer */
  {"LL2LM",       MXLP, LL2LM     },  /* LampLink 2-way Dimmer */
  {"LL2000STW",   MXLP, LL2LM     },  /* LampLinc 2000STW Dimmer */
  {"REMOTE2",     MXLS, REM2      },  /* Remote transmitter, 2 function */
  {"REMOTE3",     MXLS, REM3      },  /* Remote transmitter, 3 function */
  {"REMOTE4",     MXLS, REM4      },  /* Remote transmitter, 4 function */
  {"REMOTE6",     MXLS, REM6      },  /* Remote transmitter, 6 function */
  {"REMOTEP",     MXLP, PRESET    },  /* Remote transmitter, Preset 1-32 only */
  {"AMEXC",       MXLS, AMEXC16   },  /* AM with exclusive-16 addressing */
  {"AMEXC16",     MXLS, AMEXC16   },  /* AM with exclusive addressing */
  {"AMEXC8",      MXLS, AMEXC8    },  /* AM with exclusive-8 addressing */
  {"RAIN8",       MXLS, AMEXC8    },  /* WGL Rain8 irrigation controller */
  {"AMEXC4",      MXLS, AMEXC4    },  /* AM with exclusive-4 addressing */
  {"XM10A",       MXLS, CAMEXC4   },  /* X10 camera power supply */
  {"XM13A",       MXLS, CAMEXC4   },  /* X10 camera power supply */
  {"XM14A",       MXLS, CAMEXC4   },  /* X10 pan/tilt power supply */
  {"VIRT4",       MXLV, VIRT4     },  /* Virtual module, 4 function */
  {"VDATA",       MXLV, VIRTUAL   },  /* Virtual module data */
#ifdef HASEXT0
  {"SHUTTER",     MXLS0, XSHUT0   },  /* Extended code Type 0 shutter */
  {"SW10",        MXLS0, XSHUT0   },  /* Marmitek SW10 shutter control */
#endif
};
static int ntypes = (sizeof(modules)/sizeof(struct modules_st));

unsigned int modmask[NumModMasks][16];
unsigned char maxdimlevel[16][16];
unsigned char ondimlevel[16][16];

extern CONFIG config;

/*-------------------------------------------------------+
 | Return the index in the module table for the argument |
 | name, or -1 if not found                              |
 | The comparison is case insensitive                    |
 +-------------------------------------------------------*/ 
int lookup_module_type ( char *modelname ) 
{
   char buffer[NAME_LEN + 1], label[NAME_LEN + 1];
   int  j;

   strncpy2(buffer, modelname, sizeof(buffer) - 1);
   strupper(buffer);

   for ( j = 0; j < ntypes; j++ ) {
      strncpy2(label, modules[j].label, sizeof(label) - 1);
      strupper(label);
      if ( strcmp(buffer, label) == 0 )
         return j;
   }

   return -1;
}

/*-------------------------------------------------------+
 | Pass back through the argument list the flags and     |
 | maxlevel values for module index 'module_type'        |
 +-------------------------------------------------------*/ 
void module_attributes ( int module_type,
                      unsigned long *flags, int *maxlevel )
{
   if ( module_type >= 0 ) {
      *flags = modules[module_type].cflags;
      *maxlevel = modules[module_type].maxlevel;
   }
   else {
      *flags = 0;
      *maxlevel = 0;
   }
   return;
}

/*-------------------------------------------------------+
 | Called by add_alias() to add options specified on the |
 | ALIAS line in the config file.                        |
 +-------------------------------------------------------*/ 
int add_module_options ( int aliasindex, char **tokens, int ntokens )
{
   int    j, type, val, maxval, valoffset;
   char   *sp;
   ALIAS  *aliasp;

   aliasp = config.aliasp;
   type = aliasp[aliasindex].modtype;
   maxval = modules[type].maxlevel;
   valoffset = (modules[type].cflags & PRESET) ? 1 : 0;

   if ( (modules[type].cflags & OPTIONS) == 0 ) {
      store_error_message("No options are supported for this module");
      return 1;
   }

   for ( j = 0; j < ntokens; j++ )
      strupper(tokens[j]);

   j = 0;
   while ( ntokens > 0 ) {
      if ( !strcmp(tokens[j], "ONLEVEL") ) {
         if ( ntokens > 1 &&
              (!strcmp(tokens[j + 1], "RESUME") ||
               !strcmp(tokens[j + 1], "0"))        ) {
            aliasp[aliasindex].flags |= RESUME;
            aliasp[aliasindex].flags &= ~ONFULL;
            aliasp[aliasindex].optflags |= RESUME;
            aliasp[aliasindex].optflags &= ~ONFULL;
            aliasp[aliasindex].onlevel = maxval;
            j += 2;
            ntokens -= 2;
         }
         else if ( ntokens > 1 &&
                 (val = strtol(tokens[j + 1], &sp, 10)) &&
                 *sp == '\0' && val > 0 &&
                  val <= (maxval + valoffset) ) {
            aliasp[aliasindex].flags &= ~RESUME;
            aliasp[aliasindex].flags |= ONFULL;
            aliasp[aliasindex].optflags &= ~RESUME;
            aliasp[aliasindex].optflags |= ONFULL;
            aliasp[aliasindex].onlevel = val - valoffset;
            j += 2;
            ntokens -= 2;
         }
         else if ( ntokens > 1 ) {
            store_error_message("Invalid ONLEVEL parameter");
            return 1;
         }
         else {
            store_error_message("Missing ONLEVEL parameter");
            return 1;
         }
      }
      else {          
         store_error_message("Module option not recognized");
         return 1;
      }
   }
   return 0;
}

/*-------------------------------------------------------+
 | Display options specified for an ALIAS.               |
 +-------------------------------------------------------*/ 
char *display_module_options (int aliasindex )
{
   static   char buffer[32];
   ALIAS    *aliasp;
   long int optflags;

   aliasp = config.aliasp;

   if ( !aliasp )
      return "";

   if ( !(optflags = aliasp[aliasindex].optflags) )
      return "";

   if ( optflags & RESUME )
      sprintf(buffer, "ONLEVEL RESUME");
   else if ( optflags & ONFULL && aliasp[aliasindex].flags & PRESET)
      sprintf(buffer, "ONLEVEL %d", aliasp[aliasindex].onlevel + 1);
   else if ( optflags & ONFULL )
      sprintf(buffer, "ONLEVEL %d", aliasp[aliasindex].onlevel);
   else
      sprintf(buffer, "???");

   return buffer;
} 
   

/*-------------------------------------------------------+
 | Return a pointer to the module name corresponding to  |
 | the argument module_type.                             | 
 +-------------------------------------------------------*/ 
char *lookup_module_name ( int module_type ) 
{

   if ( module_type >= 0 && module_type < ntypes )
      return modules[module_type].label;

   return "";
}

/*-------------------------------------------------------+
 | Create the state filter determined by characteristics |
 | of each module defined in the config file.            |
 +-------------------------------------------------------*/ 
void set_module_masks ( ALIAS *aliasp )
{
   int j, ucode;
   unsigned char hcode, maxlevel, onlevel;
   unsigned int  bitmap, cflags;
   unsigned int  defined[16];

   if ( config.module_types == NO ) {
      for ( j = 0; j < NumModMasks; j++ ) {
         for ( hcode = 0; hcode < 16; hcode++ )
            modmask[j][hcode] = 0xffff;
      }
      return;
   }

   /* Default for undefined modules */
   cflags = modules[config.default_module].cflags;
   maxlevel = modules[config.default_module].maxlevel;

   /* Record housecode|units with defined module type */
   for ( j = 0; j < 16; j++ ) {
      defined[j] = 0;
   }
   j = 0;
   while ( aliasp && aliasp[j].line_no > 0 ) {
      if ( aliasp[j].modtype >= 0 ) {
         /* Module is defined */
         hcode = hc2code(aliasp[j].housecode);
         defined[hcode] |= aliasp[j].unitbmap;
      }
      j++;
   }

   /* Initialize to zero */
   for ( j = 0; j < NumModMasks; j++ ) {
      for ( hcode = 0; hcode < 16; hcode++ )
         modmask[j][hcode] = 0;
   }

   /* Set the characteristic of undefined modules */
   for ( hcode = 0; hcode < 16; hcode++ ) {
      bitmap = ~defined[hcode];
      /* Mark units which respond */   
      if ( cflags & UNOFF )
         modmask[AllOffMask][hcode] = bitmap;
      if ( cflags & LION )
         modmask[LightsOnMask][hcode] = bitmap;
      if ( cflags & MON )
         modmask[OnMask][hcode] = bitmap;
      if ( cflags & MOFF )
         modmask[OffMask][hcode] = bitmap;
      if ( cflags & DIM )
         modmask[DimMask][hcode] = bitmap;
      if ( cflags & BRI )
         modmask[BriMask][hcode] = bitmap;
      if ( cflags & BRIB4DIM )
         modmask[BriDimMask][hcode] = bitmap;
      if ( cflags & STDX10 )
         modmask[StdMask][hcode] = bitmap;
      if ( cflags & EXT3SW ) {
         modmask[Ext3Mask][hcode] = bitmap;
         modmask[AllOnMask][hcode] = bitmap;
      }
      if ( cflags & EXT3DIM ) 
         modmask[Ext3DimMask][hcode] = bitmap;
      if ( cflags & LIOFF )
         modmask[LightsOffMask][hcode] = bitmap;
      if ( cflags & PRESET )
         modmask[PresetMask][hcode] = bitmap;
      if ( cflags & STAREQ )
         modmask[StatusMask][hcode] = bitmap;
      if ( cflags & STAON )
         modmask[StatusOnMask][hcode] = bitmap;
      if ( cflags & STAOFF )
         modmask[StatusOffMask][hcode] = bitmap;
      if ( cflags & LIONFULL )
         modmask[LightsOnFullMask][hcode] = bitmap;
      if ( cflags & ONFULL )
         modmask[OnFullMask][hcode] = bitmap;
      if ( cflags & ONFULLOFF )
         modmask[OnFullOffMask][hcode] = bitmap;
      if ( cflags & ALLON )
         modmask[AllOnMask][hcode] = bitmap;
      if ( cflags & RESUME )
         modmask[ResumeMask][hcode] = bitmap;
      if ( cflags & TARG )
         modmask[TargMask][hcode] = bitmap;
      if ( cflags & EXC16 )
         modmask[Exc16Mask][hcode] = bitmap;
      if ( cflags & EXC8 )
         modmask[Exc8Mask][hcode] = bitmap;
      if ( cflags & EXC4 )
         modmask[Exc4Mask][hcode] = bitmap;
      if ( cflags & EXT0 )
         modmask[Ext0Mask][hcode] = bitmap;
      if ( cflags & VDATA )
         modmask[VdataMask][hcode] = bitmap;

      /* Set max and on dim levels for each unit */
      for ( ucode = 0; ucode < 16; ucode++ ) {
         if ( bitmap & (1 << ucode) ) {
            maxdimlevel[hcode][ucode] = maxlevel;
            ondimlevel[hcode][ucode] = maxlevel;
         }
      }
   }
   
   /* Now fill in the characteristics of defined modules */
   j = 0;
   while ( aliasp && aliasp[j].line_no > 0 ) {
      if ( aliasp[j].modtype < 0 ) {
         /* Module is undefined */
         j++;
         continue;
      }
      hcode  = hc2code(aliasp[j].housecode);
      bitmap = aliasp[j].unitbmap;
      cflags = aliasp[j].flags;
      maxlevel = aliasp[j].maxlevel;
      onlevel = aliasp[j].onlevel;

      /* Mark units which respond */   
      if ( cflags & UNOFF )
         modmask[AllOffMask][hcode] |= bitmap;
      if ( cflags & LION )
         modmask[LightsOnMask][hcode] |= bitmap;
      if ( cflags & MON )
         modmask[OnMask][hcode] |= bitmap;
      if ( cflags & MOFF )
         modmask[OffMask][hcode] |= bitmap;
      if ( cflags & DIM )
         modmask[DimMask][hcode] |= bitmap;
      if ( cflags & BRI )
         modmask[BriMask][hcode] |= bitmap;
      if ( cflags & BRIB4DIM )
         modmask[BriDimMask][hcode] |= bitmap;
      if ( cflags & STDX10 )
         modmask[StdMask][hcode] |= bitmap;
      if ( cflags & EXT3SW ) {
         modmask[Ext3Mask][hcode] |= bitmap;
         modmask[AllOnMask][hcode] |= bitmap;
      }
      if ( cflags & EXT3DIM ) 
         modmask[Ext3DimMask][hcode] |= bitmap;
      if ( cflags & LIOFF )
         modmask[LightsOffMask][hcode] |= bitmap;
      if ( cflags & PRESET )
         modmask[PresetMask][hcode] |= bitmap;
      if ( cflags & STAREQ )
         modmask[StatusMask][hcode] |= bitmap;
      if ( cflags & STAON )
         modmask[StatusOnMask][hcode] |= bitmap;
      if ( cflags & STAOFF )
         modmask[StatusOffMask][hcode] |= bitmap;
      if ( cflags & LIONFULL )
         modmask[LightsOnFullMask][hcode] |= bitmap;
      if ( cflags & ONFULL )
         modmask[OnFullMask][hcode] |= bitmap;
      if ( cflags & ONFULLOFF )
         modmask[OnFullOffMask][hcode] |= bitmap;
      if ( cflags & ALLON )
         modmask[AllOnMask][hcode] |= bitmap;
      if ( cflags & RESUME )
         modmask[ResumeMask][hcode] |= bitmap;
      if ( cflags & TARG )
         modmask[TargMask][hcode] |= bitmap;
      if ( cflags & EXC16 )
         modmask[Exc16Mask][hcode] |= bitmap;
      if ( cflags & EXC8 )
         modmask[Exc8Mask][hcode] |= bitmap;
      if ( cflags & EXC4 )
         modmask[Exc4Mask][hcode] |= bitmap;
      if ( cflags & EXT0 )
         modmask[Ext0Mask][hcode] |= bitmap;
      if ( cflags & VDATA )
         modmask[VdataMask][hcode] |= bitmap;

      /* Set max and on dim levels for each unit */
      for ( ucode = 0; ucode < 16; ucode++ ) {
         if ( bitmap & (1 << ucode) ) {
            maxdimlevel[hcode][ucode] = maxlevel;
            ondimlevel[hcode][ucode] = onlevel;
         }
      }

      j++;
   }

   return;
}


/*-------------------------------------------------------+
 | Display a table indicating the module attributes      |
 | (either defined or defaults) for each unit            |
 +-------------------------------------------------------*/ 
void show_module_mask ( unsigned char hcode )
{
   char *chr = ".*";
   int  lw = 13;
   unsigned int exclusive;

   printf("Module Attributes\n");
   printf("%*s %c\n", lw + 13, "Housecode", code2hc(hcode));
   printf("%*s  1..4...8.......16\n", lw, "Unit:");
   printf("%*s (%s)  %s\n", lw, "On", bmap2asc(modmask[OnMask][hcode], chr),
      "Accepts On command");
   printf("%*s (%s)  %s\n", lw, "Off", bmap2asc(modmask[OffMask][hcode], chr),
      "Accepts Off command");
   printf("%*s (%s)  %s\n", lw, "Dim", bmap2asc(modmask[DimMask][hcode], chr),
      "Accepts Dim (1-22) commands");
   printf("%*s (%s)  %s\n", lw, "Bright", bmap2asc(modmask[BriMask][hcode], chr),
      "Accepts Bright (1-22) commands");
   printf("%*s (%s)  %s\n", lw, "BrightB4Dim", bmap2asc(modmask[BriDimMask][hcode], chr),
      "Brightens to 100% before Dim/Bright if Off");
   printf("%*s (%s)  %s\n", lw, "LightsOn", bmap2asc(modmask[LightsOnMask][hcode], chr),
      "Accepts LightsOn (AllLightsOn) command");
   printf("%*s (%s)  %s\n", lw, "OnFullIfOff", bmap2asc(modmask[OnFullOffMask][hcode], chr),
      "LightsOn brightens to 100% only if Off");
   exclusive = modmask[Exc16Mask][hcode] | modmask[Exc8Mask][hcode] | modmask[Exc4Mask][hcode];
#if 0
   printf("%*s (%s)  %s\n", lw, "ExclusiveOn", bmap2asc(modmask[Exc16Mask][hcode], chr),
      "Only last unit turns On - other turn Off");
#endif
   printf("%*s (%s)  %s\n", lw, "ExclusiveOn", bmap2asc(exclusive, chr),
      "Last unit turns On, other units in group Off");
   printf("%*s (%s)  %s\n", lw, "LightsOff", bmap2asc(modmask[LightsOffMask][hcode], chr),
      "Accepts LightsOff command");
#if 0
   printf("%*s (%s)  %s\n", lw, "AllOn", bmap2asc(modmask[AllOnMask][hcode], chr),
      "Recognizes (heyu defined) AllOn command");
#endif
   printf("%*s (%s)  %s\n", lw, "AllOff", bmap2asc(modmask[AllOffMask][hcode], chr),
      "Accepts AllOff (AllUnitsOff) command");
   printf("%*s (%s)  %s\n", lw, "Preset", bmap2asc(modmask[PresetMask][hcode], chr),
      "Accepts Preset (1-32) commands");
   printf("%*s (%s)  %s\n", lw, "LevelResume", bmap2asc(modmask[ResumeMask][hcode], chr),
      "On command resumes previous brightness level");
   printf("%*s (%s)  %s\n", lw, "LevelFixed", bmap2asc(modmask[OnFullMask][hcode], chr),
      "On command sets to fixed brightness level");
   printf("%*s (%s)  %s\n", lw, "LightsOnFull", bmap2asc(modmask[LightsOnFullMask][hcode], chr),
      "LightsOn command always brightens to 100%");
   printf("%*s (%s)  %s\n", lw, "ExtSwitch", bmap2asc(modmask[Ext3Mask][hcode], chr),
      "Accepts Extended code switch commands");
   printf("%*s (%s)  %s\n", lw, "ExtDimmer", bmap2asc(modmask[Ext3DimMask][hcode], chr),
      "Accepts Extended code dimmer (0-63) commands");
   printf("%*s (%s)  %s\n", lw, "ExtShutter", bmap2asc(modmask[Ext0Mask][hcode], chr),
      "Accepts Extended code shutter (0-25) commands");
   printf("%*s (%s)  %s\n", lw, "StatusOn", bmap2asc(modmask[StatusOnMask][hcode], chr),
      "Sends only StatusOn in response to StatusReq");
   printf("%*s (%s)  %s\n", lw, "StatusOff", bmap2asc(modmask[StatusOffMask][hcode], chr),
      "Sends only StatusOff in response to StatusReq");
   printf("%*s (%s)  %s\n", lw, "StatusReq", bmap2asc(modmask[StatusMask][hcode], chr),
      "Responds to StatusReq command");
   printf("%*s (%s)  %s\n", lw, "Virtual", bmap2asc(modmask[VdataMask][hcode], chr),
      "Virtual Data module");
   printf("\n");

   return;
}
