/*----------------------------------------------------------------------------+
 |                                                                            |
 |                       HEYU Configuration                                   |
 |       Copyright 2002,2003,2004,2005,2006 Charles W. Sullivan               |
 |                      All Right Reserved                                    |
 |                                                                            |
 |                                                                            |
 | This software is licensed free of charge for non-commercial distribution   |
 | and for personal and internal business use only.  Inclusion of this        |
 | software or any part thereof in a commercial product is prohibited         |
 | without the prior written permission of the author.  You may copy, use,    |
 | and distribute this software subject to the following restrictions:        |
 |                                                                            |
 |  1)	You may not charge money for it.                                      |
 |  2)	You may not remove or alter this license, copyright notice, or the    |
 |      included disclaimers.                                                 |
 |  3)	You may not claim you wrote it.                                       |
 |  4)	If you make improvements (or other changes), you are requested        |
 |	to send them to the official Heyu maintainer, Charles W. Sullivan     |
 |      (cwsulliv01@heyu.org), so there's a focal point for distributing      |
 |      improved versions.                                                    |
 |                                                                            |
 | As used herein, HEYU is a trademark of Daniel B. Suthers, while X10,       | 
 | CM11A, and ActiveHome are trademarks of X-10 (USA) Inc.                    |
 | The author is not affiliated with either entity.                           |
 |                                                                            |
 | Charles W. Sullivan                                                        |
 | Greensboro, North Carolina                                                 |
 | Email: cwsulliv01@heyu.org                                                 |
 |                                                                            |
 | Disclaimers:                                                               |
 | THERE IS NO ASSURANCE THAT THIS SOFTWARE IS FREE OF DEFECTS AND IT MUST    |
 | NOT BE USED IN ANY SITUATION WHERE THERE IS ANY CHANCE THAT ITS            |
 | PERFORMANCE OR FAILURE TO PERFORM AS EXPECTED COULD RESULT IN LOSS OF      |
 | LIFE, INJURY TO PERSONS OR PROPERTY, FINANCIAL LOSS, OR LEGAL LIABILITY.   |
 |                                                                            |
 | TO THE EXTENT ALLOWED BY APPLICABLE LAW, THIS SOFTWARE IS PROVIDED "AS IS",|
 | WITH NO EXPRESS OR IMPLIED WARRANTY, INCLUDING, BUT NOT LIMITED TO, THE    |
 | IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.|
 |                                                                            |
 | IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW WILL THE AUTHOR BE LIABLE    |
 | FOR DAMAGES, INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL   |
 | DAMAGES ARISING OUT OF THE USE OR INABILITY TO USE THIS SOFTWARE EVEN IF   |
 | THE AUTHOR HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.            |
 |                                                                            |
 +----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#if defined(SYSV) || defined(FREEBSD) || defined(OPENBSD)
#include <string.h>
#else
#include <strings.h>
#endif
#include "x10.h"
#include "process.h"

extern int line_no;
extern char heyu_path[PATH_LEN + 1];
extern char alt_path[PATH_LEN + 1];
extern char heyu_script[PATH_LEN + 1];
extern long std_tzone;
extern char default_housecode;

char        statefile[PATH_LEN + 1];
char        enginelockfile[PATH_LEN + 1];
char        pfailfile[PATH_LEN + 1];

CONFIG config;

enum {
   OldAlias, Alias, UserSyn, Launcher, Script, Scene, RFBursts,
   /* Directives with multiple instances allowed */
   /* must be before 'Separator'.                */ 
   Separator,  
   Tty, TtyAux, HouseCode, ForceAddr,
   NewFormat, ScheduleFile, MaxPParms, RcsTemp, StatusTimeout,
   SpfTimeout, DispExpMac, AckHails, Mode, ProgramDays,
   AsIfDate, AsIfTime, CombineEvents, CompressMacros, FebKluge,
   Latitude, Longitude, DawnOption, DuskOption, DawnSubstitute,
   DuskSubstitute, MinDawn, MaxDawn, MinDusk, MaxDusk, CheckFiles,
   ReportPath, ReplDelay, Ignored, ResvTimers, TrigTag, XrefApp,
   MacTerm, AutoChain, ResOverlap, ModuleTypes, LaunchMode,
   FunctionMode, LaunchSrc, DefaultModule, ScriptShell, ScriptMode,
   LogDir, IsDarkOffset, EnvAliasPrefix, DevType, SunMode, Fix5A,
   AutoFetch, PfailUpdate, BitDelay, DefRFBursts, BurstSpacing,
   TimerTweak, RFPostDelay, RFFarbDelay, DispRFX, LoopCount,
   RestrictDims, StateFmt, /* StateDir, */PfailScript, CM11PostDelay,
   StartEngine, IgnoreSilent, NoSwitch, RespoolPerms, SpoolMax,
   CheckRILine, SendRetries
};

#define CMLT  1    /* Multiple instances allowed */
#define COVR  2    /* Allow override in sched file */
#define CHLP  4    /* Minimal config for help command */
#define CCAL  8    /* Minimal config for syscheck command */
#define CSTP  16   /* Minimal config for stop command */

static struct conf {
   char          *name;
   int           mintoken;
   int           maxtoken;
   unsigned char isparsed;
   unsigned char override;  /* Allow override in sched file */
   unsigned int  flags;
   unsigned char switchval;
} command[] = {
   {"(_alias_)",            3, 3, 0, 0, CMLT,    OldAlias },
   {"ALIAS",                3, 6, 0, 0, CMLT,    Alias },
   {"LAUNCHER",             2,-1, 0, 0, CMLT,    Launcher},
   {"SCRIPT",               2,-1, 0, 0, CMLT,    Script },
   {"USERSYN",              2,-1, 0, 0, CMLT,    UserSyn },
   {"SCENE",                2,-1, 0, 0, CMLT,    Scene },
   {"RF_BURSTS",            3, 3, 0, 0, CMLT,    RFBursts},
   /* Multiple instances OK for switchval above */
   {"TTY",                  2, 4, 0, 0, CSTP,    Tty },
   {"TTY_AUX",              3, 3, 0, 0, CSTP,    TtyAux},
   {"HOUSECODE",            2, 2, 0, 0, 0,       HouseCode },
   {"FORCE_ADDR",           2, 2, 0, 0, 0,       ForceAddr },
   {"NEWFORMAT",            1, 1, 0, 0, 0,       Ignored },
   {"MACROXREF",            2, 2, 0, 0, 0,       Ignored },
   {"SCHEDULE_FILE",        2, 2, 0, 0, 0,       ScheduleFile },
   {"MAX_PPARMS",           2, 2, 0, 0, 0,       MaxPParms },
   {"STATUS_TIMEOUT",       2, 2, 0, 0, 0,       StatusTimeout },
   {"SPF_TIMEOUT",          2, 2, 0, 0, 0,       SpfTimeout },
   {"RCS_TEMPERATURE",      2, 2, 0, 0, 0,       RcsTemp },
   {"RCS_DECODE",           2, 2, 0, 0, 0,       RcsTemp },
   {"DISPLAY_EXP_MACROS",   2, 2, 0, 0, 0,       Ignored }, 
   {"ACK_HAILS",            2, 2, 0, 0, 0,       AckHails }, 
   {"MODE",                 2, 2, 0, 1, COVR,    Mode },
   {"PROGRAM_DAYS",         2, 2, 0, 1, COVR,    ProgramDays },
   {"ASIF_DATE",            2, 2, 0, 1, COVR,    AsIfDate },
   {"ASIF_TIME",            2, 2, 0, 1, COVR,    AsIfTime },
   {"COMBINE_EVENTS",       2, 2, 0, 1, COVR,    CombineEvents },
   {"COMPRESS_MACROS",      2, 2, 0, 1, COVR,    CompressMacros },
   {"FEB_KLUGE",            2, 2, 0, 1, COVR,    FebKluge },
   {"LATITUDE",             2, 2, 0, 1, COVR,    Latitude },
   {"LONGITUDE",            2, 2, 0, 1, COVR,    Longitude },
   {"DAWN_OPTION",          2, 2, 0, 1, COVR,    DawnOption },
   {"DUSK_OPTION",          2, 2, 0, 1, COVR,    DuskOption },
   {"DAWN_SUBSTITUTE",      2, 2, 0, 1, COVR,    Ignored },
   {"DUSK_SUBSTITUTE",      2, 2, 0, 1, COVR,    Ignored },
   {"MIN_DAWN",             2, 2, 0, 1, COVR,    MinDawn },
   {"MAX_DAWN",             2, 2, 0, 1, COVR,    MaxDawn },
   {"MIN_DUSK",             2, 2, 0, 1, COVR,    MinDusk },
   {"MAX_DUSK",             2, 2, 0, 1, COVR,    MaxDusk },
   {"WRITE_CHECK_FILES",    2, 2, 0, 1, COVR,    CheckFiles },
   {"REPORT_PATH",          2, 2, 0, 1, COVR,    ReportPath },
   {"REPL_DELAYED_MACROS",  2, 2, 0, 1, COVR,    ReplDelay },
   {"RESERVED_TIMERS",      2, 2, 0, 1, COVR,    ResvTimers }, /* WIP */
   {"TRIGGER_TAG",          2, 2, 0, 1, COVR,    Ignored }, 
   {"XREF_APPEND",          2, 2, 0, 1, COVR,    Ignored }, 
   {"MACTERM",              2, 2, 0, 1, COVR,    MacTerm }, /* WIP */
   {"RESOLVE_OVERLAP",      2, 2, 0, 1, COVR,    ResOverlap },
   {"SCRIPT_MODE",          2, 2, 0, 0, 0,       ScriptMode },
   {"SCRIPT_SHELL",         2, 2, 0, 0, 0,       ScriptShell },
   {"LAUNCH_SOURCE",        2, 6, 0, 0, 0,       LaunchSrc },
   {"DEFAULT_MODULE",       2, 2, 0, 0, 0,       DefaultModule },
   {"LOG_DIR",              2, 2, 0, 0, 0,       LogDir },
   {"ISDARK_OFFSET",        2, 2, 0, 0, 0,       IsDarkOffset },
   {"ENV_ALIAS_PREFIX",     2, 2, 0, 0, 0,       EnvAliasPrefix },
   {"DAWNDUSK_DEF",         2, 2, 0, 1, COVR,    SunMode },
   {"FIX_5A",               2, 2, 0, 0, 0,       Fix5A },
   {"AUTOFETCH",            2, 2, 0, 0, 0,       AutoFetch },
   {"POWERFAIL_UPDATE",     2, 2, 0, 0, 0,       PfailUpdate },
   {"CM17A_BIT_DELAY",      2, 2, 0, 0, 0,       BitDelay },
   {"DEF_RF_BURSTS",        2, 2, 0, 0, 0,       DefRFBursts },
   {"TIMER_LOOPCOUNT",      2, 2, 0, 0, 0,       LoopCount },
   {"RF_BURST_SPACING",     2, 2, 0, 0, 0,       BurstSpacing },
   {"RF_TIMER_TWEAK",       2, 2, 0, 0, 0,       TimerTweak },
   {"RF_POST_DELAY",        2, 2, 0, 0, 0,       RFPostDelay },
   {"RF_FARB_DELAY",        2, 2, 0, 0, 0,       RFFarbDelay },
   {"DISPLAY_RF_XMIT",      2, 2, 0, 0, 0,       DispRFX },
   {"RESTRICT_DIMS",        2, 2, 0, 0, 0,       RestrictDims },
   {"STATE_FORMAT",         2, 2, 0, 0, 0,       StateFmt },
   {"RELAY_POWERFAIL_SCRIPT", 2,-1, 0, 0, 0,       PfailScript},
   {"CM11_POST_DELAY",      2, 2, 0, 0, 0,       CM11PostDelay},
   {"START_ENGINE",         2, 2, 0, 0, 0,       StartEngine},
   {"SECTION",              1,-1, 0, 0, COVR|CMLT, IgnoreSilent},
   {"RF_NOSWITCH",          2, 2, 0, 0, 0,       NoSwitch},
   {"RESPOOL_PERMISSIONS",  2, 2, 0, 0, 0,       RespoolPerms},
   {"SPOOLFILE_MAX",        2, 2, 0, 0, 0,       SpoolMax},
   {"CHECK_RI_LINE",        2, 2, 0, 0, 0,       CheckRILine},
   {"SEND_RETRIES",         2, 2, 0, 0, 0,       SendRetries},
};
int ncommands = ( sizeof(command)/sizeof(struct conf) );

/* Dawn/Dusk options */
static struct ddopt {
   char     *label;
   unsigned char value;
} dd_option[] = {
   {"FIRST",    FIRST },
   {"EARLIEST", EARLIEST },
   {"LATEST",   LATEST },
   {"AVERAGE",  AVERAGE },
   {"MEDIAN",   MEDIAN },
};
int nddopt = ( sizeof(dd_option)/sizeof(struct ddopt) );

/* Launch source options - set default sources of signal */
/* permitted to launch a script.                         */
static struct lsopt {
   char      *label;
   unsigned int value;
} ls_option[] = {
   {"SNDC",     SNDC   }, /* Sent from command line */
   {"SNDM",     SNDM   }, /* Sent from macro executed by Timer */
   {"SNDT",     SNDT   }, /* Sent from macro executed by Trigger */
   {"SNDP",     SNDP   }, /* Sent from a relay powerfail script */
   {"RCVI",     RCVI   }, /* Received from the interface */
   {"RCVT",     RCVT   }, /* Trigger which executed a macro */
   {"ANYSRC",   LSALL  }, /* Any of the above (SNDS and SNDP not included!) */
   {"NOSRC",    LSNONE }, /* No sources are the default */
};
int nlsopt = ( sizeof(ls_option)/sizeof(struct lsopt) );

/* CM17A function labels */
static struct cm17a_label {
   char          *label;
   unsigned char subcode;
} rf_label[] = {
  {"falloff",    0 },
  {"flightson",  1 },
  {"fon",        2 },
  {"foff",       3 },
  {"fdim",       4 },
  {"fbright",    5 },
  {"flightsoff", 6 },
  {"farb",       7 },
};
int nrflabels = ( sizeof(rf_label)/sizeof(struct cm17a_label) );

/*---------------------------------------------------------------------+
 | Reset the isparsed flags in the conf struct array.                  |
 +---------------------------------------------------------------------*/
void reset_isparsed_flags ( void )
{
   int j;

   for ( j = 0; j < ncommands; j++ )
      command[j].isparsed = 0;

   return;
}

#if 0
/*------------------------------------------------------+
 | Create a suffix for files to be locked by prefixing  |
 | the part of the device name following the last '/'   |
 | with the '.' character.  E.g., /dev/ttyS0 -> .ttyS0  |
 +------------------------------------------------------*/
void create_lock_suffix ( char *suffix, char *tty )
{
   char *sp;

   if ( (sp = strrchr(tty, '/')) != NULL ) {
      strcpy(suffix, sp);
      *suffix = '.';
   }
   else {	   
      strcpy(suffix, ".");
      strcat(suffix, tty);
   }

   return;
}
#endif

/*---------------------------------------------------------------------+
 | Return 1 if all valid characters within alias label, otherwise 0    |
 +---------------------------------------------------------------------*/
int is_valid_alias_label ( char *label, char **sp )
{
   *sp = label;
   while ( **sp ) {
      if ( isalnum((int)(**sp)) || strchr("-_.", **sp) ) {
         (*sp)++;
	 continue;
      }
      return 0;
   }
   return 1;
}   
	      
/*---------------------------------------------------------------------+
 | Initialize the CONFIG structure with default values.                |
 +---------------------------------------------------------------------*/
void initialize_config ( void )
{
   char *sp1;
   int  j;

   /* Load default configuration values */
   config.read_flag = 0;
   config.mode = DEF_MODE;
   (void) strncpy2(config.schedfile, DEF_SCHEDFILE, sizeof(config.schedfile) - 1);
   config.asif_date = -1;
   config.asif_time = -1;
   config.program_days_in = DEF_PROGRAM_DAYS;
   config.program_days = DEF_PROGRAM_DAYS;
   config.combine_events = DEF_COMBINE_EVENTS;
   config.compress_macros = DEF_COMPRESS_MACROS;
   config.feb_kluge = DEF_FEB_KLUGE;
   config.housecode = DEF_HOUSECODE;
   config.force_addr = DEF_FORCE_ADDR;
   config.loc_flag = 0;
   config.dawn_option = DEF_DAWN_OPTION;
   config.dusk_option = DEF_DUSK_OPTION;
   config.dawn_substitute = DEF_DAWN_SUBSTITUTE;
   config.dusk_substitute = DEF_DUSK_SUBSTITUTE;
   config.min_dawn = DEF_MIN_DAWN;
   config.max_dawn = DEF_MAX_DAWN;
   config.min_dusk = DEF_MIN_DUSK;
   config.max_dusk = DEF_MAX_DUSK;
   strncpy2(config.tty, DEF_TTY, sizeof(config.tty) - 1);
   config.ttyaux[0] = '\0';
   config.auxdev = 0;
   config.newformat = 0;
   config.checkfiles = DEF_CHECK_FILES;
   config.repl_delay = DEF_REPL_DELAYED_MACROS;
   config.reserved_timers = DEF_RESERVED_TIMERS;
   config.aliasp = NULL;
   config.scenep = NULL;
   config.scriptp = NULL;
   config.launcherp = NULL;
   config.max_pparms = DEF_MAX_PPARMS;
   config.rcs_temperature = DEF_RCS_TEMPERATURE;
   config.trigger_tag = DEF_TRIGGER_TAG;
   config.display_offset = DEF_DISPLAY_OFFSET;
   config.xref_append = DEF_XREF_APPEND;
   config.ack_hails = DEF_ACK_HAILS;
   config.macterm = DEF_MACTERM;
   config.status_timeout = DEF_STATUS_TIMEOUT;
   config.spf_timeout = DEF_SPF_TIMEOUT;
   config.disp_exp_mac = DEF_DISPLAY_EXP_MACROS;
   config.module_types = DEF_MODULE_TYPES;
   config.launch_mode = DEF_LAUNCH_MODE;
   config.function_mode = DEF_FUNCTION_MODE;
   config.launch_source = DEF_LAUNCH_SOURCE;
   config.res_overlap = DEF_RES_OVERLAP;
   config.default_module = lookup_module_type(DEF_DEFAULT_MODULE);
   config.script_mode = DEF_SCRIPT_MODE;
   if ( (sp1 = getenv("SHELL")) != NULL )
      strncpy2(config.script_shell, sp1, sizeof(config.script_shell) - 1);
   else
      strncpy2(config.script_shell, "/bin/sh", sizeof(config.script_shell) - 1);
   *config.logfile = '\0';
   config.isdark_offset = DEF_ISDARK_OFFSET;
   strncpy2(config.env_alias_prefix, DEF_ENV_ALIAS_PREFIX, sizeof(config.env_alias_prefix) - 1);
   config.sunmode = DEF_SUNMODE;
   config.fix_5a = DEF_FIX_5A;
   config.cm11_post_delay = DEF_CM11_POST_DELAY;
   config.pfail_update = DEF_PFAIL_UPDATE;
   config.cm17a_bit_delay = DEF_CM17A_BIT_DELAY;
   config.rf_burst_spacing = DEF_RF_BURST_SPACING;
   config.rf_timer_tweak = DEF_RF_TIMER_TWEAK;
   config.rf_post_delay = DEF_RF_POST_DELAY;
   config.disp_rf_xmit = DEF_DISP_RF_XMIT;
   config.def_rf_bursts = DEF_RF_BURSTS;
   for ( j = 0; j < nrflabels; j++ )
      config.rf_bursts[j] = DEF_RF_BURSTS;
   config.timer_loopcount = 0;
   config.restrict_dims = DEF_RESTRICT_DIMS;
   config.state_format = DEF_STATE_FORMAT;
   config.pfail_script = NULL;
   config.start_engine = DEF_START_ENGINE;
   config.rf_noswitch = DEF_RF_NOSWITCH;
   config.respool_perms = DEF_RESPOOL_PERMS;
   config.spool_max = DEF_SPOOLFILE_MAX;
   config.check_RI_line = DEF_CHECK_RI_LINE;
   config.send_retries = DEF_SEND_RETRIES;

   return;
}

/*---------------------------------------------------------------------+
 | Parse the users configuration file and save info in CONFIG struct   |
 | for only the minimal directives required for some specific commands.|
 | E.g., the 'heyu stop' command needs only the TTY directive.         |
 +---------------------------------------------------------------------*/
int parse_minimal_config ( FILE *fd_conf, unsigned char source )
{
   char   buffer[LINE_LEN];
   char   *sp1;
   int    err, errors;

   /* Make sure the isparsed flags are reset in the command list */
   reset_isparsed_flags();

   /* Load some default configuration values */
   initialize_config();

   line_no = 0;
   errors = 0;
   while ( fgets(buffer, LINE_LEN, fd_conf) != NULL ) {
      line_no++ ;
      buffer[LINE_LEN - 1] = '\0';

      /* Get rid of comments and blank lines */
      if ( (sp1 = strchr(buffer, '#')) != NULL )
         *sp1 = '\0';
      (void) strtrim(buffer);
      if ( buffer[0] == '\0' )
         continue;

      err = parse_config_tail(buffer, source);
      if ( err || *error_message() != '\0' ) { 
         fprintf(stderr, "Config Line %02d: %s\n", line_no, error_message());
         clear_error_message();
      }

      errors += err;
      if ( errors > MAX_ERRORS )
         return 1;
   }

   /* Determine and load the user's timezone */
   get_std_timezone();
   config.tzone = std_tzone;

   /* Determine if Daylight Time is ever in effect and if so,    */
   /* the minutes after midnight it goes into and out of effect. */
   config.isdst = get_dst_info(0);

   /* Add configuration items from environment */
   errors += environment_config();

   errors += finalize_config();
   if ( *error_message() != '\0' ) {
      fprintf(stderr, "%s\n", error_message());
      clear_error_message();
   }

   /* Done with config file */
   line_no = 0; 
  
   return errors;
}
         

/*---------------------------------------------------------------------+
 | Parse the users configuration file and save info in CONFIG struct.  |
 | First check to see if it's already been read.                       |
 +---------------------------------------------------------------------*/
int parse_config ( FILE *fd_conf )
{
   char   buffer[LINE_LEN];
   char   *sp1;
   int    err, errors;
   int    j;
   SCENE  *scenep;
   extern char *typename[];

   /* If the configuration file has already been read into memory, */
   /* just return.                                                 */
   if ( config.read_flag != 0 ) {
      return 0;
   }

   /* Make sure the isparsed flags are reset in the command list */
   reset_isparsed_flags();

   /* Load some default configuration values */
   initialize_config();

   line_no = 0;
   errors = 0;
   while ( fgets(buffer, LINE_LEN, fd_conf) != NULL ) {
      line_no++ ;
      buffer[LINE_LEN - 1] = '\0';

      /* Get rid of comments and blank lines */
      if ( (sp1 = strchr(buffer, '#')) != NULL )
         *sp1 = '\0';
      (void) strtrim(buffer);
      if ( buffer[0] == '\0' )
         continue;

      err = parse_config_tail(buffer, SRC_CONFIG);
      if ( err || *error_message() != '\0' ) { 
         fprintf(stderr, "Config Line %02d: %s\n", line_no, error_message());
         clear_error_message();
      }

      errors += err;
      if ( errors > MAX_ERRORS )
         return 1;
   }

   /* Everything has now been read from the config file and stored */

   /* Verify the syntax of user-defined scenes/usersyns */
   scenep = config.scenep;
   j = 0;
   while ( scenep && scenep[j].line_no > 0 ) {
      if ( verify_scene(scenep[j].body) != 0 ) {
         fprintf(stderr, "Config Line %02d: %s '%s': %s\n",
            scenep[j].line_no, typename[scenep[j].type], scenep[j].label,
            error_message());
         clear_error_message();
         if ( ++errors > MAX_ERRORS )
            return errors;
      }
      else if ( *error_message() != '\0' ) {
         /* Check warning messages */
         fprintf(stderr, "Config Line %02d: %s '%s': %s\n",
            scenep[j].line_no, typename[scenep[j].type], scenep[j].label,
            error_message());
         clear_error_message();
      }
      j++;
   }

   /* Configure module masks */
   set_module_masks( config.aliasp );

   /* Use the Heyu path if the user hasn't provided an alternate */
   if ( !(*alt_path) )
      (void)strncpy2(alt_path, heyu_path, sizeof(alt_path) - 1);

   /* Determine and load the user's timezone */
   get_std_timezone();
   config.tzone = std_tzone;

   /* Determine if Daylight Time is ever in effect and if so,    */
   /* the minutes after midnight it goes into and out of effect. */
   config.isdst = get_dst_info(0);

   /* Add configuration items from environment */
   errors += environment_config();

   errors += finalize_config();
   if ( *error_message() != '\0' ) {
      fprintf(stderr, "%s\n", error_message());
      clear_error_message();
   }

   config.read_flag = (errors == 0) ? 1 : 0;

   /* Done with config file */
   line_no = 0; 
  
   return errors;
}
         

/*---------------------------------------------------------------------+
 | Parse the config line in buffer.  Argument 'source' can be          |
 | SRC_CONFIG if called from parse_config() or SRC_SCHED if called     |
 | from parse_sched(), for configuration items which may be overridden |
 | in the latter.                                                      |
 +---------------------------------------------------------------------*/
int parse_config_tail ( char *buffer, unsigned char source ) 
{  
   char   errbuffer[80];
   char   searchstr[128];
   char   directive[128];
   char   label[(NAME_LEN + SCENE_LEN + MACRO_LEN + 1)];
   char   token[50];
   char   hc;
   char   *bufp, *sp;
   int    errors = 0;
   int    num, value, hour, minut, len, modtype;
   int    bursts;
   int    j, k, commj, switchval;
   int    tokc;
   char   **tokv = NULL;
   int    perms;

      bufp = buffer;
      get_token(directive, &bufp, " \t", sizeof(directive)/sizeof(char));
      strtrim(bufp);
      strncpy2(searchstr, directive, sizeof(searchstr) - 1);
      strupper(searchstr);


      /* Search list of configuration commands starting */
      /* past "oldalias".                               */

      commj = 0; switchval = OldAlias;
      for ( j = 1; j < ncommands; j++ ) {
         if ( !strcmp(searchstr, command[j].name) ) {
            switchval = command[j].switchval;
            commj = j;
            break;
         }
      }

      /* See if override in schedule file is permitted */
#if 0
      if ( (source != SRC_CONFIG) && command[commj].override == 0 ) {
#endif
      if ( (source == SRC_SCHED || source == SRC_ENVIRON) &&
               (command[commj].flags & COVR) == 0 ) {
         if ( commj )
            sprintf(errbuffer, "Configuration directive %s not allowed here.",
                command[commj].name);
         else
            sprintf(errbuffer, "Invalid configuration directive");
         store_error_message(errbuffer);
         return 1;
      }

      /* Minimal configurations for some commands */
      if ( source == SRC_STOP && (command[commj].flags & CSTP) == 0 )
         return 0;
      if ( source == SRC_HELP && (command[commj].flags & CHLP) == 0 )
         return 0;
      if ( source == SRC_SYSCHECK && (command[commj].flags & CCAL) == 0 )
         return 0;


      /* If the directive has a fixed number of tokens (as denoted */
      /* by maxtokens > 0) or _might_ be an old-style alias,       */
      /* tokenize the tail immediately.                            */

      if ( commj == 0 || command[j].maxtoken > 0 ) {
         tokenize(bufp, " \t", &tokc, &tokv);
      }
      else {
         tokc = (*bufp == '\0') ? 0 : 1 ;
      }


      /* If not found in list and it doesn't have the correct */
      /* number of items for an alias, reject it.             */
      if ( commj == 0  && tokc != 2 ) {
         sprintf(errbuffer, "Invalid Directive '%s'", directive);
         store_error_message(errbuffer);
         return 1;
      }

#if 0                        
      /* Unless allowed, check to see it's not a duplicate  */
      /* of an earlier directive.                           */
      if ( commj > Separator ) {
         if ( command[j].isparsed & source ) {
            sprintf(errbuffer, 
               "Directive '%s' appears more than once in the file", searchstr);
            store_error_message(errbuffer);
            return 1;
         }
         else
            command[j].isparsed |= source;
      }
#endif

#if 0
      /* Unless allowed, check to see it's not a duplicate  */
      /* of an earlier directive.                           */
      if ( commj > 0 && command[j].isparsed & source && (command[j].flags & CMLT) == 0) {
         sprintf(errbuffer, 
            "Directive '%s' appears more than once in the file", searchstr);
         store_error_message(errbuffer);
         return 1;
      }
      else {
         command[j].isparsed |= source;
      }
#endif

      /* Unless allowed, check to see it's not a duplicate  */
      /* of an earlier directive.                           */
      if ( command[commj].isparsed & source && (command[commj].flags & CMLT) == 0) {
         sprintf(errbuffer, 
            "Directive '%s' appears more than once in the file", searchstr);
         store_error_message(errbuffer);
         return 1;
      }
      else {
         command[commj].isparsed |= source;
      }

      /* Check that commands found on the list have the correct */
      /* number of tokens on the line.                          */
      if ( (tokc + 1) < command[commj].mintoken ) {
         store_error_message("Too few items on line.");
         return 1;
      }
      else if ( (command[commj].maxtoken != -1) && ((tokc + 1) > command[commj].maxtoken) ) {
         store_error_message("Too many items on line.");
         return 1;
      }

      errors = 0;
      switch ( switchval ) {
         case Ignored :
            sprintf(errbuffer,
               "Directive %s is obsolete and is being ignored.", searchstr);
            store_error_message(errbuffer);
            break;

         case Mode :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "COMPATIBLE") )
               config.mode = COMPATIBLE;
            else if ( !strcmp(tokv[0], "HEYU") )
               config.mode = HEYU_MODE;
            else {
               store_error_message("MODE must be COMPATIBLE or HEYU");
               errors++;
            }
            break;

         case OldAlias :  /* Possible alias, else invalid command */
            /* Check if it matches the form of an alias */
            if ( tokc != 2 || strlen(tokv[0]) != 1 ||
                  (hc = toupper(*tokv[0])) < 'A' || hc > 'P' ) {
               store_error_message("Invalid Directive.");
               errors++;
               break;
            }
	    if ( !is_valid_alias_label(directive, &sp) ) {
	       sprintf(errbuffer, "Invalid character '%c' in alias label.", *sp);
	       store_error_message(errbuffer);
	       errors++;
	       break;
	    }
	       
            if ( strcmp(tokv[0], "macro") == 0 ) {
               store_error_message("An alias label may not be the word \"macro\".");
               errors++;
               break;
            }

            if ( strchr("_-", *directive ) ) {
               store_error_message("Alias labels may not begin with '_' or '-'");
               errors++;
               break;
            }

            if ( add_alias(&config.aliasp, directive, line_no,
                                                 hc, tokv[1], -1) < 0 ) {
               errors++;
            }
            break;

         case Alias :  /* New alias format using ALIAS directive */
            /* Expects housecode and unit code(s) to be concatenated, */
            /* the same as they are for the command line or macro.    */

	    if ( !is_valid_alias_label(tokv[0], &sp) ) {
	       sprintf(errbuffer, "Invalid character '%c' in alias label.", *sp);
	       store_error_message(errbuffer);
	       errors++;
	       break;
	    }

            if ( strcmp(tokv[0], "macro") == 0 ) {
               store_error_message("An alias label may not be the word \"macro\".");
               errors++;
               break;
            }

            if ( strchr("_-", *tokv[0] ) ) {
               store_error_message("Alias labels may not begin with '_' or '-'");
               errors++;
               break;
            }

            hc = toupper( *tokv[1] );
            *tokv[1] = ' ';
            strtrim(tokv[1]);

            modtype = -1;
            if ( tokc > 2 ) {
               if ( (modtype = lookup_module_type(tokv[2])) < 0 ) {
                  sprintf(errbuffer, "Invalid module model '%s'", tokv[2]);
                  store_error_message(errbuffer);
                  errors++;
                  break;
               }
            }

            if ( (j = add_alias(&config.aliasp, tokv[0], line_no,
                                hc, tokv[1], modtype)) < 0 ) {
               errors++;
               break;
            }

            if ( modtype >= 0 && tokc > 3 ) {
               if ( add_module_options(j, tokv + 3, tokc - 3) != 0 ) {
                  errors++;
                  break;
               }
            }
            
            break;

         case UserSyn :
            get_token(label, &bufp, " \t", sizeof(label)/sizeof(char));
            if ( add_scene(&config.scenep, label, line_no, bufp, F_USYN) < 0 ) {
               errors++;
            }
            break;

         case Scene :
            get_token(label, &bufp, " \t", sizeof(label)/sizeof(char));
            if ( add_scene(&config.scenep, label, line_no, bufp, F_SCENE) < 0 ) {
               errors++;
            }
            break;

         case AsIfDate :
            config.asif_date = strtol(tokv[0], &sp, 10);
            if ( !strchr(" \t\n", *sp) ||
	         config.asif_date < 19700101 ||
                 config.asif_date > 20380101 )  {
               store_error_message("ASIF_DATE must be yyyymmdd between 19700101 and 20380101");
               errors++;
            }
            break;

         case AsIfTime :
            num = sscanf(tokv[0], "%d:%d", &hour, &minut) ;
            value = 60 * hour + minut ;
            config.asif_time = value;
            if ( num != 2 || value < 0 || value > 1439 ) {
               store_error_message("ASIF_TIME must be hh:mm between 00:00-23:59");
               errors++;
            }
            break;

         case ScheduleFile :
            strncpy2(config.schedfile, tokv[0], sizeof(config.schedfile) - 1);
            break;

         case ProgramDays :
            strupper(tokv[0]);
            config.program_days_in = (int)strtol(tokv[0], &sp, 10);

            if ( !strchr(" \t\n", *sp) ||
	       config.program_days_in < 1 ||
	       config.program_days_in > 366 ) {
               store_error_message("PROGRAM_DAYS outside range 1-366");
               errors++;
            }
            break;

         case CombineEvents :
            strupper(tokv[0]);
            if ( !strcmp(tokv[0], "YES") )
               config.combine_events = YES;
            else if ( !strcmp(tokv[0], "NO") )
               config.combine_events = NO;
            else {
               store_error_message("COMBINE_EVENTS must be YES or NO");
               errors++;
            }
            break;

         case CompressMacros :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "YES") )
               config.compress_macros = YES;
            else if ( !strcmp(tokv[0], "NO") )
               config.compress_macros = NO;
            else {
               store_error_message("COMPRESS_MACROS must be YES or NO");
               errors++;
            }
            break;

         case FebKluge :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "YES") )
               config.feb_kluge = YES;
            else if ( !strcmp(tokv[0], "NO") )
               config.feb_kluge = NO;
            else {
               store_error_message("FEB_KLUGE must be YES or NO");
               errors++;
            }
            break;

         case Latitude :            
            errors += parse_latitude(tokv[0]);
            break;

         case Longitude :
            errors += parse_longitude(tokv[0]);
            break;

         case DawnOption :
            (void) strupper(tokv[0]);
            for ( j = 0; j < nddopt; j++ ) {
               if ( !strcmp(tokv[0], dd_option[j].label) )
                  break;
            }
            if ( j == nddopt ) {
               store_error_message("Invalid DAWN_OPTION");
               errors++;
               break;
            }
            config.dawn_option = dd_option[j].value;
            break;

         case DuskOption :
            (void) strupper(tokv[0]);
            for ( j = 0; j < nddopt; j++ ) {
               if ( !strcmp(tokv[0], dd_option[j].label) )
                  break;
            }
            if ( j == nddopt ) {
               store_error_message("Invalid DUSK_OPTION");
               errors++;
               break;
            }
            config.dusk_option = dd_option[j].value;
            break;

         case DawnSubstitute :
            num = sscanf(tokv[0], "%d:%d", &hour, &minut) ;
            value = 60 * hour + minut ;
            if ( num != 2 || value < 0 || value > 1439 ) {
               store_error_message("DAWN_SUBSTITUTE - must be 00:00-23:59 (hh:mm)");
               errors++;
               break;
            }
            config.dawn_substitute = value;
            break;

         case DuskSubstitute :
            num = sscanf(tokv[0], "%d:%d", &hour, &minut) ;
            value = 60 * hour + minut ;
            if ( num != 2 || value < 0 || value > 1439 ) {
               store_error_message("DUSK_SUBSTITUTE must be 00:00-23:59 (hh:mm)");
               errors++;
               break;
            }
            config.dusk_substitute = value;
            break;

         case MinDawn :
            if ( strcmp(strupper(tokv[0]), "OFF") == 0 ) {
               config.min_dawn = OFF;
               break;
            }
            num = sscanf(tokv[0], "%d:%d", &hour, &minut);
            value = 60 * hour + minut ;
            if ( num != 2 || value < 0 || value > 1439 ) {
               store_error_message("MIN_DAWN must be 00:00-23:59 (hh:mm) or OFF");
               errors++;
               break;
            }
            config.min_dawn = value;
            break;

         case MaxDawn :
            if ( strcmp( strupper(tokv[0]), "OFF") == 0 ) {
               config.max_dawn = OFF;
               break;
            }
            num = sscanf(tokv[0], "%d:%d", &hour, &minut);
            value = 60 * hour + minut ;
            if ( num != 2 || value < 0 || value > 1439 ) {
               store_error_message("MAX_DAWN must be 00:00-23:59 (hh:mm) or OFF");
               errors++;
               break;
            }
            config.max_dawn = value;
            break;

         case MinDusk :
            if ( strcmp( strupper(tokv[0]), "OFF") == 0 ) {
               config.min_dusk = OFF;
               break;
            }
            num = sscanf(tokv[0], "%d:%d", &hour, &minut);
            value = 60 * hour + minut ;
            if ( num != 2 || value < 0 || value > 1439 ) {
               store_error_message("MIN_DUSK must be 00:00-23:59 (hh:mm) or OFF");
               errors++;
               break;
            }
            config.min_dusk = value;
            break;

         case MaxDusk :
            if ( strcmp( strupper(tokv[0]), "OFF") == 0 ) {
               config.max_dusk = OFF;
               break;
            }
            num = sscanf(tokv[0], "%d:%d", &hour, &minut);
            value = 60 * hour + minut ;
            if ( num != 2 || value < 0 || value > 1439 ) {
               store_error_message("MAX_DUSK must be 00:00-23:59 (hh:mm) or OFF");
               errors++;
               break;
            }
            config.max_dusk = value;
            break;

         case Tty :
            (void) strncpy2(config.tty, tokv[0], sizeof(config.tty) - 1);

	    for ( j = 1; j < tokc; j++) {
	       strupper(tokv[j]);
               if ( strncmp(tokv[j], "CM10A", 4) == 0 ) {
	          config.device_type |= DEV_CM10A;
	       }
	       else if ( strncmp(tokv[j], "CM17A", 4) == 0 ) {
                  config.device_type |= DEV_CM17A;
	       }
	       else if ( !(strncmp(tokv[j], "CM11A", 4) == 0 ||
			   strncmp(tokv[j], "CM12U", 4) == 0)    ) {
	          sprintf(errbuffer, "Unsupported interface type '%s'", tokv[j]);
		  store_error_message(errbuffer);
	          errors++;
		  break;
	       }
	    }
            break;

         case TtyAux :
            (void) strncpy2(config.ttyaux, tokv[0], sizeof(config.ttyaux) - 1);

	    strupper(tokv[1]);
            if ( strncmp(tokv[1], "W800RF32", 8) == 0 ) {
	       config.auxdev = DEV_W800RF32;
	    }
	    else if ( strncmp(tokv[1], "MR26A", 4) == 0 ) {
               config.auxdev = DEV_MR26A;
	    }
	    else {
	       sprintf(errbuffer, "Unsupported aux device '%s'", tokv[1]);
               store_error_message(errbuffer);
	       errors++;
               break;
	    }
            break;

         case HouseCode : /* Base housecode */
            hc = toupper(*tokv[0]);
            if ( (int)strlen(tokv[0]) > 1 || hc < 'A' || hc > 'Z' ) {
               store_error_message("Invalid HOUSECODE - must be A though P");
               errors++;
               break;
            }
            config.housecode = hc;
            default_housecode = hc;
            break;

         case ForceAddr :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "YES") )
               config.force_addr = YES;
            else if ( !strcmp(tokv[0], "NO") )
               config.force_addr = NO;
            else {
               store_error_message("FORCE_ADDR must be YES or NO");
               errors++;
            }
            break;

         case NewFormat : /* New Format - allow for additional choices in future */
            if ( (tokc + 1) == 1 )
               config.newformat = 1;
            else
               config.newformat = (unsigned char)strtol(tokv[0], NULL, 10);
            break;

         case CheckFiles :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "YES") )
               config.checkfiles = YES;
            else if ( !strcmp(tokv[0], "NO") )
               config.checkfiles = NO;
            else {
               store_error_message("WRITE_CHECK_FILES must be YES or NO");
               errors++;
            }
            break;

         case ReportPath : /* Alternate path for report files */
            (void)strncpy2(alt_path, tokv[0], sizeof(alt_path) - 1);
            if ( alt_path[strlen(alt_path) - 1] != '/' )
               (void)strncat(alt_path, "/", sizeof(alt_path) - 1 - strlen(alt_path));
            break;

         case ReplDelay :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "YES") )
               config.repl_delay = YES;
            else if ( !strcmp(tokv[0], "NO") )
               config.repl_delay = NO;
            else {
               store_error_message("REPL_DELAYED_MACROS must be YES or NO");
               errors++;
            }
            break;

         case ResvTimers :
            (void) strupper(tokv[0]);
            config.reserved_timers = (int)strtol(tokv[0], &sp, 10);

            if ( !strchr(" \t\n", *sp) || 
	       config.reserved_timers < 0 || config.reserved_timers > 50 ) {
               store_error_message("RESERVED_TIMERS outside range 0-50");
               errors++;
            }
            break;

         case TrigTag :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "YES") )
               config.trigger_tag = YES;
            else if ( !strcmp(tokv[0], "NO") )
               config.trigger_tag = NO;
            else {
               store_error_message("TRIGGER_TAG must be YES or NO");
               errors++;
            }
            break;

         case MaxPParms :
            (void) strupper(tokv[0]);
            config.max_pparms = (int)strtol(tokv[0], &sp, 10);

            if ( !strchr(" \t\n", *sp) ||
	       config.max_pparms < 1 || config.max_pparms > 999 ) {
               store_error_message("MAX_PPARMS outside range 1-999");
               errors++;
            }
            break;

         case RcsTemp :
            /* Housecodes for which Preset commands received from   */
            /* thermostats are to be decoded into temperature.      */
            /* Information is stored as a housecode bitmap.         */
            sp = tokv[0];
            (void) strlower(sp);
            config.rcs_temperature = 0;
            if ( !strcmp(sp, "off") || !strcmp(sp, "no") ) {
               break;
            }
            if ( !strcmp(sp, "all") || !strcmp(sp, "yes") ) {
               config.rcs_temperature = 0xffff;
               break;
            }

            len = strlen(sp);
            if ( *sp == '[' && *(sp + len - 1) == ']' ) {
               /* Create the housecode bitmap */               
               for ( j = 1; j < len - 1; j++ ) {
                  /* Ignore spaces and commas */
                  if ( sp[j] == ' ' || sp[j] == ',' )
                     continue;
                  if ( sp[j] < 'a' || sp[j] > 'p' )
                     break;
                  config.rcs_temperature |= (1 << hc2code(sp[j]));
               }
               if ( j != len - 1 ) {
                  store_error_message("Invalid character in RCS_DECODE housecode list");
                  errors++;
                  break;
               }
               break;
            }
            store_error_message("RCS_DECODE must be OFF or [<housecode list>] or ALL");
            errors++;
            break;

         case StatusTimeout :
            config.status_timeout = (int)strtol(tokv[0], &sp, 10);

            if ( !strchr(" \t\n", *sp) ||
	       config.status_timeout < 1 || config.status_timeout > 5 ) {
               store_error_message("STATUS_TIMEOUT outside range 1-5");
               errors++;
            }
            break;


         case SpfTimeout :
            config.spf_timeout = (int)strtol(tokv[0], &sp, 10);

            if ( !strchr(" \t\n", *sp) ||
	       config.spf_timeout < 1 || config.spf_timeout > 10 ) {
               store_error_message("SPF_TIMEOUT outside range 1-10");
               errors++;
            }
            break;

         case XrefApp :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "YES") )
               config.xref_append = YES;
            else if ( !strcmp(tokv[0], "NO") )
               config.xref_append = NO;
            else {
               store_error_message("XREF_APPEND must be YES or NO");
               errors++;
            }
            break;

         case AckHails :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "YES") )
               config.ack_hails = YES;
            else if ( !strcmp(tokv[0], "NO") )
               config.ack_hails = NO;
            else {
               store_error_message("ACK_HAILS must be YES or NO");
               errors++;
            }
            break;

         case DispExpMac :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "YES") )
               config.disp_exp_mac = YES;
            else if ( !strcmp(tokv[0], "NO") )
               config.disp_exp_mac = NO;
            else {
               store_error_message("DISPLAY_EXP_MACROS must be YES or NO");
               errors++;
            }
            break;

         case MacTerm :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "YES") )
               config.macterm = YES;
            else if ( !strcmp(tokv[0], "NO") )
               config.macterm = NO;
            else {
               store_error_message("MACTERM must be YES or NO");
               errors++;
            }
            break;

         case AutoChain :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "YES") )
               config.auto_chain = YES;
            else if ( !strcmp(tokv[0], "NO") )
               config.auto_chain = NO;
            else {
               store_error_message("AUTO_CHAIN must be YES or NO");
               errors++;
            }
            break;

         case Launcher :
            if ( add_launchers(&config.launcherp, line_no, bufp) < 0 )
               errors++;
            break;

         case Script :
            if ( add_script(&config.scriptp, &config.launcherp, line_no, bufp) < 0 )
               errors++;
            break;

         case ModuleTypes :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "YES") )
               config.module_types = YES;
            else if ( !strcmp(tokv[0], "NO") )
               config.module_types = NO;
            else {
               store_error_message("MODULE_TYPES must be YES or NO");
               errors++;
            }
            break;

         case FunctionMode :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "ACTUAL") )
               config.function_mode = FUNC_ACTUAL;
            else if ( !strcmp(tokv[0], "GENERIC") )
               config.function_mode = FUNC_GENERIC;
            else {
               store_error_message("FUNCTION_MODE must be ACTUAL or GENERIC");
               errors++;
            }
            break;

         case LaunchMode :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "SIGNAL") )
               config.launch_mode = TMODE_SIGNAL;
            else if ( !strcmp(tokv[0], "MODULE") )
               config.launch_mode = TMODE_MODULE;
            else {
               store_error_message("LAUNCH_MODE must be SIGNAL or MODULE");
               errors++;
            }
            break;

         case LaunchSrc :
            value = 0;
            for ( j = 0; j < tokc; j++ ) {
               strncpy2(token, tokv[j], sizeof(token));
               strupper(token);
               if ( strcmp(token, "SNDS") == 0 ) {
                  sprintf(errbuffer, "LAUNCH_SOURCE '%s' not allowed as a default.", tokv[j]);
                  store_error_message(errbuffer);
                  errors++;
                  break;
               }
               for ( k = 0; k < nlsopt; k++ ) {
                  if ( strcmp(token, ls_option[k].label) == 0 ) { 
                     value |= ls_option[k].value;
                     break;
                  }
               }
               if ( k >= nlsopt ) {
                  sprintf(errbuffer, "Invalid LAUNCH_SOURCE option '%s'", tokv[j]);
                  store_error_message(errbuffer);
                  errors++;
                  break;
               }
            }
            if ( value & LSNONE ) {
               if ( value & ~LSNONE ) 
                  store_error_message("Warning: LAUNCH_SOURCE 'nosrc' cancels all others on line.");
               config.launch_source = 0;
            }
            else
               config.launch_source = (unsigned char)value;
            break;

         case ResOverlap :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "OLD") )
               config.res_overlap = RES_OVLAP_COMBINED;
            else if ( !strcmp(tokv[0], "NEW") )
               config.res_overlap = RES_OVLAP_SEPARATE;
            else {
               store_error_message("RESOLVE_OVERLAP must be OLD or NEW");
               errors++;
            }
            break;

         case DefaultModule :
            if ( (config.default_module = lookup_module_type(tokv[0])) < 0 ) {
               sprintf(errbuffer, "Module type '%s' is unknown.", tokv[0]);
               store_error_message(errbuffer);
               errors++;
            }
            break;
        
         case ScriptShell :
            if ( access(tokv[0], X_OK) == 0 ) {
              strncpy2(config.script_shell, tokv[0], sizeof(config.script_shell) - 1);
            }
            else {
              sprintf(errbuffer,
                 "An executable shell '%s' is not found.", tokv[0]);
              store_error_message(errbuffer);
              errors++;
            }
            break;
         
         case ScriptMode :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "HEYUHELPER") )
               config.script_mode = HEYU_HELPER;
            else if ( !strncmp(tokv[0], "SCRIPT", 6) )
               config.script_mode = HEYU_SCRIPT;
            else {
               store_error_message("SCRIPT_MODE must be HEYUHELPER or SCRIPT");
               errors++;
            }
            break;

         case LogDir :
            strncpy2(token, tokv[0], sizeof(token) - 1);
            strupper(token);
            if ( strcmp(token, "NONE") == 0 )
               *config.logfile = '\0';
            else {
	       strncpy2(token, tokv[0], sizeof(token) - 1);
               strcat(token, "/");
               if ( access(token, W_OK) != 0 ) {
                  store_error_message(
		    "LOG_DIR does not exist or is not writable.");
                  errors++;
	       }
	       else {     
               sprintf(config.logfile, "%s/%s",
		 tokv[0], LOGFILE);
	       }
	    }
            break;


         case IsDarkOffset :
            config.isdark_offset = (int)strtol(tokv[0], &sp, 10);

            if ( !strchr(" \t\n", *sp) ||
	         config.isdark_offset < 1 || config.isdark_offset > 360 ) {
               store_error_message("ISDARK_OFFSET outside range 1-360 minutes");
               errors++;
            }
            break;

	 case EnvAliasPrefix :
	    strupper(tokv[0]);
	    if ( strcmp(tokv[0], "UC") == 0 )
               strncpy2(config.env_alias_prefix, "X10", sizeof(config.env_alias_prefix) - 1);
	    else if ( strcmp(tokv[0], "LC") == 0 )
	       strncpy2(config.env_alias_prefix, "x10", sizeof(config.env_alias_prefix) - 1);
	    else {
	       store_error_message("ENV_ALIAS_PREFIX must be UC or LC");
	       errors++;
	    }
	    break;

	 case SunMode :
	    strupper(tokv[0]);
	    if ( strncmp(tokv[0], "RISESET", 1) == 0 )
	       config.sunmode = RiseSet;
	    else if ( strncmp(tokv[0], "CIVILTWI", 1) == 0 )
	       config.sunmode = CivilTwi;
	    else if ( strncmp(tokv[0], "NAUTTWI", 1) == 0 )
	       config.sunmode = NautTwi;
	    else if ( strncmp(tokv[0], "ASTROTWI", 1) == 0 )
	       config.sunmode = AstroTwi;
	    else {
	       store_error_message("DAWNDUSK_DEF must be R, C, N, or A");
	       errors++;
	    }
	    break;

         case Fix5A :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "YES") )
               config.fix_5a = YES;
            else if ( !strcmp(tokv[0], "NO") )
               config.fix_5a = NO;
            else {
               store_error_message("FIX_5A must be YES or NO");
               errors++;
            }
            break;

	 case CM11PostDelay :
	    config.cm11_post_delay = (int)strtol(tokv[0], &sp, 10);
	    if ( !strchr(" \t\n", *sp) ||
                 config.cm11_post_delay < 0 ||
                 config.cm11_post_delay > 1000 ) {
	       store_error_message("CM11_POST_DELAY must be 0-1000");
	       errors++;
	    }
	    break;
	    
         case AutoFetch :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "YES") )
               config.autofetch = YES;
            else if ( !strcmp(tokv[0], "NO") )
               config.autofetch = NO;
            else {
               store_error_message("AUTOFETCH must be YES or NO");
               errors++;
            }
            break;

         case PfailUpdate :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "YES") )
               config.pfail_update = YES;
            else if ( !strcmp(tokv[0], "NO") )
               config.pfail_update = NO;
            else {
               store_error_message("POWERFAIL_UPDATE must be YES or NO");
               errors++;
            }
            break;

	 case BitDelay :
	    config.cm17a_bit_delay = (int)strtol(tokv[0], &sp, 10);
	    if ( !strchr(" \t\n", *sp) ||
                 config.cm17a_bit_delay < 100 ||
                 config.cm17a_bit_delay > 10000 ) {
	       store_error_message("CM17A_BIT_DELAY must be 100-10000");
	       errors++;
	    }
	    break;

         case BurstSpacing :
            config.rf_burst_spacing = (int)strtol(tokv[0], &sp, 10);
	    if ( !strchr(" \t\n", *sp) ||
                 config.rf_burst_spacing < 80 ||
                 config.rf_burst_spacing > 160 ) {
	       store_error_message("RF_BURST_SPACING must be 80-160");
	       errors++;
	    }
	    break;

         case TimerTweak :
            config.rf_timer_tweak = (int)strtol(tokv[0], &sp, 10);
	    if ( !strchr(" \t\n", *sp) ||
                 config.rf_timer_tweak < 0 ||
                 config.rf_timer_tweak > 50 ) {
	       store_error_message("RF_TIMER_TWEAK must be 0-50");
	       errors++;
	    }
	    break;
            	    
	 case RFPostDelay :
	    config.rf_post_delay = (int)strtol(tokv[0], &sp, 10);
	    if ( !strchr(" \t\n", *sp) ||
                 config.rf_post_delay < 0 ||
                 config.rf_post_delay > 10000 ) {
	       store_error_message("RF_POST_DELAY must be 0-10000");
	       errors++;
	    }
	    break;
	    
	 case RFFarbDelay :
	    config.rf_farb_delay = (int)strtol(tokv[0], &sp, 10);
	    if ( !strchr(" \t\n", *sp) ||
                 config.rf_farb_delay < 0 ||
                 config.rf_farb_delay > 10000 ) {
	       store_error_message("RF_FARB_DELAY must be 0-10000");
	       errors++;
	    }
	    break;
	    
         case DispRFX :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "YES") )
               config.disp_rf_xmit = YES;
            else if ( !strcmp(tokv[0], "NO") )
               config.disp_rf_xmit = NO;
	    else if ( !strcmp(tokv[0], "VERBOSE") )
	       config.disp_rf_xmit = VERBOSE;
            else {
               store_error_message("DISPLAY_RF_XMIT must be NO or YES or VERBOSE");
               errors++;
            }
            break;

	 case DefRFBursts :
	    config.def_rf_bursts = (int)strtol(tokv[0], &sp, 10);
	    if ( !strchr(" \t\n", *sp) ||
                 config.def_rf_bursts < 5 ||
                 config.def_rf_bursts > 6 ) {
	       store_error_message("DEF_RF_BURSTS must be 5 or 6");
	       errors++;
	    }
	    break;
	    
	 case RFBursts :
	    bursts = strtol(tokv[1], &sp, 10);
	    if ( !strchr(" \t\n", *sp) || bursts < 1 ) {
	       store_error_message("Invalid RF_BURSTS");
	       errors++;
	       break;
	    }
	    (void) strlower(tokv[0]);
	    for ( j = 0; j < nrflabels; j++ ) {
	       if ( !strcmp(tokv[0], rf_label[j].label) ) { 
	          config.rf_bursts[rf_label[j].subcode] = bursts;
		  break;
	       }
	    }
	    if ( j >= nrflabels ) {
	       sprintf(errbuffer, "Unknown CM17A function '%s'", tokv[0]);
	       store_error_message(errbuffer);
	       errors++;
	    }
	    break;

	 case LoopCount :
	    config.timer_loopcount = (unsigned long)strtol(tokv[0], &sp, 10);
	    if ( !strchr(" \t\n", *sp) ) {
	       store_error_message("Invalid TIMER_LOOPCOUNT");
	       errors++;
	       break;
	    }
	    break;

	 case RestrictDims :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "YES") )
               config.restrict_dims = YES;
            else if ( !strcmp(tokv[0], "NO") )
               config.restrict_dims = NO;
            else {
               store_error_message("RESTRICT_DIMS must be YES or NO");
               errors++;
            }
            break;

	 case StateFmt :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "NEW") )
               config.state_format = NEW;
            else if ( !strcmp(tokv[0], "OLD") )
               config.state_format = OLD;
            else {
               store_error_message("STATE_FORMAT must be NEW or OLD");
               errors++;
            }
            break;

         case PfailScript :
            if ( !(config.pfail_script = strdup(bufp)) ) {
               store_error_message("Memory allocation error - out of memory");
               errors++;
            }
            break;
	    
	 case StartEngine :
            (void) strupper(tokv[0]);
            if ( !strncmp(tokv[0], "MANUAL", 3) )
               config.start_engine = MANUAL;
            else if ( !strncmp(tokv[0], "AUTOMATIC", 4) )
               config.start_engine = AUTOMATIC;
            else {
               store_error_message("START_ENGINE must be MANUAL or AUTO");
               errors++;
            }
            break;

	 case IgnoreSilent :
            break;

	 case NoSwitch :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "YES") )
               config.rf_noswitch = YES;
            else if ( !strcmp(tokv[0], "NO") )
               config.rf_noswitch = NO;
            else {
               store_error_message("RF_NOSWITCH must be YES or NO");
               errors++;
            }
            break;

         case RespoolPerms :
#ifndef RESPOOL
            store_error_message(
              "The RESPOOL_PERMISSIONS directive is invalid for this operating system");
            errors++;
            break;
#endif
            perms = (int)strtol(tokv[0], &sp, 8);
            if ( !strchr(" \t\n", *sp) || perms < 0 || perms > 07777) {
               store_error_message("RESPOOL_PERMISSIONS - invalid octal number");
               errors++;
            }
            config.respool_perms = (unsigned int)perms;
            break;

	 case SpoolMax :
	    config.spool_max = strtol(tokv[0], &sp, 10);
	    if ( !strchr(" \t\n", *sp) ||
                 config.spool_max < SPOOLFILE_ABSMIN ||
                 config.spool_max > SPOOLFILE_ABSMAX ) {
               sprintf(errbuffer, "SPOOLFILE_MAX must be between %ul and %ul",
                 SPOOLFILE_ABSMIN,  SPOOLFILE_ABSMAX);
	       store_error_message(errbuffer);
	       errors++;
	    }
	    break;
	    
	 case CheckRILine :
            (void) strupper(tokv[0]);
            if ( !strcmp(tokv[0], "YES") )
               config.check_RI_line = YES;
            else if ( !strcmp(tokv[0], "NO") )
               config.check_RI_line = NO;
            else {
               store_error_message("CHECK_RI_LINE must be YES or NO");
               errors++;
            }
            break;

         case SendRetries :
	    config.send_retries = (int)strtol(tokv[0], &sp, 10);
	    if ( !strchr(" \t\n", *sp) ||
                 config.send_retries < 0 ) {
	       store_error_message("SEND_RETRIES must be > 0");
	       errors++;
	    }
	    break;

      }
      free( tokv );

      return errors;
}

/*---------------------------------------------------------------------+
 | Get certain configuration items from environment.                   |
 +---------------------------------------------------------------------*/
int environment_config ( void )
{
   int  j, retcode, errors = 0;
   char *sp;
   char buffer[32];

   static char *envars[] = {
     "LATITUDE",
     "LONGITUDE",
     "ASIF_DATE",
     "ASIF_TIME",
   };

   reset_isparsed_flags();

   for ( j = 0; j < (int)(sizeof(envars)/sizeof(char *)); j++ ) {
      if ( (sp = getenv(envars[j])) != NULL ) {
         sprintf(buffer, "%s %s", envars[j], sp);
         retcode = parse_config_tail(buffer, SRC_ENVIRON);
         errors += retcode;
         if ( retcode != 0 || *error_message() != '\0' ) {
            fprintf(stderr, "Environment variable %s.\n", error_message());
            clear_error_message();
         }
      }
   }
   return errors;
}


/*---------------------------------------------------------------------+
 | Display configuration file directives that have been overridden,    |
 | either by an environment variable or by a 'config' command in the   |
 | schedule file, as indicated by the isparsed flag.                   |
 +---------------------------------------------------------------------*/
void display_config_overrides ( FILE *fd )
{
   int  j, count = 0;
   char delim = ' ';

   fprintf(fd, "Configuration overrides:");

   for ( j = 0; j < ncommands; j++ ) {
      if ( command[j].isparsed & (SRC_ENVIRON | SRC_SCHED)) {
         count++;
         fprintf(fd, "%c %s", delim, command[j].name);
         delim = ',';
      }
   }
   if ( !count )
      fprintf(fd, " -- None --");

   fprintf(fd, "\n\n");

   return;
}

/*---------------------------------------------------------------------+
 | Open the user's X10 configuration file and read only the minimal    |
 | directives for specific commands, like 'heyu stop'.                 |
 +---------------------------------------------------------------------*/
void read_minimal_config ( unsigned int source )
{

   FILE        *fd ;
   int         error_count;
   char        confp[PATH_LEN + 1];
   extern char heyu_config[PATH_LEN + 1];
   extern int  verbose;

   find_heyu_path();

   strncpy2(confp, pathspec(heyu_config), sizeof(confp) - 1);

   if ( verbose ) 
      (void) fprintf(stdout, "Opening Heyu configuration file '%s'\n", confp); 
 
   if ( !(fd = fopen(confp, "r")) ) {
      (void)fprintf(stderr, 
              "Unable to find (or open) Heyu configuration file '%s'\n", confp);
      return;
   }

   error_count = parse_minimal_config( fd, source );

   if ( error_count != 0 ) {
      (void)fprintf(stderr, 
              "Quitting due to errors in configuration file '%s'\n", confp);
      exit(1);
   }
 
   (void) fclose( fd );

   return;
}


/*---------------------------------------------------------------------+
 | Return the index in the array of SCRIPT structures for the SCRIPT   |
 | having the argument 'label', or -1 if not found.                    |
 +---------------------------------------------------------------------*/
int lookup_script ( SCRIPT *scriptp, char *label )
{
   int    j = 0;

   while ( scriptp && scriptp[j].line_no > 0 ) {
      if ( strcmp(label, scriptp[j].label) == 0 )
         return j;
      j++;
   }
   return -1;
}

/*---------------------------------------------------------------------+
 | Return the index in the array of LAUNCHER structures for the first  |
 | LAUNCHER having the argument 'label', or -1 if not found.           |
 +---------------------------------------------------------------------*/
int lookup_launcher ( LAUNCHER *launcherp, char *label )
{
   int   j = 0;

   while ( launcherp && launcherp[j].line_no > 0 ) {
      if ( strcmp(label, launcherp[j].label) == 0 )
         return j;
      j++;
   }
   return -1;
}


/*---------------------------------------------------------------------+
 | Resolve interrelated configuration items.                           |
 +---------------------------------------------------------------------*/
int finalize_config ( void )
{
   int  finalize_launchers(void);
   int  create_file_paths(void);  

   char errmsg[80];
   char *sp;
   int  j;

   if ( config.mode == COMPATIBLE ) {
      config.program_days = 366;
   }
   else {
      config.program_days = config.program_days_in;
   }

 
   if ( config.min_dusk != OFF && config.max_dusk != OFF && 
        config.min_dusk >= config.max_dusk ) {
      store_error_message("MIN_DUSK must be less than MAX_DUSK");
      return 1;
   }

   if ( config.min_dawn != OFF && config.max_dawn != OFF && 
        config.min_dawn >= config.max_dawn ) {
      store_error_message("MIN_DAWN must be less than MAX_DAWN");
      return 1;
   }

   /* Create suffix for lock files, e.g., ".ttyS0" */

   if ( (sp = strrchr(config.tty, '/')) != NULL ) {
      strncpy2(config.suffix, sp, sizeof(config.suffix) - 1);
      *(config.suffix) = '.';
   }
   else {	   
      strncpy2(config.suffix, ".", sizeof(config.suffix) - 1);
      strncat(config.suffix, config.tty, sizeof(config.suffix) - 1 - strlen(config.suffix));
   }

   if ( config.auxdev ) {
      if ( (sp = strrchr(config.ttyaux, '/')) != NULL ) {
         strncpy2(config.suffixaux, sp, sizeof(config.suffixaux) - 1);
         *(config.suffixaux) = '.';
      }
      else {	   
         strncpy2(config.suffixaux, ".", sizeof(config.suffixaux) - 1);
         strncat(config.suffixaux, config.ttyaux, sizeof(config.suffixaux) - 1 - strlen(config.suffixaux));
      }
   }

   if ( *config.logfile == '\0' ) {
      strncpy2(config.logfile, DEF_LOGFILE, sizeof(config.logfile) - 1);
   }
   else {
      strncat(config.logfile, config.suffix, sizeof(config.logfile) - 1 - strlen(config.logfile));
   }

   create_file_paths();  

   if ( access(config.logfile, F_OK) == 0 &&
        access(config.logfile, W_OK) != 0 ) {
      sprintf(errmsg, "Log file '%s' is not writable - check permissions.",
	 config.logfile);
      store_error_message(errmsg);
      return 1;
   }      

   if ( access(statefile, F_OK) == 0 &&
        access(statefile, W_OK) != 0 ) {
      sprintf(errmsg, "State file '%s' is not writable - check permissions.",
	 statefile);
      store_error_message(errmsg);
      return 1;
   }

   for ( j = 0; j < nrflabels; j++ ) {
      if ( config.rf_bursts[j] > config.def_rf_bursts ) {
         sprintf(errmsg, "RF_BURSTS exceeds %d\n", config.def_rf_bursts);
	 store_error_message(errmsg);
	 return 1;
      }
   }
   
   if ( finalize_launchers() > 0 )
      return 1;

   return 0;
}


/*---------------------------------------------------------------------+
 | Add an alias to the array of ALIAS structures and return the index  |
 | of the new member.                                                  |
 +---------------------------------------------------------------------*/
int add_alias ( ALIAS **aliaspp, char *label, int line_no, 
                              char housecode, char *units, int modtype )
{
   static int    size, max_size;
   static int    strucsize = sizeof(ALIAS);
   int           j, maxlevel;
   int           blksize = 10;
   char          hc;
   unsigned int  bmap;
   unsigned long flags;
   char          errmsg[128];

   clear_error_message();

   /* Allocate initial block of memory */
   if ( *aliaspp == NULL ) {
      *aliaspp = (ALIAS *) calloc(blksize, strucsize );
      if ( *aliaspp == NULL ) {
         (void) fprintf(stderr, "Unable to allocate memory for Alias.\n");
         exit(1);
      }
      max_size = blksize;
      size = 0;
      /* Initialize it where necessary */
      for ( j = 0; j < max_size; j++ ) {
         (*aliaspp)[j].label[0] = '\0';
         (*aliaspp)[j].modtype = -1;
         (*aliaspp)[j].line_no = 0;
         (*aliaspp)[j].flags = 0;
         (*aliaspp)[j].optflags = 0;
      }
   }

   /* Check for a valid label length */
   if ( (int)strlen(label) > NAME_LEN ) {
      sprintf(errmsg, 
         "Alias label '%s' too long - maximum %d characters", label, NAME_LEN);
      store_error_message(errmsg);
      return -1;
   }

   /* See if the alias label is already in the list.     */
   /* If so, it's an error.                              */
   if ( (j = get_alias(*aliaspp, label, &hc, &bmap)) >= 0 ) {
      (void) sprintf(errmsg, "Duplicate alias label '%s'", label);
      store_error_message(errmsg);
      return -1;
   }

   /* Verify that the label is not 'macro' */
   if ( strcmp(label, "macro") == 0 ) {
      store_error_message("An alias may not have the label 'macro'\n");
      return -1;
   }

   /* Check the housecode */
   if ( housecode == '_' ) {
      (void) sprintf(errmsg, 
         "Alias '%s': Default housecode symbol ('_') is invalid in an alias",
              label);
      store_error_message(errmsg);
      return -1;
   }
   if ( (hc = toupper(housecode)) < 'A' || hc > 'P' ) {
      (void) sprintf(errmsg, "Alias '%s': Housecode '%c' outside range A-P",
               label, hc);
      store_error_message(errmsg);
      return -1;
   }

   /* Check the units list */
   if ( parse_units( units, &bmap ) != 0 ) {
      (void) sprintf(errmsg, "Alias '%s': ", label);
      add_error_prefix(errmsg);
      return -1;
   }

   /* Check to see that the module type (if any) specified for */
   /* this housecode|unit address doesn't conflict with that   */
   /* in a previously defined alias.                           */
   
   if ( modtype >= 0 ) {
      j = 0;
      while ( (*aliaspp)[j].line_no > 0 ) {
         if ( (*aliaspp)[j].housecode == hc && (*aliaspp)[j].unitbmap & bmap &&
              (*aliaspp)[j].modtype >= 0 && (*aliaspp)[j].modtype != modtype ) {
            sprintf(errmsg,
               "Module type conflicts with that defined for %c%s on Line %d",
                    hc, bmap2units((*aliaspp)[j].unitbmap & bmap),
                       (*aliaspp)[j].line_no);
            store_error_message(errmsg);
            return -1;
         }
         j++;
      }
   }

   /* Check to see if there's an available location          */
   /* If not, increase the size of the memory allocation.    */
   /* (Always leave room for a final termination indicator.) */
   if ( size == (max_size - 1)) {
      max_size += blksize ;
      *aliaspp = (ALIAS *) realloc(*aliaspp, max_size * strucsize );
      if ( *aliaspp == NULL ) {
         (void) fprintf(stderr, "Unable to increase size of Alias list.\n");
         exit(1);
      }

      /* Initialize the new memory allocation */
      for ( j = size; j < max_size; j++ ) {
         (*aliaspp)[j].label[0] = '\0';
         (*aliaspp)[j].modtype = -1;
         (*aliaspp)[j].line_no = 0;
         (*aliaspp)[j].flags = 0;
         (*aliaspp)[j].optflags = 0;
      }
   }

   j = size;
   size += 1;

   (void) strncpy2((*aliaspp)[j].label, label, NAME_LEN);
   (*aliaspp)[j].line_no = line_no;
   (*aliaspp)[j].housecode = toupper(housecode);
   (*aliaspp)[j].unitbmap = bmap;
   (*aliaspp)[j].modtype = modtype;
   module_attributes(modtype, &flags, &maxlevel);
   (*aliaspp)[j].flags = flags;
   (*aliaspp)[j].maxlevel = maxlevel;
   (*aliaspp)[j].onlevel = maxlevel;

   return j;
}

/*---------------------------------------------------------------------+
 | Add a scene or usersyn to the array of SCENE structures and return  |
 | the index of the new member.                                        |
 | Argument 'type' may be 1 for a scene or 2 for a usersyn             |
 +---------------------------------------------------------------------*/
int add_scene ( SCENE **scenepp, char *label, 
                 int line_no, char *body, unsigned int type )
{
   static int    size, max_size ;
   static int    strucsize = sizeof(SCENE);
   int           j;
   char          *sp;
   int           cmdc, nparms;
   char          **cmdv;
   int           blksize = 10;
   char          errmsg[128];
   extern char   *typename[];

   /* Allocate initial block of memory */
   if ( *scenepp == NULL ) {
      *scenepp = calloc(blksize, strucsize );
      if ( *scenepp == NULL ) {
         fprintf(stderr, "Unable to allocate memory for Scene/Usersyn.\n");
         exit(1);
      }
      max_size = blksize;
      size = 0;
      /* Initialize it */
      for ( j = 0; j < max_size; j++ ) {
         (*scenepp)[j].label[0] = '\0';
         (*scenepp)[j].line_no = -1;
         (*scenepp)[j].nparms = 0;
         (*scenepp)[j].type = 0;
         (*scenepp)[j].body = NULL;
      }
   }

   /* Check for a valid scene label */
   if ( (int)strlen(label) > SCENE_LEN ) {
      sprintf(errmsg, 
         "%s label too long - maximum %d characters", typename[type], SCENE_LEN);
      store_error_message(errmsg);
      return -1;
   }
   if ( strchr("+_-", *label) != NULL ) {
      sprintf(errmsg, "%s label may not may not begin with '+', '-' or '_'", typename[type]);
      store_error_message(errmsg);
      return -1;
   }
   if ( strchr(label, '$') != NULL ) {
      sprintf(errmsg, "%s label may not contain the '$' character", typename[type]);
      store_error_message(errmsg);
      return -1;
   }
   if ( strchr(label, ';') != NULL ) {
      sprintf(errmsg, "%s label may not contain the ';' character", typename[type]);
      store_error_message(errmsg);
      return -1;
   }

   /* See if the scene label is already in the list.     */
   /* If so, it's an error.                              */
   if ( (j = lookup_scene(*scenepp, label)) >= 0 ) {
      sprintf(errmsg, 
         "%s label '%s' previously defined as a %s on line %d",
          typename[type], label, typename[(*scenepp)[j].type], (*scenepp)[j].line_no);
      store_error_message(errmsg);
      return -1;
   }
   if ( is_admin_cmd(label) || is_direct_cmd(label) ) {
      sprintf(errmsg, "%s label '%s' conflicts with heyu command", typename[type], label);
      store_error_message(errmsg);
      return -1;
   }

   /* Check to see if there's an available location          */
   /* If not, increase the size of the memory allocation.    */
   /* (Always leave room for a final termination indicator.) */
   if ( size == (max_size - 1)) {
      max_size += blksize ;
      *scenepp = realloc(*scenepp, max_size * strucsize );
      if ( *scenepp == NULL ) {
         fprintf(stderr, "Unable to increase size of Scene/Usersyn list.\n");
         exit(1);
      }

      /* Initialize the new memory allocation */
      for ( j = size; j < max_size; j++ ) {
         (*scenepp)[j].label[0] = '\0';
         (*scenepp)[j].line_no = -1;
         (*scenepp)[j].nparms = 0;
         (*scenepp)[j].type = 0;
         (*scenepp)[j].body = NULL;
      }
   }

   j = size;
   size += 1;

   /* Determine the number of replaceable parameters */
   sp = strdup(body);
   tokenize(sp, " \t;", &cmdc, &cmdv);
   nparms = max_parms(cmdc, cmdv);
   free(sp);
   free(cmdv);
   if ( *error_message() != '\0' ) {
      sprintf(errmsg, "%s '%s': ", typename[type], label);
      add_error_prefix(errmsg);
   }
   if ( nparms < 0 )
      return -1;

   sp = strdup(body);
   if ( sp == NULL ) {
      fprintf(stderr,
        "Unable to allocate memory for body of Scene/Usersyn '%s'.\n", label);
         exit(1);
   }
   strtrim(sp);
  
   (void) strncpy2((*scenepp)[j].label, label, SCENE_LEN);
   (*scenepp)[j].line_no = line_no;
   (*scenepp)[j].nparms = nparms;
   (*scenepp)[j].type = type;
   (*scenepp)[j].body = sp;

   return j;
}

/*---------------------------------------------------------------------+
 | Parse Latitude string [NS+-]ddd:mm and store in struct config.      |
 | line_no <= 0 indicates string from environment, otherwise line in   |
 | configuration file.                                                 | 
 +---------------------------------------------------------------------*/
int parse_latitude ( char *string )
{
   char tmpbuff[128];
   char *sp1, *sp2;
   int  sign;

   if ( string == NULL )
      return 0;

   (void)strncpy2(tmpbuff, string, sizeof(tmpbuff) - 1);
   sp1 = tmpbuff;

   /* For compatibility with Heyu 1 */
   if ( isdigit((int)(*sp1)) ) {
      (void)strcpy(tmpbuff, "N");
      (void)strncpy2(tmpbuff + 1, string, sizeof(tmpbuff) - 2);
   }
   else if ( *sp1 == '+' ) {
      *sp1 = 'N';
   }
   else if ( *sp1 == '-' ) {
      *sp1 = 'S';
   }

   switch ( toupper(*sp1) ) {
      case 'N' :
         sign = 1;
         break;
      case 'S' :
         sign = -1;
         break;
      default :
         sign = -2;   
         break;
   }
   sp1++;

   if ( sign < -1 || !(sp2 = strchr(sp1, ':')) ) {
         store_error_message("LATITUDE invalid - must be [NS+-]dd:mm");
         config.loc_flag &= ~(LATITUDE) ;
         return 1;
   }

   config.lat_m = (int)strtol(sp2 + 1, NULL, 10);
   *sp2 = '\0';
   config.lat_d = (int)strtol(sp1, NULL, 10);
   if ( config.lat_d == 0 )
      config.lat_m *= sign ;
   else
      config.lat_d *= sign ;

   config.latitude = ( config.lat_d < 0 ) ?
         (double)config.lat_d - (double)config.lat_m/60. :
               (double)config.lat_d + (double)config.lat_m/60.;

   if ( config.latitude < -89. || config.latitude > 89. ) {
         store_error_message("LATITUDE outside range -89 to +89 degrees");
         config.loc_flag &= ~(LATITUDE);
         return 1;
   }

   config.loc_flag |= LATITUDE ;

   return 0;
}

/*---------------------------------------------------------------------+
 | Parse Longitude string [EW+-]ddd:mm and store in struct config.     |
 | line_no <= 0 indicates string from environment, otherwise line in   |
 | configuration file.                                                 | 
 +---------------------------------------------------------------------*/
int parse_longitude ( char *string )
{
   char tmpbuff[128];
   char *sp1, *sp2;
   int  sign;

   if ( string == NULL )
      return 0;

   (void)strncpy2(tmpbuff, string, sizeof(tmpbuff) - 1);
   sp1 = tmpbuff;

   /* For compatibility with Heyu 1, where positive longitude assumed West */
   if ( isdigit((int)(*sp1)) ) {
      (void)strcpy(tmpbuff, "W");
      (void)strncpy2(tmpbuff + 1, string, sizeof(tmpbuff) - 2);
   }
   else if ( *sp1 == '+' ) {
      *sp1 = 'W';
   }
   else if ( *sp1 == '-' ) {
      *sp1 = 'E';
   }

   switch ( toupper(*sp1) ) {
      case 'E' :
         sign = 1;
         break;
      case 'W' :
         sign = -1;
         break;
      default :
         sign = -2;     
         break;
   }
   sp1++;

   if ( sign < -1 || !(sp2 = strchr(sp1, ':')) ) {
         store_error_message("LONGITUDE invalid - must be [EW+-]dd:mm");
         config.loc_flag &= ~(LONGITUDE) ;
         return 1;
   }

   config.lon_m = (int)strtol(sp2 + 1, NULL, 10);
   *sp2 = '\0';
   config.lon_d = (int)strtol(sp1, NULL, 10);
   if ( config.lon_d == 0 )
      config.lon_m *= sign ;
   else
      config.lon_d *= sign ;

   config.longitude = ( config.lon_d < 0 ) ?
      (double)config.lon_d - (double)config.lon_m/60. :
               (double)config.lon_d + (double)config.lon_m/60.;

   if ( config.longitude < -180. || config.longitude > 180. ) {
         store_error_message("LONGITUDE outside range -180 to +180 degrees");
         config.loc_flag &= ~(LONGITUDE);
         return 1;
   }

   config.loc_flag |= LONGITUDE ;

   return 0;
}

/*---------------------------------------------------------------------+
 | Check executability of a file on user's PATH                        |
 +---------------------------------------------------------------------*/
int is_executable ( char *pathname )
{
   char pathbuffer[1024];
   char buffer[1024];
   char *sp;
   int  j, tokc;
   char **tokv;

   if ( access(pathname, X_OK) == 0 )
      return 1;

   if ( (sp = getenv("PATH")) == NULL ) 
      return 0;

   strncpy2(pathbuffer, sp, sizeof(pathbuffer) - 1);

   tokenize(buffer, ":", &tokc, &tokv);

   for ( j = 0; j < tokc; j++ ) {
      strncpy2(buffer, tokv[j], sizeof(buffer) - 1);
      strncat(buffer, "/", sizeof(buffer) - 1 - strlen(buffer));
      strncat(buffer, pathname, sizeof(buffer) - 1 - strlen(buffer));
      if ( access(buffer, X_OK) == 0 ) {
         free(tokv);
         return 1;
      }
   }
   free(tokv);
   return 0;
}

/*---------------------------------------------------------------------+
 | Free memory allocated for SCRIPT structure and contents.            |
 +---------------------------------------------------------------------*/
void free_scripts ( SCRIPT **scriptpp )
{
   int j = 0;

   if ( *scriptpp == NULL )
      return;

   while ( (*scriptpp)[j].line_no > 0 ) {
     if ( (*scriptpp)[j].cmdline ) {
        free((*scriptpp)[j].cmdline);
     }
     j++;
   }
   free(*scriptpp);
   *scriptpp = NULL;

   return;
}  

/*---------------------------------------------------------------------+
 |  Free the array of SCENEs and the scene bodies therein.             |
 +---------------------------------------------------------------------*/
void free_scenes ( SCENE **scenepp )
{
   int j = 0;

   if ( *scenepp == NULL )
      return;

   while ( (*scenepp)[j].line_no > 0 ) {
     if ( (*scenepp)[j].body != NULL ) {
        free((*scenepp)[j].body);
     }
     j++;
   }

   free((*scenepp));
   *scenepp = NULL;
   return;
}

/*---------------------------------------------------------------------+
 |  Free the array of ALIASES.                                         |
 +---------------------------------------------------------------------*/
void free_aliases ( ALIAS **aliaspp )
{
   if ( *aliaspp == NULL )
      return;

   free((*aliaspp));
   *aliaspp = NULL;
   return;
}

/*---------------------------------------------------------------------+
 |  Free the array of LAUNCHERS.                                       |
 +---------------------------------------------------------------------*/
void free_launchers ( LAUNCHER **launcherpp )
{
   if ( *launcherpp == NULL )
      return;

   free((*launcherpp));
   *launcherpp = NULL;
   return;
}

/*---------------------------------------------------------------------+
 |  Free arrays of ALIASes, SCENEs, SCRIPTs, and LAUNCHERs             |
 +---------------------------------------------------------------------*/
void free_all_arrays ( void )
{        
   free_aliases(&config.aliasp);
   free_scenes(&config.scenep);
   free_scripts(&config.scriptp);
   free_launchers(&config.launcherp);

   return;
}

/*---------------------------------------------------------------------+
 |  Create the file pathspecs for many files.                          |
 +---------------------------------------------------------------------*/
int create_file_paths ( void )
{
    sprintf(statefile, "%s", pathspec(STATE_FILE));
    sprintf(pfailfile, "%s/%s%s", SPOOLDIR, "heyu.pfail", config.suffix);
    sprintf(enginelockfile, "%s/LCK..%s%s", LOCKDIR, STATE_LOCKFILE, config.suffix);
    
#if 0 /* future */
    sprintf(spoolfile, "%s/%s%s", SPOOLDIR, SPOOLFILE, config.suffix);
    sprintf(relaylockfile, "%s/LCK..%s%s", LOCKDIR, RELAYFILE, config.suffix);
    sprintf(writelockfile, "%s/LCK..%s%s", LOCKDIR, WRITEFILE, config.suffix);
    sprintf(ttylockfile, "%s/LCK.%s", LOCKDIR, config.suffix);
#endif

    return 0;
}

   

   

   

