//////////////////////////////////////////////////////////////////////////////
//    Copyright 2004-2014, SenseGraphics AB
//
//    This file is part of H3D API.
//
//    H3D API is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    H3D API is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with H3D API; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//    A commercial license is also available. Please contact us at 
//    www.sensegraphics.com for more information.
//
//
/// \file PhysX3Callbacks.cpp
/// \brief Source file for PhysX3Callbacks, callback functions for
/// the PhysX3 physics engine.
///
//
//////////////////////////////////////////////////////////////////////////////

#include <H3D/H3DPhysics/PhysX3Callbacks.h>

#ifdef HAVE_PHYSX3

#include <H3D/H3DPhysics/PhysX3Joints.h>
#include <H3D/H3DPhysics/PhysX3RigidBodyOptions.h>
#include <H3D/H3DPhysics/PhysX3CollidableOptions.h>

#include <H3D/Box.h>
#include <H3D/Cone.h>
#include <H3D/Sphere.h>
#include <H3D/Cylinder.h>
#include <H3D/IndexedTriangleSet.h>
#include <H3D/Coordinate.h>

#ifdef HAVE_HACD
#include <hacdHACD.h>
#endif

using namespace H3D;

PhysicsEngineThread::PhysicsEngineRegistration
PhysX3Callbacks::registration( "PhysX3", 
                              PhysicsEngineThread::createPhysicsEngineCallbacks< PhysX3Callbacks >() );

PhysX3RigidBody::PhysX3RigidBody ( PhysicsEngineParameters::RigidBodyParameters& _params ) :
  mass ( 0 ), 
  massDensityModel ( NULL ), 
  fixed ( false ), 
  kinematic ( false ),
  autoDisable ( false ),
  disableLinearSpeed ( 0 ),
  disableAngularSpeed ( 0 ) {
  PhysX3Callbacks::PhysX3SpecificData *physx_data = 
    static_cast< PhysX3Callbacks::PhysX3SpecificData * >(_params.getEngine()->getEngineSpecificData());

  rigid_body= physx_data->sdk->createRigidDynamic(PxTransform::createIdentity());

  rigid_body->userData= this;

  /// \todo Add option to support engines own damping if it supports it 
  ///       which could be more stable
  //rigid_body->setAngularDamping ( 50 );
  //rigid_body->setLinearDamping ( 10 );
  setParameters ( _params );
}

PhysX3RigidBody::~PhysX3RigidBody () {
  // Remove geometry
  for ( ShapeVector::const_iterator i= shapes.begin(); i != shapes.end(); ++i ) {
    PhysX3CollidableShape* shape= *i;
    shape->removeFromBody();
  }

  rigid_body->release();
  rigid_body= NULL;
}

void PhysX3RigidBody::setParameters ( PhysicsEngineParameters::RigidBodyParameters& _params ) {
  // Auto disable
  if ( _params.haveAutoDisable() ) {
    autoDisable= _params.getAutoDisable();
  }

  // Auto disable linear speed
  if ( _params.haveDisableLinearSpeed() ) {
    disableLinearSpeed= _params.getDisableLinearSpeed();
  }

  // Auto disable angular speed
  if ( _params.haveDisableAngularSpeed() ) {
    disableAngularSpeed= _params.getDisableAngularSpeed();
  }

  if ( _params.haveAutoDisable() || 
       _params.haveDisableLinearSpeed() ||
       _params.haveDisableAngularSpeed() ) {
    setSleeping ( _params );
  }

  // useGlobalGravity
  if ( _params.haveUseGlobalGravity() ) {
    rigid_body->setActorFlag ( PxActorFlag::eDISABLE_GRAVITY, !_params.getUseGlobalGravity() );
  }

  // start position
  if ( _params.haveStartPosition() ) {
    rigid_body->setGlobalPose ( 
      PxTransform(toPxVec3(_params.getStartPosition()),rigid_body->getGlobalPose().q) );
  }
  // start orientation
  if ( _params.haveStartOrientation() ) {
    rigid_body->setGlobalPose ( 
      PxTransform(rigid_body->getGlobalPose().p,toPxQuat(_params.getStartOrientation())) );
  }

  // Enable / disable kinematic control
  if ( _params.haveKinematicControl() ) {
    kinematic= _params.getKinematicControl();
  }

  // Fixed
  if ( _params.haveFixed() ) {
    fixed= _params.getFixed();
  }

  // Make body kinematic if fixed or kinematic is enabled
  if ( _params.haveKinematicControl() || _params.haveFixed() ) {
    rigid_body->setRigidDynamicFlag ( PxRigidDynamicFlag::eKINEMATIC, kinematic || fixed );
  }

  // Kinematic control
  if ( kinematic ) {
    // Position
    if ( _params.havePosition() ) {
      rigid_body->setGlobalPose ( PxTransform ( toPxVec3 ( _params.getPosition() ), rigid_body->getGlobalPose().q ) );
    }

    // Orientation
    if ( _params.haveOrientation() ) {
      rigid_body->setGlobalPose ( PxTransform ( rigid_body->getGlobalPose().p, toPxQuat ( _params.getOrientation() ) ) );
    }
  }

  // geometry
  if( _params.haveGeometry() ) {

    // Remove old geometry
    for ( ShapeVector::const_iterator i= shapes.begin(); i != shapes.end(); ++i ) {
      PhysX3CollidableShape* shape= *i;
      shape->removeFromBody();
    }

    // Add new geometry
    const vector< H3DCollidableId > geom_ids = _params.getGeometry();
    for ( vector< H3DCollidableId >::const_iterator i= geom_ids.begin(); i != geom_ids.end(); ++i ) {
      PhysX3CollidableShape* shape= (PhysX3CollidableShape*)*i;

      PhysX3Callbacks::PhysX3SpecificData *physx_data = 
        static_cast< PhysX3Callbacks::PhysX3SpecificData * >(_params.getEngine()->getEngineSpecificData());

      shape->addToBody ( *this, *physx_data );
      shapes.push_back ( shape );
    }
  }

  // Mass
  if ( _params.haveMass() ) {
    mass= _params.getMass();
    rigid_body->setMass ( mass );
  }

  // Mass density model
  if ( _params.haveMassDensityModel() ) {
    massDensityModel= _params.getMassDensityModel();
  }

  // Center of mass
  if ( _params.haveCenterOfMass() ) {
    centerOfMass= _params.getCenterOfMass();
  }

  // Inertia
  if ( _params.haveInertia() ) {
    inertia= _params.getInertia();
  }

  // mass, massDensityModel, inertia and centerOfMass
  /// \todo Thread safety if mass density set after initial creation
  if( _params.haveMass() || _params.haveMassDensityModel() || 
      _params.haveCenterOfMass() || _params.haveInertia() ) {
    // Reference: http://en.wikipedia.org/wiki/List_of_moment_of_inertia_tensors
    Node *n = massDensityModel;
    if ( Box *box = dynamic_cast< Box* >( n ) ) {
      // Box moment of inertia
      Vec3f s= box->size->getValue();
      rigid_body->setMassSpaceInertiaTensor ( 
        PxVec3 ( (mass*(s.y*s.y + s.z*s.z))/12.0f,
                 (mass*(s.x*s.x + s.z*s.z))/12.0f,
                 (mass*(s.x*s.x + s.y*s.y))/12.0f ) );
    } else if ( Sphere *sph = dynamic_cast< Sphere* >( n ) ) {
      // Sphere moment of inertia
      H3DFloat r= sph->radius->getValue();
      H3DFloat m= (2*mass*r*r)/5.0f;
      rigid_body->setMassSpaceInertiaTensor ( PxVec3 ( m, m, m ) );
    } else if ( Cylinder *cyl = dynamic_cast< Cylinder* >( n ) ) {
      // Cylinder moment of inertia
      H3DFloat r= cyl->radius->getValue();
      H3DFloat h= cyl->height->getValue();
      H3DFloat m= (2*mass*r*r)/5.0f;
      rigid_body->setMassSpaceInertiaTensor ( 
        PxVec3 ( (m*(3*r*r+h*h))/12.0f,
                 (m*(3*r*r+h*h))/12.0f,
                 (m*r*r)/2.0f ) );
    } else {
      if ( n ) {
        Console(4) << "Warning: Mass density model " << n->getTypeName() << " is not supported by PhysX3!" 
                   << " Inertia will be calculated based on collision geometry instead. " << endl;

        // No mass density model supplied, calculate it automatically based on the
        // current collision shapes
        PxRigidBodyExt::setMassAndUpdateInertia  ( *rigid_body, mass );
      } else {
        // Set user defined center of mass and inertia
        rigid_body->setCMassLocalPose ( PxTransform ( toPxVec3 ( centerOfMass ) ) );
        Matrix3f m= inertia;
        rigid_body->setMassSpaceInertiaTensor ( PxVec3 ( m[0][0], m[1][1], m[2][2] ) );
      }
    }
  }

  // PhysX3 specific rigid body options
  if ( _params.haveEngineOptions() ) {
    if ( PhysX3RigidBodyParameters* options= 
      dynamic_cast<PhysX3RigidBodyParameters*>(_params.getEngineOptions()) ) {

      // contactReportThreshold
      if ( options->haveContactReportThreshold() ) {
        rigid_body->setContactReportThreshold ( options->getContactReportThreshold() );
      }

      // solverPositionIterations
      if ( options->haveSolverPositionIterations() ) {
        PxU32 p, v;
        rigid_body->getSolverIterationCounts ( p, v );
        rigid_body->setSolverIterationCounts ( options->getSolverPositionIterations(), v );
      }

      // solverVelocityIterations
      if ( options->haveSolverVelocityIterations() ) {
        PxU32 p, v;
        rigid_body->getSolverIterationCounts ( p, v );
        rigid_body->setSolverIterationCounts ( p, options->getSolverVelocityIterations() );
      }

    }
  }
}

void PhysX3RigidBody::getParameters ( PhysicsEngineParameters::RigidBodyParameters& _params ) {
  // position
  _params.setPosition( toVec3f ( rigid_body->getGlobalPose().p ) );
  // orientation  
  _params.setOrientation( toRotation ( rigid_body->getGlobalPose().q ) );
  // linear velocity
  _params.setLinearVelocity( toVec3f ( rigid_body->getLinearVelocity() ) );
  // angular velocity
  _params.setAngularVelocity( toVec3f ( rigid_body->getAngularVelocity() ) );
}

void PhysX3RigidBody::setSleeping ( PhysicsEngineParameters::RigidBodyParameters& _params ) {
  if ( _params.getAutoDisable() ) {
    H3DFloat v_l= disableLinearSpeed;
    H3DFloat v_r= disableAngularSpeed;
    H3DFloat massNormalizedK= (v_l*v_l+v_r*v_r)/2;

    // Actors whose kinetic energy divided by their mass is above this threshold will not be put to sleep.
    rigid_body->setSleepThreshold( massNormalizedK );
  } else {
    WorldParameters worldParams= _params.getEngine()->getWorldParameters();

    if ( worldParams.getAutoDisable() ) {
      H3DFloat v_l= worldParams.getDisableLinearSpeed();
      H3DFloat v_r= worldParams.getDisableAngularSpeed();
      H3DFloat massNormalizedK= (v_l*v_l+v_r*v_r)/2;

      // Actors whose kinetic energy divided by their mass is above this threshold will not be put to sleep.
      rigid_body->setSleepThreshold( massNormalizedK );
    } else {
      rigid_body->setSleepThreshold( 0 );
    }
  }
}

struct lt_PxVec3 {
  bool operator()(const PxVec3& v1, const PxVec3& v2) const {
    return (v1.x<v2.x) || (v1.x==v2.x && v1.y<v2.y) || (v1.x==v2.x && v1.y==v2.y && v1.z<v2.z);
  }
};

#ifdef HAVE_HACD
// Helper function to debug HACD by loading input directly from file
bool LoadOFF(const std::string & fileName, std::vector< HACD::Vec3<HACD::Real> > & points, std::vector< HACD::Vec3<long> > & triangles, bool invert) 
{    
  FILE * fid = fopen(fileName.c_str(), "r");
  if (fid) 
    {
    const std::string strOFF("OFF");
    char temp[1024];
    fscanf(fid, "%s", temp);
    if (std::string(temp) != strOFF)
    {
      printf( "Loading error: format not recognized \n");
            fclose(fid);

      return false;            
    }
    else
    {
      int nv = 0;
      int nf = 0;
      int ne = 0;
      fscanf(fid, "%i", &nv);
      fscanf(fid, "%i", &nf);
      fscanf(fid, "%i", &ne);
            points.resize(nv);
      triangles.resize(nf);
            HACD::Vec3<HACD::Real> coord;
      float x = 0;
      float y = 0;
      float z = 0;
      for (long p = 0; p < nv ; ++p) 
            {
        fscanf(fid, "%f", &x);
        fscanf(fid, "%f", &y);
        fscanf(fid, "%f", &z);
        points[p].X() = x;
        points[p].Y() = y;
        points[p].Z() = z;
      }        
      int i = 0;
      int j = 0;
      int k = 0;
      int s = 0;
      for (long t = 0; t < nf ; ++t) {
        fscanf(fid, "%i", &s);
        if (s == 3)
        {
          fscanf(fid, "%i", &i);
          fscanf(fid, "%i", &j);
          fscanf(fid, "%i", &k);
          triangles[t].X() = i;
          if (invert)
          {
            triangles[t].Y() = k;
            triangles[t].Z() = j;
          }
          else
          {
            triangles[t].Y() = j;
            triangles[t].Z() = k;
          }
        }
        else      // Fix me: support only triangular meshes
        {
          for(long h = 0; h < s; ++h) fscanf(fid, "%i", &s);
        }
      }
            fclose(fid);
    }
  }
  else 
    {
    printf( "Loading error: file not found \n");
    return false;
    }
  return true;
}

// Helper function to debug HACD by saving result to file
bool SaveOFF(const std::string & fileName, size_t nV, size_t nT, const HACD::Vec3<HACD::Real> * const points, const HACD::Vec3<long> * const triangles)
{
  std::cout << "Saving " <<  fileName << std::endl;
  std::ofstream fout(fileName.c_str());
  if (fout.is_open()) 
  {           
    fout <<"OFF" << std::endl;        
    fout << nV << " " << nT << " " << 0<< std::endl;    
    for(size_t v = 0; v < nV; ++v)
    {
      fout << points[v].X() << " " 
        << points[v].Y() << " " 
        << points[v].Z() << std::endl;
    }
    for(size_t f = 0; f < nT; ++f)
    {
      fout <<"3 " << triangles[f].X() << " " 
        << triangles[f].Y() << " "                                                  
        << triangles[f].Z() << std::endl;
    }
    fout.close();
    return true;
  }
  return false;
}
#endif // HAVE_HACD

PhysX3CollidableShape::PhysX3CollidableShape ( PhysicsEngineParameters::ShapeParameters& _params ) :
  body ( NULL ), collisionEnabled ( true ), exceptionGroupMask(0) {
  PhysX3Callbacks::PhysX3SpecificData *physx_data = 
    static_cast< PhysX3Callbacks::PhysX3SpecificData * >(_params.getEngine()->getEngineSpecificData());

  if ( Sphere* sphere= dynamic_cast<Sphere*>(_params.getShape()) ) {
    geometry.push_back ( new PxSphereGeometry ( sphere->radius->getValue() ) );
  } else if ( Box* box= dynamic_cast<Box*>(_params.getShape()) ) {
    Vec3f size= box->size->getValue();
    geometry.push_back (  new PxBoxGeometry ( size.x/2, size.y/2, size.z/2 ) );
  } else if ( Cylinder* cylinder= dynamic_cast<Cylinder*>(_params.getShape()) ) {
    geometry.push_back ( new PxCapsuleGeometry ( cylinder->radius->getValue(), cylinder->height->getValue()/2 ) );
    rotationOffset= Rotation(0,0,1,(H3DFloat)H3DUtil::Constants::pi/2);
  } else {
    // get triangles from geometry
    vector< HAPI::Collision::Triangle > triangles;
    _params.getShape()->boundTree->getValue()->getAllTriangles( triangles );

    PhysX3CollidableParameters options;
    if ( _params.haveEngineOptions() ) {
      if ( PhysX3CollidableParameters* p= 
        dynamic_cast<PhysX3CollidableParameters*>(_params.getEngineOptions()) ) {
        options= *p;
      }
    }

    // Create appropriate collision shape
    if ( options.getConvex() ) {
      // Mesh is already convex

      // Load the pre-cooked mesh if it exists
      string filename= options.getCookedFilename();
      bool load_cooked= filename != "";
      PxConvexMesh* convexMesh= NULL;
      if ( load_cooked ) {
        auto_ptr< PxDefaultFileInputData > input( new PxDefaultFileInputData( filename.c_str() ) );
        if ( !input->isValid() && options.getBaseURL() != "" ) {
          string filename_tmp = options.getBaseURL() + filename;
          input.reset( new PxDefaultFileInputData( filename_tmp.c_str() ) );
        }
        if( input->isValid() ) {
          convexMesh= physx_data->sdk->createConvexMesh(*input.get());
          if ( convexMesh ) {
            geometry.push_back ( new PxConvexMeshGeometry ( convexMesh ) );
          }
        }
      }

      if ( !convexMesh ) {
        // Remove duplicates
        set<PxVec3,lt_PxVec3> points;
        for( unsigned int i = 0; i < triangles.size(); ++i ) {
          points.insert( toPxVec3 ( Vec3f(triangles[i].a) ) );
          points.insert( toPxVec3 ( Vec3f(triangles[i].b) ) );
          points.insert( toPxVec3 ( Vec3f(triangles[i].c) ) );
        }

        // Copy unique points
        PxVec3* verts= new PxVec3 [points.size()];
        size_t j= 0;
        for ( set<PxVec3,lt_PxVec3>::iterator i= points.begin(); i != points.end(); ++i ) {
          verts[j++]= *i;
        }

        // Create mesh description
        PxConvexMeshDesc convexDesc;
        convexDesc.points.count     = points.size();
        convexDesc.points.stride    = sizeof(PxVec3);
        convexDesc.points.data      = verts;
        convexDesc.flags            = PxConvexFlag::eCOMPUTE_CONVEX;

        // Cook the mesh
        if ( load_cooked ) {
          bool success= false;
          {
            auto_ptr< PxDefaultFileOutputStream > buf( new PxDefaultFileOutputStream( filename.c_str() ) );
            if( !buf->isValid() && options.getBaseURL() != "" ) {
              filename = options.getBaseURL() + filename;
              buf.reset( new PxDefaultFileOutputStream( filename.c_str() ) );
            }
            if( buf->isValid() )
              success= physx_data->cooking->cookConvexMesh(convexDesc, *buf.get());
          }
          if ( !success ) {
            Console(4) << "Error: PhysX3 error cooking convex mesh!" << endl;
          } else {
            // Load the cooked mesh
            PxDefaultFileInputData input( filename.c_str() );
            if( input.isValid() ) {
              convexMesh= physx_data->sdk->createConvexMesh(input);
              geometry.push_back ( new PxConvexMeshGeometry ( convexMesh ) );
            } else
              Console(4) << "Error: PhysX3 error creating convex mesh!" << endl;
          }
        } else {
          PxDefaultMemoryOutputStream buf;
          if(!physx_data->cooking->cookConvexMesh(convexDesc, buf)) {
            Console(4) << "Error: PhysX3 error cooking convex mesh!" << endl;
          } else {
            // Load the cooked mesh
            PxDefaultMemoryInputData input(buf.getData(), buf.getSize());
            convexMesh= physx_data->sdk->createConvexMesh(input);
            geometry.push_back ( new PxConvexMeshGeometry ( convexMesh ) );
          }
        }

        delete [] verts;

      }
    } else {
#ifdef HAVE_HACD
      // If we have HACD and the convex decomposition is requested then
      // attempt to decompose the mesh into multiple convex hulls
      if ( options.getConvexDecomposition() ) {

        string filename= options.getCookedFilename();
        bool load_cooked= filename != "";

        size_t c;
        if ( load_cooked ) {
          for ( c= 0; true; ++c ) {
            stringstream ss;
            ss << filename << c;

            auto_ptr< PxDefaultFileInputData > input( new PxDefaultFileInputData( ss.str().c_str() ) );
            if( !input->isValid() && options.getBaseURL() != "" ) {
              stringstream ss1;
              ss1 << options.getBaseURL() << filename << c;
              input.reset( new PxDefaultFileInputData( ss1.str().c_str() ) );
            }
            PxConvexMesh* mesh= NULL;
            if ( input->isValid () ) {
              mesh= physx_data->sdk->createConvexMesh(*input.get());
            }
            if ( mesh ) {
              geometry.push_back ( new PxConvexMeshGeometry ( mesh ) );
            } else {
              break;
            }
          }
        }

        if ( !load_cooked || c == 0 ) {
          // This can take a long time, ouput message to console
          // Use PhysX3CollidableOptions::cookedFilename to cache the result
          if ( !load_cooked ) {
            Console(4) << "Warning: Consider caching convex decomposition using PhysX3CollidableOptions::cookedFilename" << endl;
          }

          Console(4) << "Note: Doing convex decomposition..." << endl;

          std::vector< HACD::Vec3<HACD::Real> > points;
          std::vector< HACD::Vec3<long> > tris;

          IndexedTriangleSet* triangle_set= dynamic_cast<IndexedTriangleSet*>(_params.getShape());
          Coordinate* coord= NULL;
          if ( triangle_set ) {
            coord= dynamic_cast<Coordinate*>(triangle_set->coord->getValue());
          }
          if ( coord ) {
            const std::vector<Vec3f>& tri_points= coord->point->getValue();
            const std::vector<H3DInt32>& tri_index= triangle_set->index->getValue();

            for ( std::vector<Vec3f>::const_iterator i= tri_points.begin(); i != tri_points.end(); ++i ) {
              const Vec3f& v= *i;
              points.push_back( HACD::Vec3<HACD::Real> ( v.x, v.y, v.z ) );
            }

            for ( std::vector<H3DInt32>::const_iterator i= tri_index.begin(); i != tri_index.end(); i+= 3 ) {
              tris.push_back( HACD::Vec3<long> ( *i, *(i+1), *(i+2) ) );
            }
          } else {
            size_t j= 0;
            for ( size_t i= 0; i < triangles.size(); ++i ) {
              points.push_back( HACD::Vec3<HACD::Real> ( triangles[i].a.x, triangles[i].a.y, triangles[i].a.z ) );
              points.push_back( HACD::Vec3<HACD::Real> ( triangles[i].b.x, triangles[i].b.y, triangles[i].b.z ) );
              points.push_back( HACD::Vec3<HACD::Real> ( triangles[i].c.x, triangles[i].c.y, triangles[i].c.z ) );
              tris.push_back ( j++ );
              tris.push_back ( j++ );
              tris.push_back ( j++ );
            }
          }

          HACD::HACD * const myHACD = HACD::CreateHACD();
          myHACD->SetPoints(&points[0]);
          myHACD->SetNPoints(points.size());
          myHACD->SetTriangles(&tris[0]);
          myHACD->SetNTriangles(tris.size());
        
          myHACD->SetCompacityWeight(options.getCompacityWeight());
          myHACD->SetVolumeWeight(options.getVolumeWeight());
          myHACD->SetScaleFactor (options.getScaleFactor());
          myHACD->SetNClusters(options.getNrClusters());
          myHACD->SetNVerticesPerCH(options.getNrVerticesPerCH());
          myHACD->SetConcavity(options.getConcavity());
          myHACD->SetAddExtraDistPoints(options.getAddExtraDistPoints());   
          myHACD->SetNTargetTrianglesDecimatedMesh(options.getNrTargetTrianglesDecimatedMesh());   
          myHACD->SetAddFacesPoints(options.getAddFacesPoints()); 
          myHACD->SetConnectDist(options.getConnectDist());
          myHACD->SetSmallClusterThreshold(options.getSmallClusterThreshold());

          myHACD->Compute();

          if ( options.getSaveConvexDecomposition() != "" ) {
            string convex_decomposition_file = options.getSaveConvexDecomposition();
            bool saved_composition =  myHACD->Save(convex_decomposition_file.c_str(), false);
            if( !saved_composition && options.getBaseURL() != "" ) {
              convex_decomposition_file = options.getBaseURL() + convex_decomposition_file;
              saved_composition =  myHACD->Save(convex_decomposition_file.c_str(), false);
            }

            if( saved_composition )
              Console(4) << "Debugging: Saved convex decomposition to " << convex_decomposition_file << endl;
            else 
              Console(4) << "Debugging: Error saving convex decomposition to " << options.getSaveConvexDecomposition() << endl;

            const HACD::Vec3<HACD::Real> * const decimatedPoints = myHACD->GetDecimatedPoints();
            const HACD::Vec3<long> * const decimatedTriangles    = myHACD->GetDecimatedTriangles();
            if (decimatedPoints && decimatedTriangles) {
              if ( SaveOFF( convex_decomposition_file + "-decimated.off", 
                            myHACD->GetNDecimatedPoints(), 
                            myHACD->GetNDecimatedTriangles(), decimatedPoints, decimatedTriangles) ) {
                Console(4) << "Debugging: Saved decimated mesh to " << convex_decomposition_file << "-decimated.off" << endl;
              } else {
                Console(4) << "Debugging: Error saving decimated mesh to " << options.getSaveConvexDecomposition() << "-decimated.off" << endl;
              }
            }
          }

          for ( size_t c= 0; c < myHACD->GetNClusters(); ++c ) {

            size_t nPoints = myHACD->GetNPointsCH(c);
            size_t nTriangles = myHACD->GetNTrianglesCH(c);
            HACD::Vec3<HACD::Real> * pointsCH = new HACD::Vec3<HACD::Real>[nPoints];
            HACD::Vec3<long> * trianglesCH = new HACD::Vec3<long>[nTriangles];
            myHACD->GetCH(c, pointsCH, trianglesCH);

            // Copy points
            PxVec3* verts= new PxVec3 [nPoints];
            for ( size_t i= 0; i < nPoints; ++i ) {
              verts[i]= PxVec3 ( (PxReal)pointsCH[i].X(), (PxReal)pointsCH[i].Y(), (PxReal)pointsCH[i].Z() );
            }

            // Create mesh description
            PxConvexMeshDesc convexDesc;
            convexDesc.points.count     = nPoints;
            convexDesc.points.stride    = sizeof(PxVec3);
            convexDesc.points.data      = verts;
            convexDesc.flags            = PxConvexFlag::eCOMPUTE_CONVEX;

            // Cook the mesh
            stringstream ss;
            ss << options.getCookedFilename() << c;
            string filename=  ss.str().c_str();

            if ( load_cooked ) {
              bool success= false;
              {
                // Allocate using new in order to
                auto_ptr< PxDefaultFileOutputStream > buf( new PxDefaultFileOutputStream( filename.c_str() ) );
                if( !buf->isValid() && options.getBaseURL() != "" ) {
                  filename = options.getBaseURL() + filename;
                  buf.reset( new PxDefaultFileOutputStream( filename.c_str() ) );
                }
                if( buf->isValid() )
                  success= physx_data->cooking->cookConvexMesh(convexDesc, *buf.get());
              }
              if ( !success ) {
                Console(4) << "Error: PhysX3 error cooking convex mesh!" << endl;
              } else {
                // Load the cooked mesh
                PxDefaultFileInputData input( filename.c_str() );
                PxConvexMesh* convexMesh= physx_data->sdk->createConvexMesh(input);
                if( convexMesh )
                  geometry.push_back ( new PxConvexMeshGeometry ( convexMesh ) );
                else
                  Console(4) << "Error: PhysX3 error creating convex mesh!" << endl;
              }
            } else {
              PxDefaultMemoryOutputStream buf;
              if(!physx_data->cooking->cookConvexMesh(convexDesc, buf)) {
                Console(4) << "Error: error cooking convex mesh!" << endl;
              } else {
                // Load the cooked mesh
                PxDefaultMemoryInputData input(buf.getData(), buf.getSize());
                PxConvexMesh* convexMesh= physx_data->sdk->createConvexMesh(input);
                geometry.push_back ( new PxConvexMeshGeometry ( convexMesh ) );
              }
            }

            delete [] verts;
          }
          Console(4) << "Done." << endl;
        }
      } else {
#endif
        if ( options.getConvexDecomposition() ) {
          Console(4) << "Warning: H3DPhysics was not compiled with convex decomposition capabilities (requires HACD)!" << endl;
        }

        string filename= options.getCookedFilename();
        bool load_cooked= filename != "";
        PxTriangleMesh* triMesh= NULL;
        if ( load_cooked ) {
          auto_ptr< PxDefaultFileInputData > input( new PxDefaultFileInputData( filename.c_str() ) );
          if( !input->isValid() && options.getBaseURL() != "" ) {
            filename = options.getBaseURL() + filename;
            input.reset( new PxDefaultFileInputData( filename.c_str() ) );
          }
          if ( input->isValid() ) {
            triMesh= physx_data->sdk->createTriangleMesh(*input.get());
            if ( triMesh ) {
              geometry.push_back ( new PxTriangleMeshGeometry ( triMesh ) );
            }
          }
        }

        if ( !triMesh ) {
          PxVec3* verts= new PxVec3 [ triangles.size()*3 ];
          PxU32* index= new PxU32 [ triangles.size()*3 ];
          for ( size_t i= 0; i < triangles.size(); ++i ) {
            verts[i*3]= toPxVec3 ( Vec3f(triangles[i].a) );
            index[i*3]= i*3;
            verts[i*3+1]= toPxVec3 ( Vec3f(triangles[i].b) );
            index[i*3+1]= i*3+1;
            verts[i*3+2]= toPxVec3 ( Vec3f(triangles[i].c) );
            index[i*3+2]= i*3+2;
          }

          PxTriangleMeshDesc meshDesc;
          meshDesc.points.count           = triangles.size()*3;
          meshDesc.points.stride          = sizeof(PxVec3);
          meshDesc.points.data            = verts;

          meshDesc.triangles.count        = triangles.size();
          meshDesc.triangles.stride       = 3*sizeof(PxU32);
          meshDesc.triangles.data         = index;

          // Cook the mesh
          if ( load_cooked ) {
            bool success= false;
            {
              PxDefaultFileOutputStream buf ( filename.c_str() );
              if( buf.isValid() )
                success= physx_data->cooking->cookTriangleMesh(meshDesc, buf);
            }
            if(!success) {
              Console(4) << "Error: PhysX3 error cooking convex mesh!" << endl;
            } else {
              // Load the cooked mesh
              PxDefaultFileInputData input( filename.c_str() );
              if( input.isValid() ) {
                triMesh= physx_data->sdk->createTriangleMesh(input);
                if( triMesh )
                  geometry.push_back ( new PxTriangleMeshGeometry ( triMesh ) );
              }
            }
          } else {
            PxDefaultMemoryOutputStream buf;
            if(!physx_data->cooking->cookTriangleMesh(meshDesc, buf)) {
              Console(4) << "Error: PhysX3 error cooking convex mesh!" << endl;
            } else {
              // Load the cooked mesh
              PxDefaultMemoryInputData input(buf.getData(), buf.getSize());
              triMesh= physx_data->sdk->createTriangleMesh(input);
              geometry.push_back ( new PxTriangleMeshGeometry ( triMesh ) );
            }
          }
          delete [] verts;
          delete [] index;
        }
#ifdef HAVE_HACD
      }
#endif
    }
  }
  
  setParameters ( _params );
}

PhysX3CollidableShape::~PhysX3CollidableShape () {
  removeFromBody();
  for( vector<PxGeometry*>::iterator i = geometry.begin(); i != geometry.end(); ++i )
    delete (*i);
  geometry.clear();
  for( vector<PxShape*>::iterator i = shape.begin(); i != shape.end(); ++i )
    (*i)->release();
  shape.clear();
}

void PhysX3CollidableShape::setParameters ( PhysicsEngineParameters::ShapeParameters& _params ) {
  PhysX3Callbacks::PhysX3SpecificData *physx_data = 
    static_cast< PhysX3Callbacks::PhysX3SpecificData * >(_params.getEngine()->getEngineSpecificData());

  if ( _params.haveTranslation() ) {
    translation= _params.getTranslation();
  }

  if ( _params.haveRotation() ) {
    rotation= _params.getRotation();
  }

  if ( _params.haveTranslation() || _params.haveRotation() ) {
    for ( vector<PxShape*>::iterator i= shape.begin(); i != shape.end(); ++i ) {
      (*i)->setLocalPose ( PxTransform ( toPxVec3 ( translation ), toPxQuat ( rotation*rotationOffset ) ) );
    }
  }

  if ( _params.haveEnabled() ) {
    collisionEnabled = _params.getEnabled();
  }

  if ( _params.haveCollidableExceptionGroupList() ) {
    exceptionGroupMask = 0;
    CollidableParameters::CollidableExceptionGroupList gList = _params.getCollidableExceptionGroupList();
    for( unsigned int i=0; i<gList.size(); i++ ){
      unsigned int groupid = 1<<gList[i];
      exceptionGroupMask |= groupid;
    }
  }

  if( _params.haveClipPlanes() ) {
    clipPlanes_mutex.lock();
    clipPlanes = _params.getClipPlanes();
    clipPlanes_mutex.unlock();
  }
  
  if( _params.haveEnabled() || _params.haveCollidableExceptionGroupList() || _params.haveClipPlanes() ){
    updateFilterData(physx_data);
  }

}

void PhysX3CollidableShape::getParameters ( PhysicsEngineParameters::ShapeParameters& _params ) {
}


void PhysX3CollidableShape::addToBody ( PhysX3RigidBody& _body, PhysX3Callbacks::PhysX3SpecificData& physx_data ) {
  if ( !body ) {
    body= &_body;
    addToBodyInternal ( _body, physx_data );
  }
}

void PhysX3CollidableShape::removeFromBody () {
  if ( body ) {
    removeFromBodyInternal();
    body= NULL;
  }
}

void PhysX3CollidableShape::addToBodyInternal ( PhysX3RigidBody& _body, PhysX3Callbacks::PhysX3SpecificData& physx_data ) {
  for ( vector<PxGeometry*>::iterator i= geometry.begin(); i != geometry.end(); ++i ) {
    PxShape* s= body->getActor()->createShape ( **i, *physx_data.default_material, 
      PxTransform ( PxTransform ( toPxVec3 ( translation ), toPxQuat ( rotation*rotationOffset ) ) ) );
    s->userData= this;
    shape.push_back ( s );
  }
  updateFilterData(&physx_data);
}

void PhysX3CollidableShape::removeFromBodyInternal () {
  for ( vector<PxShape*>::iterator i= shape.begin(); i != shape.end(); ++i ) {
    (*i)->release();
  }
  shape.clear();
}

void PhysX3CollidableShape::updateFilterData( PhysX3Callbacks::PhysX3SpecificData * physx_data ){

  PxFilterData d;
  // Enabled or diabled.
  d.word0= collisionEnabled;
  // Clipped or not.
  d.word2 = false;
  // Mask for exception groups which should be ignored.
  d.word3 = exceptionGroupMask;

  if( clipPlanes.size() > 0 )
    d.word2= true;

  for ( vector<PxShape*>::iterator i= shape.begin(); i != shape.end(); ++i ) {
    (*i)->setSimulationFilterData ( d );
#if PX_PHYSICS_VERSION_MAJOR < 4 && PX_PHYSICS_VERSION_MINOR < 3
    (*i)->resetFiltering();
#endif
  }
  //Console (4) << "In UpdateFilterData: "<< d.word0 <<"  "<< d.word3 <<endl;
#if PX_PHYSICS_VERSION_MAJOR >= 3 && PX_PHYSICS_VERSION_MINOR >= 3
  if( body && body->getActor() )
    physx_data->scene->resetFiltering(body->getActor() );
#endif
}

void PhysX3Callbacks::SimulationEventCallback::onContact (const PxContactPairHeader &pairHeader, const PxContactPair *pairs, PxU32 nbPairs) {
  for(PxU32 i=0; i < nbPairs; ++i) {
    const PxContactPair& cp = pairs[i];

    if( cp.events & PxPairFlag::eNOTIFY_TOUCH_PERSISTS ) {
      PxContactPairPoint *userBuffer= new PxContactPairPoint[cp.contactCount];
      cp.extractContacts (userBuffer, cp.contactCount);

      for ( size_t c= 0; c < cp.contactCount; ++c ) {
        PxContactPairPoint ct= userBuffer[c];
        if ( ct.separation < 0 ) {
          PhysicsEngineParameters::ContactParameters contact;

          contact.body1_id= (H3DBodyId)pairHeader.actors[0]->userData;
          contact.body2_id= (H3DBodyId)pairHeader.actors[1]->userData;

          contact.geom1_id= (H3DCollidableId)cp.shapes[0]->userData;
          contact.geom2_id= (H3DCollidableId)cp.shapes[1]->userData;

          contact.contact_normal= toVec3f(ct.normal);
          contact.position= toVec3f(ct.position);
          contact.depth= -ct.separation;

          contacts.push_back ( contact );
        }
      }

      delete [] userBuffer;
    }
  }
}

void PhysX3Callbacks::SimulationEventCallback::clearContacts () {
  contacts.clear();
}

void PhysX3Callbacks::SimulationEventCallback::getContacts ( list<PhysicsEngineParameters::ContactParameters>& _contacts ) {
  _contacts.insert ( _contacts.begin(), contacts.begin(), contacts.end() );
}

void PhysX3Callbacks::ContactModifyCallback::onContactModify(PxContactModifyPair* const pairs, PxU32 count){
  for(PxU32 i=0; i<count; i++){
    PxContactModifyPair& cmp = pairs[i];

    // Add all the clip planes from both shapes
    std::vector<Vec4f> planes;

    PhysX3CollidableShape* shape= (PhysX3CollidableShape*)cmp.shape[0]->userData;
    shape->clipPlanes_mutex.lock();
    planes.insert( planes.end(), shape->clipPlanes.begin(), shape->clipPlanes.end());
    shape->clipPlanes_mutex.unlock();

    shape= (PhysX3CollidableShape*)cmp.shape[1]->userData;
    shape->clipPlanes_mutex.lock();
    planes.insert( planes.end(), shape->clipPlanes.begin(), shape->clipPlanes.end());
    shape->clipPlanes_mutex.unlock();

    for(PxU32 n=0; n<cmp.contacts.size(); n++){

      for( unsigned int p=0; p<planes.size(); p++ ){
        if (cmp.contacts.getPoint(n).dot(toPxVec3 ( Vec3f( planes[p].x, planes[p].y, planes[p].z ) ) ) + planes[p].w < 0.0 ){
          cmp.contacts.ignore(n);
          break;
        }
      }
    }
  }
}

static PxDefaultErrorCallback gDefaultErrorCallback;
static PxDefaultAllocator gDefaultAllocatorCallback;
static PxSimulationFilterShader gDefaultFilterShader=PxDefaultSimulationFilterShader;

// A 'shader' that is executed by the collision detection system to
// determine how to handle contacts
PxFilterFlags physX3FilterShader (
  PxFilterObjectAttributes attributes0, PxFilterData filterData0, 
  PxFilterObjectAttributes attributes1, PxFilterData filterData1,
  PxPairFlags& pairFlags, const void* constantBlock, PxU32 constantBlockSize) {
  // let triggers through
  if(PxFilterObjectIsTrigger(attributes0) || PxFilterObjectIsTrigger(attributes1))
  {
    pairFlags = PxPairFlag::eTRIGGER_DEFAULT;
    return PxFilterFlag::eDEFAULT;
  }

  if( filterData0.word3 & filterData1.word3 )
    return PxFilterFlag::eKILL;

  // generate contacts for all that were not filtered above
  pairFlags = PxPairFlag::eCONTACT_DEFAULT;

  if( filterData0.word2 || filterData1.word2 ) {
    pairFlags |= PxPairFlag::eMODIFY_CONTACTS;
  }

  // Are the collision shapes both enabled
  if( filterData0.word0 && filterData1.word0 ) {
    pairFlags |= PxPairFlag::eNOTIFY_CONTACT_POINTS;
    pairFlags |= PxPairFlag::eNOTIFY_TOUCH_PERSISTS;
  } else {
    return PxFilterFlag::eSUPPRESS;
  }

  return PxFilterFlag::eDEFAULT;
}

PeriodicThread::CallbackCode PhysX3Callbacks::initEngine( void *data ) {
  PhysicsEngineThread *pt = static_cast< PhysicsEngineThread * >( data );
  PhysX3Callbacks::PhysX3SpecificData *physx_data = 
    new PhysX3Callbacks::PhysX3SpecificData;
  pt->setEngineSpecificData( physx_data );

  // Initialise the SDK
  physx_data->foundation= PxCreateFoundation(PX_PHYSICS_VERSION, gDefaultAllocatorCallback, gDefaultErrorCallback);
  if ( !physx_data->foundation ) { 
    Console(4) << "Error: Unable to initialize the PhysX3 foundation" << endl; 
    return PeriodicThread::CALLBACK_DONE;
  }

  physx_data->sdk = PxCreatePhysics(PX_PHYSICS_VERSION, *physx_data->foundation, PxTolerancesScale());
  if ( !physx_data->sdk ) { 
    Console(4) << "Error: Unable to initialize the PhysX3 SDK" << endl; 
    return PeriodicThread::CALLBACK_DONE;
  }

  // Initialise cooking
  physx_data->cooking = PxCreateCooking(PX_PHYSICS_VERSION, *physx_data->foundation, PxCookingParams());
  if ( !physx_data->cooking ) {
    Console(4) << "Error: Unable to initialize the Cooking PhysX3 SDK" << endl; 
    return PeriodicThread::CALLBACK_DONE;
  }

  if (!PxInitExtensions(*physx_data->sdk)) {
    Console(4) << "Error: Unable to initialize PhysX3 extensions" << endl; 
    return PeriodicThread::CALLBACK_DONE;
  }

  // Create scene
  PxSceneDesc sceneDesc (physx_data->sdk->getTolerancesScale());
  if(!sceneDesc.cpuDispatcher) {
    physx_data->cpu_dispatcher= PxDefaultCpuDispatcherCreate(1);
    sceneDesc.cpuDispatcher= physx_data->cpu_dispatcher;
    if(!sceneDesc.cpuDispatcher) {
      Console(4) << "PxDefaultCpuDispatcherCreate failed!" <<endl;
      return PeriodicThread::CALLBACK_DONE;
    }
  } 
  if(!sceneDesc.filterShader)
   sceneDesc.filterShader= gDefaultFilterShader;

  physx_data->simulation_events= new SimulationEventCallback;
  sceneDesc.simulationEventCallback= physx_data->simulation_events;

  physx_data->contact_modifier= new ContactModifyCallback;
  sceneDesc.contactModifyCallback = physx_data->contact_modifier;

  sceneDesc.filterShader= physX3FilterShader;

  physx_data->scene = physx_data->sdk->createScene( sceneDesc );

  GlobalContactParameters p= pt->getGlobalContactParameters();
  physx_data->default_material= physx_data->sdk->createMaterial ( 
    p.friction_coefficients.x, p.friction_coefficients.y, p.bounce );

  return PeriodicThread::CALLBACK_DONE;
}

PeriodicThread::CallbackCode PhysX3Callbacks::deInitEngine( void *data ) {
  PhysicsEngineThread *pt = static_cast< PhysicsEngineThread *>( data );
  PhysX3SpecificData *physx_data = static_cast< PhysX3SpecificData * >( pt->getEngineSpecificData() );
  if ( physx_data->sdk != NULL ){
    physx_data->default_material->release();
    physx_data->default_material = NULL;

    physx_data->scene->release();
    physx_data->scene = NULL;

    delete physx_data->simulation_events;
    physx_data->simulation_events= NULL;

    delete physx_data->contact_modifier;
    physx_data->contact_modifier= NULL;

    physx_data->cpu_dispatcher->release();
    physx_data->cpu_dispatcher= NULL;

    PxCloseExtensions();

    physx_data->cooking->release();
    physx_data->cooking = NULL;

    physx_data->sdk->release();
    physx_data->sdk = NULL;

    physx_data->foundation->release();
    physx_data->foundation = NULL;
  }
  return PeriodicThread::CALLBACK_DONE;
}

PeriodicThread::CallbackCode PhysX3Callbacks::doSimulationSteps(void *data) {
  PhysicsEngineThread * physics_thread = 
    static_cast< PhysicsEngineThread * >( data );
  PhysX3SpecificData *physx_data = 
    static_cast< PhysX3SpecificData * >(physics_thread->getEngineSpecificData());
  if( !physx_data->sdk )
    return PeriodicThread::CALLBACK_CONTINUE;

  TimeStamp t;

  // Default rigid body material properties
  GlobalContactParameters p= physics_thread->getGlobalContactParameters();
  physx_data->default_material->setStaticFriction ( p.friction_coefficients.x );
  physx_data->default_material->setDynamicFriction ( p.friction_coefficients.y );
  physx_data->default_material->setRestitution ( p.bounce );

  physx_data->simulation_events->clearContacts ();

  // Step the simulation
  H3DTime dt= TimeStamp()-physx_data->last_update_time;
  physx_data->last_update_time= TimeStamp();
  H3DTime max_dt= physics_thread->getStepSize();

  dt= std::min ( dt, 2*max_dt );
  while ( dt >= max_dt ) {
    physx_data->scene->simulate ( PxReal(max_dt) );
    physx_data->scene->fetchResults ( true );
    dt-= max_dt;
  }
  physx_data->last_update_time -= dt;
  physics_thread->setLastLoopTime( TimeStamp() - t );
  
  return PeriodicThread::CALLBACK_CONTINUE;
}

PeriodicThread::CallbackCode PhysX3Callbacks::setWorldParameters( void *data ) {
  PhysicsEngineParameters::WorldParameters *params = 
    static_cast< PhysicsEngineParameters::WorldParameters *>( data );
  PhysX3SpecificData *physx_data = static_cast< PhysX3SpecificData * >( 
    params->getEngine()->getEngineSpecificData() );
  if( !physx_data->sdk )
    return PeriodicThread::CALLBACK_DONE;

  // gravity
  if( params->haveGravity() ) {
    physx_data->scene->setGravity( toPxVec3 ( params->getGravity() ) ); 
  }

  // disabling bodies
  if( params->haveAutoDisable() || 
      params->haveDisableLinearSpeed() || 
      params->haveDisableAngularSpeed() ){
    list< H3DBodyId > bodies;
    params->getEngine()->getCurrentRigidBodies( bodies );
    for( list< H3DBodyId >::iterator i = bodies.begin(); 
         i != bodies.end(); ++i ) {
      PhysicsEngineParameters::RigidBodyParameters p;
      params->getEngine()->getRigidBodyParameters( *i, p );
      ((PhysX3RigidBody*)(*i))->setSleeping ( p );
    }
  }

  /// \todo contactSurfaceThickness, disableTime, errorCorrection, massCorrectionSpeed

  delete params;
  return PeriodicThread::CALLBACK_DONE;
}

H3DUtil::PeriodicThread::CallbackCode PhysX3Callbacks::addRigidBody( void *data ) {
  RigidBodyParameters *params = 
    static_cast< RigidBodyParameters *>( data );
  PhysX3SpecificData *physx_data = 
    static_cast< PhysX3SpecificData * >(params->getEngine()->getEngineSpecificData());

  PhysX3RigidBody* body= new PhysX3RigidBody ( *params );
  physx_data->scene->addActor ( *body->getActor() );
  params->setBodyId ( (H3DBodyId)body );

  return PeriodicThread::CALLBACK_DONE;
}

PeriodicThread::CallbackCode PhysX3Callbacks::removeRigidBody( void *data ) {
  RigidBodyParameters *params = static_cast< RigidBodyParameters * >( data );
  PhysX3Callbacks::PhysX3SpecificData *physx_data = 
    static_cast< PhysX3Callbacks::PhysX3SpecificData *>( 
      params->getEngine()->getEngineSpecificData() );

  PhysX3RigidBody* body= (PhysX3RigidBody*)params->getBodyId();
  physx_data->scene->removeActor ( *body->getActor() );
  delete body;

  return PeriodicThread::CALLBACK_DONE;
}

H3DUtil::PeriodicThread::CallbackCode PhysX3Callbacks::setRigidBodyParameters( void *data ) {
  PhysicsEngineParameters::RigidBodyParameters *params = 
    static_cast< PhysicsEngineParameters::RigidBodyParameters *>( data );
  PhysX3SpecificData *physx_data = 
    static_cast< PhysX3SpecificData * >(params->getEngine()->getEngineSpecificData());
  PhysX3RigidBody* body= (PhysX3RigidBody*)params->getBodyId();
  body->setParameters ( *params );
  delete params;
  return PeriodicThread::CALLBACK_DONE;
}

H3DUtil::PeriodicThread::CallbackCode PhysX3Callbacks::getRigidBodyParameters( void *data ) {
  PhysicsEngineParameters::RigidBodyParameters *params = 
    static_cast< PhysicsEngineParameters::RigidBodyParameters *>( data );
  PhysX3RigidBody* body= (PhysX3RigidBody*)params->getBodyId();
  body->getParameters ( *params );
  return PeriodicThread::CALLBACK_DONE;
}

H3DUtil::PeriodicThread::CallbackCode PhysX3Callbacks::addCollidable( void *data ) {
  PhysicsEngineParameters::CollidableParameters *data_params = 
    static_cast< PhysicsEngineParameters::CollidableParameters *>( data );
  PhysicsEngineParameters::ShapeParameters *params =
    dynamic_cast< PhysicsEngineParameters::ShapeParameters *>( data_params );
  if( !params ) {
    /// \todo Support CollidableOffset node
    Console(3) << "Warning: Only CollidableShapes are supported by PhysX3!" << endl;
    return PeriodicThread::CALLBACK_DONE;
  }
  PhysX3SpecificData *physx_data = 
    static_cast< PhysX3SpecificData * >(params->getEngine()->getEngineSpecificData());

  PhysX3CollidableShape* shape= new PhysX3CollidableShape ( *params );
  params->setCollidableId ( (H3DCollidableId)shape );

  return PeriodicThread::CALLBACK_DONE;
}

PeriodicThread::CallbackCode PhysX3Callbacks::removeCollidable( void *data ) {
  ShapeParameters *params = static_cast< ShapeParameters * >( data );
  PhysX3Callbacks::PhysX3SpecificData *physx_data = 
    static_cast< PhysX3SpecificData * >(params->getEngine()->getEngineSpecificData());
  
  PhysX3CollidableShape* shape= (PhysX3CollidableShape*)params->getCollidableId();
  delete shape;

  return PeriodicThread::CALLBACK_DONE;
}

H3DUtil::PeriodicThread::CallbackCode PhysX3Callbacks::setCollidableParameters( void *data ) {
  PhysicsEngineParameters::ShapeParameters *params = 
    static_cast< PhysicsEngineParameters::ShapeParameters *>( data );

  PhysX3CollidableShape* shape= (PhysX3CollidableShape*)params->getCollidableId();
  shape->setParameters ( *params );
  delete params;
  return PeriodicThread::CALLBACK_DONE;
}

H3DUtil::PeriodicThread::CallbackCode PhysX3Callbacks::getCollidableParameters( void *data ) {
  PhysicsEngineParameters::ShapeParameters *params = 
    static_cast< PhysicsEngineParameters::ShapeParameters *>( data );

  PhysX3CollidableShape* shape= (PhysX3CollidableShape*)params->getCollidableId();
  shape->getParameters ( *params );
  return PeriodicThread::CALLBACK_DONE;
}

H3DUtil::PeriodicThread::CallbackCode PhysX3Callbacks::addGlobalExternalForceAndTorque( void *data ) {
  PhysicsEngineParameters::ExternalForceTorqueParameters *params = 
    static_cast< PhysicsEngineParameters::ExternalForceTorqueParameters *>( data );
  PhysX3SpecificData *physx_data = 
    static_cast< PhysX3SpecificData * >(params->engine_thread->getEngineSpecificData());

  PhysX3RigidBody* body= (PhysX3RigidBody*)params->body_id;
  body->getActor()->addForce ( toPxVec3 ( params->force ) );
  body->getActor()->addTorque ( toPxVec3 ( params->torque ) );

  return PeriodicThread::CALLBACK_DONE;
}

H3DUtil::PeriodicThread::CallbackCode PhysX3Callbacks::addConstraint( void *data ) {
  PhysicsEngineParameters::JointParameters *params = 
    static_cast< PhysicsEngineParameters::JointParameters *>( data );
  PhysX3SpecificData *physx_data = 
    static_cast< PhysX3SpecificData * >(params->getEngine()->getEngineSpecificData());
  
  PhysX3Joint* joint= PhysX3Joint::createJoint ( *params );

  if ( !joint ) {
    // Constraint creation failed
    Console ( 3 ) << "Warning: Constraint type " << params->getType() << " not supported by PhysX3 in H3DPhysics!" << endl;
  }

  return PeriodicThread::CALLBACK_DONE;
}

PeriodicThread::CallbackCode PhysX3Callbacks::removeConstraint( void *data ) {
  ConstraintParameters *params = static_cast< ConstraintParameters *>( data );
  PhysX3Joint* joint= (PhysX3Joint*)params->getConstraintId();
  delete joint;
  return PeriodicThread::CALLBACK_DONE;
}

PeriodicThread::CallbackCode PhysX3Callbacks::setConstraintParameters( void *data ) {
  ConstraintParameters *params = static_cast< ConstraintParameters *>( data );
  PhysX3Joint* joint= (PhysX3Joint*)params->getConstraintId();
  joint->setParameters ( *params );
  delete params;
  return PeriodicThread::CALLBACK_DONE;
}

PeriodicThread::CallbackCode PhysX3Callbacks::getConstraintParameters( void *data ) {
  ConstraintParameters *params = static_cast< ConstraintParameters *>( data );
  PhysX3Joint* joint= (PhysX3Joint*)params->getConstraintId();
  joint->getParameters ( *params );
  return PeriodicThread::CALLBACK_DONE;
}

PeriodicThread::CallbackCode PhysX3Callbacks::getCurrentContacts( void *data ) {
  typedef pair< list< PhysicsEngineParameters::ContactParameters  >*,
                PhysicsEngineThread * > InputType;
  InputType *params = 
    static_cast< InputType *>( data );
  PhysX3SpecificData *physX_data = 
    static_cast< PhysX3SpecificData * >(params->second->getEngineSpecificData());
  physX_data->simulation_events->getContacts ( *params->first );
  return PeriodicThread::CALLBACK_DONE;
}


PeriodicThread::CallbackCode PhysX3Callbacks::addSpace( void *data ) {
  return PeriodicThread::CALLBACK_DONE;
}

PeriodicThread::CallbackCode PhysX3Callbacks::removeSpace( void *data ) {
  return PeriodicThread::CALLBACK_DONE;
}

PeriodicThread::CallbackCode PhysX3Callbacks::setSpaceParameters( void *data ) {
  return PeriodicThread::CALLBACK_DONE;
}

PeriodicThread::CallbackCode PhysX3Callbacks::getSpaceParameters( void *data ) {
  return PeriodicThread::CALLBACK_DONE;
}



// soft body callback functions
//

H3DUtil::PeriodicThread::CallbackCode PhysX3Callbacks::addSoftBody ( void *data ) {
  return PeriodicThread::CALLBACK_DONE;
}

H3DUtil::PeriodicThread::CallbackCode PhysX3Callbacks::removeSoftBody ( void *data ) {
  return PeriodicThread::CALLBACK_DONE;
}

H3DUtil::PeriodicThread::CallbackCode PhysX3Callbacks::setSoftBodyParameters ( void *data ) {
  return PeriodicThread::CALLBACK_DONE;
}

H3DUtil::PeriodicThread::CallbackCode PhysX3Callbacks::getSoftBodyParameters ( void *data ) {
  return PeriodicThread::CALLBACK_DONE;
}

H3DUtil::PeriodicThread::CallbackCode PhysX3Callbacks::applyExternalForces ( void *data ) {
  return PeriodicThread::CALLBACK_DONE;
}

#endif // HAVE_PHYSX3


