/*
 *  $Id: graph_stats.c 21891 2019-02-19 18:13:37Z yeti-dn $
 *  Copyright (C) 2016-2019 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <stdlib.h>
#include <string.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libprocess/linestats.h>
#include <libgwydgets/gwygraphmodel.h>
#include <libgwydgets/gwycombobox.h>
#include <libgwydgets/gwynullstore.h>
#include <libgwydgets/gwystock.h>
#include <libgwydgets/gwydgetutils.h>
#include <libgwymodule/gwymodule-graph.h>
#include <app/gwymoduleutils.h>
#include <app/gwyapp.h>

enum { NSIMPLEPARAMS = 8 };

typedef struct {
    gint curve;
    gdouble from;
    gdouble to;
    GwyResultsReportType report_style;
} StatsArgs;

static const gchar *guivalues[] = {
    "min", "max", "avg", "median",
    "ra", "rms", "skew", "kurtosis", "projlen", "length", "variation",
    "integralavg", "integral", "integralp", "integraln", "integral2",
};

typedef struct {
    GwyGraphModel *parent_gmodel;
    StatsArgs *args;
    GwySIValueFormat *xvf;
    gulong recalculate_id;
    gboolean same_units;

    GtkWidget *dialogue;
    GtkWidget *graph;
    GtkWidget *curve;
    GtkWidget *from;
    GtkWidget *to;
    GtkWidget *npts;

    GwyResults *results;
    GtkWidget *guivalues[G_N_ELEMENTS(guivalues)];
    GtkWidget *rexport;
} StatsControls;

static gboolean module_register       (void);
static void     graph_stats           (GwyGraph *graph);
static void     graph_stats_dialogue  (GwyContainer *data,
                                       GwyGraphModel *gmodel,
                                       StatsArgs *args);
static gint     attach_value_block    (StatsControls *controls,
                                       GtkTable *table,
                                       gint row,
                                       const gchar *name,
                                       guint from,
                                       guint to);
static void     curve_changed         (GtkComboBox *combo,
                                       StatsControls *controls);
static void     range_changed         (GtkWidget *entry,
                                       StatsControls *controls);
static void     stats_limit_selection (StatsControls *controls,
                                       gboolean curve_switch);
static void     stats_get_full_x_range(StatsControls *controls,
                                       gdouble *xmin,
                                       gdouble *xmax);
static void     graph_selected        (GwySelection* selection,
                                       gint i,
                                       StatsControls *controls);
static void     report_style_changed  (StatsControls *controls,
                                       GwyResultsExport *rexport);
static gboolean update_stats_table    (gpointer user_data);
static void     invalidate            (StatsControls *controls);
static guint    compute_stats         (GwyGraphCurveModel *gcmodel,
                                       GwyResults *results,
                                       gboolean same_units,
                                       gdouble from,
                                       gdouble to);
static void     graph_stats_load_args (GwyContainer *container,
                                       StatsArgs *args);
static void     graph_stats_save_args (GwyContainer *container,
                                       StatsArgs *args);

static const StatsArgs graph_stats_defaults = {
    0, -G_MAXDOUBLE, G_MAXDOUBLE,
    GWY_RESULTS_REPORT_COLON,
};

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Calculates simple graph curve statistics."),
    "Yeti <yeti@gwyddion.net>",
    "2.0",
    "David Nečas (Yeti)",
    "2017",
};

GWY_MODULE_QUERY2(module_info, graph_stats)

static gboolean
module_register(void)
{
    gwy_graph_func_register("graph_stats",
                            (GwyGraphFunc)&graph_stats,
                            N_("/_Statistics..."),
                            NULL,
                            GWY_MENU_FLAG_GRAPH,
                            N_("Calculate graph curve statistics"));

    return TRUE;
}

static void
graph_stats(GwyGraph *graph)
{
    GwyContainer *data;
    StatsArgs args;

    gwy_app_data_browser_get_current(GWY_APP_CONTAINER, &data, NULL);

    graph_stats_load_args(gwy_app_settings_get(), &args);
    graph_stats_dialogue(data, gwy_graph_get_model(graph), &args);
    graph_stats_save_args(gwy_app_settings_get(), &args);
}

/* This seems to be the safe way of doing things.  Calling g_source_remove()
 * and gtk_widget_destroy() seems a race. */
static void
cancel_update(gpointer p, G_GNUC_UNUSED GObject *dialogue)
{
    gulong *sid = (gulong*)p;

    if (*sid) {
        g_source_remove(*sid);
        *sid = 0;
    }
}

static void
graph_stats_dialogue(GwyContainer *data,
                     GwyGraphModel *parent_gmodel,
                     StatsArgs *args)
{
    GtkWidget *dialogue, *hbox, *table, *hbox2, *label;
    GwyGraphArea *area;
    GwyGraphModel *gmodel;
    GwySelection *selection;
    StatsControls controls;
    GwyResultsExport *rexport;
    GwyResults *results;
    gint row;
    gdouble xrange, min, max;
    GwySIUnit *xunit, *yunit;

    gwy_clear(&controls, 1);
    controls.args = args;
    controls.parent_gmodel = parent_gmodel;
    gmodel = gwy_graph_model_new_alike(parent_gmodel);

    g_object_get(parent_gmodel, "si-unit-x", &xunit, "si-unit-y", &yunit, NULL);
    gwy_graph_model_get_x_range(parent_gmodel, &min, &max);
    xrange = MAX(fabs(max), fabs(min));
    controls.xvf
        = gwy_si_unit_get_format_with_digits(xunit, GWY_SI_UNIT_FORMAT_VFMARKUP,
                                             xrange, 3, NULL);
    controls.same_units = gwy_si_unit_equal(xunit, yunit);

    controls.results = results = gwy_results_new();
    gwy_results_add_header(results, N_("Graph Statistics"));
    gwy_results_add_value_str(results, "file", N_("File"));
    gwy_results_add_value_str(results, "graph", N_("Graph"));
    gwy_results_add_value_str(results, "curve", N_("Curve"));
    gwy_results_add_format(results, "range", N_("Range"), TRUE,
    /* TRANSLATORS: %{from}v and %{to}v are ids, do NOT translate them. */
                           N_("%{from}v to %{to}v"),
                           "power-x", 1,
                           NULL);
    gwy_results_add_value_int(results, "npts", N_("Number of points"));

    gwy_results_add_separator(results);
    gwy_results_add_header(results, _("Simple Parameters"));
    gwy_results_add_value_z(results, "min", N_("Minimum"));
    gwy_results_add_value_z(results, "max", N_("Maximum"));
    gwy_results_add_value_z(results, "avg", N_("Mean value"));
    gwy_results_add_value_z(results, "median", N_("Median"));
    gwy_results_add_value_z(results, "ra", N_("Ra"));
    gwy_results_add_value_z(results, "rms", N_("Rms (Rq)"));
    gwy_results_add_value_plain(results, "skew", N_("Skew"));
    gwy_results_add_value_plain(results, "kurtosis", N_("Excess kurtosis"));

    gwy_results_add_separator(results);
    gwy_results_add_header(results, _("Integrals"));
    gwy_results_add_value_x(results, "projlen", N_("Projected length"));
    gwy_results_add_value_x(results, "length", N_("Developed length"));
    gwy_results_add_value_z(results, "variation", N_("Variation"));
    gwy_results_add_value_z(results, "integralavg", N_("Mean value"));
    gwy_results_add_value(results, "integral", N_("Area under curve"),
                          "type", GWY_RESULTS_VALUE_FLOAT,
                          "power-x", 1, "power-z", 1,
                          NULL);
    gwy_results_add_value(results, "integralp", N_("Positive area"),
                          "type", GWY_RESULTS_VALUE_FLOAT,
                          "power-x", 1, "power-z", 1,
                          NULL);
    gwy_results_add_value(results, "integraln", N_("Negative area"),
                          "type", GWY_RESULTS_VALUE_FLOAT,
                          "power-x", 1, "power-z", 1,
                          NULL);
    gwy_results_add_value_z(results, "integral2", N_("Root mean square"));

    gwy_results_set_unit(results, "x", xunit);
    gwy_results_set_unit(results, "z", yunit);
    g_object_unref(xunit);
    g_object_unref(yunit);

    gwy_results_fill_filename(results, "file", data);
    gwy_results_fill_graph(results, "graph", gmodel);

    dialogue = gtk_dialog_new_with_buttons(_("Graph Statistics"),
                                           NULL, 0, NULL);
    controls.dialogue = dialogue;
    gtk_dialog_add_button(GTK_DIALOG(dialogue), GTK_STOCK_OK, GTK_RESPONSE_OK);
    gwy_help_add_to_graph_dialog(GTK_DIALOG(dialogue), GWY_HELP_DEFAULT);
    gtk_dialog_set_default_response(GTK_DIALOG(dialogue), GTK_RESPONSE_OK);

    hbox = gtk_hbox_new(FALSE, 2);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialogue)->vbox), hbox,
                       TRUE, TRUE, 0);

    /* Parameters */
    table = gtk_table_new(6 + G_N_ELEMENTS(guivalues), 2, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_box_pack_start(GTK_BOX(hbox), table, FALSE, FALSE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    row = 0;

    controls.curve = gwy_combo_box_graph_curve_new(G_CALLBACK(curve_changed),
                                                   &controls,
                                                   parent_gmodel, args->curve);
    gwy_table_attach_adjbar(table, row, _("_Graph curve:"), NULL,
                            GTK_OBJECT(controls.curve),
                            GWY_HSCALE_WIDGET_NO_EXPAND);
    row++;

    /* Range */
    hbox2 = gtk_hbox_new(FALSE, 6);
    gtk_table_attach(GTK_TABLE(table), hbox2,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    label = gtk_label_new(_("Range:"));
    gtk_box_pack_start(GTK_BOX(hbox2), label, FALSE, FALSE, 0);

    controls.from = gtk_entry_new();
    g_object_set_data(G_OBJECT(controls.from), "id", (gpointer)"from");
    gtk_entry_set_width_chars(GTK_ENTRY(controls.from), 8);
    gtk_box_pack_start(GTK_BOX(hbox2), controls.from, FALSE, FALSE, 0);
    g_signal_connect(controls.from, "activate",
                     G_CALLBACK(range_changed), &controls);
    gwy_widget_set_activate_on_unfocus(controls.from, TRUE);

    label = gtk_label_new(gwy_sgettext("range|to"));
    gtk_box_pack_start(GTK_BOX(hbox2), label, FALSE, FALSE, 0);

    controls.to = gtk_entry_new();
    g_object_set_data(G_OBJECT(controls.to), "id", (gpointer)"to");
    gtk_entry_set_width_chars(GTK_ENTRY(controls.to), 8);
    gtk_box_pack_start(GTK_BOX(hbox2), controls.to, FALSE, FALSE, 0);
    g_signal_connect(controls.to, "activate",
                     G_CALLBACK(range_changed), &controls);
    gwy_widget_set_activate_on_unfocus(controls.to, TRUE);

    label = gtk_label_new(NULL);
    gtk_label_set_markup(GTK_LABEL(label), controls.xvf->units);
    gtk_box_pack_start(GTK_BOX(hbox2), label, FALSE, FALSE, 0);

    label = gtk_label_new(_("Number of points:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
                     GTK_EXPAND | GTK_FILL, 0, 0, 0);

    controls.npts = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(controls.npts), 1.0, 0.5);
    gtk_label_set_selectable(GTK_LABEL(controls.npts), TRUE);
    gtk_table_attach(GTK_TABLE(table), controls.npts, 1, 2, row, row+1,
                     GTK_FILL, 0, 0, 0);
    row++;

    /* Results */
    row = attach_value_block(&controls, GTK_TABLE(table), row,
                             _("Simple Parameters"),
                             0, NSIMPLEPARAMS);
    row = attach_value_block(&controls, GTK_TABLE(table), row,
                             _("Integrals"),
                             NSIMPLEPARAMS, G_N_ELEMENTS(guivalues));

    controls.rexport = gwy_results_export_new(args->report_style);
    rexport = GWY_RESULTS_EXPORT(controls.rexport);
    gwy_results_export_set_title(rexport, _("Save Parameters"));
    gwy_results_export_set_results(rexport, results);
    gtk_table_attach(GTK_TABLE(table), controls.rexport, 0, 2, row, row+1,
                     GTK_FILL, 0, 0, 0);
    g_signal_connect_swapped(rexport, "format-changed",
                             G_CALLBACK(report_style_changed), &controls);
    row++;

    /* Graph */
    controls.graph = gwy_graph_new(gmodel);
    g_object_unref(gmodel);
    gtk_widget_set_size_request(controls.graph, 400, 300);

    gwy_graph_enable_user_input(GWY_GRAPH(controls.graph), FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), controls.graph, TRUE, TRUE, 0);
    gwy_graph_set_status(GWY_GRAPH(controls.graph), GWY_GRAPH_STATUS_XSEL);

    area = GWY_GRAPH_AREA(gwy_graph_get_area(GWY_GRAPH(controls.graph)));
    selection = gwy_graph_area_get_selection(area, GWY_GRAPH_STATUS_XSEL);
    gwy_selection_set_max_objects(selection, 1);
    g_signal_connect(selection, "changed",
                     G_CALLBACK(graph_selected), &controls);

    curve_changed(GTK_COMBO_BOX(controls.curve), &controls);
    graph_selected(selection, -1, &controls);

    g_object_weak_ref(G_OBJECT(dialogue), cancel_update,
                      &controls.recalculate_id);
    gtk_widget_show_all(dialogue);
    gtk_dialog_run(GTK_DIALOG(dialogue));
    gtk_widget_destroy(dialogue);

    gwy_si_unit_value_format_free(controls.xvf);
    g_object_unref(controls.results);
}

static gint
attach_value_block(StatsControls *controls,
                   GtkTable *table, gint row,
                   const gchar *name,
                   guint from, guint to)
{
    GwyResults *results = controls->results;
    GtkWidget *label;
    GString *str;
    guint i;

    if (row)
        gtk_table_set_row_spacing(table, row-1, 8);
    gtk_table_attach(table, gwy_label_new_header(name),
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    str = g_string_new(NULL);
    for (i = from; i < to; i++) {
        g_string_assign(str, gwy_results_get_label_with_symbol(results,
                                                               guivalues[i]));
        g_string_append_c(str, ':');
        label = gtk_label_new(str->str);
        gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
        gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
        gtk_table_attach(table, label, 0, 1, row, row+1, GTK_FILL, 0, 0, 0);

        label = controls->guivalues[i] = gtk_label_new(NULL);
        gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
        gtk_label_set_selectable(GTK_LABEL(label), TRUE);
        gtk_table_attach(table, label, 1, 2, row, row+1,
                         GTK_EXPAND | GTK_FILL, 0, 0, 0);
        row++;
    }
    g_string_free(str, TRUE);

    return row;
}

static void
curve_changed(GtkComboBox *combo, StatsControls *controls)
{
    StatsArgs *args = controls->args;
    GwyGraphModel *gmodel;
    GwyGraphCurveModel *gcmodel;

    args->curve = gwy_enum_combo_box_get_active(combo);
    gmodel = gwy_graph_get_model(GWY_GRAPH(controls->graph));
    gwy_graph_model_remove_all_curves(gmodel);
    gcmodel = gwy_graph_model_get_curve(controls->parent_gmodel, args->curve);
    if (gwy_graph_curve_model_is_ordered(gcmodel))
        gwy_graph_model_add_curve(gmodel, gcmodel);
    else {
        gcmodel = gwy_graph_curve_model_duplicate(gcmodel);
        gwy_graph_curve_model_enforce_order(gcmodel);
        gwy_graph_model_add_curve(gmodel, gcmodel);
        g_object_unref(gcmodel);
    }
    invalidate(controls);
}

static void
range_changed(GtkWidget *entry, StatsControls *controls)
{
    const gchar *id;
    gdouble *x, newval;

    id = g_object_get_data(G_OBJECT(entry), "id");
    if (gwy_strequal(id, "from"))
        x = &controls->args->from;
    else
        x = &controls->args->to;

    newval = atof(gtk_entry_get_text(GTK_ENTRY(entry)));
    newval *= controls->xvf->magnitude;
    if (newval == *x)
        return;
    *x = newval;
    stats_limit_selection(controls, FALSE);
}

static void
stats_limit_selection(StatsControls *controls, gboolean curve_switch)
{
    GwySelection *selection;
    GwyGraphArea *area;
    gdouble xmin, xmax;

    area = GWY_GRAPH_AREA(gwy_graph_get_area(GWY_GRAPH(controls->graph)));
    selection = gwy_graph_area_get_selection(area, GWY_GRAPH_STATUS_XSEL);

    if (curve_switch && !gwy_selection_get_data(selection, NULL)) {
        graph_selected(selection, -1, controls);
        return;
    }

    stats_get_full_x_range(controls, &xmin, &xmax);
    controls->args->from = CLAMP(controls->args->from, xmin, xmax);
    controls->args->to = CLAMP(controls->args->to, xmin, xmax);

    if (controls->args->from == xmin && controls->args->to == xmax)
        gwy_selection_clear(selection);
    else {
        gdouble range[2];

        range[0] = controls->args->from;
        range[1] = controls->args->to;
        gwy_selection_set_object(selection, 0, range);
    }
}

static void
stats_get_full_x_range(StatsControls *controls,
                       gdouble *xmin, gdouble *xmax)
{
    GwyGraphModel *gmodel;
    GwyGraphCurveModel *gcmodel;

    gmodel = gwy_graph_get_model(GWY_GRAPH(controls->graph));
    gcmodel = gwy_graph_model_get_curve(gmodel, 0);
    gwy_graph_curve_model_get_x_range(gcmodel, xmin, xmax);
}

static void
graph_selected(GwySelection* selection,
               gint i,
               StatsControls *controls)
{
    GwySIValueFormat *xvf = controls->xvf;
    StatsArgs *args;
    gchar buffer[24];
    gdouble range[2];
    gint nselections;
    gdouble power10;

    g_return_if_fail(i <= 0);

    args = controls->args;
    nselections = gwy_selection_get_data(selection, NULL);
    gwy_selection_get_object(selection, 0, range);

    if (nselections <= 0 || range[0] == range[1])
        stats_get_full_x_range(controls, &args->from, &args->to);
    else {
        args->from = MIN(range[0], range[1]);
        args->to = MAX(range[0], range[1]);
    }
    power10 = pow10(xvf->precision);
    g_snprintf(buffer, sizeof(buffer), "%.*f",
               xvf->precision,
               floor(args->from*power10/xvf->magnitude)/power10);
    gtk_entry_set_text(GTK_ENTRY(controls->from), buffer);
    g_snprintf(buffer, sizeof(buffer), "%.*f",
               xvf->precision,
               ceil(args->to*power10/xvf->magnitude)/power10);
    gtk_entry_set_text(GTK_ENTRY(controls->to), buffer);

    invalidate(controls);
}

static void
report_style_changed(StatsControls *controls, GwyResultsExport *rexport)
{
    controls->args->report_style = gwy_results_export_get_format(rexport);
}

static void
invalidate(StatsControls *controls)
{
    if (controls->recalculate_id)
        return;

    controls->recalculate_id = g_idle_add(update_stats_table, controls);
}

static gboolean
update_stats_table(gpointer user_data)
{
    StatsControls *controls = (StatsControls*)user_data;
    StatsArgs *args = controls->args;
    GwyGraphModel *gmodel;
    GwyGraphCurveModel *gcmodel;
    GwyResults *results = controls->results;
    GwyResultsExport *rexport = GWY_RESULTS_EXPORT(controls->rexport);
    gchar buffer[16];
    guint i, npts;

    gmodel = gwy_graph_get_model(GWY_GRAPH(controls->graph));
    gcmodel = gwy_graph_model_get_curve(gmodel, 0);
    npts = compute_stats(gcmodel, results, controls->same_units,
                         args->from, args->to);

    for (i = 0; i < G_N_ELEMENTS(guivalues); i++) {
        gtk_label_set_markup(GTK_LABEL(controls->guivalues[i]),
                             npts > 0
                             ? gwy_results_get_full(results, guivalues[i])
                             : "");
    }
    gwy_results_export_set_actions_sensitive(rexport, npts > 0);

    g_snprintf(buffer, sizeof(buffer), "%u", npts);
    gtk_label_set_text(GTK_LABEL(controls->npts), buffer);

    controls->recalculate_id = 0;
    return FALSE;
}

static guint
compute_stats(GwyGraphCurveModel *gcmodel,
              GwyResults *results,
              gboolean same_units,
              gdouble from, gdouble to)
{
    GwyDataLine *dline;
    const gdouble *xdata, *ydata;
    guint ndata, i, pos, npts;
    gdouble min, max, projlen, length, variation;
    gdouble integralp, integraln, integral, integral2;

    xdata = gwy_graph_curve_model_get_xdata(gcmodel);
    ydata = gwy_graph_curve_model_get_ydata(gcmodel);
    ndata = gwy_graph_curve_model_get_ndata(gcmodel);
    gwy_results_fill_graph_curve(results, "curve", gcmodel);

    for (pos = 0; pos < ndata && xdata[pos] < from; pos++)
        pos++;
    for (npts = ndata; npts && xdata[npts-1] > to; npts--)
        npts--;

    gwy_results_set_nav(results, G_N_ELEMENTS(guivalues), guivalues);
    if (npts <= pos)
        return 0;

    npts -= pos;
    gwy_results_fill_values(results, "npts", npts, NULL);
    gwy_results_fill_format(results, "range", "from", from, "to", to, NULL);

    /* Calculate simple quantities only depending on the value distribution
     * using DataLine methods. */
    dline = gwy_data_line_new(npts, 1.0, FALSE);
    gwy_assign(gwy_data_line_get_data(dline), ydata + pos, npts);
    gwy_data_line_get_min_max(dline, &min, &max);
    gwy_results_fill_values(results,
                            "min", min, "max", max,
                            "avg", gwy_data_line_get_avg(dline),
                            "median", gwy_data_line_get_median(dline),
                            NULL);
    if (npts > 1) {
        gwy_results_fill_values(results,
                                "rms", gwy_data_line_get_rms(dline),
                                "ra", gwy_data_line_get_ra(dline),
                                "skew", gwy_data_line_get_skew(dline),
                                "kurtosis", gwy_data_line_get_kurtosis(dline),
                                NULL);
    }
    g_object_unref(dline);
    if (npts < 2)
        return npts;

    projlen = xdata[pos + npts-1] - xdata[pos];
    integralp = integraln = integral = integral2 = length = variation = 0.0;
    for (i = 0; i < npts-1; i++) {
        gdouble y1 = ydata[pos + i], y2 = ydata[pos + i+1];
        gdouble dx = xdata[pos + i+1] - xdata[pos + i];
        gdouble x, dpos = 0.0, dneg = 0.0, d2 = 0.0;

        length += sqrt((y2 - y1)*(y2 - y1) + dx*dx);
        variation += fabs(y2 - y1);
        if (dx <= 0.0)
            continue;

        if (y1 >= 0.0 && y2 >= 0.0) {
            dpos = (y1 + y2)*dx;
            d2 = (y1*y1 + y2*y2)*dx;
        }
        else if (y1 <= 0.0 && y2 <= 0.0) {
            dneg = (y1 + y2)*dx;
            d2 = (y1*y1 + y2*y2)*dx;
        }
        else if (y1 > 0.0 && y2 < 0.0) {
            x = y1/(y1 - y2)*dx;
            dpos = y1*x;
            dneg = y2*(dx - x);
            d2 = (y1*y1*x + y2*y2*(dx - x));
        }
        else if (y1 < 0.0 && y2 > 0.0) {
            x = y2/(y2 - y1)*dx;
            dpos = y2*x;
            dneg = y1*(dx - x);
            d2 = (y1*y1*(dx - x) + y2*y2*x);
        }
        else {
            g_warning("Impossible curve value signs.");
            continue;
        }
        integralp += dpos;
        integraln += dneg;
        integral += dpos + dneg;
        integral2 += d2;
    }

    integral *= 0.5;
    integralp *= 0.5;
    integraln *= 0.5;
    integral2 *= 0.5;
    gwy_results_fill_values(results,
                            "projlen", projlen,
                            "variation", variation,
                            "integralp", integralp,
                            "integraln", integraln,
                            "integral", integral,
                            "integralavg", integral/projlen,
                            "integral2", sqrt(integral2/projlen),
                            NULL);
    if (same_units)
        gwy_results_fill_values(results, "length", length, NULL);

    return npts;
}

static const gchar report_style_key[]  = "/module/graph_stats/report_style";

static void
graph_stats_load_args(GwyContainer *container, StatsArgs *args)
{
    *args = graph_stats_defaults;

    gwy_container_gis_enum_by_name(container, report_style_key,
                                   &args->report_style);
}

static void
graph_stats_save_args(GwyContainer *container, StatsArgs *args)
{
    gwy_container_set_enum_by_name(container, report_style_key,
                                   args->report_style);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
