/*
 *  @(#) $Id: rank.c 18537 2016-03-28 18:11:19Z yeti-dn $
 *  Copyright (C) 2014-2015 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libprocess/stats.h>
#include <libprocess/filters.h>
#include <libprocess/elliptic.h>
#include <libgwydgets/gwyradiobuttons.h>
#include <libgwydgets/gwydgetutils.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwyapp.h>
#include "preview.h"

#define RANK_RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

enum {
    MAX_SIZE = 129,
    BLOCK_SIZE = 200,
};

typedef enum {
    FILTER_RANK          = 0,
    FILTER_NORMALIZATION = 1,
    FILTER_RANGE         = 2,
    FILTER_NFILTERS,
} FilterType;

typedef struct {
    FilterType type;
    guint size;
} RankArgs;

typedef struct {
    RankArgs *args;
    GSList *type;
    GtkObject *size;
} RankControls;

static gboolean module_register    (void);
static void     rank               (GwyContainer *data,
                                    GwyRunType run);
static gboolean rank_dialog        (RankArgs *args);
static void     rank_dialog_reset  (RankControls *controls);
static void     size_changed       (RankControls *controls,
                                    GtkAdjustment *adj);
static void     filter_type_changed(GtkToggleButton *button,
                                    RankControls *controls);
static void     rank_do            (GwyContainer *data,
                                    RankArgs *args);
static gdouble  local_rank         (GwyDataField *data_field,
                                    gint size,
                                    const gint *xsize,
                                    gint col,
                                    gint row);
static void     minmax_do          (GwyContainer *data,
                                    RankArgs *args);
static void     load_args          (GwyContainer *container,
                                    RankArgs *args);
static void     save_args          (GwyContainer *container,
                                    RankArgs *args);
static void     sanitize_args      (RankArgs *args);

static const RankArgs rank_defaults = {
    FILTER_RANK, 15,
};

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Enhances local contrast using a rank transform."),
    "Yeti <yeti@gwyddion.net>",
    "2.0",
    "David Nečas (Yeti) & Petr Klapetek",
    "2014",
};

GWY_MODULE_QUERY(module_info)

static gboolean
module_register(void)
{
    gwy_process_func_register("rank",
                              (GwyProcessFunc)&rank,
                              N_("/_Presentation/_Rank..."),
                              NULL,
                              RANK_RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              N_("Presentation with local contrast "
                                 "ehnanced using a rank transform"));

    return TRUE;
}

static void
rank(GwyContainer *data, GwyRunType run)
{
    RankArgs args;
    gint id;

    g_return_if_fail(run & RANK_RUN_MODES);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD_ID, &id, 0);
    load_args(gwy_app_settings_get(), &args);
    if (run == GWY_RUN_INTERACTIVE) {
        gboolean ok = rank_dialog(&args);
        save_args(gwy_app_settings_get(), &args);
        if (!ok)
            return;
    }

    if (args.type == FILTER_RANK)
        rank_do(data, &args);
    else
        minmax_do(data, &args);

    gwy_app_channel_log_add_proc(data, id, id);
}

static gboolean
rank_dialog(RankArgs *args)
{
    static const GwyEnum types[] = {
        { N_("Rank transform"),      FILTER_RANK,          },
        { N_("Local normalization"), FILTER_NORMALIZATION, },
        { N_("Value range"),         FILTER_RANGE,         },
    };

    GtkWidget *dialog, *table, *label;
    RankControls controls;
    gint response;
    gint row;

    controls.args = args;

    dialog = gtk_dialog_new_with_buttons(_("Rank Transform"),
                                         NULL, 0,
                                         _("_Reset"), RESPONSE_RESET,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                         GTK_STOCK_OK, GTK_RESPONSE_OK,
                                         NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialog), GWY_HELP_DEFAULT);

    table = gtk_table_new(2 + G_N_ELEMENTS(types), 4, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), table,
                       FALSE, FALSE, 4);
    row = 0;

    label = gtk_label_new(_("Filter type:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 0, 3, row, row+1,
                     GTK_FILL, 0, 0, 0);
    row++;

    controls.type = gwy_radio_buttons_create(types, G_N_ELEMENTS(types),
                                             G_CALLBACK(filter_type_changed),
                                             &controls,
                                             args->type);
    row = gwy_radio_buttons_attach_to_table(controls.type, GTK_TABLE(table),
                                            3, row);

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    controls.size = gtk_adjustment_new(args->size, 1, MAX_SIZE, 1, 10, 0);
    gwy_table_attach_hscale(table, row++, _("Kernel _size:"), "px",
                            controls.size, 0);
    g_signal_connect_swapped(controls.size, "value-changed",
                             G_CALLBACK(size_changed), &controls);

    gtk_widget_show_all(dialog);
    do {
        response = gtk_dialog_run(GTK_DIALOG(dialog));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(dialog);
            case GTK_RESPONSE_NONE:
            return FALSE;
            break;

            case GTK_RESPONSE_OK:
            break;

            case RESPONSE_RESET:
            rank_dialog_reset(&controls);
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    gtk_widget_destroy(dialog);

    return TRUE;
}

static void
rank_dialog_reset(RankControls *controls)
{
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->size),
                             rank_defaults.size);
}

static void
size_changed(RankControls *controls, GtkAdjustment *adj)
{
    controls->args->size = gwy_adjustment_get_int(adj);
}

static void
filter_type_changed(GtkToggleButton *button,
                    RankControls *controls)
{
    if (!gtk_toggle_button_get_active(button))
        return;

    controls->args->type = gwy_radio_buttons_get_current(controls->type);
}

static void
rank_do(GwyContainer *data, RankArgs *args)
{
    GwyDataField *dfield, *showfield;
    GQuark dquark, squark;
    gdouble *show;
    gint xres, yres, i, j, size, id;
    gint *xsize;
    guint count, step;
    gdouble q;

    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD_KEY, &dquark,
                                     GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     GWY_APP_SHOW_FIELD_KEY, &squark,
                                     GWY_APP_SHOW_FIELD, &showfield,
                                     0);
    g_return_if_fail(dfield && dquark && squark);
    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);

    size = 2*args->size + 1;

    gwy_app_wait_start(gwy_app_find_window_for_channel(data, id),
                       _("Rank transform..."));
    gwy_app_wait_set_fraction(0.0);

    showfield = gwy_data_field_new_alike(dfield, FALSE);
    gwy_si_unit_set_from_string(gwy_data_field_get_si_unit_z(showfield), "");
    show = gwy_data_field_get_data(showfield);

    xsize = g_new(gint, size);
    for (i = -(gint)args->size; i <= (gint)args->size; i++) {
        gdouble x = sqrt(0.25*size*size - i*i);
        xsize[i + args->size] = (gint)floor(x);
    }

    step = MAX(10000, xres*yres/100);
    count = 0;
    q = 1.0/(xres*yres);
    for (i = 0; i < yres; i++) {
        for (j = 0; j < xres; j++) {
            show[i*xres + j] = local_rank(dfield, args->size, xsize, j, i);
            if ((++count) % step == 0) {
                if (!gwy_app_wait_set_fraction(q*count))
                    goto cancelled;
            }
        }
    }

    gwy_data_field_normalize(showfield);
    if (!gwy_app_wait_set_fraction(1.0))
        goto cancelled;

    gwy_app_undo_qcheckpointv(data, 1, &squark);
    gwy_container_set_object(data, squark, showfield);
    gwy_data_field_data_changed(showfield);

cancelled:
    g_free(xsize);
    g_object_unref(showfield);
    gwy_app_wait_finish();
}

static gdouble
local_rank(GwyDataField *data_field,
           gint size, const gint *xsize,
           gint col, gint row)
{
    gint xres, yres, i, j, yfrom, yto;
    guint r, hr, t;
    const gdouble *data;
    gdouble v;

    xres = data_field->xres;
    yres = data_field->yres;
    data = data_field->data;
    v = data[row*xres + col];

    yfrom = MAX(0, row - size);
    yto = MIN(yres-1, row + size);

    r = hr = t = 0;
    for (i = yfrom; i <= yto; i++) {
        gint xr = xsize[i - row + size];
        gint xfrom = MAX(0, col - xr);
        gint xto = MIN(xres-1, col + xr);
        guint xlen = xto - xfrom + 1;
        const gdouble *d = data + i*xres + xfrom;

        for (j = xlen; j; j--, d++) {
            if (*d <= v) {
                r++;
                if (G_UNLIKELY(*d == v))
                    hr++;
            }
            t++;
        }
    }

    return (r - 0.5*hr)/t;
}

static void
minmax_do(GwyContainer *data, RankArgs *args)
{
    GwyDataField *dfield, *showfield, *kernel;
    GwyMinMaxFilterType filtertype;
    GQuark dquark, squark;
    gint size, id;

    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD_KEY, &dquark,
                                     GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     GWY_APP_SHOW_FIELD_KEY, &squark,
                                     GWY_APP_SHOW_FIELD, &showfield,
                                     0);
    g_return_if_fail(dfield && dquark && squark);

    size = 2*args->size + 1;
    filtertype = (args->type == FILTER_NORMALIZATION
                  ? GWY_MIN_MAX_FILTER_NORMALIZATION
                  : GWY_MIN_MAX_FILTER_RANGE);
    kernel = gwy_data_field_new(size, size, size, size, TRUE);
    gwy_data_field_elliptic_area_fill(kernel, 0, 0, size, size, 1.0);

    showfield = gwy_data_field_duplicate(dfield);
    gwy_si_unit_set_from_string(gwy_data_field_get_si_unit_z(showfield), "");
    gwy_data_field_area_filter_min_max(showfield, kernel, filtertype,
                                       0, 0, showfield->xres, showfield->yres);

    gwy_app_undo_qcheckpointv(data, 1, &squark);
    gwy_container_set_object(data, squark, showfield);
    gwy_data_field_data_changed(showfield);

    g_object_unref(kernel);
    g_object_unref(showfield);
}

static const gchar size_key[] = "/module/rank/size";
static const gchar type_key[] = "/module/rank/type";

static void
sanitize_args(RankArgs *args)
{
    args->size = CLAMP(args->size, 1, MAX_SIZE);
    args->type = CLAMP(args->type, 0, FILTER_NFILTERS-1);
}

static void
load_args(GwyContainer *container,
          RankArgs *args)
{
    *args = rank_defaults;

    gwy_container_gis_int32_by_name(container, size_key, &args->size);
    gwy_container_gis_enum_by_name(container, type_key, &args->type);
    sanitize_args(args);
}

static void
save_args(GwyContainer *container,
          RankArgs *args)
{
    gwy_container_set_int32_by_name(container, size_key, args->size);
    gwy_container_set_enum_by_name(container, type_key, args->type);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
