#include <zlib.h>
#include <iostream>
#include <fstream>
#include <cerrno>
#include <cstring>
#include <sstream>

#include "xmltrainingplan.h"

/*
 * EveMon skill plan example, thanks to Chruker on #eve-dev@Coldfront.
 * Direct export from EveMon with both, Planned and Prereq skills.
 * http://pastebin.com/f13baf757
 */

void
XmlTrainingPlanImport::import_from_file (std::string const& filename)
{
  /* Check extension. */
  if (filename.size() < 4)
    throw Exception("Error detecting file type. "
        "Valid types are \"emp\" and \"xml\".");

  std::string extension = filename.substr(filename.size() - 4);
  if (extension == ".emp" || extension == ".xml")
  {
    std::cout << "Parsing XML: " << filename << " ..." << std::endl;
    XmlDocumentPtr xmldoc = XmlDocument::create_from_file(filename);
    this->parse_xml(xmldoc);
  }
  else
  {
    throw Exception("Error detecting file type. "
        "Valid types are \"emp\" and \"xml\".");
  }
}

/* ---------------------------------------------------------------- */

void
XmlTrainingPlanImport::parse_xml (XmlDocumentPtr xmldoc)
{
  xmlNodePtr root = xmldoc->get_root_element();

  if (root->type != XML_ELEMENT_NODE
      || xmlStrcmp(root->name, (xmlChar const*)"plan"))
    throw Exception("Invalid XML root. Expecting <plan> node.");

  ApiSkillTreePtr tree = ApiSkillTree::request();

  /* Iterate over all plan entries. */
  xmlNodePtr node;
  for (node = root->children; node != 0; node = node->next)
  {
    if (node->type != XML_ELEMENT_NODE)
      continue;

    if (xmlStrcmp(node->name, (xmlChar const*)"entry"))
      continue;

    int skill_id = this->get_property_int(node, "skillID");
    int skill_level = this->get_property_int(node, "level");
    std::string skill_name = this->get_property(node, "skill");
    std::string entry_type = this->get_property(node, "type");
    bool prereq = (entry_type == "Prerequisite");

    if (skill_level < 0)
      throw Exception("Skill entry without destination level");

    ApiSkill const* skill = 0;
    if (skill_id >= 0)
      skill = tree->get_skill_for_id(skill_id);
    else if (!skill_name.empty())
      skill = tree->get_skill_for_name(skill_name);
    else
      throw Exception("Cannot find skill ID or name in entry");

    if (skill == 0)
    {
      std::cout << "Error finding skill \"" << skill_name
          << "\" (ID " << skill_id << ")" << std::endl;
      continue;
    }

    /* Try to get user notes. */
    std::string notes;
    for (xmlNodePtr iter = node->children; iter != 0; iter = iter->next)
    {
      if (iter->type != XML_ELEMENT_NODE)
        continue;
      this->set_string_if_node_text(iter, "notes", notes);
    }

    /* We got a valid skill description. Add it. */
    this->plan.push_back(XmlTrainingItem(skill, skill_level, prereq, notes));
  }
}

/* ---------------------------------------------------------------- */

void
XmlTrainingPlanExport::write_to_file (std::string const& filename)
{
  if (filename.size() <= 4)
    throw Exception("Invalid filename for plan export");

  /* Generate the export XML string. */

  std::stringstream xmldata;
  xmldata << "<?xml version=\"1.0\"?>" << std::endl;
  xmldata << "<!-- Generated by GtkEveMon "
      << "(http://gtkevemon.battleclinic.com) -->" << std::endl;
  xmldata << "<plan xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\""
      << " xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\""
      << " name=\"GtkEveMon exported\""
      << " owner=\"00000000-0000-0000-0000-000000000000\""
      << " revision=\"2186\">" << std::endl;
  xmldata << "  <sorting criteria=\"None\" order=\"None\""
      << " optimizeLearning=\"false\""
      << " groupByPriority=\"false\" />" << std::endl;

  for (std::size_t i = 0; i < this->plan.size(); ++i)
  {
    XmlTrainingItem const& item = this->plan[i];
    xmldata << "  <entry skillID=\"" << item.skill->id << "\""
        << " skill=\"" << item.skill->name << "\""
        << " level=\"" << item.level << "\""
        << " priority=\"1\" type=\""
        << (item.prerequisite ? "Prerequisite" : "Planned")
        << "\"><notes>" << item.user_notes << "</notes></entry>"
        << std::endl;
  }

  xmldata << "</plan>" << std::endl;

  /* Write the string stream to file. */

  std::string xmldata_str = xmldata.str();
  std::string extension = filename.substr(filename.size() - 4);
  if (extension == ".emp")
  {
    /* Write XML to gzipped file. */
    ::gzFile out = ::gzopen(filename.c_str(), "wb");
    if (out == 0)
      throw Exception("Error writing to file: "
          + std::string(::strerror(errno))); // FIXME Use errno?
    ::gzwrite(out, xmldata_str.c_str(), (unsigned int)xmldata_str.size());
    ::gzclose(out);
  }
  else if (extension == ".xml")
  {
    /* Write XML to unzipped, plain file. */
    std::ofstream out(filename.c_str());
    if (!out.good())
      throw Exception("Error writing to file: "
          + std::string(::strerror(errno)));
    out.write(xmldata_str.c_str(), xmldata_str.size());
    out.close();
  }
  else
  {
    throw Exception("Invalid file extension for plan export. "
        "Valid extensions are \"emp\" and \"xml\".");
  }
}
