/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


/*
 * This source block creates the baseband RDS signal.
 * 
 * It reads its configuration from an XML file; composes the infowords;
 * calculates checkwords; merges the two into blocks and then groups;
 * and streams out the resulting buffer.
 */


/*
 * config.h is generated by configure.  It contains the results
 * of probing for features, options etc.  It should be the first
 * file included in your .cc file.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

//#define DEBUG

#ifdef DEBUG
#define DBG(x) x
#else
#define DBG(x)
#endif

#include <rds/data_encoder.h>
#include <rds/constants.h>
#include <gnuradio/io_signature.h>
#include <math.h>
#include <ctype.h>
#include <time.h>

using namespace gr::rds;

data_encoder::sptr
data_encoder::make (const char *xmlfile) {
	return gnuradio::get_initial_sptr(new data_encoder(xmlfile));
}

data_encoder::data_encoder (const char *xmlfile)
  : gr::sync_block ("gr_rds_data_encoder",
			gr::io_signature::make (0, 0, 0),
			gr::io_signature::make (1, 1, sizeof(unsigned char)))
{
	message_port_register_in(pmt::mp("rds in"));
	set_msg_handler(pmt::mp("rds in"), boost::bind(&data_encoder::rds_in, this, _1));

	xml = (char*)malloc(strlen(xmlfile) + 1);
	memcpy(xml, xmlfile, strlen(xmlfile) + 1);

	init();
}

void data_encoder::init(char *txt, int len) {
	gr::thread::scoped_lock lock(d_mutex);
	// initializes the library, checks for potential ABI mismatches
	LIBXML_TEST_VERSION
	int i=0, j=0;
	reset_rds_data();
	read_xml(xml);

	if(txt) {
		memset(radiotext,' ',sizeof(radiotext));
		memcpy(radiotext, txt, len);
		radiotext[len] = '\0';
	}

	groups[4]=1;      // group 4a: clocktime
	count_groups();

	// allocate memory for nbuffers buffers of 104 unsigned chars each
	buffer = (unsigned char **)malloc(nbuffers*sizeof(unsigned char *));
	for(i=0;i<nbuffers;i++){
		buffer[i] = (unsigned char *)malloc(104*sizeof(unsigned char));
		for(int j=0; j<104; j++) buffer[i][j]=0;
	}
	printf("%i buffers allocated\n", nbuffers);

	// prepare each of the groups
	for(i=0; i<32; i++){
		if(groups[i]==1){
			create_group(i%16, (i<16)?false:true);
			if(i%16==0)  // if group is type 0, call 3 more times
				for(j=0; j<3; j++) create_group(i%16, (i<16)?false:true);
			if(i%16==2) // if group type is 2, call 15 more times
				for(j=0; j<15; j++) create_group(i%16, (i<16)?false:true);
		}
	}
	d_current_buffer=0;

}

void data_encoder::rds_in(pmt::pmt_t msg) {
	if(!pmt::is_pair(msg)) {
		return;
	}

	int msg_len = pmt::blob_length(pmt::cdr(msg));
	std::string text = std::string((char*)pmt::blob_data(pmt::cdr(msg)), msg_len);
	std::cout << std::endl << "new rds text: " << text << std::endl;
	init((char *)pmt::blob_data(pmt::cdr(msg)), msg_len);

}

data_encoder::~data_encoder () {
	xmlCleanupParser();  // Cleanup function for the XML library
	xmlMemoryDump();     // this is to debug memory for regression tests
	free(buffer);
}

void data_encoder::reset_rds_data(){
	int i=0;
	for(i=0; i<4; i++) {infoword[i]=0; checkword[i]=0;}
	for(i=0; i<32; i++) groups[i]=0;
	ngroups=0;
	nbuffers=0;
	d_g0_counter=0;
	d_g2_counter=0;
	d_current_buffer=0;
	d_buffer_bit_counter=0;

	PI=0;
	TP=false;
	PTY=0;
	TA=false;
	MuSp=false;
	MS=false;
	AH=false;
	compressed=false;
	static_pty=false;
	memset(PS,' ',sizeof(PS));
	memset(radiotext,' ',sizeof(radiotext));

	printf("RDS data reset\n");
}


////////////////////  READING XML FILE  //////////////////

/* assinging the values from the xml file to our variables
 * i could do some more checking for invalid inputs,
 * but leaving as is for now */
void data_encoder::assign_from_xml
(const char *field, const char *value, const int length){
	if(!strcmp(field, "PI")){
		if(length!=4) printf("invalid PI string length: %i\n", length);
		else PI=strtol(value, NULL, 16);
	}
	else if(!strcmp(field, "TP")){
		if(!strcmp(value, "true")) TP=true;
		else if(!strcmp(value, "false")) TP=false;
		else printf("unrecognized TP value: %s\n", value);
	}
	else if(!strcmp(field, "PTY")){
		if((length!=1)&&(length!=2))
			printf("invalid TPY string length: %i\n", length);
		else PTY=atol(value);
	}
	else if(!strcmp(field, "TA")){
		if(!strcmp(value, "true")) TA=true;
		else if(!strcmp(value, "false")) TA=false;
		else printf("unrecognized TA value: %s\n", value);
	}
	else if(!strcmp(field, "MuSp")){
		if(!strcmp(value, "true")) MuSp=true;
		else if(!strcmp(value, "false")) MuSp=false;
		else printf("unrecognized MuSp value: %s\n", value);
	}
	else if(!strcmp(field, "AF1")) AF1=atof(value);
	else if(!strcmp(field, "AF2")) AF2=atof(value);
/* need to copy a char arrays here */
	else if(!strcmp(field, "PS")){
		if(length!=8) printf("invalid PS string length: %i\n", length);
		else for(int i=0; i<8; i++)
			PS[i]=value[i];
	}
	else if(!strcmp(field, "RadioText")){
		if(length>64) printf("invalid RadioText string length: %i\n", length);
		else for(int i=0; i<length; i++) radiotext[i]=value[i];
	}
	else if(!strcmp(field, "DP"))
		DP=atol(value);
	else if(!strcmp(field, "extent"))
		extent=atol(value);
	else if(!strcmp(field, "event"))
		event=atol(value);
	else if(!strcmp(field, "location"))
		location=atol(value);
	else printf("unrecognized field type: %s\n", field);
}

/* recursively print the xml nodes */
void data_encoder::print_element_names(xmlNode * a_node){
	xmlNode *cur_node = NULL;
	char *node_name='\0', *attribute='\0', *value='\0';
	int length=0;
	int tmp=0;

	for (cur_node = a_node; cur_node; cur_node = cur_node->next) {
		if (cur_node->type == XML_ELEMENT_NODE){
			node_name=(char*)cur_node->name;
			if(!strcmp(node_name, "rds")) ;		//move on
			else if(!strcmp(node_name, "group")){
				attribute=(char*)xmlGetProp(cur_node, (const xmlChar *)"type");
				/* check that group type is 0-16, A or B */
				if(isdigit(attribute[0])&&((attribute[1]=='A')||(attribute[1]=='B'))){
					printf("\ngroup type: %s  ###  ", attribute);
					tmp=(attribute[0]-48);//+(attribute[1]=='A'?0:16);
					groups[tmp]=1;
				}
				else{
					printf("\ninvalid group type: %s\n", attribute);
					break;
				}
			}
			else if(!strcmp(node_name, "field")){
				attribute=(char*)xmlGetProp(cur_node, (const xmlChar *)"name");
				value=(char*)xmlNodeGetContent(cur_node);
				length=xmlUTF8Strlen(xmlNodeGetContent(cur_node));
				printf("%s: %s # ", attribute, value);
				assign_from_xml(attribute, value, length);
			}
			else printf("invalid node name: %s\n", node_name);
		}
		print_element_names(cur_node->children);
	}
}

/* open the xml file, confirm that the root element is "rds",
 * then recursively print it and assign values to the variables.
 * for now, this runs once at startup. in the future, i might want
 * to read periodically (say, each 5 sec?) so as to change values
 * in the xml file and see the results in the "air"... */
int data_encoder::read_xml (const char *xmlfile){
	xmlDoc *doc;
	xmlNode *root_element = NULL;

	doc = xmlParseFile(xmlfile);
	if (doc == NULL) {
		fprintf(stderr, "Failed to parse %s\n", xmlfile);
		return 1;
	}
	root_element = xmlDocGetRootElement(doc);
// The root element MUST be "rds"
	if(strcmp((char*)root_element->name, "rds")){
		fprintf(stderr, "invalid XML root element!\n");
		return 1;
	}
	print_element_names(root_element);
	printf("\n");

	xmlFreeDoc(doc);
	return 0;
}



//////////////////////  CREATE DATA GROUPS  ///////////////////////

/* see Annex B, page 64 of the standard */
unsigned int data_encoder::calc_syndrome(unsigned long message, 
			unsigned char mlen){
	unsigned long reg=0;
	unsigned int i;
	const unsigned long poly=0x5B9;
	const unsigned char plen=10;

	for (i=mlen;i>0;i--)  {
		reg=(reg<<1) | ((message>>(i-1)) & 0x01);
		if (reg & (1<<plen)) reg=reg^poly;
	}
	for (i=plen;i>0;i--) {
		reg=reg<<1;
		if (reg & (1<<plen)) reg=reg^poly;
	}
	return (reg & ((1<<plen)-1));
}

/* see page 41 in the standard; this is an implementation of AF method A
 * FIXME need to add code that declares the number of AF to follow... */
unsigned int data_encoder::encode_af(const double af){
	unsigned int af_code=0;
	if((af>=87.6)&&(af<=107.9))
		af_code=nearbyint((af-87.5)*10);
	else if((af>=153)&&(af<=279))
		af_code=nearbyint((af-144)/9);
	else if((af>=531)&&(af<=1602))
		af_code=nearbyint((af-531)/9+16);
	else
		printf("invalid alternate frequency: %f\n", af);
	return(af_code);
}

/* count and print present groups */
void data_encoder::count_groups(void){
	printf("groups present: ");
	for(int i=0; i<32; i++){
		if(groups[i]==1){
			ngroups++;
			printf("%i%c ", i%16, (i<16)?'A':'B');
			if(i%16==0)				// group 0
				nbuffers+=4;
			else if(i%16==2)		// group 2
				nbuffers+=16;
			else
				nbuffers++;
		}
	}
	printf("(%i groups)\n", ngroups);
}

/* create the 4 infowords, according to group type.
 * then calculate checkwords and put everything in the groups */
void data_encoder::create_group(const int group_type, const bool AB){
	int i=0;
	
	infoword[0]=PI;
	infoword[1]=(((group_type&0xf)<<12)|(AB<<11)|(TP<<10)|(PTY<<5));

	if(group_type==0) prepare_group0(AB);
	else if(group_type==2) prepare_group2(AB);
	else if(group_type==4) prepare_group4a();
	else if(group_type==8) prepare_group8a();
	else printf("preparation of group %i not yet supported\n", group_type);
	printf("data: %04X %04X %04X %04X, ",
		infoword[0], infoword[1], infoword[2], infoword[3]);

	for(i=0;i<4;i++){
		checkword[i]=calc_syndrome(infoword[i], 16);
		block[i]=((infoword[i]&0xffff)<<10)|(checkword[i]&0x3ff);
		// add the offset word
		if((i==2)&&AB) block[2]^=offset_word[4];
		else block[i]^=offset_word[i];
	}
	printf("group: %04X %04X %04X %04X\n",
		block[0], block[1], block[2], block[3]);

	prepare_buffer(d_current_buffer);
	d_current_buffer++;
}

void data_encoder::prepare_group0(const bool AB){
	infoword[1]=infoword[1]|(TA<<4)|(MuSp<<3);
	if(d_g0_counter==3)
		infoword[1]=infoword[1]|0x5;	// d0=1 (stereo), d1-3=0
	infoword[1]=infoword[1]|(d_g0_counter&0x3);
	if(!AB)
		infoword[2]=((encode_af(AF1)&0xff)<<8)|(encode_af(AF2)&0xff);
	else
		infoword[2]=PI;
	infoword[3]=(PS[2*d_g0_counter]<<8)|PS[2*d_g0_counter+1];
	d_g0_counter++;
	if(d_g0_counter>3) d_g0_counter=0;
}

void data_encoder::prepare_group2(const bool AB){
	infoword[1]=infoword[1]|((AB<<4)|(d_g2_counter&0xf));
	if(!AB){
		infoword[2]=(radiotext[d_g2_counter*4]<<8|radiotext[d_g2_counter*4+1]);
		infoword[3]=(radiotext[d_g2_counter*4+2]<<8|radiotext[d_g2_counter*4+3]);
	}
	else{
		infoword[2]=PI;
		infoword[3]=(radiotext[d_g2_counter*2]<<8|radiotext[d_g2_counter*2+1]);
	}
	d_g2_counter++;
	//if(d_g2_counter>15) d_g2_counter=0;
	d_g2_counter%=16;
}

/* see page 28 and Annex G, page 81 in the standard */
/* FIXME this is supposed to be transmitted only once per minute, when 
 * the minute changes */
void data_encoder::prepare_group4a(void){
	time_t rightnow;
	tm *utc;
	
	time(&rightnow);
	printf("%s", asctime(localtime(&rightnow)));

/* we're supposed to send UTC time; the receiver should then add the
 * local timezone offset */
	utc=gmtime(&rightnow);
	int m=utc->tm_min;
	int h=utc->tm_hour;
	int D=utc->tm_mday;
	int M=utc->tm_mon+1;	// January: M=0
	int Y=utc->tm_year;
	double toffset=localtime(&rightnow)->tm_hour-h;
	
	int L=((M==1)||(M==2))?1:0;
	int mjd=14956+D+int((Y-L)*365.25)+int((M+1+L*12)*30.6001);
	
	infoword[1]=infoword[1]|((mjd>>15)&0x3);
	infoword[2]=(((mjd>>7)&0xff)<<8)|((mjd&0x7f)<<1)|((h>>4)&0x1);
	infoword[3]=((h&0xf)<<12)|(((m>>2)&0xf)<<8)|((m&0x3)<<6)|
		((toffset>0?0:1)<<5)|((abs(toffset*2))&0x1f);
}

// for now single-group only
void data_encoder::prepare_group8a(void){
	infoword[1]=infoword[1]|(1<<3)|(DP&0x7);
	infoword[2]=(1<<15)|((extent&0x7)<<11)|(event&0x7ff);
	infoword[3]=location;
}

void data_encoder::prepare_buffer(int which){
	int q=0, i=0, j=0, a=0, b=0;
	unsigned char temp[13];	// 13*8=104
	for(i=0; i<13; i++) temp[i]=0;
	
	for (q=0;q<104;q++){
		a=floor(q/26); b=25-q%26;
		buffer[which][q]=(unsigned char)(block[a]>>b)&0x1;
		i=floor(q/8); j=7-q%8;
		temp[i]=temp[i]|(buffer[which][q]<<j);
	}
	printf("buffer[%i]: ", which);
	for(i=0;i<13;i++) printf("%02X", temp[i]);
	printf("\n");
}

//////////////////////// WORK ////////////////////////////////////
int data_encoder::work (int noutput_items,
					gr_vector_const_void_star &input_items,
					gr_vector_void_star &output_items)
{
	gr::thread::scoped_lock lock(d_mutex);
	unsigned char *out = (unsigned char *) output_items[0];
	
	for(int i=0; i<noutput_items; i++){
		out[i]=buffer[d_current_buffer][d_buffer_bit_counter];
		if(++d_buffer_bit_counter>103){
			d_buffer_bit_counter=0;
			d_current_buffer++;
			d_current_buffer=d_current_buffer%nbuffers;
		}
	}
	
	return noutput_items;
}
