/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street,
 * Boston, MA 02110-1301, USA.
 */

/*
 * gr-baz by Balint Seeber (http://spench.net/contact)
 * Information, documentation & samples: http://wiki.spench.net/wiki/gr-baz
 */

/*
 * config.h is generated by configure.  It contains the results
 * of probing for features, options etc.  It should be the first
 * file included in your .cc file.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <baz_print_char.h>
#include <gnuradio/io_signature.h>
#include <stdio.h>

baz_print_char_sptr BAZ_API baz_make_print_char (float threshold /*= 0.0*/, int limit /*= -1*/, const char* file /*= NULL*/, int padding /*= 0*/, bool fixed_limit /*= false*/, bool append /*= false*/)
{
  return baz_print_char_sptr (new baz_print_char (threshold, limit, file, padding, fixed_limit, append));
}

static const int MIN_IN = 1;	// mininum number of input streams
static const int MAX_IN = 2;	// maximum number of input streams
static const int MIN_OUT = 0;	// minimum number of output streams
static const int MAX_OUT = 0;	// maximum number of output streams

typedef unsigned char BYTE;
typedef BYTE *LPBYTE;

baz_print_char::baz_print_char (float threshold, int limit, const char* file, int padding, bool fixed_limit, bool append)
  : gr::sync_block ("print_char",
	      gr::io_signature::make2 (MIN_IN, MAX_IN, sizeof (BYTE), sizeof(float)),
	      gr::io_signature::make (MIN_OUT, MAX_OUT, 0))
  , d_threshold(threshold)
  , d_limit(limit)
  , d_length(0)
  , d_file(NULL)
  , d_padding(1)
  , d_fixed_limit(fixed_limit)
{
  if (file)
  {
	d_file = fopen(file, (append ? "w+" : "w"));
	if (d_file == NULL)
	  fprintf(stderr, "Failed to open symbol output file: %s (%i - %s)\n", file, errno, strerror(errno));
  }

  d_padding = std::min(9, std::max(1, padding));

  snprintf(d_format_string, sizeof(d_format_string), "%%0%dX", d_padding);
}

baz_print_char::~baz_print_char ()
{
  if (d_file)
  {
	fclose(d_file);
	d_file = NULL;
  }
}

int baz_print_char::work (int noutput_items, gr_vector_const_void_star &input_items, gr_vector_void_star &output_items)
{
  const LPBYTE data = (const LPBYTE) input_items[0];
  const float *level = NULL;
  if (input_items.size() >= 2)
	level = (const float *) input_items[1];
  
  for (int i = 0; i < noutput_items; i++)
  {
    if ((level != NULL) && (level[i] < d_threshold))
	{
	  if (d_length)
	  {
		if (d_file == NULL)
		{
		  printf(" [%i symbols]\n", d_length);
		  fflush(stdout);
		}
		else
		  fprintf(d_file, "\n");
		
		d_length = 0;
	  }
	  
	  continue;
	}
	
	if ((d_limit == -1) || (d_length < d_limit))
	{
	  if (d_file)
	  {
		fprintf(d_file, d_format_string, data[i]);
	  }
	  else
	  {
		printf(d_format_string, data[i]);
		fflush(stdout);
	  }
	}
	else if ((d_limit > -1) && (d_length == d_limit))
	{
	  if (d_file == NULL)
	  {
		printf("...");
		fflush(stdout);
	  }
	}
	
	++d_length;

	if ((d_fixed_limit) && (d_length == d_limit))
	{
		if (d_file == NULL)
		{
		  printf(" [%i symbol limit]\n", d_length);
		  fflush(stdout);
		}
		else
		  fprintf(d_file, "\n");

		d_length = 0;
	}
  }

  return noutput_items;	// Tell runtime system how many output items we produced.
}
