/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 *
 * This file is part of GNU Radio
 *
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 *
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street,
 * Boston, MA 02110-1301, USA.
 */

/*
 * gr-baz by Balint Seeber (http://spench.net/contact)
 * Information, documentation & samples: http://wiki.spench.net/wiki/gr-baz
 */

/*
 * config.h is generated by configure.  It contains the results
 * of probing for features, options etc.  It should be the first
 * file included in your .cc file.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <baz_native_mux.h>
#include <gnuradio/io_signature.h>

#include <stdio.h>
//#include <typeinfo>

/*
 * Create a new instance of baz_native_mux and return
 * a boost shared_ptr.  This is effectively the public constructor.
 */
baz_native_mux_sptr
baz_make_native_mux (int item_size, int input_count, int trigger_count)
{
  return baz_native_mux_sptr (new baz_native_mux (item_size, input_count, trigger_count));
}

/*
 * The private constructor
 */
baz_native_mux::baz_native_mux (int item_size, int input_count, int trigger_count)
  : /*gr::sync_block*/gr::block ("native_mux",
		   gr::io_signature::make (input_count, input_count, item_size),
		   gr::io_signature::make (1, 1, item_size))
  , d_item_size(item_size), d_input_count(input_count), d_trigger_count(trigger_count)
  , d_selected_input(0), d_trigger_countdown(0), d_value_index(0)
  , d_last_noutput_items(0)
{
  fprintf(stderr, "[%s] Trigger count %d\n", name().c_str(), trigger_count);

  d_values.push_back(0.5);
  d_values.push_back(0.9);
  d_values.push_back(1.3);
}

void baz_native_mux::callback(float f, unsigned long samples_processed)
{/*
  int i = (int)f;
  if ((i < 0) || (i >= d_input_count))
    return;
  d_selected_input = i;  // TODO: No concurrency protection
  if (d_trigger_count > -1)
  {
    if (d_trigger_countdown > 0) {
      if (d_trigger_countdown != d_trigger_count)
        fprintf(stderr, "[%s] Setting trigger countdown to %d when still %d\n", name().c_str(), d_trigger_countdown, d_trigger_count);
      else {
        //fprintf(stderr, "."); // Callback while not in work
      }
    }
    else {
      d_value_index = (d_value_index + 1) % d_values.size();  // Only change when we've run work since the last callback
    }
    d_trigger_countdown = d_trigger_count;
  }
  //fprintf(stderr, "[%s] Selecting input %d (countdown: %d)\n", name().c_str(), d_selected_input, d_trigger_countdown);
*/
  const unsigned long latency = 16384*2*2*2 + 2048; // samples
  d_switch_time.push_back(samples_processed + latency);
  //fprintf(stderr, "[%s] Scheduling %d (sample: %d, local: %d)\n", name().c_str(), (samples_processed + latency), samples_processed, d_samples_processed);
}

/*
 * Our virtual destructor.
 */
baz_native_mux::~baz_native_mux ()
{
}

int
baz_native_mux::general_work (int noutput_items,
                              gr_vector_int &ninput_items,
			gr_vector_const_void_star &input_items,
			gr_vector_void_star &output_items)
{
  if (d_last_noutput_items != noutput_items) {
    fprintf(stderr, "[%s] Work output items: %d\n", name().c_str(), noutput_items);
    d_last_noutput_items = noutput_items;
  }

  char *out = (char*)output_items[0];
  std::vector<int> used(ninput_items.size());

  for (int i = 0; i < ninput_items.size(); ++i)
  {
      if (ninput_items[i] < noutput_items)
        fprintf(stderr, "[%s] Not enough input items\n", name().c_str());
  }

  for (int i = 0; i < noutput_items; i++) {

    /////////////////////////////////////////////

    for (int j = 0; j < d_switch_time.size(); ++j)
    {
      unsigned long next_time = d_switch_time[j];
      if (next_time < d_samples_processed)
      {
        fprintf(stderr, "[%s] Late %d (processed: %d, next time: %d)\n", name().c_str(), ((int)d_samples_processed - (int)next_time), d_samples_processed, next_time);
        d_switch_time.erase(d_switch_time.begin());
        --j;
        continue;
      }
      else if (next_time > d_samples_processed)
        break;

      d_selected_input = 1;
      d_trigger_countdown = d_trigger_count;
      d_value_index = (d_value_index + 1) % d_values.size();
      d_switch_time.erase(d_switch_time.begin());
      break;
    }

    /////////////////////////////////////////////

    if (d_trigger_count > -1) {
      if (d_trigger_countdown == 0)
        d_selected_input = 0;
      else
        --d_trigger_countdown;
    }

    const char *in = (char*)input_items[d_selected_input];
    memcpy(out + (i * d_item_size), in + (i * d_item_size), d_item_size);

    /////////////////////////////////////////////
    if (d_selected_input == 1)
    {
        float* f = (float*)(out + (i * d_item_size));
        *f = d_values[d_value_index];
    }
    /////////////////////////////////////////////

    used[d_selected_input] += 1;

    ++d_samples_processed;
  }

  consume(0, noutput_items);

  for (int i = 1; i < used.size(); ++i)
    consume(i, used[i]);

  return noutput_items;
}
